<?php
/**
 * Plugin Name: Static Cache Wrangler - Headless Assistant
 * Plugin URI: https://moderncli.dev/code/stcw-headless-assistant/
 * Description: Convert Static Cache Wrangler HTML output to headless CMS import formats
 * Version: 2.1.0
 * Author: Derick Schaefer
 * Author URI: https://moderncli.dev/author/
 * Text Domain: stcw-headless-assistant
 * Requires at least: 6.0
 * Tested up to: 6.9
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Requires Plugins: static-cache-wrangler
 */

if (!defined('ABSPATH')) {
    exit;
}

define('STCW_HEADLESS_VERSION', '2.1.0');
define('STCW_HEADLESS_DIR', plugin_dir_path(__FILE__));
define('STCW_HEADLESS_URL', plugin_dir_url(__FILE__));

/**
 * PSR-4 Autoloader (WordPress Naming Convention)
 */
spl_autoload_register(function ($class) {
    $prefix = 'STCW\\Headless\\';
    
    if (strpos($class, $prefix) !== 0) {
        return;
    }
    
    $relative_class = substr($class, strlen($prefix));
    $parts = explode('\\', $relative_class);
    $class_name = array_pop($parts);
    
    $file_prefix = 'class';
    if (strpos($class_name, 'Interface') !== false) {
        $file_prefix = 'interface';
    }
    
    $filename_base = strtolower(preg_replace('/([a-z])([A-Z])/', '$1-$2', $class_name));
    $filename = $file_prefix . '-stcw-headless-' . $filename_base . '.php';
    
    $directory = STCW_HEADLESS_DIR . 'includes/';
    if (!empty($parts)) {
        $directory .= implode('/', $parts) . '/';
    }
    
    $file = $directory . $filename;
    
    if (file_exists($file)) {
        require_once $file;
    }
});

require_once STCW_HEADLESS_DIR . 'includes/functions.php';

/**
 * Check if Static Cache Wrangler is active
 *
 * @return bool True if SCW is active
 */
function stcw_headless_check_scw_active() {
    return class_exists('STCW_Core');
}

/**
 * Display admin notice if SCW is not active
 */
function stcw_headless_admin_notice_missing_scw() {
    if (!stcw_headless_check_scw_active()) {
        echo '<div class="notice notice-error"><p><strong>' . 
            esc_html__('Static Cache Wrangler – Headless Assistant', 'stcw-headless-assistant') . 
            '</strong> ' . 
            esc_html__('requires Static Cache Wrangler to be installed and activated.', 'stcw-headless-assistant') . 
            '</p></div>';
    }
}
add_action('admin_notices', 'stcw_headless_admin_notice_missing_scw');

/**
 * Add WP-CLI requirement notice to plugin row meta
 *
 * @param array  $plugin_meta An array of the plugin's metadata.
 * @param string $plugin_file Path to the plugin file relative to the plugins directory.
 * @return array Modified plugin meta array
 */
function stcw_headless_plugin_row_meta($plugin_meta, $plugin_file) {
    if (plugin_basename(__FILE__) === $plugin_file) {
        $plugin_meta[] = '<strong style="color: #d63638;">' . 
            esc_html__('⚡ Requires WP-CLI', 'stcw-headless-assistant') . 
            '</strong>';
        
        $plugin_meta[] = '<a href="https://wp-cli.org/#installing" target="_blank" rel="noopener noreferrer">' . 
            esc_html__('Install WP-CLI', 'stcw-headless-assistant') . 
            '</a>';
    }
    
    return $plugin_meta;
}
add_filter('plugin_row_meta', 'stcw_headless_plugin_row_meta', 10, 2);

/**
 * Initialize the plugin
 */
function stcw_headless_init() {
    if (!stcw_headless_check_scw_active()) {
        return;
    }
    
    $core = \STCW\Headless\Core::instance(); 
    $core->init();
    
    // Register Kadence block extractors (v1.3.1: deprecated in favor of Patterns/Kadence.php)
    // Priority 25 to run AFTER load_patterns() at priority 20
    add_action('init', 'stcw_headless_register_kadence_extractors', 25);
}
add_action('plugins_loaded', 'stcw_headless_init');

/**
 * Register Kadence block extractors
 *
 * DEPRECATED in v1.3.1+: Kadence patterns are now registered in
 * Patterns/Kadence.php with semantic type support (sanity_target).
 * The old KadenceExtractors class is kept for backward compatibility
 * but is no longer called by default.
 *
 * This function now checks if the new pattern system exists and skips
 * registration if it does, preventing duplicate pattern registration
 * and ensuring semantic types are properly applied.
 *
 * @since 2.0.0
 * @deprecated 2.0.9 Use Patterns/Kadence.php instead
 */
function stcw_headless_register_kadence_extractors() {
    // Skip if new pattern system is available (v1.3.1+)
    $new_kadence_patterns = STCW_HEADLESS_DIR . 'includes/Engine/Detector/Patterns/Kadence.php';
    
    if (file_exists($new_kadence_patterns)) {
        if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log('STCW Headless: Using Patterns/Kadence.php (v1.3.1+) instead of KadenceExtractors');
        }
        return; // New system takes precedence
    }
    
    // Fallback to old system if new patterns don't exist (backward compatibility)
    $kadence_file = STCW_HEADLESS_DIR . 'includes/Engine/Extractor/class-stcw-headless-kadence-extractors.php';
    
    if (file_exists($kadence_file)) {
        require_once $kadence_file;
        
        if (class_exists('\STCW\Headless\Engine\Extractor\KadenceExtractors')) {
            \STCW\Headless\Engine\Extractor\KadenceExtractors::register_all();
            
            if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('STCW Headless: Using legacy KadenceExtractors (pre-v1.3.1)');
            }
        }
    }
}

/**
 * Plugin activation hook
 */
register_activation_hook(__FILE__, function() {
    if (!stcw_headless_check_scw_active()) {
        wp_die(
            esc_html__('This plugin requires Static Cache Wrangler to be installed and activated.', 'stcw-headless-assistant'),
            esc_html__('Plugin Activation Error', 'stcw-headless-assistant'),
            ['back_link' => true]
        );
    }
});

/**
 * Load WP-CLI commands
 */
if (defined('WP_CLI') && WP_CLI) {
    require_once STCW_HEADLESS_DIR . 'includes/class-stcw-headless-cli.php';
    WP_CLI::add_command('scw-headless', 'STCW\Headless\CLI');
}
