<?php
/**
 * Core functionality for STCW Headless Assistant
 *
 * Coordinates scanning cached files, conversion to CMS format,
 * export package generation, and extension registration.
 *
 * @package STCWHeadlessAssistant
 * @since 2.0.0
 */

namespace STCW\Headless;

if (!defined('ABSPATH')) exit;

class Core {
    
    /**
     * Singleton instance
     * @var Core
     */
    private static $instance = null;
    
    /**
     * Scanner instance
     * @var Scanner
     */
    private $scanner;
    
    /**
     * Parser instance
     * @var Engine\Parser
     */
    private $parser;
    
    /**
     * Registered detector extensions
     * @var array
     */
    private $detectors = [];
    
    /**
     * Registered CMS target extensions
     * @var array
     */
    private $targets = [];
    
    /**
     * Get singleton instance
     *
     * @return Core
     */
    public static function instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Initialize core functionality
     */
    public function init() {
        // Register built-in detectors FIRST
        $this->register_builtin_detectors();
        
        // Load pattern definitions
        $this->load_patterns();
        
        // Update detector pattern counts after patterns are loaded
        $this->update_detector_pattern_counts();
        
        // Register CMS targets
        $this->register_targets();
        
        // Then initialize components
        $this->scanner = new Scanner();
        $this->parser = new Engine\Parser();
        
        // Load admin interface
        if (is_admin()) {
            require_once STCW_HEADLESS_DIR . 'admin/class-stcw-headless-admin.php';
            $admin = new Admin();
            $admin->init();
        }
        
        // Fire action for extensions to register
        do_action('stcw_headless_core_loaded');
    }
    
    /**
     * Register built-in detectors
     *
     * Registers the core Gutenberg and Kadence detectors that ship with the plugin.
     * Pattern counts are calculated dynamically from PatternRegistry after patterns are loaded.
     *
     * @since 2.0.0
     */
    private function register_builtin_detectors() {
        // Gutenberg Core Blocks
        $this->register_detector('gutenberg', [
            'name' => 'Gutenberg Core Blocks',
            'version' => STCW_HEADLESS_VERSION,
            'type' => 'built-in',
            'description' => 'WordPress core block patterns',
            'patterns' => 0,  // Will be updated dynamically after patterns load
            'status' => 'active',
        ]);
        
        // Kadence Blocks
        $this->register_detector('kadence', [
            'name' => 'Kadence Blocks',
            'version' => STCW_HEADLESS_VERSION,
            'type' => 'built-in',
            'description' => 'Kadence Blocks pattern detection',
            'patterns' => 0,  // Will be updated dynamically after patterns load
            'status' => 'active',
        ]);
    }
    
    /**
     * Update detector pattern counts dynamically
     * 
     * Called after patterns are loaded to count actual patterns from PatternRegistry
     *
     * @since 2.0.9
     */
    private function update_detector_pattern_counts() {
        if (!class_exists('\\STCW\\Headless\\Engine\\Detector\\PatternRegistry')) {
            return;
        }
        
        $patterns = Engine\Detector\PatternRegistry::get_patterns();
        
        // Count Gutenberg patterns
        $gutenberg_count = 0;
        $kadence_count = 0;
        
        foreach ($patterns as $pattern_name => $pattern_config) {
            if (strpos($pattern_name, 'kadence_') === 0) {
                $kadence_count++;
            } elseif (
                strpos($pattern_name, 'core/') === 0 ||
                in_array($pattern_name, [
                    'heading', 'paragraph', 'list', 'quote', 'image', 'gallery',
                    'video', 'separator', 'button', 'buttons', 'table', 'code'
                ], true)
            ) {
                $gutenberg_count++;
            }
        }
        
        // Update detector counts
        if (isset($this->detectors['gutenberg'])) {
            $this->detectors['gutenberg']['patterns'] = $gutenberg_count;
        }
        
        if (isset($this->detectors['kadence'])) {
            $this->detectors['kadence']['patterns'] = $kadence_count;
        }
        
        if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log(sprintf(
                'STCW Headless: Updated pattern counts - Gutenberg: %d, Kadence: %d',
                $gutenberg_count,
                $kadence_count
            ));
        }
    }
    
    /**
     * Register a detector extension
     *
     * Allows premium/third-party plugins to register themselves.
     * This makes them visible in CLI and admin dashboard.
     *
     * @param string $slug Unique detector slug
     * @param array $data Detector metadata
     * @return bool Success
     */
    public function register_detector($slug, $data) {
        // Validate slug
        if (empty($slug) || !is_string($slug)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('STCW Headless: Invalid detector slug');
            }
            return false;
        }
        
        // Check for duplicates
        if (isset($this->detectors[$slug])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log(sprintf('STCW Headless: Detector "%s" already registered', $slug));
            }
            return false;
        }
        
        // Set defaults
        $defaults = [
            'name' => ucfirst($slug),
            'version' => '1.0.0',
            'type' => 'extension',
            'description' => '',
            'author' => '',
            'patterns' => 0,
            'status' => 'active',
        ];
        
        $data = array_merge($defaults, $data);
        
        // Store detector
        $this->detectors[$slug] = $data;
        
        // Fire action for other plugins to hook into
        do_action('stcw_headless_detector_registered', $slug, $data);
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log(sprintf('STCW Headless: Registered detector "%s" (%s)', $data['name'], $slug));
        }
        
        return true;
    }
    
    /**
     * Get all registered detectors
     *
     * @return array Detectors array
     */
    public function get_detectors() {
        return apply_filters('stcw_headless_detectors', $this->detectors);
    }
    
    /**
     * Get specific detector
     *
     * @param string $slug Detector slug
     * @return array|null Detector data or null
     */
    public function get_detector($slug) {
        return $this->detectors[$slug] ?? null;
    }
    
    /**
     * Get detector count by type
     *
     * @param string|null $type Detector type (built-in, extension, premium, test) or null for all
     * @return int Detector count
     */
    public function get_detector_count($type = null) {
        if ($type === null) {
            return count($this->detectors);
        }
        
        return count(array_filter($this->detectors, function($detector) use ($type) {
            return ($detector['type'] ?? 'extension') === $type;
        }));
    }
    
    /**
     * Register a CMS target extension
     *
     * @param string $slug Unique target slug
     * @param array $data Target metadata
     * @return bool Success
     */
    public function register_target($slug, $data) {
        // Similar implementation to register_detector
        // For future CMS target extensions
        
        $this->targets[$slug] = $data;
        do_action('stcw_headless_target_registered', $slug, $data);
        
        return true;
    }
    
    /**
     * Get all registered targets
     *
     * @return array Targets array
     */
    public function get_targets() {
        return apply_filters('stcw_headless_targets', $this->targets);
    }
    
    /**
     * Register CMS targets
     *
     * Registers built-in targets and allows custom targets via action.
     *
     * @since 2.0.0
     */
    private function register_targets() {
        // Manually load Target classes in correct order (autoloader loading order issue)
        $target_files = [
            STCW_HEADLESS_DIR . 'includes/Engine/Target/interface-stcw-headless-target.php',
            STCW_HEADLESS_DIR . 'includes/Engine/Target/class-stcw-headless-target-registry.php',
            STCW_HEADLESS_DIR . 'includes/Engine/Target/Sanity/class-stcw-headless-sanity-converter.php',
            STCW_HEADLESS_DIR . 'includes/Engine/Target/Sanity/class-stcw-headless-sanity-schema-generator.php',
            STCW_HEADLESS_DIR . 'includes/Engine/Target/Sanity/class-stcw-headless-sanity-exporter.php',
            STCW_HEADLESS_DIR . 'includes/Engine/Target/Sanity/class-stcw-headless-sanity-target.php',
        ];
        
        foreach ($target_files as $file) {
            if (file_exists($file)) {
                require_once $file;
            }
        }
        
        // Register Sanity target
        $sanity = new Engine\Target\Sanity\SanityTarget();
        Engine\Target\TargetRegistry::register($sanity);
        
        /**
         * Action to register custom CMS targets
         */
        do_action('stcw_headless_register_targets');
    }
    
    /**
     * Load pattern definition files
     *
     * Patterns are loaded in priority order:
     * 1. Core Gutenberg patterns
     * 2. Kadence patterns
     * 3. Custom patterns (via filter)
     *
     * @since 1.1.0
     */
    private function load_patterns() {
        // Core pattern files
        $pattern_files = [
            STCW_HEADLESS_DIR . 'includes/Engine/Detector/Patterns/Gutenberg.php',
            STCW_HEADLESS_DIR . 'includes/Engine/Detector/Patterns/Kadence.php',
        ];
        
        foreach ($pattern_files as $file) {
            if (file_exists($file)) {
                require_once $file;
            }
        }
        
        /**
         * Filter to add custom pattern files
         */
        $custom_files = apply_filters('stcw_headless_pattern_files', []);
        
        foreach ($custom_files as $file) {
            if (file_exists($file)) {
                require_once $file;
            }
        }
        
        /**
         * Action after all patterns loaded
         */
        do_action('stcw_headless_patterns_loaded');
    }
    
    /**
     * Get scanner instance
     * 
     * @return Scanner
     */
    public function get_scanner() {
        return $this->scanner;
    }
    
    /**
     * Get parser instance
     * 
     * @return Engine\Parser
     */
    public function get_parser() {
        return $this->parser;
    }
    
    /**
     * Get export directory path
     * 
     * @return string Absolute path to exports directory
     */
    public static function get_export_dir() {
	// Use cache directory (not uploads) for consistency with Static Cache Wrangler
	$cache_dir = WP_CONTENT_DIR . '/cache/';
	return trailingslashit($cache_dir) . 'stcw-headless-exports/';
     } 
    /**
     * Get Static Cache Wrangler static directory
     * 
     * @return string Path to SCW static files
     */
    public static function get_scw_static_dir() {
        if (defined('STCW_STATIC_DIR')) {
            return STCW_STATIC_DIR;
        }
        return '';
    }
    
    /**
     * Get file permissions constant
     * 
     * @return int File permissions mode
     */
    private static function get_file_chmod() {
        if (!defined('FS_CHMOD_FILE')) {
            return 0644;
        }
        return FS_CHMOD_FILE;
    }
    
    /**
     * Create export directory if it doesn't exist
     * 
     * @return bool True on success
     */
    public static function ensure_export_dir() {
        $dir = self::get_export_dir();
        
        if (!is_dir($dir)) {
            wp_mkdir_p($dir);
        }
        
        // Add .htaccess protection
        $htaccess = $dir . '.htaccess';
        if (!file_exists($htaccess)) {
            $htaccess_content = "# Protect export files\nDeny from all\n";
            
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
            file_put_contents($htaccess, $htaccess_content);
            
            if (file_exists($htaccess)) {
                // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_chmod
                chmod($htaccess, self::get_file_chmod());
            }
        }
        
        return is_dir($dir);
    }
}
