<?php
/**
 * WP-CLI Commands for Headless Assistant
 *
 * CLI-first interface for scanning, analyzing, and converting
 * WordPress content for headless CMS workflows.
 *
 * @package STCWHeadlessAssistant
 * @since 2.0.0
 */

namespace STCW\Headless;

use STCW\Headless\Engine\Normalizer;
use STCW\Headless\Engine\Parser;
use STCW\Headless\Engine\Detector\PatternRegistry;
use STCW\Headless\Engine\Detector\PatternDetector;

if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('WP_CLI')) {
    return;
}

/**
 * Headless Assistant CLI Commands
 */
class CLI {

    /**
     * Show comprehensive plugin information and statistics
     *
     * Displays detailed overview of cached files, registered patterns,
     * and system status.
     *
     * ## OPTIONS
     *
     * [--format=<format>]
     * : Output format (table, json)
     * ---
     * default: table
     * options:
     *   - table
     *   - json
     * ---
     *
     * ## EXAMPLES
     *
     *     # Show full overview
     *     $ wp scw-headless info
     *
     *     # Get data as JSON
     *     $ wp scw-headless info --format=json
     *
     * @when after_wp_load
     */
    public function info($args, $assoc_args) {
        $format = $assoc_args['format'] ?? 'table';

        // Gather all data
        $scanner = new Scanner();
        $scan_result = $scanner->scan_cached_files();
        $stats = $scanner->get_scan_stats();

        // Get pattern info
        $total_patterns = PatternRegistry::count_patterns();
        $enabled_patterns = PatternRegistry::count_patterns(true);

        // Get detector info
        $detectors = \STCW_Headless()->get_detectors();
        $total_detectors = count($detectors);

        // Get CMS targets
        $targets = \STCW\Headless\Engine\Target\TargetRegistry::get_all();
        $total_targets = count($targets);

        // Get directory paths
        $static_dir = Core::get_scw_static_dir();
        $export_dir = Core::get_export_dir();

        // Build info array
        $info = [
            'plugin_version' => STCW_HEADLESS_VERSION,
            'static_cache_dir' => $static_dir ?: 'Not configured',
            'export_dir' => $export_dir,
            'cached_files' => $stats['total_files'],
            'cache_size' => $stats['formatted_size'],
            'oldest_file' => $stats['oldest_file'],
            'newest_file' => $stats['newest_file'],
            'total_patterns' => $total_patterns,
            'enabled_patterns' => $enabled_patterns,
            'disabled_patterns' => $total_patterns - $enabled_patterns,
            'total_detectors' => $total_detectors,
            'total_targets' => $total_targets,
        ];

        if ($format === 'json') {
            \WP_CLI::line(wp_json_encode($info, JSON_PRETTY_PRINT));
            return;
        }

        // Display formatted output
        \WP_CLI::line('');
        \WP_CLI::line(\WP_CLI::colorize('%B=== Headless Assistant Info ===%n'));
        \WP_CLI::line('');

        // Plugin Section
        \WP_CLI::line(\WP_CLI::colorize('%Y» Plugin%n'));
        \WP_CLI::line(sprintf('  Version:            %s', \WP_CLI::colorize('%G' . $info['plugin_version'] . '%n')));
        \WP_CLI::line('');

        // Directories Section
        \WP_CLI::line(\WP_CLI::colorize('%Y» Directories%n'));
        \WP_CLI::line(sprintf('  Static Cache:       %s', $info['static_cache_dir']));
        \WP_CLI::line(sprintf('  Export:             %s', $info['export_dir']));
        \WP_CLI::line('');

        // Cached Files Section
        \WP_CLI::line(\WP_CLI::colorize('%Y» Cached Files%n'));
        \WP_CLI::line(sprintf('  Total Files:        %s', \WP_CLI::colorize('%G' . $info['cached_files'] . '%n')));
        \WP_CLI::line(sprintf('  Cache Size:         %s', $info['cache_size']));
        \WP_CLI::line(sprintf('  Oldest File:        %s', $info['oldest_file'] ?: 'N/A'));
        \WP_CLI::line(sprintf('  Newest File:        %s', $info['newest_file'] ?: 'N/A'));
        \WP_CLI::line('');

        // Pattern Registry Section
        \WP_CLI::line(\WP_CLI::colorize('%Y» Pattern Registry%n'));
        \WP_CLI::line(sprintf('  Total Patterns:     %s', \WP_CLI::colorize('%G' . $info['total_patterns'] . '%n')));
        \WP_CLI::line(sprintf('  Enabled:            %s', \WP_CLI::colorize('%G' . $info['enabled_patterns'] . '%n')));

        if ($info['disabled_patterns'] > 0) {
            \WP_CLI::line(sprintf('  Disabled:           %s', \WP_CLI::colorize('%R' . $info['disabled_patterns'] . '%n')));
        }

        \WP_CLI::line('');

        // Detector Modules Section
        \WP_CLI::line(\WP_CLI::colorize('%Y» Detector Modules%n'));
        \WP_CLI::line(sprintf('  Total Detectors:    %s', \WP_CLI::colorize('%G' . $info['total_detectors'] . '%n')));
        \WP_CLI::line('');

        // CMS Targets Section
        \WP_CLI::line(\WP_CLI::colorize('%Y» CMS Targets%n'));
        \WP_CLI::line(sprintf('  Registered Targets: %s', \WP_CLI::colorize('%G' . $info['total_targets'] . '%n')));

        // List target names
        if ($total_targets > 0) {
            foreach ($targets as $slug => $target) {
                \WP_CLI::line(sprintf('    • %s', $target->get_name()));
            }
        }

        \WP_CLI::line('');

        \WP_CLI::success('System ready for pattern detection');
    }

    /**
     * Scan cached HTML files from Static Cache Wrangler
     *
     * ## OPTIONS
     *
     * [--format=<format>]
     * : Output format (table, json, csv, yaml)
     * ---
     * default: table
     * options:
     *   - table
     *   - json
     *   - csv
     *   - yaml
     * ---
     *
     * ## EXAMPLES
     *
     *     # Show all cached files in table format
     *     $ wp scw-headless scan
     *
     *     # Get file list as JSON
     *     $ wp scw-headless scan --format=json
     *
     * @when after_wp_load
     */
    public function scan($args, $assoc_args) {
        $format = $assoc_args['format'] ?? 'table';

        \WP_CLI::line('');
        \WP_CLI::line(\WP_CLI::colorize('%B=== Scanning Static Cache Files ===%n'));
        \WP_CLI::line('');

        $scanner = new Scanner();
        $result = $scanner->scan_cached_files();

        if (!$result['success']) {
            \WP_CLI::error($result['message']);
        }

        $files = $result['files'];

        if ($format === 'json') {
            \WP_CLI::line(wp_json_encode($result, JSON_PRETTY_PRINT));
            return;
        }

        if (empty($files)) {
            \WP_CLI::warning('No cached files found. Generate some static pages first.');
            return;
        }

        // Display results
        if ($format === 'table') {
            $table_data = [];
            foreach ($files as $file) {
                $table_data[] = [
                    'path' => $file['url_path'],
                    'size' => size_format($file['size']),
                    'modified' => $file['modified_formatted'],
                ];
            }

            \WP_CLI\Utils\format_items('table', $table_data, ['path', 'size', 'modified']);

            \WP_CLI::line('');
            \WP_CLI::success(sprintf(
                'Found %s cached files ready for analysis',
                \WP_CLI::colorize('%G' . count($files) . '%n')
            ));
        } else {
            \WP_CLI\Utils\format_items($format, $files, array_keys($files[0]));
        }
    }

    /**
     * Analyze patterns in cached HTML file
     *
     * Scans HTML file and identifies registered content patterns.
     * Useful for debugging pattern detection before conversion.
     *
     * ## OPTIONS
     *
     * <file>
     * : Path to HTML file. Accepts multiple formats:
     *   - URL path from scan output (e.g., /contact/)
     *   - Short form without slashes (e.g., contact)
     *   - Explicit file path (e.g., contact/index.html)
     *   - Absolute file path
     *
     * [--pattern=<name>]
     * : Detect only specific pattern
     *
     * [--format=<format>]
     * : Output format (table, json, yaml)
     * ---
     * default: table
     * options:
     *   - table
     *   - json
     *   - yaml
     * ---
     *
     * [--verbose]
     * : Show detailed detection info including XPath queries
     *
     * [--min-confidence=<float>]
     * : Only show patterns with confidence >= this value (0.0-1.0)
     * ---
     * default: 0.0
     * ---
     *
     * ## EXAMPLES
     *
     *     # Analyze using URL path from scan output
     *     $ wp scw-headless analyze /contact/
     *
     *     # Analyze using short form
     *     $ wp scw-headless analyze contact
     *
     *     # Analyze with explicit file path
     *     $ wp scw-headless analyze contact/index.html
     *
     *     # Analyze specific pattern only
     *     $ wp scw-headless analyze /contact/ --pattern=accordion
     *
     *     # Show detailed output
     *     $ wp scw-headless analyze /contact/ --verbose
     *
     *     # Export as JSON
     *     $ wp scw-headless analyze /contact/ --format=json
     *
     * @when after_wp_load
     */
    public function analyze($args, $assoc_args) {
        $file_path = $args[0] ?? '';
        $format = $assoc_args['format'] ?? 'table';
        $verbose = isset($assoc_args['verbose']);
        $pattern_filter = $assoc_args['pattern'] ?? null;
        $min_confidence = isset($assoc_args['min-confidence']) ? floatval($assoc_args['min-confidence']) : 0.0;

        if (empty($file_path)) {
            \WP_CLI::error('Please provide a file path');
        }

        // Resolve file path - now supports multiple formats
        $resolved_path = $this->resolve_file_path($file_path);

        if (!$resolved_path || !file_exists($resolved_path)) {
            $static_dir = Core::get_scw_static_dir();
            $error_msg = sprintf('File not found: %s', $file_path);
            
            if (!empty($static_dir)) {
                $error_msg .= sprintf("\n\nStatic cache directory: %s", $static_dir);
                $error_msg .= "\n\nTried paths:";
                $error_msg .= sprintf("\n  - %s", trailingslashit($static_dir) . trim($file_path, '/'));
                $error_msg .= sprintf("\n  - %s", trailingslashit($static_dir) . trailingslashit(trim($file_path, '/')) . 'index.html');
            } else {
                $error_msg .= "\n\nStatic Cache Wrangler directory not configured (STCW_STATIC_DIR not defined)";
            }
            
            \WP_CLI::error($error_msg);
        }

        \WP_CLI::line('');
        \WP_CLI::line(\WP_CLI::colorize('%B=== Pattern Analysis ===%n'));
        \WP_CLI::line('');

        // Debug: Show resolved path
        if ($verbose) {
            \WP_CLI::line(sprintf('Resolved path: %s', $resolved_path));
            \WP_CLI::line(sprintf('File exists: %s', file_exists($resolved_path) ? 'Yes' : 'No'));
            \WP_CLI::line(sprintf('File size: %s', file_exists($resolved_path) ? size_format(filesize($resolved_path)) : 'N/A'));
            \WP_CLI::line('');
        }

        // Parse file
        $parser = new Parser();
        $result = $parser->parse_file($resolved_path);

        if (!$result['success']) {
            \WP_CLI::error($result['message']);
        }

        $detected = $result['patterns'];

        // Filter by pattern name if specified
        if ($pattern_filter) {
            $detected = array_filter($detected, function ($match) use ($pattern_filter) {
                return $match['pattern'] === $pattern_filter;
            });
        }

        // Filter by confidence
        if ($min_confidence > 0) {
            $detected = array_filter($detected, function ($match) use ($min_confidence) {
                return $match['confidence'] >= $min_confidence;
            });
        }

        // Get statistics
        $stats = $result['detection_stats'];

        // JSON output
        if ($format === 'json') {
            $output = [
                'file' => basename($resolved_path),
                'size' => size_format(filesize($resolved_path)),
                'patterns_found' => count($detected),
                'statistics' => $stats,
                'metadata' => $result['metadata'],
                'matches' => array_map(function ($match) {
                    return [
                        'pattern' => $match['pattern'],
                        'confidence' => $match['confidence'],
                        'xpath' => $match['xpath'],
                        'html_preview' => substr($match['html'], 0, 200),
                    ];
                }, $detected),
            ];

            \WP_CLI::line(wp_json_encode($output, JSON_PRETTY_PRINT));
            return;
        }

        // Table output
        \WP_CLI::line(sprintf('File: %s (%s)', basename($resolved_path), size_format(filesize($resolved_path))));
        \WP_CLI::line(sprintf('Title: %s', $result['metadata']['title'] ?: 'N/A'));
        \WP_CLI::line(sprintf('Patterns Found: %s', \WP_CLI::colorize('%G' . count($detected) . '%n')));
        \WP_CLI::line('');

        if (empty($detected)) {
            \WP_CLI::warning('No patterns detected');
            return;
        }

        // Prepare table data with consolidation
        $table_data = [];
        $pattern_counts = [];

        foreach ($detected as $match) {
            $pattern_name = $match['pattern'];

            if (!isset($pattern_counts[$pattern_name])) {
                $pattern_counts[$pattern_name] = [
                    'count' => 0,
                    'confidence' => $match['confidence'],
                    'xpath' => $match['xpath'],
                ];
            }

            $pattern_counts[$pattern_name]['count']++;
        }

        foreach ($pattern_counts as $name => $data) {
            $confidence_color = $data['confidence'] >= 0.95 ? '%G' : ($data['confidence'] >= 0.85 ? '%Y' : '%R');

            $row = [
                'pattern' => $name,
                'count' => $data['count'],
                'confidence' => \WP_CLI::colorize($confidence_color . number_format($data['confidence'], 2) . '%n'),
            ];

            if ($verbose) {
                $row['xpath'] = $data['xpath'];
            }

            $table_data[] = $row;
        }

        // Sort by count (descending)
        usort($table_data, function ($a, $b) {
            return $b['count'] - $a['count'];
        });

        // Display table
        $fields = $verbose ? ['pattern', 'count', 'confidence', 'xpath'] : ['pattern', 'count', 'confidence'];
        \WP_CLI\Utils\format_items('table', $table_data, $fields);

        \WP_CLI::line('');

        // Show statistics
        if ($verbose) {
            \WP_CLI::line(\WP_CLI::colorize('%Y» Confidence Distribution%n'));
            \WP_CLI::line(sprintf('  High (≥0.95):   %s', \WP_CLI::colorize('%G' . $stats['by_confidence']['high'] . '%n')));
            \WP_CLI::line(sprintf('  Medium (0.85+): %s', \WP_CLI::colorize('%Y' . $stats['by_confidence']['medium'] . '%n')));
            \WP_CLI::line(sprintf('  Low (<0.85):    %s', \WP_CLI::colorize('%R' . $stats['by_confidence']['low'] . '%n')));
            \WP_CLI::line('');
        }

        \WP_CLI::success(sprintf('Analyzed %d patterns', count($detected)));
    }

    /**
     * Normalize HTML file(s) to generic Portable Text format
     *
     * **ENTERPRISE FEATURE** - Requires Pattern Library Pro license
     *
     * Converts cached HTML to CMS-agnostic Portable Text that can be
     * consumed by Go CLI converters for any target CMS.
     *
     * ## OPTIONS
     *
     * <path>
     * : Path to HTML file, directory, or * for all files
     *
     * [--output=<path>]
     * : Output file or directory (required for multiple files)
     *
     * [--recursive]
     * : Process directories recursively
     *
     * [--bundle]
     * : Output single bundled JSON file (requires --output)
     *
     * [--verbose]
     * : Show detailed conversion statistics and block type breakdown
     *
     * ## EXAMPLES
     *
     *     # Single file to STDOUT
     *     $ wp scw-headless normalize /
     *
     *     # Single file to file
     *     $ wp scw-headless normalize /about/ --output=about.json
     *
     *     # All files to directory
     *     $ wp scw-headless normalize * --output=/tmp/export/
     *
     *     # All files to bundled ZIP
     *     $ wp scw-headless normalize * --output=export.json --bundle
     *
     *     # Directory recursively
     *     $ wp scw-headless normalize /blog/ --recursive --output=/tmp/blog/
     *
     * @when after_wp_load
     */
    public function normalize( $args, $assoc_args ) {
        // ==========================================
        // ENTERPRISE GATE: Check ID at the door
        // ==========================================
        $is_enterprise = apply_filters( 'stcw_headless_is_enterprise', false );

        if ( ! $is_enterprise ) {
            \WP_CLI::line( '' );
            \WP_CLI::line( \WP_CLI::colorize( '%R═══════════════════════════════════════════════════════════%n' ) );
            \WP_CLI::line( \WP_CLI::colorize( '%R  ENTERPRISE FEATURE: normalize command%n' ) );
            \WP_CLI::line( \WP_CLI::colorize( '%R═══════════════════════════════════════════════════════════%n' ) );
            \WP_CLI::line( '' );
            \WP_CLI::line( 'The normalize command outputs CMS-agnostic PortableText' );
            \WP_CLI::line( 'for use with Go CLI converters (Sanity, Strapi, Contentful, etc.)' );
            \WP_CLI::line( '' );
            \WP_CLI::line( \WP_CLI::colorize( '%YThis feature requires STCW Pattern Library Pro.%n' ) );
            \WP_CLI::line( '' );
            \WP_CLI::line( \WP_CLI::colorize( '%G→ FREE ALTERNATIVE:%n' ) );
            \WP_CLI::line( '  wp scw-headless convert --cms=sanity' );
            \WP_CLI::line( '  (Direct WordPress → Sanity conversion, always free)' );
            \WP_CLI::line( '' );
            \WP_CLI::line( \WP_CLI::colorize( '%G→ UNLOCK ENTERPRISE:%n' ) );
            \WP_CLI::line( '  1. Install STCW Pattern Library Pro' );
            \WP_CLI::line( '  2. Activate license: wp scw-patterns license <code>' );
            \WP_CLI::line( '  3. Run: wp scw-headless normalize' );
            \WP_CLI::line( '' );
            \WP_CLI::line( 'Questions? https://moderncli.dev/' );
            \WP_CLI::line( '' );

            \WP_CLI::error( 'Enterprise license required', false );
            exit( 1 );
        }

        // ==========================================
        // LICENSE VALID - CONVERT TO GENERIC FORMAT
        // ==========================================

        $input       = $args[0] ?? '';
        $output_path = $assoc_args['output'] ?? null;
        $recursive   = isset( $assoc_args['recursive'] );
        $bundle      = isset( $assoc_args['bundle'] );
        $verbose     = isset( $assoc_args['verbose'] );

        if ( empty( $input ) ) {
            \WP_CLI::error( 'Please provide a path or * for all files' );
        }

        // Build file list
        $files = $this->build_file_list( $input, $recursive );
        $total = count( $files );

        \WP_CLI::line( '' );
        \WP_CLI::line( \WP_CLI::colorize( '%B=== Generic Portable Text Conversion ===%n' ) );
        \WP_CLI::line( '' );
        \WP_CLI::line( sprintf( 'Files to process: %s', \WP_CLI::colorize( '%G' . $total . '%n' ) ) );
        \WP_CLI::line( '' );

        // Process files with progress bar
        $progress = \WP_CLI\Utils\make_progress_bar( 'Converting files', $total );

        $results       = array();
        $success_count = 0;
        $error_count   = 0;

        foreach ( $files as $file ) {
            $result = $this->normalize_single_file( $file['path'], $verbose );

            if ( $result ) {
                $results[]      = $result;
                $success_count++;

                if ( $verbose ) {
                    \WP_CLI::log(
                        sprintf(
                            'Converted: %s (%d blocks)',
                            \WP_CLI::colorize( '%G' . ( $file['url_path'] ?? basename( $file['path'] ) ) . '%n' ),
                            $result['stats']['total_blocks']
                        )
                    );
                }
            } else {
                $error_count++;
                if ( $verbose ) {
                    \WP_CLI::warning( sprintf( 'Failed: %s', $file['url_path'] ?? basename( $file['path'] ) ) );
                }
            }

            $progress->tick();
        }

        $progress->finish();

        \WP_CLI::line( '' );
        \WP_CLI::line( \WP_CLI::colorize( '%B=== Conversion Complete ===%n' ) );
        \WP_CLI::line( '' );
        \WP_CLI::line( sprintf( 'Successfully converted: %s', \WP_CLI::colorize( '%G' . $success_count . '%n' ) ) );

        if ( $error_count > 0 ) {
            \WP_CLI::line( sprintf( 'Errors:                %s', \WP_CLI::colorize( '%R' . $error_count . '%n' ) ) );
        }

        \WP_CLI::line( '' );

        // Handle output based on mode
        if ( count( $results ) === 1 && ! $bundle ) {
            // SINGLE FILE MODE - existing behavior
            $json = wp_json_encode( $results[0], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE );

            if ( $output_path ) {
                // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
                file_put_contents( $output_path, $json );
                \WP_CLI::success( sprintf( 'Generic Portable Text saved to: %s', $output_path ) );
            } else {
                // Output to STDOUT
                \WP_CLI::line( $json );
            }
        } elseif ( $bundle ) {
            // BUNDLE MODE - single JSON with all pages
            if ( ! $output_path ) {
                \WP_CLI::error( '--bundle requires --output=<file.json>' );
            }

            $bundled = array(
                'version'   => '1.0.0',
                'format'    => 'generic-portable-text-bundle',
                'generator' => array(
                    'name'    => 'stcw-headless-assistant',
                    'version' => STCW_HEADLESS_VERSION,
                ),
                'pages'     => $results,
                'stats'     => array(
                    'total_pages'  => count( $results ),
                    'total_blocks' => array_sum( array_column( array_column( $results, 'stats' ), 'total_blocks' ) ),
                    'total_assets' => array_sum( array_column( array_column( $results, 'stats' ), 'total_assets' ) ),
                ),
            );

            $json = wp_json_encode( $bundled, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE );

            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
            file_put_contents( $output_path, $json );
            \WP_CLI::success( sprintf( 'Bundled export saved to: %s', $output_path ) );
        } else {
            // MULTI-FILE MODE - create ZIP package
            if ( ! $output_path ) {
                \WP_CLI::error( 'Multiple files require --output=<directory> or --bundle' );
            }

            \WP_CLI::line( 'Creating export package...' );

            // Create export directory
            $timestamp   = gmdate( 'Y-m-d-H-i-s' );
            $package_dir = trailingslashit( Core::get_export_dir() ) . 'generic-export-' . $timestamp . '/';
            $pages_dir   = $package_dir . 'pages/';

            wp_mkdir_p( $pages_dir );

            // Write individual JSON files
            foreach ( $results as $result ) {
                $filename  = $this->generate_output_filename( $result );
                $file_path = $pages_dir . $filename;
                $json      = wp_json_encode( $result, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE );

                // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
                file_put_contents( $file_path, $json );
            }

            // Create manifest
            $manifest = array(
                'version'   => '1.0.0',
                'format'    => 'generic-portable-text-multi',
                'generator' => array(
                    'name'    => 'stcw-headless-assistant',
                    'version' => STCW_HEADLESS_VERSION,
                ),
                'files'     => array_map(
                    function( $result ) {
                        return array(
                            'filename' => $this->generate_output_filename( $result ),
                            'title'    => $result['page']['title'],
                            'slug'     => $result['page']['slug'],
                            'url'      => $result['page']['url'],
                        );
                    },
                    $results
                ),
                'stats'     => array(
                    'total_pages'  => count( $results ),
                    'total_blocks' => array_sum( array_column( array_column( $results, 'stats' ), 'total_blocks' ) ),
                    'total_assets' => array_sum( array_column( array_column( $results, 'stats' ), 'total_assets' ) ),
                ),
            );

            $manifest_json = wp_json_encode( $manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE );
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
            file_put_contents( $package_dir . 'manifest.json', $manifest_json );

            // Create README
            $readme = "Generic Portable Text Export\n";
            $readme .= "=============================\n\n";
            $readme .= "Generated: " . gmdate( 'Y-m-d H:i:s' ) . " UTC\n";
            $readme .= "Total Pages: " . count( $results ) . "\n\n";
            $readme .= "Usage:\n";
            $readme .= "------\n";
            $readme .= "Import these files using Go CLI converters:\n";
            $readme .= "  go run . convert --source=pages/ --target=sanity\n\n";

            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
            file_put_contents( $package_dir . 'README.txt', $readme );

            // Create ZIP
            $zip_filename = 'generic-export-' . $timestamp . '.zip';
            $zip_path     = Core::get_export_dir() . '/' . $zip_filename;

            $zip = new \ZipArchive();
            if ( $zip->open( $zip_path, \ZipArchive::CREATE ) !== true ) {
                \WP_CLI::error( 'Failed to create ZIP file' );
            }

            // Add all files to ZIP
            $files_to_zip = new \RecursiveIteratorIterator(
                new \RecursiveDirectoryIterator( $package_dir ),
                \RecursiveIteratorIterator::LEAVES_ONLY
            );

            foreach ( $files_to_zip as $file ) {
                if ( ! $file->isDir() ) {
                    $file_path     = $file->getRealPath();
                    $relative_path = substr( $file_path, strlen( $package_dir ) );
                    $zip->addFile( $file_path, $relative_path );
                }
            }

            $zip->close();

            \WP_CLI::success( sprintf( 'Export package created: %s', $zip_filename ) );
            \WP_CLI::line( 'Package location: ' . $package_dir );
        }

        \WP_CLI::success( 'Normalization complete' );
    }

    /**
     * Show all registered patterns
     *
     * Displays pattern registry with inheritance, confidence, and status.
     *
     * ## OPTIONS
     *
     * [--format=<format>]
     * : Output format (table, json, yaml)
     * ---
     * default: table
     * options:
     *   - table
     *   - json
     *   - yaml
     * ---
     *
     * [--enabled-only]
     * : Show only enabled patterns
     *
     * [--verbose]
     * : Show detailed pattern configuration
     *
     * ## EXAMPLES
     *
     *     # Show all patterns in table format
     *     $ wp scw-headless patterns
     *
     *     # Show only enabled patterns
     *     $ wp scw-headless patterns --enabled-only
     *
     *     # Get detailed JSON output
     *     $ wp scw-headless patterns --format=json --verbose
     *
     * @when after_wp_load
     */
    public function patterns($args, $assoc_args) {
        $format = $assoc_args['format'] ?? 'table';
        $enabled_only = isset($assoc_args['enabled-only']);
        $verbose = isset($assoc_args['verbose']);

        $patterns = PatternRegistry::get_patterns();

        if ($enabled_only) {
            $patterns = array_filter($patterns, function ($pattern) {
                return $pattern['enabled'] ?? true;
            });
        }

        if ($format === 'json') {
            \WP_CLI::line(wp_json_encode($verbose ? $patterns : array_keys($patterns), JSON_PRETTY_PRINT));
            return;
        }

        if ($format === 'yaml') {
            // format_items() prints directly; also PatternRegistry is associative, so normalize to rows.
            $yaml_rows = [];
            foreach ($patterns as $name => $config) {
                $yaml_rows[] = array_merge(['name' => $name], $config);
            }
            $fields = array_keys($yaml_rows[0] ?? ['name']);
            \WP_CLI\Utils\format_items('yaml', $yaml_rows, $fields);
            return;
        }

        // Table format
        \WP_CLI::line('');
        \WP_CLI::line(\WP_CLI::colorize('%B=== Registered Patterns ===%n'));
        \WP_CLI::line('');

        // Summary stats
        $total = count(PatternRegistry::get_patterns());
        $enabled = PatternRegistry::count_patterns(true);
        $disabled = $total - $enabled;

        \WP_CLI::line(sprintf('Total Patterns:    %s', \WP_CLI::colorize('%G' . $total . '%n')));
        \WP_CLI::line(sprintf('Enabled:           %s', \WP_CLI::colorize('%G' . $enabled . '%n')));

        if ($disabled > 0) {
            \WP_CLI::line(sprintf('Disabled:          %s', \WP_CLI::colorize('%R' . $disabled . '%n')));
        }

        \WP_CLI::line('');

        // Prepare table data
        $table_data = [];

        foreach ($patterns as $name => $config) {
            $status = ($config['enabled'] ?? true) ? \WP_CLI::colorize('%G✓%n') : \WP_CLI::colorize('%R✗%n');
            $confidence = number_format($config['confidence'] ?? 1.0, 2);
            $selectors = count($config['selectors'] ?? []);
            $extends = $config['extends'] ?? '-';

            $row = [
                'status' => $status,
                'name' => $name,
                'priority' => $config['priority'] ?? 5,
                'confidence' => $confidence,
                'selectors' => $selectors,
            ];

            if ($verbose) {
                $row['extends'] = $extends;
                $row['description'] = $config['description'] ?? '';
            }

            $table_data[] = $row;
        }

        // Sort by priority (descending)
        usort($table_data, function ($a, $b) {
            return $b['priority'] - $a['priority'];
        });

        // Display table
        $fields = $verbose
            ? ['status', 'name', 'priority', 'confidence', 'selectors', 'extends', 'description']
            : ['status', 'name', 'priority', 'confidence', 'selectors'];

        \WP_CLI\Utils\format_items('table', $table_data, $fields);

        \WP_CLI::line('');

        if ($verbose && !$enabled_only) {
            \WP_CLI::line(\WP_CLI::colorize('%YTip:%n Use --enabled-only to hide disabled patterns'));
        }
    }

    /**
     * List all registered detector modules
     *
     * Shows built-in and extension detector plugins with metadata.
     *
     * ## OPTIONS
     *
     * [--format=<format>]
     * : Output format (table, json, yaml)
     * ---
     * default: table
     * options:
     *   - table
     *   - json
     *   - yaml
     * ---
     *
     * [--type=<type>]
     * : Filter by detector type (built-in, extension, premium, test)
     *
     * ## EXAMPLES
     *
     *     # List all detectors
     *     $ wp scw-headless detectors
     *
     *     # Show only extension detectors
     *     $ wp scw-headless detectors --type=extension
     *
     *     # Get as JSON
     *     $ wp scw-headless detectors --format=json
     *
     * @when after_wp_load
     */
    public function detectors($args, $assoc_args) {
        $format = $assoc_args['format'] ?? 'table';
        $type_filter = $assoc_args['type'] ?? null;

        $detectors = \STCW_Headless()->get_detectors();

        // Filter by type if specified
        if ($type_filter) {
            $detectors = array_filter($detectors, function ($detector) use ($type_filter) {
                return ($detector['type'] ?? 'extension') === $type_filter;
            });
        }

        if (empty($detectors)) {
            \WP_CLI::warning('No detector modules found');
            return;
        }

        // JSON output
        if ($format === 'json') {
            \WP_CLI::line(wp_json_encode($detectors, JSON_PRETTY_PRINT));
            return;
        }

        // Table output
        \WP_CLI::line('');
        \WP_CLI::line(\WP_CLI::colorize('%B=== Registered Detector Modules ===%n'));
        \WP_CLI::line('');

        $table_data = [];
        foreach ($detectors as $slug => $detector) {
            $type = $detector['type'] ?? 'extension';
            $status = $detector['status'] ?? 'active';

            // Color code status
            $status_display = $status === 'active'
                ? \WP_CLI::colorize('%G✓%n')
                : \WP_CLI::colorize('%R○%n');

            // Color code type
            $type_display = $type;
            if ($type === 'built-in') {
                $type_display = \WP_CLI::colorize('%C' . $type . '%n');
            } elseif ($type === 'premium') {
                $type_display = \WP_CLI::colorize('%Y' . $type . '%n');
            } elseif ($type === 'test') {
                $type_display = \WP_CLI::colorize('%M' . $type . '%n');
            }

            $table_data[] = [
                'status' => $status_display,
                'slug' => $slug,
                'name' => $detector['name'] ?? $slug,
                'version' => $detector['version'] ?? 'N/A',
                'type' => $type_display,
                'patterns' => $detector['patterns'] ?? 0,
            ];
        }

        \WP_CLI\Utils\format_items('table', $table_data, [
            'status', 'slug', 'name', 'version', 'type', 'patterns',
        ]);

        \WP_CLI::line('');

        // Summary
        $total = count($detectors);
        $builtin = \STCW_Headless()->get_detector_count('built-in');
        $extensions = \STCW_Headless()->get_detector_count('extension');
        $premium = \STCW_Headless()->get_detector_count('premium');
        $test = \STCW_Headless()->get_detector_count('test');

        \WP_CLI::line(\WP_CLI::colorize('%Y» Summary%n'));
        \WP_CLI::line(sprintf('  Total Detectors: %s', \WP_CLI::colorize('%G' . $total . '%n')));
        if ($builtin > 0) {
            \WP_CLI::line(sprintf('  Built-in:        %s', $builtin));
        }
        if ($extensions > 0) {
            \WP_CLI::line(sprintf('  Extensions:      %s', $extensions));
        }
        if ($premium > 0) {
            \WP_CLI::line(sprintf('  Premium:         %s', $premium));
        }
        if ($test > 0) {
            \WP_CLI::line(sprintf('  Test/Dev:        %s', $test));
        }

        \WP_CLI::line('');
        \WP_CLI::success('Detector registry active');
    }

    /**
     * Convert cached files to CMS format
     *
     * Parses cached HTML files, detects patterns, converts to CMS format,
     * and exports importable package.
     *
     * ## OPTIONS
     *
     * [--cms=<target>]
     * : Target CMS (default: sanity)
     *
     * [--limit=<number>]
     * : Limit number of files to convert
     *
     * [--verbose]
     * : Show detailed conversion output
     *
     * [--prepare-assets]
     * : Copy assets from static cache to uploads before converting
     *
     * ## EXAMPLES
     *
     *     # Convert with asset preparation
     *     $ wp scw-headless convert --cms=sanity --prepare-assets
     *
     *     # Convert 10 files with assets
     *     $ wp scw-headless convert --limit=10 --prepare-assets --verbose*
     * @when after_wp_load
     */
    public function convert($args, $assoc_args) {
        $cms = $assoc_args['cms'] ?? 'sanity';
        $limit = isset($assoc_args['limit']) ? absint($assoc_args['limit']) : 0;
	$verbose = isset($assoc_args['verbose']);
	$prepare_assets = isset($assoc_args['prepare-assets']);

	if ($prepare_assets) {
            \WP_CLI::line('');
            \WP_CLI::line(\WP_CLI::colorize('%Y» Preparing assets first...%n'));
            $this->prepare_assets([], ['verbose' => $verbose]);
            \WP_CLI::line('');
	}

        \WP_CLI::line('');
        \WP_CLI::line(\WP_CLI::colorize('%B=== Converting to ' . ucfirst($cms) . ' Format ===%n'));
        \WP_CLI::line('');

        // Get CMS target
        $target = \STCW\Headless\Engine\Target\TargetRegistry::get($cms);

        if (!$target) {
            $available = implode(', ', \STCW\Headless\Engine\Target\TargetRegistry::get_slugs());
            \WP_CLI::error("CMS target '{$cms}' not found. Available: {$available}");
        }

        \WP_CLI::line(sprintf('Using target: %s', \WP_CLI::colorize('%G' . $target->get_name() . '%n')));
        \WP_CLI::line('');

        // Scan files
        $scanner = new Scanner();
        $scan_result = $scanner->scan_cached_files();

        if (!$scan_result['success']) {
            \WP_CLI::error($scan_result['message']);
        }

        $files = $scan_result['files'];

        if (empty($files)) {
            \WP_CLI::error('No files to convert. Run: wp scw-headless scan');
        }

        // Apply limit
        if ($limit > 0) {
            $files = array_slice($files, 0, $limit);
        }

        $total = count($files);
        \WP_CLI::line(sprintf('Converting %d files...', $total));
        \WP_CLI::line('');

        $progress = \WP_CLI\Utils\make_progress_bar('Converting files', $total);

        $parser = new Parser();
        $converted_pages = [];
        $success_count = 0;
        $error_count = 0;

        foreach ($files as $file) {
            // Parse HTML file
            $parse_result = $parser->parse_file($file['path']);

            if (!$parse_result['success']) {
                $error_count++;
                if ($verbose) {
                    \WP_CLI::warning(sprintf(
                        'Failed to parse: %s - %s',
                        $file['url_path'],
                        $parse_result['message']
                    ));
                }
                $progress->tick();
                continue;
            }

            // Convert using target
            $converted = $target->convert($parse_result['patterns'], $parse_result['metadata']);
            $converted_pages[] = $converted;
            $success_count++;

            if ($verbose) {
                \WP_CLI::log(sprintf(
                    'Converted: %s (%d patterns)',
                    \WP_CLI::colorize('%G' . $file['url_path'] . '%n'),
                    count($parse_result['patterns'])
                ));
            }

            $progress->tick();
        }

        $progress->finish();

        \WP_CLI::line('');
        \WP_CLI::line(\WP_CLI::colorize('%B=== Conversion Complete ===%n'));
        \WP_CLI::line('');
        \WP_CLI::line(sprintf('Successfully converted: %s', \WP_CLI::colorize('%G' . $success_count . '%n')));

        if ($error_count > 0) {
            \WP_CLI::line(sprintf('Errors:              %s', \WP_CLI::colorize('%R' . $error_count . '%n')));
        }

        \WP_CLI::line('');

        // Create export package
        \WP_CLI::line('Creating export package...');

        $export_result = $target->export($converted_pages, Core::get_export_dir());

        if ($export_result['success']) {
            \WP_CLI::success(sprintf(
                'Export package created: %s',
                basename($export_result['zip_file'])
            ));
            \WP_CLI::line('Package location: ' . $export_result['package_dir']);
        } else {
            \WP_CLI::error('Failed to create export package');
        }
    }

 /**
 * Prepare assets for export by copying from static cache to uploads
 *
 * @subcommand prepare-assets
 *
 * Copies assets from Static Cache Wrangler's /assets/ directory to
 * /wp-content/uploads/stcw-assets/ for predictable URLs.
 *
 * ## OPTIONS
 *
 * [--force]
 * : Overwrite existing assets
 *
 * [--verbose]
 * : Show detailed output
 *
 * ## EXAMPLES
 *
 *     # Copy assets
 *     $ wp scw-headless prepare-assets
 *
 *     # Force overwrite existing
 *     $ wp scw-headless prepare-assets --force
 *
 *     # Show details
 *     $ wp scw-headless prepare-assets --verbose
 *
 * @when after_wp_load
 */
public function prepare_assets($args, $assoc_args) {
    $force = isset($assoc_args['force']);
    $verbose = isset($assoc_args['verbose']);

    \WP_CLI::line('');
    \WP_CLI::line(\WP_CLI::colorize('%B=== Preparing Assets ===%n'));
    \WP_CLI::line('');

    // Get source directory (SCW static cache)
    $static_dir = Core::get_scw_static_dir();
    if (empty($static_dir)) {
        \WP_CLI::error('Static Cache Wrangler directory not found');
    }

    $source_dir = trailingslashit($static_dir) . 'assets/';

    if (!is_dir($source_dir)) {
        \WP_CLI::error('No assets directory found in static cache: ' . $source_dir);
    }

    // Get destination directory
    $dest_dir = WP_CONTENT_DIR . '/uploads/stcw-assets/';

    // Create destination if doesn't exist
    if (!is_dir($dest_dir)) {
        wp_mkdir_p($dest_dir);
        \WP_CLI::log('Created directory: ' . $dest_dir);
    }

    // Copy assets
    $result = $this->recursive_copy_assets($source_dir, $dest_dir, $force, $verbose);

    \WP_CLI::line('');
    \WP_CLI::line(\WP_CLI::colorize('%B=== Copy Complete ===%n'));
    \WP_CLI::line('');
    \WP_CLI::line(sprintf('Total files:     %s', \WP_CLI::colorize('%G' . $result['total'] . '%n')));
    \WP_CLI::line(sprintf('Copied:          %s', \WP_CLI::colorize('%G' . $result['copied'] . '%n')));
    \WP_CLI::line(sprintf('Skipped:         %s', \WP_CLI::colorize('%Y' . $result['skipped'] . '%n')));

    if ($result['errors'] > 0) {
        \WP_CLI::line(sprintf('Errors:          %s', \WP_CLI::colorize('%R' . $result['errors'] . '%n')));
    }

    \WP_CLI::line('');
    \WP_CLI::success(sprintf(
        'Assets prepared in: %s',
        $dest_dir
    ));
}

/**
 * Recursively copy assets from source to destination
 *
 * @param string $source Source directory
 * @param string $dest Destination directory
 * @param bool $force Force overwrite
 * @param bool $verbose Show verbose output
 * @return array Copy statistics
 */
private function recursive_copy_assets($source, $dest, $force = false, $verbose = false) {
    $stats = [
        'total' => 0,
        'copied' => 0,
        'skipped' => 0,
        'errors' => 0,
    ];

    // Use DirectoryIterator for better performance
    $iterator = new \RecursiveIteratorIterator(
        new \RecursiveDirectoryIterator($source, \RecursiveDirectoryIterator::SKIP_DOTS),
        \RecursiveIteratorIterator::SELF_FIRST
    );

    foreach ($iterator as $item) {
        $stats['total']++;

        if ($item->isDir()) {
            continue;
        }

    // NEW: Only copy image and PDF files
    $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg', 'pdf', 'ico'];
    $extension = strtolower(pathinfo($item->getFilename(), PATHINFO_EXTENSION));
    
    if (!in_array($extension, $allowed_extensions, true)) {
        $stats['skipped']++;
        if ($verbose) {
            \WP_CLI::log(sprintf('Skipped: %s (not an image/PDF)', substr($item->getPathname(), strlen($source))));
        }
        continue;
    }

        // Get relative path
        $relative_path = substr($item->getPathname(), strlen($source));
        $dest_file = $dest . $relative_path;

        // Create subdirectories if needed
        $dest_dir = dirname($dest_file);
        if (!is_dir($dest_dir)) {
            wp_mkdir_p($dest_dir);
        }

        // Check if file exists
        if (file_exists($dest_file) && !$force) {
            $stats['skipped']++;
            if ($verbose) {
                \WP_CLI::log(sprintf('Skipped: %s (already exists)', $relative_path));
            }
            continue;
        }

        // Copy file
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_copy -- CLI context
        if (copy($item->getPathname(), $dest_file)) {
            $stats['copied']++;
            if ($verbose) {
                \WP_CLI::log(sprintf('Copied: %s', \WP_CLI::colorize('%G' . $relative_path . '%n')));
            }
        } else {
            $stats['errors']++;
            \WP_CLI::warning(sprintf('Failed to copy: %s', $relative_path));
        }
    }

    return $stats;
}

    /**
     * List available CMS targets
     *
     * Shows all registered CMS targets that can be used with the convert command.
     *
     * ## OPTIONS
     *
     * [--format=<format>]
     * : Output format (table, json, yaml)
     * ---
     * default: table
     * options:
     *   - table
     *   - json
     *   - yaml
     * ---
     *
     * ## EXAMPLES
     *
     *     # List available targets
     *     $ wp scw-headless targets
     *
     *     # Get targets as JSON
     *     $ wp scw-headless targets --format=json
     *
     * @when after_wp_load
     */
    public function targets($args, $assoc_args) {
        $format = $assoc_args['format'] ?? 'table';

        $targets = \STCW\Headless\Engine\Target\TargetRegistry::get_all();

        if (empty($targets)) {
            \WP_CLI::warning('No CMS targets registered');
            return;
        }

        if ($format === 'json') {
            $output = [];
            foreach ($targets as $slug => $target) {
                $output[] = [
                    'slug' => $target->get_slug(),
                    'name' => $target->get_name(),
                ];
            }
            \WP_CLI::line(wp_json_encode($output, JSON_PRETTY_PRINT));
            return;
        }

        // Table format
        \WP_CLI::line('');
        \WP_CLI::line(\WP_CLI::colorize('%B=== Available CMS Targets ===%n'));
        \WP_CLI::line('');

        $table_data = [];
        foreach ($targets as $slug => $target) {
            $table_data[] = [
                'slug' => $target->get_slug(),
                'name' => $target->get_name(),
            ];
        }

        \WP_CLI\Utils\format_items('table', $table_data, ['slug', 'name']);

        \WP_CLI::line('');
        \WP_CLI::line('Use with: wp scw-headless convert --cms=<slug>');
    }

    /**
     * Resolve file path relative to static cache directory
     *
     * Handles multiple path formats flexibly:
     * - URL paths from scan output (/contact/, /publishing/)
     * - Homepage (/)
     * - Short form without slashes (contact, publishing)
     * - Explicit file paths (contact/index.html)
     * - Absolute paths (if they exist)
     *
     * @param string $path User-provided path
     * @return string|false Resolved absolute path or false
     */
    private function resolve_file_path($path) {
        // Get static cache directory first
        $static_dir = Core::get_scw_static_dir();

        if (empty($static_dir) || !is_dir($static_dir)) {
            return false;
        }

        // Special case: Handle homepage "/" first
        if ($path === '/' || trim($path) === '') {
            $homepage = trailingslashit($static_dir) . 'index.html';
            if (file_exists($homepage)) {
                return $homepage;
            }
            return false;
        }

        // Strategy 1: Already an absolute path that exists
        if (file_exists($path)) {
            // If it's a directory, append index.html
            if (is_dir($path)) {
                $index = trailingslashit($path) . 'index.html';
                if (file_exists($index)) {
                    return $index;
                }
            }
            return $path;
        }

        // Clean the input path
        $clean_path = trim($path, '/');

        // Strategy 2: Try path as-is relative to static dir
        $resolved = trailingslashit($static_dir) . $clean_path;
        
        // If resolved path is a directory, append index.html
        if (is_dir($resolved)) {
            $index = trailingslashit($resolved) . 'index.html';
            if (file_exists($index)) {
                return $index;
            }
        }
        
        // If resolved path is a file, return it
        if (file_exists($resolved)) {
            return $resolved;
        }

        // Strategy 3: Try with trailing slash + index.html
        $with_index = trailingslashit($static_dir) . trailingslashit($clean_path) . 'index.html';
        if (file_exists($with_index)) {
            return $with_index;
        }

        return false;
    }

/**
     * Build file list from input argument
     *
     * @param string $input User input (path, *, /)
     * @param bool $recursive Allow directory traversal
     * @return array Array of file info arrays
     */

    private function build_file_list( $input, $recursive = false ) {
    $scanner = new Scanner();

    // Special case: / with --recursive means all files
    if ( $input === '/' && $recursive ) {
        $scan_result = $scanner->scan_cached_files();

        if ( ! $scan_result['success'] ) {
            \WP_CLI::error( $scan_result['message'] );
        }

        return $scan_result['files'];
    }

    // Check if input looks like a directory path (ends with /)
    $looks_like_directory = substr( $input, -1 ) === '/';

    // If it looks like a directory AND --recursive is set, treat as URL prefix filter
    if ( $looks_like_directory && $recursive ) {
        $scan_result = $scanner->scan_cached_files();

        if ( ! $scan_result['success'] ) {
            \WP_CLI::error( $scan_result['message'] );
        }

        // Build URL prefix from input
        $url_prefix = '/' . trim( $input, '/' ) . '/';

        // Filter files by URL path prefix
        $filtered = array_filter(
            $scan_result['files'],
            function( $file ) use ( $url_prefix ) {
                $file_url = $file['url_path'] ?? '';
                return strpos( $file_url, $url_prefix ) === 0;
            }
        );

        if ( empty( $filtered ) ) {
            \WP_CLI::error( sprintf( 'No files found under: %s', $input ) );
        }

        return array_values( $filtered );
    }

    // Resolve path for single file
    $resolved_path = $this->resolve_file_path( $input );

    if ( ! $resolved_path || ! file_exists( $resolved_path ) ) {
        \WP_CLI::error( sprintf( 'Path not found: %s', $input ) );
    }

    // Single file
    return array(
        array(
            'path'     => $resolved_path,
            'url_path' => $input,
        ),
    );
    }
    
    /**
     * Normalize single HTML file to generic portable text
     *
     * @param string $file_path Path to HTML file
     * @param bool $verbose Show detailed output
     * @return array|null Normalized page data or null on failure
     */
    private function normalize_single_file( $file_path, $verbose = false ) {
        // Parse HTML file
        $parser = new Parser();
        $parse_result = $parser->parse_file( $file_path );
        
        if ( ! $parse_result['success'] ) {
            return null;
        }
        
        $detected      = $parse_result['patterns'] ?? array();
        $page_metadata = $parse_result['metadata'] ?? array();
        
        // Load Generic Portable Text Converter
        require_once STCW_HEADLESS_DIR . 'includes/Engine/Target/Generic/class-generic-portable-text-converter.php';
        
        $converter = new \STCW\Headless\Engine\Target\Generic\GenericPortableTextConverter();
        
        // Convert patterns to generic format
        $blocks = $converter->convert_patterns( $detected, $page_metadata );
        $assets = $converter->get_asset_references();
        
        // Build page structure
        return array(
            'version'   => '1.0.0',
            'format'    => 'generic-portable-text',
            'generator' => array(
                'name'    => 'stcw-headless-assistant',
                'version' => STCW_HEADLESS_VERSION,
            ),
            'page'      => array(
                'title'       => html_entity_decode( $page_metadata['title'] ?? 'Untitled', ENT_QUOTES | ENT_HTML5, 'UTF-8' ),
                'slug'        => $page_metadata['slug'] ?? '',
                'description' => html_entity_decode( $page_metadata['description'] ?? '', ENT_QUOTES | ENT_HTML5, 'UTF-8' ),
                'url'         => $page_metadata['url'] ?? '',
                'source'      => array(
                    'type'         => 'wordpress',
                    'permalink'    => $page_metadata['source']['permalink'] ?? '',
                    'wp_post_id'   => $page_metadata['source']['wp_post_id'] ?? null,
                    'wp_post_type' => $page_metadata['source']['wp_post_type'] ?? 'unknown',
                    'og_type'      => $page_metadata['source']['og_type'] ?? '',
                    'template'     => $page_metadata['source']['template'] ?? '',
                    'published'    => $page_metadata['source']['published'] ?? '',
                    'modified'     => $page_metadata['source']['modified'] ?? '',
                    'cached'       => $page_metadata['source']['cached'] ?? '',
                    'scw_version'  => $page_metadata['source']['scw_version'] ?? '',
                    'hash'         => md5( $parse_result['normalized_html'] ?? '' ),
                ),
            ),
            'blocks'    => $blocks,
            'assets'    => $assets,
            'stats'     => array(
                'total_blocks' => count( $blocks ),
                'total_assets' => count( $assets ),
            ),
        );
    }
    
    /**
     * Generate unique output filename from page metadata
     *
     * @param array $page_data Normalized page data
     * @return string Filename (e.g., "homepage-a3f2e1b4.json")
     */
    private function generate_output_filename( $page_data ) {
        $permalink = $page_data['page']['source']['permalink'] ?? '';
        
        if ( empty( $permalink ) ) {
            // Fallback: random filename
            return 'page-' . wp_rand( 10000, 99999 ) . '.json';
        }
        
        // Create readable filename with hash suffix for uniqueness
        $slug = $page_data['page']['slug'] ?? 'untitled';
        $slug = trim( $slug, '/' );
        
        if ( empty( $slug ) ) {
            $slug = 'homepage';
        }
        
        $hash = substr( md5( $permalink ), 0, 8 );
        
        return sanitize_file_name( $slug . '-' . $hash ) . '.json';
    }

}

\WP_CLI::add_command('scw-headless', CLI::class);

