<?php
/**
 * CMS Target Registry
 *
 * Central registry for managing available CMS targets.
 * Allows plugins to register custom targets via filters.
 *
 * @package STCWHeadlessAssistant
 * @since 2.0.0
 */

namespace STCW\Headless\Engine\Target;

if (!defined('ABSPATH')) exit;

class TargetRegistry {
    
    /**
     * Registered targets
     * @var array
     */
    private static $targets = [];
    
    /**
     * Register a CMS target
     *
     * @param TargetInterface $target Target implementation
     * @return bool True on success
     */
    public static function register(TargetInterface $target): bool {
        $slug = $target->get_slug();
        
        if (isset(self::$targets[$slug])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log(
                    sprintf(
                        /* translators: %s: Target slug */
                        esc_html('CMS target "%s" is already registered'),
                        esc_html($slug)
                    )
                );
            }
            return false;
        }
        
        self::$targets[$slug] = $target;
        
        return true;
    }
    
    /**
     * Get specific target by slug
     *
     * @param string $slug Target slug (e.g., 'sanity')
     * @return TargetInterface|null Target instance or null
     */
    public static function get(string $slug): ?TargetInterface {
        return self::$targets[$slug] ?? null;
    }
    
    /**
     * Get all registered targets
     *
     * @return array Array of TargetInterface instances
     */
    public static function get_all(): array {
        return self::$targets;
    }
    
    /**
     * Get all target slugs
     *
     * @return array Array of registered slugs
     */
    public static function get_slugs(): array {
        return array_keys(self::$targets);
    }
    
    /**
     * Check if target exists
     *
     * @param string $slug Target slug
     * @return bool True if target is registered
     */
    public static function has(string $slug): bool {
        return isset(self::$targets[$slug]);
    }
    
    /**
     * Clear all registered targets (for testing)
     *
     * @return void
     */
    public static function clear(): void {
        self::$targets = [];
    }
}
