<?php
/**
 * Sanity CMS Target
 *
 * Main implementation of TargetInterface for Sanity CMS.
 * Orchestrates conversion, schema generation, and export.
 *
 * @package STCWHeadlessAssistant
 * @since 2.0.0
 */

namespace STCW\Headless\Engine\Target\Sanity;

use STCW\Headless\Engine\Target\TargetInterface;

if (!defined('ABSPATH')) exit;

class SanityTarget implements TargetInterface {
    
    /**
     * Converter instance
     * @var Converter
     */
    private $converter;
    
    /**
     * Schema generator instance
     * @var SchemaGenerator
     */
    private $schema_generator;
    
    /**
     * Exporter instance
     * @var Exporter
     */
    private $exporter;
    
    /**
     * Constructor - initialize components
     */
    public function __construct() {
        $this->converter = new Converter();
        $this->schema_generator = new SchemaGenerator();
        $this->exporter = new Exporter();
    }
    
    /**
     * Get target display name
     *
     * @return string Human-readable name
     */
    public function get_name(): string {
        return 'Sanity® CMS';
    }
    
    /**
     * Get target slug
     *
     * @return string Slug for CLI usage
     */
    public function get_slug(): string {
        return 'sanity';
    }
    
    /**
     * Convert detected patterns to Sanity format
     *
     * @param array $patterns Detected patterns from PatternDetector
     * @param array $metadata Page metadata (title, description, etc.)
     * @return array Converted page document in Sanity format
     */
    public function convert(array $patterns, array $metadata): array {
        // Convert patterns to Sanity Portable Text
        $content = $this->converter->convert_patterns_to_sanity($patterns, $metadata);
        
        // Get asset references collected during conversion
        $asset_refs = $this->converter->get_asset_references();
        
        // Build Sanity page document
        $page_document = [
            '_id' => $this->generate_page_id($metadata),
            '_type' => 'page',
            'title' => html_entity_decode($metadata['title'] ?? 'Untitled', ENT_QUOTES | ENT_HTML5, 'UTF-8'),
            'slug' => [
                '_type' => 'slug',
                'current' => $this->extract_slug($metadata),
            ],
            'description' => html_entity_decode($metadata['description'] ?? '', ENT_QUOTES | ENT_HTML5, 'UTF-8'),
            'content' => $content,
            '_assetReferences' => $asset_refs,
            'publishedAt' => gmdate('c'),
        ];
        
        // Add OG image if present
        // Temporarily commented out along with all image references
        //if (!empty($metadata['og_image'])) {
        //    $page_document['ogImage'] = [
        //        '_type' => 'image',
        //        'asset' => [
        //            '_type' => 'reference',
        //            '_ref' => $this->generate_image_ref($metadata['og_image']),
        //        ],
        //    ];
       // }
        
        return $page_document;
    }
    
    /**
     * Generate CMS schema files
     *
     * @param string $output_dir Directory to write schema files
     * @return array Result with generated file paths
     */
    public function generate_schemas(string $output_dir): array {
        return $this->schema_generator->generate_schemas($output_dir);
    }
    
    /**
     * Export converted data to importable format
     *
     * @param array $converted_pages Array of converted page data
     * @param string $output_dir Export directory
     * @return array Result with export package information
     */
    public function export(array $converted_pages, string $output_dir): array {
        return $this->exporter->create_export_package($converted_pages);
    }
    
    /**
     * Generate page ID from metadata
     *
     * @param array $metadata Page metadata
     * @return string Sanity document ID
     */
    private function generate_page_id($metadata) {
        $slug = $this->extract_slug($metadata);
        
        // Clean slug for ID (remove slashes, use hyphens)
        $id = str_replace('/', '-', trim($slug, '/'));
        $id = $id ?: 'home';
        
        return 'page-' . sanitize_title($id);
    }
    
    /**
     * Extract slug from metadata
     *
     * @param array $metadata Page metadata
     * @return string Page slug/path
     */
    private function extract_slug($metadata) {
        // Try canonical URL first
        if (!empty($metadata['canonical'])) {
            $path = wp_parse_url($metadata['canonical'], PHP_URL_PATH);
            if ($path) {
                return $path;
            }
        }
        
        // Fallback to title-based slug
        if (!empty($metadata['title'])) {
            return '/' . sanitize_title($metadata['title']) . '/';
        }
        
        return '/';
    }
    
    /**
     * Generate image reference ID
     *
     * @param string $url Image URL
     * @return string Sanity image reference
     */
    private function generate_image_ref($url) {
        $filename = basename(wp_parse_url($url, PHP_URL_PATH));
        $clean_name = sanitize_title($filename);
        return 'image-' . $clean_name;
    }
}
