<?php
/**
 * Sanity Schema Generator - v1.3 Option C
 *
 * Generates Sanity Studio schema definition files for imported content.
 * v1.3: Generates imageBlock schema with migration metadata support.
 *
 * @package STCWHeadlessAssistant
 * @since 1.3.0
 */

namespace STCW\Headless\Engine\Target\Sanity;

if (!defined('ABSPATH')) exit;

class SchemaGenerator {
    
    /**
     * Generate all Sanity schema files
     * 
     * Creates schema definitions for page documents and custom blocks
     * that will be used in Sanity Studio.
     * 
     * @param string $output_dir Export package directory
     * @return array Result with success status and file paths
     */
    public function generate_schemas($output_dir) {
        $schema_dir = trailingslashit($output_dir) . 'schemas/';
        
        // Create schemas directory
        wp_mkdir_p($schema_dir);
        
        $files = [];
        
        // Generate individual schema files
        $files['page'] = $this->generate_page_schema($schema_dir);
        $files['separator'] = $this->generate_separator_schema($schema_dir);
        $files['customButton'] = $this->generate_custom_button_schema($schema_dir);
        $files['rawHtml'] = $this->generate_raw_html_schema($schema_dir);
        $files['accordion'] = $this->generate_accordion_schema($schema_dir);
        $files['imageBlock'] = $this->generate_image_block_schema($schema_dir);  // v1.3
        $files['codeBlock'] = $this->generate_code_block_schema($schema_dir);    // v1.3.1: Semantic types
        $files['accordionBlock'] = $this->generate_accordion_block_schema($schema_dir);  // v1.3.1
        $files['tableBlock'] = $this->generate_table_block_schema($schema_dir);  // v1.3.1
        $files['index'] = $this->generate_index_file($schema_dir);
        
        return [
            'success' => true,
            'files' => $files,
            'directory' => $schema_dir,
        ];
    }
    
    /**
     * Write schema file
     * 
     * @param string $file_path File path
     * @param string $content File content
     * @return bool True on success
     */
    private function write_schema_file($file_path, $content) {
        // Use direct file writing for schema files (CLI context)
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- CLI context, generating export files
        $result = file_put_contents($file_path, $content);
        
        if ($result !== false && file_exists($file_path)) {
            // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_chmod -- Setting secure permissions
            chmod($file_path, 0644);
            return true;
        }
        
        return false;
    }
    
    /**
     * Generate page document schema
     * 
     * @param string $schema_dir Schema directory path
     * @return string Path to generated file
     */
    private function generate_page_schema($schema_dir) {
        $schema = $this->build_page_schema_content();
        $file_path = $schema_dir . 'page.js';
        $this->write_schema_file($file_path, $schema);
        return $file_path;
    }
    
    /**
     * Build page schema JavaScript content
     * 
     * @return string JavaScript schema definition
     */
    private function build_page_schema_content() {
        $lines = array(
            'export default {',
            '  name: \'page\',',
            '  title: \'Page\',',
            '  type: \'document\',',
            '  fields: [',
            '    {',
            '      name: \'title\',',
            '      title: \'Title\',',
            '      type: \'string\',',
            '      validation: Rule => Rule.required()',
            '    },',
            '    {',
            '      name: \'slug\',',
            '      title: \'Slug\',',
            '      type: \'slug\',',
            '      options: {',
            '        source: \'title\'',
            '      },',
            '      validation: Rule => Rule.required()',
            '    },',
            '    {',
            '      name: \'description\',',
            '      title: \'Meta Description\',',
            '      type: \'text\',',
            '      rows: 3',
            '    },',
            '    {',
            '      name: \'ogImage\',',
            '      title: \'OG Image\',',
            '      type: \'image\'',
            '    },',
            '    {',
            '      name: \'content\',',
            '      title: \'Content\',',
            '      type: \'array\',',
            '      of: [',
            '        {',
            '          type: \'block\',',
            '          marks: {',
            '            annotations: [',
            '              {',
            '                name: \'link\',',
            '                type: \'object\',',
            '                title: \'Link\',',
            '                fields: [',
            '                  {',
            '                    name: \'href\',',
            '                    type: \'string\',',
            '                    title: \'URL\'',
            '                  }',
            '                ]',
            '              }',
            '            ]',
            '          }',
            '        },',
            '        {type: \'imageBlock\'},',  // v1.3: Image with migration metadata
            '        {type: \'codeBlock\'},',  // v1.3.1: Semantic types
            '        {type: \'accordionBlock\'},',  // v1.3.1
            '        {type: \'tableBlock\'},',  // v1.3.1
            '        {type: \'separator\'},',
            '        {type: \'customButton\'},',
            '        {type: \'accordion\'},',  // Legacy - keep for backward compatibility
            '        {type: \'rawHtml\'}',  // Fallback
            '      ]',
            '    },',
            '    {',
            '      name: \'publishedAt\',',
            '      title: \'Published At\',',
            '      type: \'datetime\'',
            '    }',
            '  ],',
            '  preview: {',
            '    select: {',
            '      title: \'title\',',
            '      subtitle: \'slug.current\'',
            '    }',
            '  }',
            '}',
        );
        
        return implode("\n", $lines) . "\n";
    }
    
    /**
     * Generate separator block schema
     * 
     * @param string $schema_dir Schema directory path
     * @return string Path to generated file
     */
    private function generate_separator_schema($schema_dir) {
        $schema = $this->build_separator_schema_content();
        $file_path = $schema_dir . 'separator.js';
        $this->write_schema_file($file_path, $schema);
        return $file_path;
    }
    
    /**
     * Build separator schema content
     * 
     * @return string JavaScript schema definition
     */
    private function build_separator_schema_content() {
        $lines = array(
            'export default {',
            '  name: \'separator\',',
            '  title: \'Separator\',',
            '  type: \'object\',',
            '  fields: [',
            '    {',
            '      name: \'style\',',
            '      title: \'Style\',',
            '      type: \'string\',',
            '      options: {',
            '        list: [',
            '          {title: \'Default\', value: \'default\'},',
            '          {title: \'Wide\', value: \'wide\'},',
            '          {title: \'Dots\', value: \'dots\'}',
            '        ]',
            '      }',
            '    }',
            '  ],',
            '  preview: {',
            '    prepare() {',
            '      return {',
            '        title: \'--- Separator ---\'',
            '      }',
            '    }',
            '  }',
            '}',
        );
        
        return implode("\n", $lines) . "\n";
    }
    
    /**
     * Generate custom button block schema
     * 
     * @param string $schema_dir Schema directory path
     * @return string Path to generated file
     */
    private function generate_custom_button_schema($schema_dir) {
        $schema = $this->build_custom_button_schema_content();
        $file_path = $schema_dir . 'customButton.js';
        $this->write_schema_file($file_path, $schema);
        return $file_path;
    }
    
    /**
     * Build custom button schema content
     * 
     * @return string JavaScript schema definition
     */
    private function build_custom_button_schema_content() {
        $lines = array(
            'export default {',
            '  name: \'customButton\',',
            '  title: \'Button\',',
            '  type: \'object\',',
            '  fields: [',
            '    {',
            '      name: \'text\',',
            '      title: \'Button Text\',',
            '      type: \'string\',',
            '      validation: Rule => Rule.required()',
            '    },',
            '    {',
            '      name: \'url\',',
            '      title: \'URL\',',
            '      type: \'url\'',
            '    },',
            '    {',
            '      name: \'style\',',
            '      title: \'Style\',',
            '      type: \'string\',',
            '      options: {',
            '        list: [',
            '          {title: \'Primary\', value: \'primary\'},',
            '          {title: \'Secondary\', value: \'secondary\'},',
            '          {title: \'Outline\', value: \'outline\'}',
            '        ]',
            '      }',
            '    }',
            '  ]',
            '}',
        );
        
        return implode("\n", $lines) . "\n";
    }
    
    /**
     * Generate accordion block schema
     * 
     * @param string $schema_dir Schema directory path
     * @return string Path to generated file
     */
    private function generate_accordion_schema($schema_dir) {
        $schema = $this->build_accordion_schema_content();
        $file_path = $schema_dir . 'accordion.js';
        $this->write_schema_file($file_path, $schema);
        return $file_path;
    }

    /**
     * Generate imageBlock schema (v1.3 Option C)
     *
     * @param string $schema_dir Schema directory path
     * @return string Path to generated file
     */
    private function generate_image_block_schema($schema_dir) {
        $schema = $this->build_image_block_schema_content();
        $file_path = $schema_dir . 'imageBlock.js';
        $this->write_schema_file($file_path, $schema);
        return $file_path;
    }

    /**
     * Build imageBlock schema content (v1.3 Option C - Sanity Compliant)
     *
     * @return string JavaScript schema definition
     */
    private function build_image_block_schema_content() {
        $lines = array(
            '// v1.3 Option C (Hybrid): Supports Phase 1 (external URL) and Phase 2 (Sanity asset)',
            '',
            'export default {',
            '  name: \'imageBlock\',',
            '  title: \'Image Block\',',
            '  type: \'object\',',
            '  fields: [',
            '    {',
            '      name: \'url\',',
            '      title: \'Image URL\',',
            '      type: \'url\',',
            '      description: \'External URL (Phase 1) or fallback\',',
            '      validation: Rule => Rule.required()',
            '    },',
            '    {',
            '      name: \'asset\',',
            '      title: \'Sanity Asset\',',
            '      type: \'image\',',  // FIXED: use 'image' not 'reference'
            '      description: \'Sanity CDN asset (Phase 2) - populated after migration\'',
            '    },',
            '    {',
            '      name: \'alt\',',
            '      title: \'Alt Text\',',
            '      type: \'string\',',
            '      description: \'Alternative text for accessibility\',',
            '    },',
            '    {',
            '      name: \'width\',',
            '      title: \'Width\',',
            '      type: \'number\',',
            '      description: \'Image width in pixels\'',
            '    },',
            '    {',
            '      name: \'height\',',
            '      title: \'Height\',',
            '      type: \'number\',',
            '      description: \'Image height in pixels\'',
            '    },',
            '    {',
            '      name: \'link\',',
            '      title: \'Link\',',
            '      type: \'object\',',
            '      description: \'Optional link wrapping the image\',',
            '      fields: [',
            '        {',
            '          name: \'href\',',
            '          title: \'URL\',',
            '          type: \'string\'',
            '        },',
            '        {',
            '          name: \'target\',',
            '          title: \'Target\',',
            '          type: \'string\',',
            '          options: {',
            '            list: [',
            '              {title: \'Same Window\', value: \'_self\'},',
            '              {title: \'New Window\', value: \'_blank\'}',
            '            ]',
            '          }',
            '        }',
            '      ]',
            '    },',
            '    {',
            '      name: \'assetId\',',  // FIXED: no underscore
            '      title: \'Asset ID\',',
            '      type: \'string\',',
            '      description: \'Stable identifier for queries and Phase 2 migration\',',
            '      readOnly: true,',
            '      hidden: true',
            '    },',
            '    {',
            '      name: \'migration\',',  // FIXED: no underscore
            '      title: \'Migration Metadata\',',
            '      type: \'object\',',
            '      description: \'Trustworthy metadata for Phase 2 migration scripts\',',
            '      readOnly: true,',
            '      hidden: true,',
            '      fields: [',
            '        {',
            '          name: \'phase\',',
            '          title: \'Migration Phase\',',
            '          type: \'number\',',
            '          description: \'1 = External URL, 2 = Sanity CDN\'',
            '        },',
            '        {',
            '          name: \'placeholderRef\',',
            '          title: \'Placeholder Reference\',',
            '          type: \'string\',',
            '          description: \'Temporary reference for Phase 2 replacement\'',
            '        },',
            '        {',
            '          name: \'sourceUrl\',',
            '          title: \'Source URL\',',
            '          type: \'string\',',
            '          description: \'Original WordPress URL (reference)\'',
            '        },',
            '        {',
            '          name: \'uploaded\',',
            '          title: \'Upload Status\',',
            '          type: \'boolean\',',
            '          description: \'Has this been uploaded to Sanity CDN?\'',
            '        },',
            '        {',
            '          name: \'sanityAssetId\',',
            '          title: \'Sanity Asset ID\',',
            '          type: \'string\',',
            '          description: \'Actual Sanity asset ID after upload\'',
            '        },',
            '        {',
            '          name: \'uploadedAt\',',
            '          title: \'Upload Timestamp\',',
            '          type: \'datetime\',',
            '          description: \'When was this uploaded?\'',
            '        }',
            '      ]',
            '    }',
            '  ],',
            '  preview: {',
            '    select: {',
            '      asset: \'asset\',',
            '      url: \'url\',',
            '      alt: \'alt\',',
            '      width: \'width\',',
            '      height: \'height\',',
            '      phase: \'migration.phase\',',  // FIXED: no underscore
            '      uploaded: \'migration.uploaded\'',  // FIXED: no underscore
            '    },',
            '    prepare({asset, url, alt, width, height, phase, uploaded}) {',
            '      // Determine migration status',
            '      let status = \'⚠️ Phase 1 (External URL)\';',
            '      if (asset || phase === 2) {',
            '        status = \'✅ Phase 2 (Sanity CDN)\';',
            '      }',
            '      ',
            '      // Build subtitle with dimensions and status',
            '      const dimensions = width && height ? `${width}×${height}` : \'Unknown size\';',
            '      const subtitle = `${dimensions} • ${status}`;',
            '      ',
            '      return {',
            '        title: alt || \'Image\',',
            '        subtitle,',
            '        // Show Sanity asset if available, otherwise external URL',
            '        media: asset || {imageUrl: url}',
            '      }',
            '    }',
            '  }',
            '}',
        );

        return implode("\n", $lines) . "\n";
    }
    
    /**
     * Build accordion schema content
     * 
     * @return string JavaScript schema definition
     */
    private function build_accordion_schema_content() {
        $lines = array(
            'export default {',
            '  name: \'accordion\',',
            '  title: \'Accordion\',',
            '  type: \'object\',',
            '  fields: [',
            '    {',
            '      name: \'items\',',
            '      title: \'Accordion Items\',',
            '      type: \'array\',',
            '      of: [',
            '        {',
            '          type: \'object\',',
            '          fields: [',
            '            {',
            '              name: \'title\',',
            '              title: \'Title\',',
            '              type: \'string\',',
            '              validation: Rule => Rule.required()',
            '            },',
            '            {',
            '              name: \'body\',',
            '              title: \'Content\',',
            '              type: \'text\',',
            '              rows: 5',
            '            }',
            '          ]',
            '        }',
            '      ]',
            '    }',
            '  ],',
            '  preview: {',
            '    select: {',
            '      items: \'items\'',
            '    },',
            '    prepare({items}) {',
            '      return {',
            '        title: `Accordion (${items?.length || 0} items)`',
            '      }',
            '    }',
            '  }',
            '}',
        );
        
        return implode("\n", $lines) . "\n";
    }
    
    /**
     * Generate raw HTML fallback schema
     * 
     * @param string $schema_dir Schema directory path
     * @return string Path to generated file
     */
    private function generate_raw_html_schema($schema_dir) {
        $schema = $this->build_raw_html_schema_content();
        $file_path = $schema_dir . 'rawHtml.js';
        $this->write_schema_file($file_path, $schema);
        return $file_path;
    }
    
    /**
     * Build raw HTML schema content
     * 
     * @return string JavaScript schema definition
     */
    private function build_raw_html_schema_content() {
        $lines = array(
            'export default {',
            '  name: \'rawHtml\',',
            '  title: \'Raw HTML\',',
            '  type: \'object\',',
            '  fields: [',
            '    {',
            '      name: \'html\',',
            '      title: \'HTML Content\',',
            '      type: \'text\',',
            '      rows: 10',
            '    },',
            '    {',
            '      name: \'blockType\',',
            '      title: \'Original Block Type\',',
            '      type: \'string\',',
            '      readOnly: true',
            '    }',
            '  ],',
            '  preview: {',
            '    select: {',
            '      blockType: \'blockType\'',
            '    },',
            '    prepare({blockType}) {',
            '      return {',
            '        title: `Raw HTML Block${blockType ? ` (${blockType})` : \'\'}`',
            '      }',
            '    }',
            '  }',
            '}',
        );
        
        return implode("\n", $lines) . "\n";
    }
    
    /**
     * Generate index.ts export file
     * 
     * @param string $schema_dir Schema directory path
     * @return string Path to generated file
     */
    private function generate_index_file($schema_dir) {
        $schema = $this->build_index_content();
        $file_path = $schema_dir . 'index.ts';
        $this->write_schema_file($file_path, $schema);
        return $file_path;
    }

    /**
     * Generate codeBlock schema (v1.3.1 - Semantic type for all code patterns)
     *
     * @param string $schema_dir Schema directory path
     * @return string Path to generated file
     */
    private function generate_code_block_schema($schema_dir) {
        $schema = $this->build_code_block_schema_content();
        $file_path = $schema_dir . 'codeBlock.js';
        $this->write_schema_file($file_path, $schema);
        return $file_path;
    }

    /**
     * Build codeBlock schema content
     *
     * @return string JavaScript schema definition
     */
    private function build_code_block_schema_content() {
        $lines = array(
            '// codeBlock.js - Semantic type for all code patterns',
            '// Handles: code_block_pro, wp/code, prism, highlight.js, etc.',
            '',
            'export default {',
            '  name: \'codeBlock\',',
            '  title: \'Code Block\',',
            '  type: \'object\',',
            '  fields: [',
            '    {',
            '      name: \'code\',',
            '      title: \'Code\',',
            '      type: \'text\',',
            '      description: \'The source code\',',
            '      validation: Rule => Rule.required(),',
            '      rows: 10',
            '    },',
            '    {',
            '      name: \'language\',',
            '      title: \'Language\',',
            '      type: \'string\',',
            '      description: \'Programming language for syntax highlighting\',',
            '      options: {',
            '        list: [',
            '          {title: \'JavaScript\', value: \'javascript\'},',
            '          {title: \'TypeScript\', value: \'typescript\'},',
            '          {title: \'Python\', value: \'python\'},',
            '          {title: \'PHP\', value: \'php\'},',
            '          {title: \'HTML\', value: \'html\'},',
            '          {title: \'CSS\', value: \'css\'},',
            '          {title: \'Bash\', value: \'bash\'},',
            '          {title: \'JSON\', value: \'json\'},',
            '          {title: \'SQL\', value: \'sql\'},',
            '          {title: \'Java\', value: \'java\'},',
            '          {title: \'C++\', value: \'cpp\'},',
            '          {title: \'C#\', value: \'csharp\'},',
            '          {title: \'Go\', value: \'go\'},',
            '          {title: \'Ruby\', value: \'ruby\'},',
            '          {title: \'Rust\', value: \'rust\'},',
            '          {title: \'Swift\', value: \'swift\'},',
            '          {title: \'Kotlin\', value: \'kotlin\'},',
            '          {title: \'Plain Text\', value: \'text\'}',
            '        ]',
            '      }',
            '    },',
            '    {',
            '      name: \'filename\',',
            '      title: \'Filename\',',
            '      type: \'string\',',
            '      description: \'Optional filename to display (e.g., "index.js")\'',
            '    },',
            '    {',
            '      name: \'lineNumbers\',',
            '      title: \'Show Line Numbers\',',
            '      type: \'boolean\',',
            '      description: \'Display line numbers in code block\',',
            '      initialValue: false',
            '    },',
            '    {',
            '      name: \'highlightLines\',',
            '      title: \'Highlighted Lines\',',
            '      type: \'string\',',
            '      description: \'Lines to highlight (e.g., "1-3,5,7-9")\'',
            '    },',
            '    {',
            '      name: \'blockType\',',
            '      title: \'Source Pattern\',',
            '      type: \'string\',',
            '      description: \'Original WordPress pattern (e.g., "code_block_pro")\',',
            '      readOnly: true,',
            '      hidden: true',
            '    }',
            '  ],',
            '  preview: {',
            '    select: {',
            '      code: \'code\',',
            '      language: \'language\',',
            '      filename: \'filename\',',
            '      blockType: \'blockType\'',
            '    },',
            '    prepare({code, language, filename, blockType}) {',
            '      const firstLine = code ? code.split(\'\\n\')[0].substring(0, 50) : \'Empty code block\';',
            '      const lang = language || \'text\';',
            '      const source = blockType ? ` (${blockType})` : \'\';',
            '      ',
            '      return {',
            '        title: filename || `${lang} code${source}`,',
            '        subtitle: firstLine + (code && code.length > 50 ? \'...\' : \'\'),',
            '        media: () => \'💻\'',
            '      }',
            '    }',
            '  }',
            '}',
        );

        return implode("\n", $lines) . "\n";
    }
    
    /**
     * Generate accordionBlock schema (v1.3.1 - Semantic type for accordions AND tabs)
     *
     * @param string $schema_dir Schema directory path
     * @return string Path to generated file
     */
    private function generate_accordion_block_schema($schema_dir) {
        $schema = $this->build_accordion_block_schema_content();
        $file_path = $schema_dir . 'accordionBlock.js';
        $this->write_schema_file($file_path, $schema);
        return $file_path;
    }

    /**
     * Build accordionBlock schema content
     *
     * FIXED v2.1.1: Changed body field from 'text' to 'array' for Portable Text support
     *
     * @return string JavaScript schema definition
     */
    private function build_accordion_block_schema_content() {
        $lines = array(
            '// accordionBlock.js - Semantic type for all accordion/tab patterns',
            '// Handles: kadence_accordion, kadence_tabs, wp/details, custom accordions/tabs',
            '// v2.1.1: body field now supports Portable Text blocks (rich content)',
            '',
            'export default {',
            '  name: \'accordionBlock\',',
            '  title: \'Accordion Block\',',
            '  type: \'object\',',
            '  fields: [',
            '    {',
            '      name: \'items\',',
            '      title: \'Accordion Items\',',
            '      type: \'array\',',
            '      description: \'Expandable accordion items (also handles tabs)\',',
            '      validation: Rule => Rule.required().min(1),',
            '      of: [',
            '        {',
            '          type: \'object\',',
            '          fields: [',
            '            {',
            '              name: \'title\',',
            '              title: \'Title\',',
            '              type: \'string\',',
            '              description: \'Accordion title / tab label\',',
            '              validation: Rule => Rule.required()',
            '            },',
            '            {',
            '              name: \'body\',',
            '              title: \'Content\',',
            '              type: \'array\',',
            '              description: \'Accordion content / tab panel (Portable Text)\',',
            '              of: [{type: \'block\'}],',
            '              validation: Rule => Rule.required()',
            '            },',
            '            {',
            '              name: \'expanded\',',
            '              title: \'Initially Expanded\',',
            '              type: \'boolean\',',
            '              description: \'Should this item be expanded by default?\',',
            '              initialValue: false',
            '            }',
            '          ],',
            '          preview: {',
            '            select: {',
            '              title: \'title\',',
            '              body: \'body\'',
            '            },',
            '            prepare({title, body}) {',
            '              // Extract text from Portable Text blocks for preview',
            '              let preview = \'No content\';',
            '              if (body && Array.isArray(body) && body.length > 0) {',
            '                const firstBlock = body[0];',
            '                if (firstBlock.children && firstBlock.children.length > 0) {',
            '                  const text = firstBlock.children.map(child => child.text || \'\').join(\'\');',
            '                  preview = text.substring(0, 60);',
            '                }',
            '              }',
            '              return {',
            '                title: title || \'Untitled item\',',
            '                subtitle: preview + (preview.length >= 60 ? \'...\' : \'\')',
            '              }',
            '            }',
            '          }',
            '        }',
            '      ]',
            '    },',
            '    {',
            '      name: \'renderStyle\',',
            '      title: \'Render Style\',',
            '      type: \'string\',',
            '      description: \'How to display this block in frontend\',',
            '      options: {',
            '        list: [',
            '          {title: \'Accordion (expand/collapse)\', value: \'accordion\'},',
            '          {title: \'Tabs (horizontal navigation)\', value: \'tabs\'},',
            '          {title: \'Auto (detect from source)\', value: \'auto\'}',
            '        ]',
            '      },',
            '      initialValue: \'auto\'',
            '    },',
            '    {',
            '      name: \'allowMultiple\',',
            '      title: \'Allow Multiple Open\',',
            '      type: \'boolean\',',
            '      description: \'Allow multiple items to be expanded at once\',',
            '      initialValue: false',
            '    },',
            '    {',
            '      name: \'blockType\',',
            '      title: \'Source Pattern\',',
            '      type: \'string\',',
            '      description: \'Original WordPress pattern (e.g., "kadence_accordion", "kadence_tabs")\',',
            '      readOnly: true,',
            '      hidden: true',
            '    }',
            '  ],',
            '  preview: {',
            '    select: {',
            '      items: \'items\',',
            '      renderStyle: \'renderStyle\',',
            '      blockType: \'blockType\'',
            '    },',
            '    prepare({items, renderStyle, blockType}) {',
            '      const count = items?.length || 0;',
            '      const style = renderStyle === \'tabs\' ? \'Tabs\' : \'Accordion\';',
            '      const source = blockType ? ` (${blockType})` : \'\';',
            '      ',
            '      return {',
            '        title: `${style}${source}`,',
            '        subtitle: `${count} item${count !== 1 ? \'s\' : \'\'}`,',
            '        media: renderStyle === \'tabs\' ? () => \'📑\' : () => \'📋\'',
            '      }',
            '    }',
            '  }',
            '}',
        );

        return implode("\n", $lines) . "\n";
    }
    
    /**
     * Generate tableBlock schema (v1.3.1 - Semantic type for all table patterns)
     *
     * @param string $schema_dir Schema directory path
     * @return string Path to generated file
     */
    private function generate_table_block_schema($schema_dir) {
        $schema = $this->build_table_block_schema_content();
        $file_path = $schema_dir . 'tableBlock.js';
        $this->write_schema_file($file_path, $schema);
        return $file_path;
    }

    /**
     * Build tableBlock schema content
     *
     * @return string JavaScript schema definition
     */
    private function build_table_block_schema_content() {
        $lines = array(
            '// tableBlock.js - Semantic type for all table patterns',
            '// Handles: kadence_table, wp/table, tablepress, custom tables',
            '',
            'export default {',
            '  name: \'tableBlock\',',
            '  title: \'Table Block\',',
            '  type: \'object\',',
            '  fields: [',
            '    {',
            '      name: \'headers\',',
            '      title: \'Column Headers\',',
            '      type: \'array\',',
            '      description: \'Table column headers\',',
            '      of: [{type: \'string\'}],',
            '      validation: Rule => Rule.required().min(1)',
            '    },',
            '    {',
            '      name: \'rows\',',
            '      title: \'Table Rows\',',
            '      type: \'array\',',
            '      description: \'Table data rows\',',
            '      validation: Rule => Rule.required().min(1),',
            '      of: [',
            '        {',
            '          type: \'object\',',
            '          fields: [',
            '            {',
            '              name: \'cells\',',
            '              title: \'Cells\',',
            '              type: \'array\',',
            '              description: \'Cell values for this row\',',
            '              of: [{type: \'string\'}],',
            '              validation: Rule => Rule.required()',
            '            }',
            '          ],',
            '          preview: {',
            '            select: {',
            '              cells: \'cells\'',
            '            },',
            '            prepare({cells}) {',
            '              const preview = cells?.join(\' | \') || \'Empty row\';',
            '              return {',
            '                title: preview.substring(0, 60) + (preview.length > 60 ? \'...\' : \'\')',
            '              }',
            '            }',
            '          }',
            '        }',
            '      ]',
            '    },',
            '    {',
            '      name: \'caption\',',
            '      title: \'Table Caption\',',
            '      type: \'string\',',
            '      description: \'Optional table caption\'',
            '    },',
            '    {',
            '      name: \'hasHeaderRow\',',
            '      title: \'Has Header Row\',',
            '      type: \'boolean\',',
            '      description: \'First row contains headers\',',
            '      initialValue: true',
            '    },',
            '    {',
            '      name: \'hasFooter\',',
            '      title: \'Has Footer Row\',',
            '      type: \'boolean\',',
            '      description: \'Last row is a footer (totals, etc.)\',',
            '      initialValue: false',
            '    },',
            '    {',
            '      name: \'striped\',',
            '      title: \'Striped Rows\',',
            '      type: \'boolean\',',
            '      description: \'Alternate row background colors\',',
            '      initialValue: false',
            '    },',
            '    {',
            '      name: \'blockType\',',
            '      title: \'Source Pattern\',',
            '      type: \'string\',',
            '      description: \'Original WordPress pattern (e.g., "kadence_table", "tablepress")\',',
            '      readOnly: true,',
            '      hidden: true',
            '    }',
            '  ],',
            '  preview: {',
            '    select: {',
            '      headers: \'headers\',',
            '      rows: \'rows\',',
            '      caption: \'caption\',',
            '      blockType: \'blockType\'',
            '    },',
            '    prepare({headers, rows, caption, blockType}) {',
            '      const cols = headers?.length || 0;',
            '      const rowCount = rows?.length || 0;',
            '      const source = blockType ? ` (${blockType})` : \'\';',
            '      ',
            '      return {',
            '        title: caption || `Table${source}`,',
            '        subtitle: `${cols} column${cols !== 1 ? \'s\' : \'\'} × ${rowCount} row${rowCount !== 1 ? \'s\' : \'\'}`,',
            '        media: () => \'📊\'',
            '      }',
            '    }',
            '  }',
            '}',
        );

        return implode("\n", $lines) . "\n";
    }
    
    /**
     * Build index.ts content (v1.3.1: includes semantic block types)
     * 
     * @return string JavaScript exports
     */
    private function build_index_content() {
        $lines = array(
            '// Auto-generated Sanity schema types',
            '// Import in sanity.config.ts: import {schemaTypes} from \'./schemas\'',
            '',
            'import page from \'./page\'',
            'import separator from \'./separator\'',
            'import customButton from \'./customButton\'',
            'import imageBlock from \'./imageBlock\'',  // v1.3
            'import accordion from \'./accordion\'',
            'import rawHtml from \'./rawHtml\'',
            'import codeBlock from \'./codeBlock\'',  // v1.3.1: Semantic types
            'import accordionBlock from \'./accordionBlock\'',  // v1.3.1
            'import tableBlock from \'./tableBlock\'',  // v1.3.1
            '',
            'export const schemaTypes = [',
            '  page,',
            '  separator,',
            '  customButton,',
            '  imageBlock,',  // v1.3
            '  accordion,',
            '  rawHtml,',
            '  codeBlock,',  // v1.3.1: Semantic types
            '  accordionBlock,',  // v1.3.1
            '  tableBlock',  // v1.3.1
            ']',
            '',
            'export default schemaTypes',
        );
        
        return implode("\n", $lines) . "\n";
    }
}
