<?php
/**
 * Sanity Exporter - v1.3
 *
 * Generates export packages for Sanity import.
 * v1.3 Changes:
 * - Clean asset manifest (rsync paths, no assets in ZIP)
 * - Priority calculation based on usage
 * - Simplified Phase 1 output (external URLs only)
 *
 * @package STCWHeadlessAssistant
 * @since 1.3.0
 */

namespace STCW\Headless\Engine\Target\Sanity;

use STCW\Headless\Core;

if (!defined('ABSPATH')) exit;

class Exporter {
    
    /**
     * Create export package from converted data
     *
     * @param array $pages Converted pages
     * @param array $options Export options (unused, for compatibility)
     * @return array Result with success status and file paths
     */
    public function create_export_package($pages, $options = []) {
        // Get export directory
        $export_base_dir = \STCW\Headless\Core::get_export_dir();
        
        $timestamp = gmdate('Y-m-d-H-i-s');
        $package_dir = trailingslashit($export_base_dir) . 'sanity-export-' . $timestamp . '/';
        
        // Create package directory
        if (!is_dir($package_dir)) {
            wp_mkdir_p($package_dir);
        }
        
        // Generate export files
        $ndjson_file = $this->generate_ndjson($pages, $package_dir);
        $manifest_file = $this->generate_asset_manifest($pages, $package_dir);
        $readme_file = $this->generate_readme($package_dir, count($pages));
        
        // Generate schemas
        $schema_generator = new SchemaGenerator();
        $schema_result = $schema_generator->generate_schemas($package_dir);
        
        // Create ZIP package
        $zip_file = $this->create_zip_package($package_dir, $timestamp, $export_base_dir);
        
        return [
            'success' => true,
            'package_dir' => $package_dir,
            'zip_file' => $zip_file,
            'files' => [
                'ndjson' => $ndjson_file,
                'manifest' => $manifest_file,
                'readme' => $readme_file,
                'schemas' => $schema_result['files'],
            ],
        ];
    }
    
    /**
     * Generate NDJSON file (newline-delimited JSON)
     * 
     * @param array $pages Converted pages
     * @param string $package_dir Package directory
     * @return string Path to generated file
     */
    private function generate_ndjson($pages, $package_dir) {
        $ndjson = '';
        $seen_ids = [];
        
        foreach ($pages as $page) {
            // Skip duplicates based on _id
            $page_id = $page['_id'] ?? null;
            
            if ($page_id && isset($seen_ids[$page_id])) {
		// Duplicate ID detected - skip this page
		if (defined('WP_CLI') && WP_CLI) {
    			\WP_CLI::debug("Skipping duplicate page ID: {$page_id}");
		}
                continue;
            }
            
            // Mark as seen
            if ($page_id) {
                $seen_ids[$page_id] = true;
            }
            
            $ndjson .= wp_json_encode($page, JSON_UNESCAPED_SLASHES) . "\n";
        }
        
        $file_path = $package_dir . 'data.ndjson';
        
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- CLI context, export file
        file_put_contents($file_path, $ndjson);
        
        return $file_path;
    }
    
    /**
     * Generate asset manifest JSON (v1.3: rsync-ready)
     * 
     * @param array $pages Converted pages
     * @param string $package_dir Package directory
     * @return string Path to generated file
     */
    private function generate_asset_manifest($pages, $package_dir) {
        // Collect all asset references from pages
        $all_assets = [];
        $usage_map = []; // Track which pages use each asset
        
        foreach ($pages as $page) {
            $page_id = $page['_id'];
            
            if (isset($page['_assetReferences'])) {
                foreach ($page['_assetReferences'] as $ref) {
                    $asset_id = $ref['assetId'];
                    
                    // Track usage
                    if (!isset($usage_map[$asset_id])) {
                        $usage_map[$asset_id] = [
                            'documentIds' => [],
                            'count' => 0,
                        ];
                    }
                    
                    $usage_map[$asset_id]['documentIds'][] = $page_id;
                    $usage_map[$asset_id]['count']++;
                    
                    // Store asset (will dedupe later)
                    $all_assets[$asset_id] = $ref;
                }
            }
        }
        
        // Build final asset list with usage data
        $assets = [];
        foreach ($all_assets as $asset_id => $ref) {
            $usage = $usage_map[$asset_id] ?? ['documentIds' => [], 'count' => 0];
            
            // Calculate priority (deterministic: based on usage count)
            $priority = $this->calculate_priority($usage['count']);
            
            // Get file metadata
            $file_info = $this->get_file_info($ref['source']['original']);
            
            $assets[] = [
                'assetId' => $asset_id,
                'type' => $ref['type'],
                'source' => [
                    'relativePath' => $this->get_relative_stcw_path($ref['source']['original']),
                    'absolutePath' => $this->get_absolute_stcw_path($ref['source']['original']),
                    'url' => $ref['source']['url'],
                ],
                'metadata' => array_merge(
                    $ref['metadata'],
                    $file_info
                ),
                'usage' => [
                    'documentIds' => array_unique($usage['documentIds']),
                    'totalReferences' => $usage['count'],
                    'priority' => $priority,
                ],
            ];
        }
        
        // Sort by priority (high to low)
        usort($assets, function($a, $b) {
            $priority_order = ['high' => 3, 'medium' => 2, 'low' => 1];
            return $priority_order[$b['usage']['priority']] <=> $priority_order[$a['usage']['priority']];
        });
        
        // Build manifest
        $manifest = [
            'version' => '2.0',
            'generatedAt' => gmdate('c'),
            'wordpress' => [
                'siteUrl' => home_url(),
                'assetsPath' => $this->get_stcw_assets_path(),
                'totalAssets' => count($assets),
                'totalSize' => $this->format_bytes($this->calculate_total_size($assets)),
            ],
            'assets' => $assets,
            'instructions' => [
                'phase1' => 'Import data.ndjson immediately - images will load from WordPress server URLs',
                'phase2' => 'Use rsync to copy assets from WordPress server, then upload to your CMS',
                'example' => 'rsync -avz --progress user@' . wp_parse_url(home_url(), PHP_URL_HOST) . ':' . $this->get_stcw_assets_path() . '/ ./assets/',
            ],
        ];
        
        $file_path = $package_dir . 'asset-manifest.json';
        
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- CLI context, export file
        file_put_contents(
            $file_path,
            wp_json_encode($manifest, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)
        );
        
        return $file_path;
    }
    
    /**
     * Calculate asset priority based on usage count
     *
     * Deterministic calculation:
     * - High: Used 3+ times
     * - Medium: Used 2 times
     * - Low: Used 1 time
     *
     * @param int $usage_count Number of times asset is referenced
     * @return string Priority level
     */
    private function calculate_priority($usage_count) {
        if ($usage_count >= 3) {
            return 'high';
        } elseif ($usage_count >= 2) {
            return 'medium';
        } else {
            return 'low';
        }
    }
    
    /**
     * Get file metadata (size, format, dimensions if image)
     *
     * @param string $src Original source path
     * @return array File metadata
     */
    private function get_file_info($src) {
        $absolute_path = $this->get_absolute_stcw_path($src);
        
        $info = [
            'format' => strtolower(pathinfo($src, PATHINFO_EXTENSION)),
            'fileSize' => file_exists($absolute_path) ? filesize($absolute_path) : 0,
        ];
        
        // Get image dimensions if available
        if (file_exists($absolute_path) && in_array($info['format'], ['jpg', 'jpeg', 'png', 'gif', 'webp'], true)) {
            $image_size = @getimagesize($absolute_path);
            if ($image_size) {
                $info['dimensions'] = [
                    'width' => $image_size[0],
                    'height' => $image_size[1],
                ];
            }
        }
        
        return $info;
    }
    
    /**
     * Get relative path in stcw-assets directory
     *
     * @param string $src Original source path
     * @return string Relative path
     */
    private function get_relative_stcw_path($src) {
        // Extract filename from various path formats
        if (strpos($src, 'assets/') === 0) {
            return str_replace('assets/', '', $src);
        }
        
        return basename($src);
    }
    
    /**
     * Get absolute filesystem path to asset in stcw-assets
     *
     * @param string $src Original source path
     * @return string Absolute path
     */
    private function get_absolute_stcw_path($src) {
        $relative = $this->get_relative_stcw_path($src);
        return $this->get_stcw_assets_path() . '/' . $relative;
    }
    
    /**
     * Get stcw-assets directory path
     *
     * @return string Absolute path to stcw-assets
     */
    private function get_stcw_assets_path() {
        return WP_CONTENT_DIR . '/uploads/stcw-assets';
    }
    
    /**
     * Calculate total size of all assets
     *
     * @param array $assets Asset array
     * @return int Total size in bytes
     */
    private function calculate_total_size($assets) {
        $total = 0;
        foreach ($assets as $asset) {
            $total += $asset['metadata']['fileSize'] ?? 0;
        }
        return $total;
    }
    
    /**
     * Format bytes to human-readable size
     *
     * @param int $bytes Size in bytes
     * @return string Formatted size
     */
    private function format_bytes($bytes) {
        $units = ['B', 'KB', 'MB', 'GB'];
        $factor = floor((strlen($bytes) - 1) / 3);
        return sprintf("%.1f %s", $bytes / pow(1024, $factor), $units[$factor]);
    }
    
    /**
     * Generate README file
     * 
     * @param string $package_dir Package directory
     * @param int $page_count Number of pages
     * @return string Path to generated file
     */
    private function generate_readme($package_dir, $page_count) {
        $site_url = home_url();
        $domain = wp_parse_url($site_url, PHP_URL_HOST);
        
        $lines = [
            '# Sanity Import Package',
            '',
            'Generated by Static Cache Wrangler – Headless Assistant',
            'Date: ' . gmdate('Y-m-d H:i:s') . ' UTC',
            'Pages: ' . $page_count,
            '',
            '## Phase 1: Import Content (5 minutes)',
            '',
            'Content is ready to import immediately. Images load from WordPress server.',
            '',
            '```bash',
            'sanity dataset import data.ndjson production',
            '```',
            '',
            '✅ **Done!** Content is live in Sanity Studio.',
            '⚠️ Images load from: `' . $site_url . '/wp-content/uploads/stcw-assets/`',
            '',
            '---',
            '',
            '## Phase 2: Migrate Assets (Optional)',
            '',
            'Assets remain on WordPress server. You have options:',
            '',
            '### Option A: Keep External URLs',
            '- **Effort:** None',
            '- **Good for:** Small sites, temporary migrations',
            '- **Note:** WordPress server must stay online',
            '',
            '### Option B: Upload to Sanity CDN',
            '',
            '**Step 1: Copy Assets from WordPress Server**',
            '',
            '```bash',
            '# Use rsync to copy (can take hours for large sites)',
            'rsync -avz --progress \\',
            '  user@' . $domain . ':' . $this->get_stcw_assets_path() . '/ \\',
            '  ./assets/',
            '',
            '# Tip: Use screen for long transfers',
            'screen -S asset-sync',
            'rsync -avz --progress ...',
            '# Detach: Ctrl+A, D',
            '```',
            '',
            '**Step 2: Upload to Sanity**',
            '',
            'See `asset-manifest.json` for full asset inventory with priorities.',
            '',
            '```bash',
            '# Option 1: Sanity CLI',
            'sanity assets upload assets/',
            '',
            '# Option 2: Custom script',
            '# See example scripts in documentation',
            '```',
            '',
            '**Step 3: Update References**',
            '',
            'After uploading, update document references from external URLs to Sanity assets.',
            '',
            '---',
            '',
            '## Asset Manifest',
            '',
            'See `asset-manifest.json` for:',
            '- rsync paths (absolute + relative)',
            '- Asset metadata (dimensions, file size, format)',
            '- Usage tracking (which documents use each asset)',
            '- Priority hints (high-usage assets listed first)',
            '',
            '---',
            '',
            '## Need Help?',
            '',
            '- Sanity Docs: https://www.sanity.io/docs/importing-data',
            '- Plugin Support: https://github.com/your-org/stcw-headless-assistant',
            '',
        ];
        
        $readme = implode("\n", $lines);
        
        $file_path = $package_dir . 'README.md';
        
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- CLI context, export file
        file_put_contents($file_path, $readme);
        
        return $file_path;
    }
    
    /**
     * Create ZIP package of export
     * 
     * @param string $package_dir Package directory
     * @param string $timestamp Timestamp for filename
     * @param string $export_base_dir Base export directory
     * @return string|false Path to ZIP file or false on failure
     */
    private function create_zip_package($package_dir, $timestamp, $export_base_dir) {
        if (!class_exists('ZipArchive')) {
            return false;
        }
        
        $zip_file = trailingslashit($export_base_dir) . 'sanity-export-' . $timestamp . '.zip';
        
        $zip = new \ZipArchive();
        if ($zip->open($zip_file, \ZipArchive::CREATE | \ZipArchive::OVERWRITE) !== true) {
            return false;
        }
        
        // Add all files recursively (NO asset files, just data + manifests + schemas)
        $files = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($package_dir),
            \RecursiveIteratorIterator::LEAVES_ONLY
        );
        
        foreach ($files as $file) {
            if (!$file->isDir()) {
                $file_path = $file->getRealPath();
                $relative_path = substr($file_path, strlen($package_dir));
                $zip->addFile($file_path, $relative_path);
            }
        }
        
        $zip->close();
        
        return $zip_file;
    }
}
