<?php
/**
 * Sanity Converter - v1.3 (Option C - Hybrid)
 *
 * Converts detected patterns to Sanity Portable Text format.
 * v1.3 Changes:
 * - Image deduplication (per-page)
 * - Simplified asset references (external URLs for Phase 1)
 * - Migration metadata for Phase 2 (trustworthy, queryable)
 * - Clean asset tracking for manifest generation
 *
 * @package STCWHeadlessAssistant
 * @since 1.3.0
 */

namespace STCW\Headless\Engine\Target\Sanity;

if (!defined('ABSPATH')) exit;

class Converter {
    
    /**
     * Asset references collected during conversion
     * @var array
     */
    private $asset_references = [];
    
    /**
     * Track seen images to prevent duplicates (per-page)
     * @var array
     */
    private $seen_images = [];
    
    /**
     * Convert patterns to Sanity Portable Text format
     *
     * @param array $patterns Detected patterns from PatternDetector
     * @param array $metadata Page metadata
     * @return array Sanity Portable Text blocks
     */
    public function convert_patterns_to_sanity($patterns, $metadata = []) {
        $sanity_blocks = [];
        
        // Reset tracking for each page
        $this->asset_references = [];
        $this->seen_images = [];
        
        foreach ($patterns as $pattern_match) {
            $converted = $this->convert_pattern($pattern_match);
            
            // Skip null blocks (deduplicated images)
            if ($converted === null) {
                continue;
            }
            
            // Handle both single blocks and arrays (e.g., lists)
            if (isset($converted[0]) && is_array($converted[0])) {
                $sanity_blocks = array_merge($sanity_blocks, $converted);
            } else {
                $sanity_blocks[] = $converted;
            }
        }
        
        return $sanity_blocks;
    }
    
    /**
     * Convert a single pattern to Portable Text
     *
     * @param array $pattern_match Pattern match data
     * @return array|null Portable Text block or null if deduplicated
     */
    private function convert_pattern($pattern_match) {
        $pattern_name = $pattern_match['pattern'];
        $extracted = $pattern_match['extracted'] ?? [];
        
        if (empty($extracted)) {
            return $this->convert_fallback($pattern_match);
	}

	return $this->to_portable_text($extracted, $pattern_name);
    }
    
    /**
     * Convert extracted content to Portable Text
     * 
     * v1.3.1: Uses semantic types with pattern declarations
     *
     * @param array $extracted Extracted content
     * @return array|null Portable Text block or null if deduplicated
     */
     private function to_portable_text($extracted, $pattern_name = null) {
        $type = $extracted['type'] ?? 'unknown';
    
        // Use passed pattern_name or fall back to type
        if ($pattern_name === null) {
            $pattern_name = $type;
	}  

            // Step 1: Check if pattern declares a Sanity target
            $sanity_target = null;
            if (class_exists('\\STCW\\Headless\\Engine\\Detector\\PatternRegistry')) {
                $sanity_target = \STCW\Headless\Engine\Detector\PatternRegistry::get_sanity_target($pattern_name);
        }
        
        // Step 2: If pattern declared a target, use it
        if ($sanity_target) {
            return $this->convert_to_semantic_type($sanity_target, $extracted, $pattern_name);
        }
        
        // Step 3: Fallback to built-in type mapping
        switch ($type) {
            case 'heading':
                return $this->heading_to_portable_text($extracted);
            
            case 'paragraph':
                return $this->paragraph_to_portable_text($extracted);
            
            case 'list':
                return $this->list_to_portable_text($extracted);
            
            case 'quote':
                return $this->quote_to_portable_text($extracted);
            
            case 'image':
                // v1.3: Deduplication + migration metadata
                return $this->image_to_portable_text($extracted);
            
            case 'button':
                return $this->button_to_portable_text($extracted);
            
            // Legacy accordion (keep for backward compatibility)
            case 'accordion':
                return $this->accordion_to_portable_text($extracted);
            
            default:
                // Smart fallback: try to guess semantic type from extracted data
                return $this->smart_fallback($extracted, $pattern_name);
        }
    }
    
    /**
     * Convert to semantic type based on pattern's declaration
     * 
     * @param string $semantic_type Target semantic type (codeBlock, accordionBlock, etc.)
     * @param array $extracted Extracted data
     * @param string $pattern_name Source pattern name
     * @return array Portable Text block
     */
    private function convert_to_semantic_type($semantic_type, $extracted, $pattern_name) {
        switch ($semantic_type) {
            case 'codeBlock':
                return $this->to_code_block($extracted, $pattern_name);
            
            case 'accordionBlock':
                return $this->to_accordion_block($extracted, $pattern_name);
            
            case 'tableBlock':
                return $this->to_table_block($extracted, $pattern_name);
            
            case 'imageBlock':
                return $this->image_to_portable_text($extracted);
            
            case 'rawHtml':
                return $this->to_raw_html($extracted, $pattern_name);
            
            default:
                // Unknown semantic type - use rawHtml
                return $this->to_raw_html($extracted, $pattern_name);
        }
    }
    
    /**
     * Convert to codeBlock (semantic type for all code patterns)
     * 
     * @param array $extracted Extracted data
     * @param string $pattern_name Source pattern
     * @return array codeBlock
     */
    private function to_code_block($extracted, $pattern_name) {
        return [
            '_type' => 'codeBlock',
            '_key' => $this->generate_key(),
            'code' => $extracted['code'] ?? '',
            'language' => $extracted['language'] ?? 'text',
            'filename' => $extracted['filename'] ?? null,
            'lineNumbers' => $extracted['lineNumbers'] ?? false,
            'highlightLines' => $extracted['highlightLines'] ?? null,
            'blockType' => $pattern_name,  // Preserve source
        ];
    }

    /**
    * Convert to accordionBlock (semantic type for accordions AND tabs)
    *
    * FIXED v2.1.1: Now normalizes content field to handle portable text blocks
    *
    * @param array $extracted Extracted data
    * @param string $pattern_name Source pattern
    * @return array accordionBlock
    */
    private function to_accordion_block($extracted, $pattern_name) {
        // Determine render style from pattern name
        $render_style = 'auto';
        if (strpos($pattern_name, 'tab') !== false) {
            $render_style = 'tabs';
        } elseif (strpos($pattern_name, 'accordion') !== false) {
            $render_style = 'accordion';
        }

        // Normalize items to ensure content is portable text blocks
        $items = $extracted['items'] ?? [];
        $normalized_items = array_map(
            function($item) {
                return [
                    'title' => $item['title'] ?? '',
                    'body' => $this->normalize_content_field($item['content'] ?? ''),
                ];
            },
            $items
        );

         return [
            '_type' => 'accordionBlock',
            '_key' => $this->generate_key(),
            'items' => $normalized_items,
            'renderStyle' => $render_style,
            'allowMultiple' => $extracted['allowMultiple'] ?? false,
            'blockType' => $pattern_name,
        ];
    }

    /**
     * Normalize content field to portable text blocks
     *
     * Handles both string content and block arrays from extractors.
     * Same helper as Generic converter for consistency.
     *
     * @param mixed $content Content from extractor (string or array).
     * @return array Portable text blocks
     */
    private function normalize_content_field($content) {
        // Already portable text blocks - return as-is
        if (is_array($content) && !empty($content)) {
            return $content;
        }

        // String content - wrap in simple paragraph block
        if (is_string($content)) {
            return [
                [
                    '_type' => 'block',
                    'style' => 'normal',
                    'children' => [
                        [
                            '_type' => 'span',
                            'text' => $content,
                            'marks' => [],
                        ],
                    ],
                ],
            ];
        }

        // Fallback for null/empty - return empty array
        return [];
    }
       
    /**
     * Convert to tableBlock (semantic type for all table patterns)
     * 
     * @param array $extracted Extracted data
     * @param string $pattern_name Source pattern
     * @return array tableBlock
     */
    private function to_table_block($extracted, $pattern_name) {
        return [
            '_type' => 'tableBlock',
            '_key' => $this->generate_key(),
            'headers' => $extracted['headers'] ?? [],
            'rows' => $extracted['rows'] ?? [],
            'caption' => $extracted['caption'] ?? null,
            'hasHeaderRow' => $extracted['hasHeaderRow'] ?? true,
            'hasFooter' => $extracted['hasFooter'] ?? false,
            'striped' => $extracted['striped'] ?? false,
            'blockType' => $pattern_name,  // Preserve source
        ];
    }
    
    /**
     * Convert to rawHtml with metadata
     * 
     * @param array $extracted Extracted data
     * @param string $pattern_name Source pattern
     * @return array rawHtml block
     */
    private function to_raw_html($extracted, $pattern_name) {
        $note = null;
        if (class_exists('\\STCW\\Headless\\Engine\\Detector\\PatternRegistry')) {
            $note = \STCW\Headless\Engine\Detector\PatternRegistry::get_sanity_note($pattern_name);
        }
        
        return [
            '_type' => 'rawHtml',
            '_key' => $this->generate_key(),
            'html' => $extracted['html'] ?? '',
            'blockType' => $pattern_name,
            'note' => $note,
        ];
    }
    
    /**
     * Smart fallback - try to guess semantic type from extracted data structure
     * 
     * @param array $extracted Extracted data
     * @param string $pattern_name Source pattern
     * @return array Portable Text block
     */
    private function smart_fallback($extracted, $pattern_name) {
        // Try to detect semantic type from data structure
        if (isset($extracted['code'])) {
            return $this->to_code_block($extracted, $pattern_name);
        }
        
        if (isset($extracted['items']) && is_array($extracted['items'])) {
            // Check if items have title/body (accordion-like)
            $first_item = $extracted['items'][0] ?? [];
            if (isset($first_item['title']) && isset($first_item['body'])) {
                return $this->to_accordion_block($extracted, $pattern_name);
            }
        }
        
        if (isset($extracted['headers']) && isset($extracted['rows'])) {
            return $this->to_table_block($extracted, $pattern_name);
        }
        
        // Last resort: rawHtml
        return $this->to_raw_html($extracted, $pattern_name);
    }
    
    /**
     * Convert heading to Portable Text
     */
    private function heading_to_portable_text($extracted) {
        return [
            '_type' => 'block',
            'style' => 'h' . $extracted['level'],
            'children' => [
                [
                    '_type' => 'span',
                    'text' => $extracted['text'],
                    'marks' => [],
                ],
            ],
        ];
    }
    
    /**
     * Convert paragraph to Portable Text
     */
    private function paragraph_to_portable_text($extracted) {
        return [
            '_type' => 'block',
            'style' => 'normal',
            'children' => [
                [
                    '_type' => 'span',
                    'text' => $extracted['text'],
                    'marks' => [],
                ],
            ],
        ];
    }
    
    /**
     * Convert list to Portable Text
     */
    private function list_to_portable_text($extracted) {
        $blocks = [];

        foreach ($extracted['items'] as $item) {
            if (is_array($item)) {
                $text = $item['text'];
                $marks = [];
                $markDefs = [];

                if (!empty($item['href'])) {
                    $link_key = $this->generate_key();
                    $marks[] = $link_key; // Just the key reference
                    $markDefs[] = [
                        '_type' => 'link',
                        '_key' => $link_key,
                        'href' => $this->normalize_url($item['href']),
                    ];
                }
            } else {
                $text = $item;
                $marks = [];
                $markDefs = [];
            }

            $block = [
                '_type' => 'block',
                'style' => 'normal',
                'listItem' => $extracted['ordered'] ? 'number' : 'bullet',
                'children' => [
                    [
                        '_type' => 'span',
                        'text' => $text,
                        'marks' => $marks,
                    ],
                ],
            ];
            
            // Add markDefs only if there are marks
            if (!empty($markDefs)) {
                $block['markDefs'] = $markDefs;
            }
            
            $blocks[] = $block;
        }

        return $blocks;
    }
    
    /**
     * Convert quote to Portable Text
     */
    private function quote_to_portable_text($extracted) {
        return [
            '_type' => 'block',
            'style' => 'blockquote',
            'children' => [
                [
                    '_type' => 'span',
                    'text' => $extracted['text'],
                    'marks' => [],
                ],
            ],
        ];
    }
    
    /**
     * Convert image to Portable Text (v1.3: with deduplication + migration metadata)
     *
     * @param array $extracted Extracted image data
     * @return array|null Image block or null if duplicate
     */
    private function image_to_portable_text($extracted) {
        $full_url = $this->build_asset_url($extracted['src'] ?? '');
        
        // v1.3: Deduplication logic
        // Create unique key: URL + link (same image with different link = keep both)
        $link_href = $extracted['link']['href'] ?? '';
        $image_key = md5($full_url . '|' . $link_href);
        
        // Check if we've seen this image+link combo already
        if (isset($this->seen_images[$image_key])) {
            return null; // Skip duplicate
        }
        
        // Mark as seen
        $this->seen_images[$image_key] = true;
        
        // Generate asset ID for manifest linking
        $asset_id = $this->generate_asset_ref($extracted['src'] ?? '', 'image');
        
        // Build image block with migration metadata (Option C - Hybrid)
        $image_block = [
            '_type' => 'imageBlock',
            
            // Phase 1: External URL (works immediately)
            'url' => $full_url,
            'alt' => $extracted['alt'] ?? '',
            'width' => $extracted['width'] ? (int) $extracted['width'] : null,
            'height' => $extracted['height'] ? (int) $extracted['height'] : null,
            
            // Phase 2: Migration metadata (trustworthy, queryable)
            'assetId' => $asset_id,  // FIXED: No underscore (Sanity compliant)
            'migration' => [  // FIXED: No underscore (Sanity compliant)
                'phase' => 1,  // Current phase (1 = external URL, 2 = Sanity CDN)
                'placeholderRef' => 'asset-placeholder-' . $asset_id,  // For Phase 2 replacement
                'sourceUrl' => $full_url,  // Original WP URL (reference)
                'uploaded' => false,  // Track upload status
                'sanityAssetId' => null,  // Will be populated after upload
            ],
        ];
        
        // Add link if present
        if (!empty($extracted['link']['href'])) {
            $image_block['link'] = [
                'href' => $this->normalize_url($extracted['link']['href']),
                'target' => $extracted['link']['target'] ?? null,
            ];
        }
        
        // Track for asset manifest
        $this->add_asset_reference('image', $extracted['src'] ?? '', $extracted, $asset_id);
        
        return $image_block;
    }
    
    /**
     * Convert button to Portable Text
     */
    private function button_to_portable_text($extracted) {
        return [
            '_type' => 'customButton',
            'text' => $extracted['text'],
            'url' => $extracted['url'],
            'target' => $extracted['target'] ?? '_self',
        ];
    }
    
    /**
     * Convert accordion to Portable Text
     */
    private function accordion_to_portable_text($extracted) {
        return [
            '_type' => 'accordion',
            'items' => $extracted['items'],
        ];
    }
    
    /**
     * Build full asset URL from relative or absolute path
     *
     * v1.3: Maps to /stcw-assets/ location (rsync-ready)
     *
     * @param string $src Image source path
     * @return string Full URL
     */
    private function build_asset_url($src) {
        // Already a full URL
        if (strpos($src, 'http://') === 0 || strpos($src, 'https://') === 0) {
            return $src;
        }
        
        $base_url = get_site_url();
        
        // Clean up leading ../ and ./
        $clean_src = $src;
        while (strpos($clean_src, '../') === 0) {
            $clean_src = substr($clean_src, 3);
        }
        while (strpos($clean_src, './') === 0) {
            $clean_src = substr($clean_src, 2);
        }
        
        // Handle Static Cache Wrangler assets
        // Convert: assets/strategy.png → https://oc2.co/wp-content/uploads/stcw-assets/strategy.png
        if (strpos($clean_src, 'assets/') === 0) {
            $filename = str_replace('assets/', '', $clean_src);
            return trailingslashit($base_url) . 'wp-content/uploads/stcw-assets/' . $filename;
        }
        
        // Handle WordPress uploads
        if (strpos($clean_src, 'wp-content/uploads/') === 0) {
            return trailingslashit($base_url) . $clean_src;
        }
        
        // Other relative paths - assume root-relative
        $clean_src = ltrim($clean_src, '/');
        return trailingslashit($base_url) . $clean_src;
    }
    
    /**
     * Generate unique key for marks
     *
     * @return string Unique key
     */
    private function generate_key() {
        return substr(md5(uniqid(wp_rand(), true)), 0, 12);
    }
    
    /**
     * Normalize URL to clean relative paths
     *
     * @param string $url URL to normalize
     * @return string Normalized URL
     */
    private function normalize_url($url) {
        $site_url = trailingslashit(get_site_url());
        
        // Convert absolute URLs to relative
        if (strpos($url, $site_url) === 0) {
            $url = '/' . substr($url, strlen($site_url));
        }
        
        // Clean up path
        $clean_url = str_replace('//', '/', $url);
        
        // Ensure trailing slash for directory paths
        if (substr($clean_url, -1) !== '/' && strpos(basename($clean_url), '.') === false) {
            $clean_url .= '/';
        }
        
        return $clean_url;
    }
    
    /**
     * Fallback converter for unknown patterns
     * 
     * @param array $pattern_match Pattern match data
     * @return array Sanity block
     */
    private function convert_fallback($pattern_match) {
        return [
            '_type' => 'rawHtml',
            'html' => $pattern_match['html'],
            'blockType' => $pattern_match['pattern'],
        ];
    }
    
    /**
     * Get collected asset references
     * 
     * @return array Asset references
     */
    public function get_asset_references() {
        return $this->asset_references;
    }
    
    /**
     * Add an asset reference (v1.3: enhanced tracking)
     * 
     * @param string $type Asset type (image, video, file)
     * @param string $src Original source path
     * @param array $extracted Extracted data (metadata)
     * @param string $asset_id Asset ID for manifest linking
     */
    private function add_asset_reference($type, $src, $extracted, $asset_id) {
        // Build reference object
        $reference = [
            'assetId' => $asset_id,
            'type' => $type,
            'source' => [
                'original' => $src,
                'url' => $this->build_asset_url($src),
            ],
            'metadata' => [
                'alt' => $extracted['alt'] ?? '',
                'width' => $extracted['width'] ?? null,
                'height' => $extracted['height'] ?? null,
            ],
            'migration' => [
                'placeholderRef' => 'asset-placeholder-' . $asset_id,
                'phase' => 1,
            ],
        ];
        
        // Add to tracking array
        $this->asset_references[] = $reference;
    }
    
    /**
     * Generate Sanity asset reference ID
     * 
     * @param string $src Asset source path
     * @param string $type Asset type
     * @return string Asset reference ID
     */
    private function generate_asset_ref($src, $type) {
        // Extract filename from path
        $path_parts = wp_parse_url($src);
        $path = $path_parts['path'] ?? $src;
        $filename = basename($path);
        
        // Clean filename for ID
        $clean_name = sanitize_title(pathinfo($filename, PATHINFO_FILENAME));
        
        return $type . '-' . $clean_name;
    }
}
