<?php
/**
 * Generic Portable Text Converter
 *
 * Converts detected patterns to CMS-agnostic Portable Text format.
 * This generic format can be consumed by Go CLI converters for
 * Sanity, Strapi, Contentful, Payload, and other headless CMS platforms.
 *
 * Output format follows standard Portable Text specification but
 * remains CMS-neutral for maximum compatibility.
 *
 * v2.1.1: Fixed accordion/tabs content to preserve portable text blocks
 *
 * @package STCWHeadlessAssistant
 * @since 2.1.0
 */

namespace STCW\Headless\Engine\Target\Generic;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class GenericPortableTextConverter
 */
class GenericPortableTextConverter {

	/**
	 * Asset references collected during conversion
	 *
	 * @var array
	 */
	private $asset_references = array();

	/**
	 * Track seen images to prevent duplicates (per-page)
	 *
	 * @var array
	 */
	private $seen_images = array();

	/**
	 * Page metadata for deterministic key generation
	 *
	 * @var array
	 */
	private $page_metadata = array();

	/**
	 * Convert patterns to generic Portable Text format
	 *
	 * @param array $patterns Detected patterns from PatternDetector.
	 * @param array $metadata Page metadata (enhanced with source envelope).
	 * @return array Generic Portable Text blocks
	 */
	public function convert_patterns( $patterns, $metadata = array() ) {
		$blocks = array();

		// Reset tracking for each page
		$this->asset_references = array();
		$this->seen_images      = array();
		$this->page_metadata    = $metadata;

		foreach ( $patterns as $index => $pattern_match ) {
			// Pass block index for deterministic keys
			$converted = $this->convert_pattern( $pattern_match, $index );

			// Skip null blocks (deduplicated images, etc.)
			if ( $converted === null ) {
				continue;
			}

			// Handle both single blocks and arrays (e.g., lists)
			if ( isset( $converted[0] ) && is_array( $converted[0] ) ) {
				$blocks = array_merge( $blocks, $converted );
			} else {
				$blocks[] = $converted;
			}
		}

		return $blocks;
	}

	/**
	 * Convert a single pattern to generic block format
	 *
	 * @param array $pattern_match Pattern match data.
	 * @param int   $block_index   Block position for deterministic keys.
	 * @return array|null Generic block or null if deduplicated
	 */
	private function convert_pattern( $pattern_match, $block_index ) {
		$pattern_name = $pattern_match['pattern'];
		$extracted    = $pattern_match['extracted'] ?? array();

		if ( empty( $extracted ) ) {
			return $this->convert_fallback( $pattern_match, $block_index );
		}

		return $this->to_generic_block( $extracted, $pattern_name, $block_index );
	}

	/**
	 * Convert extracted content to generic block
	 *
	 * @param array  $extracted    Extracted content.
	 * @param string $pattern_name Pattern name.
	 * @param int    $block_index  Block position.
	 * @return array|null Generic block or null if deduplicated
	 */
	private function to_generic_block( $extracted, $pattern_name = null, $block_index = null ) {
		$type = $extracted['type'] ?? 'unknown';

		// Use passed pattern_name or fall back to type
		if ( $pattern_name === null ) {
			$pattern_name = $type;
		}

		// Map to generic block types
		switch ( $type ) {
			case 'heading':
				return $this->convert_heading( $extracted, $block_index );

			case 'paragraph':
				return $this->convert_paragraph( $extracted, $block_index );

			case 'list':
				return $this->convert_list( $extracted, $block_index );

			case 'quote':
				return $this->convert_quote( $extracted, $block_index );

			case 'image':
				return $this->convert_image( $extracted, $block_index );

			case 'button':
				return $this->convert_button( $extracted, $block_index );

			case 'code':
				return $this->convert_code( $extracted, $block_index );

			case 'table':
				return $this->convert_table( $extracted, $block_index );

			case 'accordion':
				return $this->convert_accordion( $extracted, $block_index );

			case 'separator':
				return $this->convert_separator( $extracted, $block_index );

			default:
				// Fallback for unrecognized patterns
				return $this->convert_fallback_with_metadata( $extracted, $pattern_name, $block_index );
		}
	}

	/**
	 * Convert heading to generic format
	 *
	 * @param array $extracted   Extracted heading data.
	 * @param int   $block_index Block position.
	 * @return array Generic heading block
	 */
	private function convert_heading( $extracted, $block_index = null ) {
		$level = isset( $extracted['level'] ) ? (int) $extracted['level'] : 2;
		$text  = $extracted['text'] ?? '';

		return array(
			'_type'    => 'block',
			'_key'     => $this->generate_key( $block_index ),
			'style'    => 'h' . $level,
			'children' => array(
				array(
					'_type' => 'span',
					'text'  => html_entity_decode( $text, ENT_QUOTES | ENT_HTML5, 'UTF-8' ),
					'marks' => array(),
				),
			),
			'_meta'    => array(
				'pattern' => 'heading',
				'level'   => $level,
			),
		);
	}

	/**
	 * Convert paragraph to generic format
	 *
	 * @param array $extracted   Extracted paragraph data.
	 * @param int   $block_index Block position.
	 * @return array Generic paragraph block
	 */
	private function convert_paragraph( $extracted, $block_index = null ) {
		$text = $extracted['text'] ?? '';

		return array(
			'_type'    => 'block',
			'_key'     => $this->generate_key( $block_index ),
			'style'    => 'normal',
			'children' => array(
				array(
					'_type' => 'span',
					'text'  => html_entity_decode( $text, ENT_QUOTES | ENT_HTML5, 'UTF-8' ),
					'marks' => array(),
				),
			),
			'_meta'    => array(
				'pattern' => 'paragraph',
			),
		);
	}

	/**
	 * Convert list to generic format
	 *
	 * @param array $extracted   Extracted list data.
	 * @param int   $block_index Block position.
	 * @return array Array of list item blocks
	 */
	private function convert_list( $extracted, $block_index = null ) {
		$list_type = $extracted['list_type'] ?? 'bullet';
		$items     = $extracted['items'] ?? array();
		$blocks    = array();

		foreach ( $items as $item_index => $item ) {
			// Handle both string items and array items
			$text = is_array( $item ) ? ( $item['text'] ?? '' ) : $item;

			$blocks[] = array(
				'_type'    => 'block',
				'_key'     => $this->generate_key( $block_index !== null ? $block_index + $item_index : null ),
				'style'    => 'normal',
				'listItem' => $list_type,
				'children' => array(
					array(
						'_type' => 'span',
						'text'  => html_entity_decode( $text, ENT_QUOTES | ENT_HTML5, 'UTF-8' ),
						'marks' => array(),
					),
				),
				'_meta'    => array(
					'pattern' => 'list',
				),
			);
		}

		return $blocks;
	}

	/**
	 * Convert quote to generic format
	 *
	 * @param array $extracted   Extracted quote data.
	 * @param int   $block_index Block position.
	 * @return array Generic quote block
	 */
	private function convert_quote( $extracted, $block_index = null ) {
		$text   = $extracted['text'] ?? '';
		$author = $extracted['author'] ?? '';

		$block = array(
			'_type'    => 'block',
			'_key'     => $this->generate_key( $block_index ),
			'style'    => 'blockquote',
			'children' => array(
				array(
					'_type' => 'span',
					'text'  => html_entity_decode( $text, ENT_QUOTES | ENT_HTML5, 'UTF-8' ),
					'marks' => array(),
				),
			),
			'_meta'    => array(
				'pattern' => 'quote',
			),
		);

		if ( ! empty( $author ) ) {
			$block['_meta']['author'] = $author;
		}

		return $block;
	}

	/**
	 * Convert image to generic format
	 *
	 * @param array $extracted   Extracted image data.
	 * @param int   $block_index Block position.
	 * @return array|null Generic image block or null if duplicate
	 */
	private function convert_image( $extracted, $block_index = null ) {
		$src = $extracted['src'] ?? '';
		$alt = $extracted['alt'] ?? '';

		// Deduplicate images within the same page
		if ( in_array( $src, $this->seen_images, true ) ) {
			return null;
		}

		$this->seen_images[] = $src;

		// Track asset reference
		$asset_id = 'img-' . md5( $src );
		$this->add_asset_reference( 'image', $src, $extracted, $asset_id );

		return array(
			'_type'  => 'image',
			'_key'   => $this->generate_key( $block_index ),
			'url'    => $src,
			'alt'    => html_entity_decode( $alt, ENT_QUOTES | ENT_HTML5, 'UTF-8' ),
			'width'  => $extracted['width'] ?? null,
			'height' => $extracted['height'] ?? null,
			'_meta'  => array(
				'pattern'  => 'image',
				'asset_id' => $asset_id,
			),
		);
	}

	/**
	 * Convert button to generic format
	 *
	 * @param array $extracted   Extracted button data.
	 * @param int   $block_index Block position.
	 * @return array Generic button block
	 */
	private function convert_button( $extracted, $block_index = null ) {
		return array(
			'_type'  => 'button',
			'_key'   => $this->generate_key( $block_index ),
			'text'   => html_entity_decode( $extracted['text'] ?? '', ENT_QUOTES | ENT_HTML5, 'UTF-8' ),
			'url'    => $extracted['url'] ?? '',
			'target' => $extracted['target'] ?? '_self',
			'_meta'  => array(
				'pattern' => 'button',
			),
		);
	}

	/**
	 * Convert code block to generic format
	 *
	 * @param array $extracted   Extracted code data.
	 * @param int   $block_index Block position.
	 * @return array Generic code block
	 */
	private function convert_code( $extracted, $block_index = null ) {
		return array(
			'_type'    => 'code',
			'_key'     => $this->generate_key( $block_index ),
			'code'     => $extracted['code'] ?? '',
			'language' => $extracted['language'] ?? 'text',
			'filename' => $extracted['filename'] ?? null,
			'_meta'    => array(
				'pattern' => 'code',
			),
		);
	}

	/**
	 * Convert table to generic format
	 *
	 * @param array $extracted   Extracted table data.
	 * @param int   $block_index Block position.
	 * @return array Generic table block
	 */
	private function convert_table( $extracted, $block_index = null ) {
		return array(
			'_type'   => 'table',
			'_key'    => $this->generate_key( $block_index ),
			'headers' => $extracted['headers'] ?? array(),
			'rows'    => $extracted['rows'] ?? array(),
			'_meta'   => array(
				'pattern' => 'table',
			),
		);
	}

	/**
	 * Convert accordion to generic format
	 *
	 * FIXED v2.1.1: Preserves portable text blocks in content field
	 *
	 * @param array $extracted   Extracted accordion data.
	 * @param int   $block_index Block position.
	 * @return array Generic accordion block
	 */
	private function convert_accordion( $extracted, $block_index = null ) {
		$items = $extracted['items'] ?? array();

		return array(
			'_type' => 'accordion',
			'_key'  => $this->generate_key( $block_index ),
			'items' => array_map(
				function ( $item ) {
					return array(
						'title'   => html_entity_decode( $item['title'] ?? '', ENT_QUOTES | ENT_HTML5, 'UTF-8' ),
						'content' => $this->normalize_content_field( $item['content'] ?? '' ),
					);
				},
				$items
			),
			'_meta' => array(
				'pattern' => 'accordion',
			),
		);
	}

	/**
	 * Convert separator to generic format
	 *
	 * @param array $extracted   Extracted separator data.
	 * @param int   $block_index Block position.
	 * @return array Generic separator block
	 */
	private function convert_separator( $extracted, $block_index = null ) {
		return array(
			'_type' => 'separator',
			'_key'  => $this->generate_key( $block_index ),
			'style' => $extracted['style'] ?? 'solid',
			'_meta' => array(
				'pattern' => 'separator',
			),
		);
	}

	/**
	 * Normalize content field to portable text blocks
	 *
	 * Handles both string content and block arrays from extractors.
	 * This ensures consistent portable text output regardless of
	 * extractor implementation.
	 *
	 * Strategy:
	 * - If already blocks: return as-is (preserve rich structure)
	 * - If string: wrap in simple paragraph block
	 *
	 * This preserves semantic meaning for CMSs that support blocks
	 * while allowing simple CMSs to flatten on their end.
	 *
	 * @param mixed $content Content from extractor (string or array).
	 * @return array Portable text blocks
	 */
	private function normalize_content_field( $content ) {
		// Already portable text blocks - return as-is
		if ( is_array( $content ) && ! empty( $content ) ) {
			return $content;
		}

		// String content - wrap in simple paragraph block
		if ( is_string( $content ) ) {
			return array(
				array(
					'_type'    => 'block',
					'_key'     => $this->generate_key(),
					'style'    => 'normal',
					'children' => array(
						array(
							'_type' => 'span',
							'text'  => html_entity_decode( $content, ENT_QUOTES | ENT_HTML5, 'UTF-8' ),
							'marks' => array(),
						),
					),
				),
			);
		}

		// Fallback for null/empty - return empty array
		return array();
	}

	/**
	 * Fallback converter with metadata
	 *
	 * @param array  $extracted    Extracted data.
	 * @param string $pattern_name Pattern name.
	 * @param int    $block_index  Block position.
	 * @return array Generic fallback block
	 */
	private function convert_fallback_with_metadata( $extracted, $pattern_name, $block_index = null ) {
		return array(
			'_type' => 'rawHtml',
			'_key'  => $this->generate_key( $block_index ),
			'html'  => $extracted['html'] ?? '',
			'_meta' => array(
				'pattern' => $pattern_name,
				'note'    => 'Fallback: pattern converter not implemented',
				'type'    => $extracted['type'] ?? 'unknown',
			),
		);
	}

	/**
	 * Fallback converter for pattern match
	 *
	 * @param array $pattern_match Pattern match data.
	 * @param int   $block_index   Block position.
	 * @return array Generic fallback block
	 */
	private function convert_fallback( $pattern_match, $block_index = null ) {
		return array(
			'_type' => 'rawHtml',
			'_key'  => $this->generate_key( $block_index ),
			'html'  => $pattern_match['html'] ?? '',
			'_meta' => array(
				'pattern' => $pattern_match['pattern'] ?? 'unknown',
				'note'    => 'Fallback: no extracted data',
			),
		);
	}

	/**
	 * Generate deterministic key for blocks
	 *
	 * Uses permalink + block index for stable keys across runs.
	 * Same content = same key = idempotent imports.
	 *
	 * @param int $block_index Optional block position.
	 * @return string Block key
	 */
	private function generate_key( $block_index = null ) {
		$permalink = $this->page_metadata['permalink'] ?? '';

		if ( $block_index !== null ) {
			$seed = $permalink . '-block-' . $block_index;
		} else {
			// Fallback: use random for nested/dynamic blocks
			$seed = $permalink . '-' . wp_rand( 1000, 9999 );
		}

		return substr( md5( $seed ), 0, 12 );
	}

	/**
	 * Add asset reference for tracking
	 *
	 * @param string $type     Asset type (image, video, file).
	 * @param string $source   Source URL.
	 * @param array  $metadata Additional asset metadata.
	 * @param string $asset_id Asset ID.
	 */
	private function add_asset_reference( $type, $source, $metadata = array(), $asset_id = null ) {
		if ( $asset_id === null ) {
			$asset_id = md5( $source );
		}

		$this->asset_references[] = array(
			'id'       => $asset_id,
			'type'     => $type,
			'source'   => array(
				'original' => $source,
			),
			'metadata' => $metadata,
		);
	}

	/**
	 * Get collected asset references
	 *
	 * @return array Asset references
	 */
	public function get_asset_references() {
		return $this->asset_references;
	}
}
