<?php
/**
 * Kadence Block Extractors
 *
 * Extracts content from Kadence Blocks cached HTML into Sanity-compatible JSON.
 * Based on real-world HTML analysis from production Kadence installations.
 *
 * v2.1.1: Fixed extract_inner_blocks() to output proper portable text format
 *
 * @package STCWHeadlessAssistant
 * @since 2.0.0
 */

namespace STCW\Headless\Engine\Extractor;

if (!defined('ABSPATH')) exit;

class KadenceExtractors {

    /**
     * Register all Kadence block extractors
     *
     * @return void
     */
    public static function register_all() {
        // Row Layout (container)
        self::register_rowlayout();
        
        // Column (container)
        self::register_column();
        
        // Advanced Heading
        self::register_advanced_heading();
        
        // Accordion
        self::register_accordion();
        
        // Advanced Gallery
        self::register_advanced_gallery();
        
        // Advanced Button
        self::register_advanced_button();
        
        // Countdown
        self::register_countdown();
        
        // Count Up
        self::register_countup();
        
        // Form
        self::register_form();
        
        // Google Maps
        self::register_google_maps();
        
        // Icon
        self::register_icon();
        
        // Icon List
        self::register_icon_list();
        
        // Info Box
        self::register_infobox();
        
        // Image
        self::register_image();
        
        // Show More
        self::register_show_more();
        
        // Lottie Animation
        self::register_lottie();
        
        // Identity (Logo/Site Title)
        self::register_identity();
        
        // Posts
        self::register_posts();
        
        // Progress Bar
        self::register_progress_bar();
        
        // Search
        self::register_search();
        
        // Spacer/Divider
        self::register_spacer();
        
        // Table
        self::register_table();
        
        // Table of Contents
        self::register_table_of_contents();
        
        // Tabs
        self::register_tabs();
        
        // Testimonials
        self::register_testimonials();
        
        // Navigation
        self::register_navigation();
        
        // Video Popup
        self::register_video_popup();
        
        // Vector (SVG)
        self::register_vector();
    }
    
    /**
     * Register Row Layout extractor
     * 
     * HTML: <div class="kb-row-layout-wrap kb-row-layout-id* wp-block-kadence-rowlayout">
     */
    private static function register_rowlayout() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_rowlayout', [
            'selectors' => [
                '.wp-block-kadence-rowlayout',
                '.kb-row-layout-wrap',
            ],
            'extractor' => [__CLASS__, 'extract_rowlayout'],
            'priority' => 5,
            'confidence' => 0.95,
            'description' => 'Kadence Row Layout (container with columns)',
        ]);
    }
    
    public static function extract_rowlayout($node, $xpath = null) {
        if (null === $xpath) { $xpath = new \DOMXPath($node->ownerDocument); }
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $columns = $xpath->query('.//div[contains(@class, "wp-block-kadence-column")]', $node);
        
        $extracted_columns = [];
        foreach ($columns as $column) {
            $extracted_columns[] = [
                '_type' => 'block',
                'blockType' => 'kadence.column',
                'content' => self::extract_inner_blocks($column, $xpath),
            ];
        }
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.rowLayout',
            'columns' => $extracted_columns,
            'columnCount' => count($extracted_columns),
        ];
    }
    
    /**
     * Register Column extractor
     * 
     * HTML: <div class="wp-block-kadence-column kadence-column*">
     */
    private static function register_column() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_column', [
            'selectors' => [
                '.wp-block-kadence-column',
            ],
            'extractor' => [__CLASS__, 'extract_column'],
            'priority' => 4,
            'confidence' => 0.95,
            'description' => 'Kadence Column (single column within row)',
        ]);
    }
    
    public static function extract_column($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        return [
            '_type' => 'block',
            'blockType' => 'kadence.column',
            'content' => self::extract_inner_blocks($node, $xpath),
        ];
    }
    
    /**
     * Register Advanced Heading extractor
     * 
     * HTML: <h2 class="kt-adv-heading* wp-block-kadence-advancedheading" data-kb-block="*">
     */
    private static function register_advanced_heading() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_advanced_heading', [
            'selectors' => [
                '.wp-block-kadence-advancedheading',
                '.kt-adv-heading',
            ],
            'extractor' => [__CLASS__, 'extract_advanced_heading'],
            'priority' => 6,
            'confidence' => 0.98,
            'description' => 'Kadence Advanced Heading with styling',
        ]);
    }
    
    public static function extract_advanced_heading($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $text = trim($node->textContent);
        $level = (int) str_replace('h', '', strtolower($node->nodeName));
        
        return [
            'type' => 'heading',
            
            'style' => 'h' . $level,
            'text' => $text,
            'level' => $level,
            'kadence' => true,
        ];
    }
    
    /**
     * Register Accordion extractor
     * 
     * HTML: <div class="wp-block-kadence-accordion">
     *         <div class="kt-accordion-wrap">
     *           <div class="wp-block-kadence-pane">
     */
    private static function register_accordion() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_accordion', [
            'selectors' => [
                '.wp-block-kadence-accordion',
                '.kt-accordion-wrap',
            ],
            'extractor' => [__CLASS__, 'extract_accordion'],
            'priority' => 7,
            'confidence' => 0.98,
            'description' => 'Kadence Accordion with expandable panes',
        ]);
    }
    
    public static function extract_accordion($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $panes = $xpath->query('.//div[contains(@class, "wp-block-kadence-pane")]', $node);
        
        $items = [];
        foreach ($panes as $pane) {
            // Get title from button
            $title_node = $xpath->query('.//span[@class="kt-blocks-accordion-title"]', $pane)->item(0);
            $title = $title_node ? trim($title_node->textContent) : '';
            
            // Get content from panel inner
            $content_node = $xpath->query('.//div[@class="kt-accordion-panel-inner"]', $pane)->item(0);
            $content = $content_node ? self::extract_inner_blocks($content_node, $xpath) : [];
            
            $items[] = [
                'title' => $title,
                'content' => $content,
            ];
        }
        
        return [
            'type' => 'accordion',
            
            'items' => $items,
        ];
    }
    
    /**
     * Register Advanced Gallery extractor
     * 
     * HTML: <div class="wp-block-kadence-advancedgallery kb-gallery-wrap-id-*">
     *         <ul class="kb-gallery-ul">
     */
    private static function register_advanced_gallery() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_advanced_gallery', [
            'selectors' => [
                '.wp-block-kadence-advancedgallery',
                '.kb-gallery-wrap-id-',
            ],
            'extractor' => [__CLASS__, 'extract_advanced_gallery'],
            'priority' => 7,
            'confidence' => 0.95,
            'description' => 'Kadence Advanced Gallery with masonry layout',
        ]);
    }
    
    public static function extract_advanced_gallery($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $gallery_items = $xpath->query('.//li[contains(@class, "kadence-blocks-gallery-item")]//img', $node);
        
        $images = [];
        foreach ($gallery_items as $img) {
            $src = $img->getAttribute('src');
            $full_src = $img->getAttribute('data-full-image') ?: $src;
            $alt = $img->getAttribute('alt');
            $width = (int) $img->getAttribute('width');
            $height = (int) $img->getAttribute('height');
            
            $images[] = [
                'url' => $full_src,
                'alt' => $alt,
                'width' => $width,
                'height' => $height,
            ];
        }
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.gallery',
            'images' => $images,
        ];
    }
    
    /**
     * Register Advanced Button extractor
     * 
     * HTML: <div class="wp-block-kadence-advancedbtn kb-buttons-wrap">
     *         <span class="kb-button kt-button">
     */
    private static function register_advanced_button() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_advanced_button', [
            'selectors' => [
                '.wp-block-kadence-advancedbtn',
                '.kb-buttons-wrap',
            ],
            'extractor' => [__CLASS__, 'extract_advanced_button'],
            'priority' => 6,
            'confidence' => 0.97,
            'description' => 'Kadence Advanced Button with styling',
        ]);
    }
    
    public static function extract_advanced_button($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $buttons = $xpath->query('.//span[contains(@class, "kb-button")] | .//button[contains(@class, "kb-button")]', $node);
        
        $button_items = [];
        foreach ($buttons as $button) {
            $text_node = $xpath->query('.//span[@class="kt-btn-inner-text"]', $button)->item(0);
            $text = $text_node ? trim($text_node->textContent) : trim($button->textContent);
            
            $button_items[] = [
                'text' => $text,
            ];
        }
        
        return [
            'type' => 'button',
            
            'buttons' => $button_items,
        ];
    }
    
    /**
     * Register Countdown extractor
     * 
     * HTML: <div class="wp-block-kadence-countdown kb-countdown-container">
     */
    private static function register_countdown() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_countdown', [
            'selectors' => [
                '.wp-block-kadence-countdown',
                '.kb-countdown-container',
            ],
            'extractor' => [__CLASS__, 'extract_countdown'],
            'priority' => 6,
            'confidence' => 0.95,
            'description' => 'Kadence Countdown Timer',
        ]);
    }
    
    public static function extract_countdown($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $data_id = $node->getAttribute('data-id');
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.countdown',
            'countdownId' => $data_id,
            'note' => 'Dynamic countdown timer - requires JavaScript in Sanity frontend',
        ];
    }
    
    /**
     * Register Count Up extractor
     * 
     * HTML: <div class="wp-block-kadence-countup kb-count-up">
     */
    private static function register_countup() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_countup', [
            'selectors' => [
                '.wp-block-kadence-countup',
                '.kb-count-up',
            ],
            'extractor' => [__CLASS__, 'extract_countup'],
            'priority' => 6,
            'confidence' => 0.95,
            'description' => 'Kadence Count Up animation',
        ]);
    }
    
    public static function extract_countup($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $start = (int) $node->getAttribute('data-start');
        $end = (int) $node->getAttribute('data-end');
        $duration = (float) $node->getAttribute('data-duration');
        $prefix = $node->getAttribute('data-prefix');
        $suffix = $node->getAttribute('data-suffix');
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.countup',
            'start' => $start,
            'end' => $end,
            'duration' => $duration,
            'prefix' => $prefix,
            'suffix' => $suffix,
        ];
    }
    
    /**
     * Register Form extractor
     * 
     * HTML: <div class="wp-block-kadence-form kadence-form-*">
     */
    private static function register_form() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_form', [
            'selectors' => [
                '.wp-block-kadence-form',
            ],
            'extractor' => [__CLASS__, 'extract_form'],
            'priority' => 7,
            'confidence' => 0.95,
            'description' => 'Kadence Form with fields',
        ]);
    }
    
    public static function extract_form($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $fields = $xpath->query('.//div[contains(@class, "kadence-blocks-form-field")]', $node);
        
        $form_fields = [];
        foreach ($fields as $field) {
            $label_node = $xpath->query('.//label', $field)->item(0);
            $input_node = $xpath->query('.//input | .//textarea', $field)->item(0);
            
            if ($input_node) {
                $form_fields[] = [
                    'label' => $label_node ? trim($label_node->textContent) : '',
                    'type' => $input_node->getAttribute('type') ?: $input_node->nodeName,
                    'required' => $input_node->getAttribute('data-required') === 'yes',
                    'placeholder' => $input_node->getAttribute('placeholder'),
                ];
            }
        }
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.form',
            'fields' => $form_fields,
            'note' => 'Form functionality requires backend integration',
        ];
    }
    
    /**
     * Register Google Maps extractor
     * 
     * HTML: <div class="kb-google-maps-container wp-block-kadence-googlemaps">
     */
    private static function register_google_maps() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_google_maps', [
            'selectors' => [
                '.wp-block-kadence-googlemaps',
                '.kb-google-maps-container',
            ],
            'extractor' => [__CLASS__, 'extract_google_maps'],
            'priority' => 6,
            'confidence' => 0.95,
            'description' => 'Kadence Google Maps embed',
        ]);
    }
    
    public static function extract_google_maps($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $iframe = $xpath->query('.//iframe', $node)->item(0);
        $src = $iframe ? $iframe->getAttribute('src') : '';
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.googleMaps',
            'embedUrl' => $src,
            'note' => 'Requires Google Maps API key in Sanity frontend',
        ];
    }
    
    /**
     * Register Icon extractor
     * 
     * HTML: <div class="wp-block-kadence-icon kt-svg-icons">
     */
    private static function register_icon() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_icon', [
            'selectors' => [
                '.wp-block-kadence-icon',
                '.kt-svg-icons',
            ],
            'extractor' => [__CLASS__, 'extract_icon'],
            'priority' => 5,
            'confidence' => 0.90,
            'description' => 'Kadence Icon (single SVG icon)',
        ]);
    }
    
    public static function extract_icon($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $icon_wrap = $xpath->query('.//span[contains(@class, "kb-svg-icon-wrap")]', $node)->item(0);
        $icon_class = $icon_wrap ? $icon_wrap->getAttribute('class') : '';
        
        // Extract icon name from class (e.g., "kb-svg-icon-fe_aperture" -> "aperture")
        preg_match('/kb-svg-icon-([^\s]+)/', $icon_class, $matches);
        $icon_name = $matches[1] ?? 'unknown';
        
        $svg = $xpath->query('.//svg', $node)->item(0);
        $svg_code = $svg ? $node->ownerDocument->saveHTML($svg) : '';
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.icon',
            'iconName' => $icon_name,
            'svgCode' => $svg_code,
        ];
    }
    
    /**
     * Register Icon List extractor
     * 
     * HTML: <div class="wp-block-kadence-iconlist kt-svg-icon-list-items">
     *         <ul class="kt-svg-icon-list">
     */
    private static function register_icon_list() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_icon_list', [
            'selectors' => [
                '.wp-block-kadence-iconlist',
                '.kt-svg-icon-list-items',
            ],
            'extractor' => [__CLASS__, 'extract_icon_list'],
            'priority' => 7,
            'confidence' => 0.97,
            'description' => 'Kadence Icon List with SVG icons',
        ]);
    }
    
    public static function extract_icon_list($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $items = $xpath->query('.//li[contains(@class, "wp-block-kadence-listitem")]', $node);
        
        $list_items = [];
        foreach ($items as $item) {
            $text_node = $xpath->query('.//span[@class="kt-svg-icon-list-text"]', $item)->item(0);
            $text = $text_node ? trim($text_node->textContent) : '';
            
            $icon_wrap = $xpath->query('.//span[contains(@class, "kb-svg-icon-wrap")]', $item)->item(0);
            $icon_class = $icon_wrap ? $icon_wrap->getAttribute('class') : '';
            preg_match('/kb-svg-icon-([^\s]+)/', $icon_class, $matches);
            $icon_name = $matches[1] ?? 'unknown';
            
            $list_items[] = [
                'text' => $text,
                'icon' => $icon_name,
            ];
        }
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.iconList',
            'items' => $list_items,
        ];
    }
    
    /**
     * Register Info Box extractor
     * 
     * HTML: <div class="wp-block-kadence-infobox kt-info-box*">
     */
    private static function register_infobox() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_infobox', [
            'selectors' => [
                '.wp-block-kadence-infobox',
                '.kt-info-box',
            ],
            'extractor' => [__CLASS__, 'extract_infobox'],
            'priority' => 7,
            'confidence' => 0.97,
            'description' => 'Kadence Info Box with icon/media, title, and text',
        ]);
    }
    
    public static function extract_infobox($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $title_node = $xpath->query('.//h2[@class="kt-blocks-info-box-title"]', $node)->item(0);
        $title = $title_node ? trim($title_node->textContent) : '';
        
        $text_node = $xpath->query('.//p[@class="kt-blocks-info-box-text"]', $node)->item(0);
        $text = $text_node ? trim($text_node->textContent) : '';
        
        $icon_wrap = $xpath->query('.//span[contains(@class, "kb-svg-icon-wrap")]', $node)->item(0);
        $icon_class = $icon_wrap ? $icon_wrap->getAttribute('class') : '';
        preg_match('/kb-svg-icon-([^\s]+)/', $icon_class, $matches);
        $icon_name = $matches[1] ?? null;
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.infoBox',
            'title' => $title,
            'text' => $text,
            'icon' => $icon_name,
        ];
    }
    
    /**
     * Register Image extractor
     * 
     * HTML: <figure class="wp-block-kadence-image kb-image*">
     */
    private static function register_image() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_image', [
            'selectors' => [
                'figure.wp-block-kadence-image',
                '.kb-image',
            ],
            'extractor' => [__CLASS__, 'extract_image'],
            'priority' => 6,
            'confidence' => 0.95,
            'description' => 'Kadence Advanced Image',
        ]);
    }
    
    public static function extract_image($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $img = $xpath->query('.//img', $node)->item(0);
        
        if (!$img) {
            return null;
        }
        
        return [
            'type' => 'image',
            'url' => $img->getAttribute('src'),
            'alt' => $img->getAttribute('alt'),
            'width' => (int) $img->getAttribute('width'),
            'height' => (int) $img->getAttribute('height'),
            'kadence' => true,
        ];
    }
    
    /**
     * Register Show More extractor
     * 
     * HTML: <div class="wp-block-kadence-show-more kb-block-show-more-container">
     */
    private static function register_show_more() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_show_more', [
            'selectors' => [
                '.wp-block-kadence-show-more',
                '.kb-block-show-more-container',
            ],
            'extractor' => [__CLASS__, 'extract_show_more'],
            'priority' => 6,
            'confidence' => 0.90,
            'description' => 'Kadence Show More toggle',
        ]);
    }
    
    public static function extract_show_more($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $content_node = $xpath->query('.//div[contains(@class, "kb-show-more-content")]', $node)->item(0);
        $content = $content_node ? self::extract_inner_blocks($content_node, $xpath) : [];
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.showMore',
            'content' => $content,
            'note' => 'Requires JavaScript toggle in Sanity frontend',
        ];
    }
    
    /**
     * Register Lottie Animation extractor
     * 
     * HTML: <div class="kb-lottie-container">
     */
    private static function register_lottie() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_lottie', [
            'selectors' => [
                '.kb-lottie-container',
            ],
            'extractor' => [__CLASS__, 'extract_lottie'],
            'priority' => 6,
            'confidence' => 0.95,
            'description' => 'Kadence Lottie Animation player',
        ]);
    }
    
    public static function extract_lottie($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $player = $xpath->query('.//dotlottie-player', $node)->item(0);
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.lottie',
            'src' => $player ? $player->getAttribute('src') : '',
            'loop' => $player ? $player->getAttribute('loop') === 'true' : false,
            'autoplay' => $player ? $player->getAttribute('autoplay') === 'true' : false,
        ];
    }
    
    /**
     * Register Identity extractor
     * 
     * HTML: <div class="kb-identity wp-block-kadence-identity">
     */
    private static function register_identity() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_identity', [
            'selectors' => [
                '.wp-block-kadence-identity',
                '.kb-identity',
            ],
            'extractor' => [__CLASS__, 'extract_identity'],
            'priority' => 6,
            'confidence' => 0.95,
            'description' => 'Kadence Identity (logo/site title)',
        ]);
    }
    
    public static function extract_identity($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $title_node = $xpath->query('.//p[@class="wp-block-site-title"]', $node)->item(0);
        $title = $title_node ? trim($title_node->textContent) : '';
        
        $link_node = $xpath->query('.//a', $node)->item(0);
        $href = $link_node ? $link_node->getAttribute('href') : '';
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.identity',
            'siteTitle' => $title,
            'homeUrl' => $href,
        ];
    }
    
    /**
     * Register Posts extractor
     * 
     * HTML: <ul class="wp-block-kadence-posts kb-posts">
     */
    private static function register_posts() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_posts', [
            'selectors' => [
                '.wp-block-kadence-posts',
                '.kb-posts',
            ],
            'extractor' => [__CLASS__, 'extract_posts'],
            'priority' => 7,
            'confidence' => 0.90,
            'description' => 'Kadence Posts block (post loop)',
        ]);
    }
    
    public static function extract_posts($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $items = $xpath->query('.//li[contains(@class, "kb-post-list-item")]', $node);
        
        $posts = [];
        foreach ($items as $item) {
            $title_node = $xpath->query('.//h2[@class="entry-title"]//a', $item)->item(0);
            $title = $title_node ? trim($title_node->textContent) : '';
            $link = $title_node ? $title_node->getAttribute('href') : '';
            
            $excerpt_node = $xpath->query('.//div[@class="entry-summary"]', $item)->item(0);
            $excerpt = $excerpt_node ? trim($excerpt_node->textContent) : '';
            
            $posts[] = [
                'title' => $title,
                'url' => $link,
                'excerpt' => $excerpt,
            ];
        }
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.posts',
            'posts' => $posts,
            'note' => 'Dynamic post query - consider replacing with Sanity query',
        ];
    }
    
    /**
     * Register Progress Bar extractor
     * 
     * HTML: <div class="kb-progress-bar-container">
     */
    private static function register_progress_bar() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_progress_bar', [
            'selectors' => [
                '.kb-progress-bar-container',
            ],
            'extractor' => [__CLASS__, 'extract_progress_bar'],
            'priority' => 6,
            'confidence' => 0.97,
            'description' => 'Kadence Progress Bar',
        ]);
    }
    
    public static function extract_progress_bar($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $label_node = $xpath->query('.//span[contains(@class, "kb-current-progress")]', $node)->item(0);
        $label_text = $label_node ? trim($label_node->textContent) : '';
        
        // Extract percentage from label (e.g., "90%")
        preg_match('/(\d+)%/', $label_text, $matches);
        $percentage = isset($matches[1]) ? (int) $matches[1] : 0;
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.progressBar',
            'percentage' => $percentage,
            'label' => $label_text,
        ];
    }
    
    /**
     * Register Search extractor
     * 
     * HTML: <div class="kb-search wp-block-kadence-search">
     */
    private static function register_search() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_search', [
            'selectors' => [
                '.wp-block-kadence-search',
                '.kb-search',
            ],
            'extractor' => [__CLASS__, 'extract_search'],
            'priority' => 6,
            'confidence' => 0.95,
            'description' => 'Kadence Search form',
        ]);
    }
    
    public static function extract_search($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $input = $xpath->query('.//input[@type="text"]', $node)->item(0);
        $placeholder = $input ? $input->getAttribute('placeholder') : '';
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.search',
            'placeholder' => $placeholder,
            'note' => 'Requires search functionality in Sanity frontend',
        ];
    }
    
    /**
     * Register Spacer extractor
     * 
     * HTML: <div class="wp-block-kadence-spacer kt-block-spacer">
     */
    private static function register_spacer() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_spacer', [
            'selectors' => [
                '.wp-block-kadence-spacer',
                '.kt-block-spacer',
            ],
            'extractor' => [__CLASS__, 'extract_spacer'],
            'priority' => 5,
            'confidence' => 0.95,
            'description' => 'Kadence Spacer/Divider',
        ]);
    }
    
    public static function extract_spacer($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $has_divider = $xpath->query('.//hr[@class="kt-divider"]', $node)->length > 0;
        
        return [
            'type' => 'separator',
            
            'style' => $has_divider ? 'line' : 'space',
            'kadence' => true,
        ];
    }
    
    /**
     * Register Table extractor
     * 
     * HTML: <div class="kb-table-container wp-block-kadence-table">
     */
    private static function register_table() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_table', [
            'selectors' => [
                '.wp-block-kadence-table',
                '.kb-table-container',
            ],
            'extractor' => [__CLASS__, 'extract_table'],
            'priority' => 7,
            'confidence' => 0.95,
            'description' => 'Kadence Table',
        ]);
    }
    
    public static function extract_table($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $table = $xpath->query('.//table', $node)->item(0);
        
        if (!$table) {
            return null;
        }
        
        $rows = $xpath->query('.//tr', $table);
        $table_data = [];
        
        foreach ($rows as $row) {
            $cells = $xpath->query('.//td | .//th', $row);
            $row_data = [];
            
            foreach ($cells as $cell) {
                $row_data[] = trim($cell->textContent);
            }
            
            $table_data[] = $row_data;
        }
        
        return [
            'type' => 'table',
            
            'rows' => $table_data,
            'kadence' => true,
        ];
    }
    
    /**
     * Register Table of Contents extractor
     * 
     * HTML: <nav class="wp-block-kadence-tableofcontents kb-table-of-content-nav">
     */
    private static function register_table_of_contents() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_table_of_contents', [
            'selectors' => [
                '.wp-block-kadence-tableofcontents',
                '.kb-table-of-content-nav',
            ],
            'extractor' => [__CLASS__, 'extract_table_of_contents'],
            'priority' => 7,
            'confidence' => 0.95,
            'description' => 'Kadence Table of Contents',
        ]);
    }
    
    public static function extract_table_of_contents($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $title_node = $xpath->query('.//span[@class="kb-table-of-contents-title"]', $node)->item(0);
        $title = $title_node ? trim($title_node->textContent) : '';
        
        $links = $xpath->query('.//a[contains(@class, "kb-table-of-contents__entry")]', $node);
        $items = [];
        
        foreach ($links as $link) {
            $items[] = [
                'text' => trim($link->textContent),
                'anchor' => $link->getAttribute('href'),
            ];
        }
        
        return [
            'type' => 'table',
            'blockType' => 'kadence.tableOfContents',
            'title' => $title,
            'items' => $items,
        ];
    }
    
    /**
     * Register Tabs extractor
     * 
     * HTML: <div class="wp-block-kadence-tabs">
     *         <div class="kt-tabs-wrap">
     */
    private static function register_tabs() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_tabs', [
            'selectors' => [
                '.wp-block-kadence-tabs',
                '.kt-tabs-wrap',
            ],
            'extractor' => [__CLASS__, 'extract_tabs'],
            'priority' => 7,
            'confidence' => 0.98,
            'description' => 'Kadence Tabs with multiple panels',
        ]);
    }
    
    public static function extract_tabs($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $title_items = $xpath->query('.//ul[@class="kt-tabs-title-list"]//li', $node);
        $tab_panels = $xpath->query('.//div[contains(@class, "wp-block-kadence-tab")]', $node);
        
        $tabs = [];
        foreach ($title_items as $index => $title_item) {
            $title_link = $xpath->query('.//span[@class="kt-title-text"]', $title_item)->item(0);
            $title = $title_link ? trim($title_link->textContent) : '';
            
            $content = [];
            if (isset($tab_panels[$index])) {
                $content = self::extract_inner_blocks($tab_panels[$index], $xpath);
            }
            
            $tabs[] = [
                'title' => $title,
                'content' => $content,
            ];
        }
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.tabs',
            'tabs' => $tabs,
        ];
    }
    
    /**
     * Register Testimonials extractor
     * 
     * HTML: <div class="kt-blocks-testimonials-wrap* wp-block-kadence-testimonials">
     */
    private static function register_testimonials() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_testimonials', [
            'selectors' => [
                '.wp-block-kadence-testimonials',
                '.kt-blocks-testimonials-wrap',
            ],
            'extractor' => [__CLASS__, 'extract_testimonials'],
            'priority' => 7,
            'confidence' => 0.95,
            'description' => 'Kadence Testimonials',
        ]);
    }
    
    public static function extract_testimonials($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $items = $xpath->query('.//li[@class="kt-testimonial-grid-item"]', $node);
        
        $testimonials = [];
        foreach ($items as $item) {
            $title_node = $xpath->query('.//h2[@class="kt-testimonial-title"]', $item)->item(0);
            $content_node = $xpath->query('.//blockquote[@class="kt-testimonial-content"]', $item)->item(0);
            $name_node = $xpath->query('.//div[@class="kt-testimonial-name"]', $item)->item(0);
            $occupation_node = $xpath->query('.//div[@class="kt-testimonial-occupation"]', $item)->item(0);
            
            $testimonials[] = [
                'title' => $title_node ? trim($title_node->textContent) : '',
                'content' => $content_node ? trim($content_node->textContent) : '',
                'name' => $name_node ? trim($name_node->textContent) : '',
                'occupation' => $occupation_node ? trim($occupation_node->textContent) : '',
            ];
        }
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.testimonials',
            'testimonials' => $testimonials,
        ];
    }
    
    /**
     * Register Navigation extractor
     * 
     * HTML: <div class="wp-block-kadence-navigation">
     */
    private static function register_navigation() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_navigation', [
            'selectors' => [
                '.wp-block-kadence-navigation',
            ],
            'extractor' => [__CLASS__, 'extract_navigation'],
            'priority' => 7,
            'confidence' => 0.95,
            'description' => 'Kadence Navigation menu',
        ]);
    }
    
    public static function extract_navigation($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $links = $xpath->query('.//li[contains(@class, "wp-block-kadence-navigation-link")]//a', $node);
        
        $menu_items = [];
        foreach ($links as $link) {
            $menu_items[] = [
                'text' => trim($link->textContent),
                'url' => $link->getAttribute('href'),
            ];
        }
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.navigation',
            'items' => $menu_items,
        ];
    }
    
    /**
     * Register Video Popup extractor
     * 
     * HTML: <div class="wp-block-kadence-videopopup kadence-video-popup*">
     */
    private static function register_video_popup() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_video_popup', [
            'selectors' => [
                '.wp-block-kadence-videopopup',
            ],
            'extractor' => [__CLASS__, 'extract_video_popup'],
            'priority' => 7,
            'confidence' => 0.95,
            'description' => 'Kadence Video Popup',
        ]);
    }
    
    public static function extract_video_popup($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $link = $xpath->query('.//a[@class="kadence-video-popup-link"]', $node)->item(0);
        $poster = $xpath->query('.//img[@class="kadence-video-poster"]', $node)->item(0);
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.videoPopup',
            'videoUrl' => $link ? $link->getAttribute('href') : '',
            'posterUrl' => $poster ? $poster->getAttribute('src') : '',
        ];
    }
    
    /**
     * Register Vector extractor
     * 
     * HTML: <div class="kb-vector-container wp-block-kadence-vector">
     */
    private static function register_vector() {
        \STCW\Headless\Engine\Detector\PatternRegistry::register('kadence_vector', [
            'selectors' => [
                '.wp-block-kadence-vector',
                '.kb-vector-container',
            ],
            'extractor' => [__CLASS__, 'extract_vector'],
            'priority' => 6,
            'confidence' => 0.95,
            'description' => 'Kadence Vector (SVG graphic)',
        ]);
    }
    
    public static function extract_vector($node, $xpath = null) {
        if ($xpath === null) { $xpath = new \DOMXPath($node->ownerDocument); }
        $svg = $xpath->query('.//svg', $node)->item(0);
        $svg_code = $svg ? $node->ownerDocument->saveHTML($svg) : '';
        
        return [
            '_type' => 'block',
            'blockType' => 'kadence.vector',
            'svgCode' => $svg_code,
        ];
    }
    
    /**
     * Helper: Extract inner blocks recursively
     * 
     * FIXED v2.1.1: Now outputs proper portable text block format
     * that the Generic converter can consume.
     * 
     * @param \DOMNode $node Parent node
     * @param \DOMXPath $xpath XPath instance
     * @return array Array of portable text blocks
     */
    private static function extract_inner_blocks($node, $xpath) {
        $blocks = [];
        
        // Extract headings
        $headings = $xpath->query('.//h1 | .//h2 | .//h3 | .//h4 | .//h5 | .//h6', $node);
        foreach ($headings as $heading) {
            $blocks[] = [
                '_type' => 'block',
                'style' => strtolower($heading->nodeName),  // h1, h2, etc.
                'children' => [
                    [
                        '_type' => 'span',
                        'text' => trim($heading->textContent),
                        'marks' => [],
                    ],
                ],
            ];
        }
        
        // Extract paragraphs
        $paragraphs = $xpath->query('.//p', $node);
        foreach ($paragraphs as $p) {
            $text = trim($p->textContent);
            if (!empty($text)) {
                $blocks[] = [
                    '_type' => 'block',
                    'style' => 'normal',
                    'children' => [
                        [
                            '_type' => 'span',
                            'text' => $text,
                            'marks' => [],
                        ],
                    ],
                ];
            }
        }
        
        return $blocks;
    }
}
