<?php
/**
 * Pattern Registry
 *
 * Central pattern registration and management with support for:
 * - Pattern inheritance
 * - Confidence scoring
 * - Opt-out filtering
 * - Priority ordering
 *
 * @package STCWHeadlessAssistant
 * @since 2.0.0
 */

namespace STCW\Headless\Engine\Detector;

if (!defined('ABSPATH')) exit;

class PatternRegistry {
    
    /**
     * Registered patterns
     * @var array
     */
    private static $patterns = [];
    
    /**
     * Pattern inheritance cache
     * @var array
     */
    private static $inheritance_cache = [];
    
    /**
     * Register a content pattern
     *
     * @param string $name Pattern identifier (e.g., 'accordion')
     * @param array $config Pattern configuration
     * @return bool True on success
     */
    public static function register($name, $config) {
        // Validate required fields
        if (empty($config['selectors'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log(
                    sprintf(
                        /* translators: %s: Pattern name */
                        esc_html('Pattern "%s" requires selectors'),
                        esc_html($name)
                    )
                );
            }
            return false;
        }
        
        // Set defaults
        $defaults = [
            'name' => $name,
            'selectors' => [],
            'extractor' => null,
            'priority' => 5,
            'confidence' => 1.0,
            'enabled' => true,
            'extends' => null,
            'validators' => [],
	    'description' => '',
	    'sanity_target' => null,
    	    'sanity_note' => null,
        ];
        
        $config = array_merge($defaults, $config);
        
        // Validate confidence score
        if ($config['confidence'] < 0 || $config['confidence'] > 1) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log(
                    sprintf(
                        /* translators: %s: Pattern name */
                        esc_html('Pattern "%s" confidence must be 0-1'),
                        esc_html($name)
                    )
                );
            }
            $config['confidence'] = 1.0;
        }
        
        self::$patterns[$name] = $config;
        
        // Clear inheritance cache
        self::$inheritance_cache = [];
        
        return true;
    }

    /**
 * Get Sanity target type for a pattern
 *
 * @param string $pattern_name Pattern identifier
 * @return string|null Target type or null
 */
public static function get_sanity_target($pattern_name) {
    if (!isset(self::$patterns[$pattern_name])) {
        return null;
    }

    $pattern = self::$patterns[$pattern_name];

    // Apply filter for override
    return apply_filters(
        'stcw_pattern_sanity_target',
        $pattern['sanity_target'] ?? null,
        $pattern_name,
        $pattern
    );
}

    /**
     * Get Sanity note for a pattern (optional metadata)
     *
     * @param string $pattern_name Pattern identifier
     * @return string|null Note or null
     */
    public static function get_sanity_note($pattern_name) {
        if (!isset(self::$patterns[$pattern_name])) {
            return null;
        }

        return self::$patterns[$pattern_name]['sanity_note'] ?? null;
    }

    /**
     * Get all patterns with a specific Sanity target
     *
     * @param string $target Target type (e.g., 'codeBlock')
     * @return array Pattern names
     */
    public static function get_patterns_by_sanity_target($target) {
        $patterns = [];

        foreach (self::$patterns as $name => $config) {
            if (isset($config['sanity_target']) && $config['sanity_target'] === $target) {
                $patterns[] = $name;
            }
        }

        return $patterns;
    }
    
    /**
     * Get all registered patterns with filters applied
     *
     * @return array Patterns array
     */
    public static function get_patterns() {
        $patterns = self::$patterns;
        
        /**
         * Filter registered patterns
         *
         * @param array $patterns All registered patterns
         */
        $patterns = apply_filters('stcw_headless_patterns', $patterns);
        
        // Apply opt-out filter
        $patterns = self::apply_opt_out_filter($patterns);
        
        // Resolve inheritance
        $patterns = self::resolve_inheritance($patterns);
        
        return $patterns;
    }
    
    /**
     * Get specific pattern by name
     *
     * @param string $name Pattern name
     * @return array|null Pattern config or null
     */
    public static function get_pattern($name) {
        $patterns = self::get_patterns();
        return $patterns[$name] ?? null;
    }
    
    /**
     * Apply opt-out filter to patterns
     *
     * Defaults to opt-out (all enabled unless explicitly disabled)
     *
     * @param array $patterns Patterns array
     * @return array Filtered patterns
     */
    private static function apply_opt_out_filter($patterns) {
        /**
         * Filter to disable specific patterns
         *
         * Return array of pattern names to DISABLE.
         *
         * @param array $disabled_patterns Pattern names to disable
         */
        $disabled = apply_filters('stcw_headless_disabled_patterns', []);
        
        foreach ($disabled as $pattern_name) {
            if (isset($patterns[$pattern_name])) {
                $patterns[$pattern_name]['enabled'] = false;
            }
        }
        
        return $patterns;
    }
    
    /**
     * Resolve pattern inheritance
     *
     * Patterns can extend other patterns to inherit selectors and config.
     *
     * @param array $patterns Patterns array
     * @return array Patterns with inheritance resolved
     */
    private static function resolve_inheritance($patterns) {
        foreach ($patterns as $name => $config) {
            if (!empty($config['extends'])) {
                $patterns[$name] = self::resolve_single_inheritance($name, $config, $patterns);
            }
        }
        
        return $patterns;
    }
    
    /**
     * Resolve inheritance for single pattern
     *
     * @param string $name Pattern name
     * @param array $config Pattern config
     * @param array $all_patterns All patterns
     * @return array Resolved config
     */
    private static function resolve_single_inheritance($name, $config, $all_patterns) {
        // Check cache
        if (isset(self::$inheritance_cache[$name])) {
            return self::$inheritance_cache[$name];
        }
        
        $parent_name = $config['extends'];
        
        if (!isset($all_patterns[$parent_name])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log(
                    sprintf(
                        /* translators: 1: Child pattern name, 2: Parent pattern name */
                        esc_html('Pattern "%1$s" extends non-existent pattern "%2$s"'),
                        esc_html($name),
                        esc_html($parent_name)
                    )
                );
            }
            return $config;
        }
        
        $parent = $all_patterns[$parent_name];
        
        // Recursive inheritance
        if (!empty($parent['extends'])) {
            $parent = self::resolve_single_inheritance($parent_name, $parent, $all_patterns);
        }
        
        // Merge selectors (child overrides parent)
        $merged_selectors = array_unique(array_merge(
            $parent['selectors'] ?? [],
            $config['selectors'] ?? []
        ));
        
        // Merge validators
        $merged_validators = array_unique(array_merge(
            $parent['validators'] ?? [],
            $config['validators'] ?? []
        ));
        
        // Child config takes precedence
        $resolved = array_merge($parent, $config, [
            'selectors' => $merged_selectors,
            'validators' => $merged_validators,
        ]);
        
        // Cache result
        self::$inheritance_cache[$name] = $resolved;
        
        return $resolved;
    }
    
    /**
     * Get patterns sorted by priority (descending)
     *
     * @return array Sorted patterns
     */
    public static function get_patterns_by_priority() {
        $patterns = self::get_patterns();
        
        uasort($patterns, function($a, $b) {
            $priority_diff = ($b['priority'] ?? 5) - ($a['priority'] ?? 5);
            
            if ($priority_diff !== 0) {
                return $priority_diff;
            }
            
            // Secondary sort by confidence
            return ($b['confidence'] ?? 1.0) <=> ($a['confidence'] ?? 1.0);
        });
        
        return $patterns;
    }
    
    /**
     * Clear all registered patterns (for testing)
     *
     * @return void
     */
    public static function clear_patterns() {
        self::$patterns = [];
        self::$inheritance_cache = [];
    }
    
    /**
     * Get pattern count
     *
     * @param bool $enabled_only Count only enabled patterns
     * @return int Pattern count
     */
    public static function count_patterns($enabled_only = false) {
        $patterns = self::get_patterns();
        
        if (!$enabled_only) {
            return count($patterns);
        }
        
        return count(array_filter($patterns, function($pattern) {
            return $pattern['enabled'] ?? true;
        }));
    }
}
