<?php
/**
 * Content Extractors - COMPLETE WITH ALL KADENCE METHODS
 *
 * Static methods for extracting structured content from detected patterns.
 * Each extractor receives a DOMNode and returns structured data.
 *
 * v1.3.1: Added 21 missing Kadence extractors + fixed accordion extraction
 *
 * @package STCWHeadlessAssistant
 * @since 2.0.0
 */

namespace STCW\Headless\Engine\Detector;

if (!defined('ABSPATH')) exit;

class Extractors {
    
    // ==========================================
    // GUTENBERG CORE BLOCKS
    // ==========================================
    
    /**
     * Extract heading content
     *
     * @param \DOMNode $node DOM node
     * @return array Structured heading data
     */
    public static function extract_heading($node) {
        $level = (int) substr($node->nodeName, 1); // h2 -> 2
        
        return [
            'type' => 'heading',
            'level' => $level,
            'text' => html_entity_decode(trim($node->textContent), ENT_QUOTES | ENT_HTML5, 'UTF-8'),
        ];
    }

    /**
     * Extract paragraph content
     *
     * @param \DOMNode $node DOM node
     * @return array Structured paragraph data
     */
    public static function extract_paragraph($node) {
        return [
            'type' => 'paragraph',
            'text' => html_entity_decode(trim($node->textContent), ENT_QUOTES | ENT_HTML5, 'UTF-8'),
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }

    /**
     * Extract image content
     *
     * @param \DOMNode $node DOM node
     * @return array|null Structured image data or null
     */
    public static function extract_image($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        
        // Handle <img> tag directly
        if (strtolower($node->nodeName) === 'img') {
            $img = $node;
            $parent = $img->parentNode;
        } else {
            // Find <img> inside wrapper
            $img = $xpath->query('.//img', $node)->item(0);
            $parent = $node;
        }
        
        if (!$img) {
            return null;
        }
        
        // Check if image is wrapped in a link
        $link = null;
        if ($parent && strtolower($parent->nodeName) === 'a') {
            $link = $parent;
        } else {
            // Check for <a> wrapper in the node
            $link = $xpath->query('.//a[.//img]', $node)->item(0);
        }
        
        $image_data = [
            'type' => 'image',
            'src' => $img->getAttribute('src'),
            'alt' => $img->getAttribute('alt'),
            'srcset' => $img->getAttribute('srcset'),
            'sizes' => $img->getAttribute('sizes'),
            'width' => $img->getAttribute('width'),
            'height' => $img->getAttribute('height'),
        ];
        
        // Add link data if image is clickable
        if ($link) {
            $image_data['link'] = [
                'href' => $link->getAttribute('href'),
                'target' => $link->getAttribute('target') ?: null,
                'title' => $link->getAttribute('title') ?: null,
            ];
        }
        
        return $image_data;
    }

    /**
     * Extract list content
     *
     * @param \DOMNode $node DOM node
     * @return array Structured list data
     */
    public static function extract_list($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        $items = [];
        
        $list_items = $xpath->query('.//li', $node);
        
        foreach ($list_items as $li) {
            // Check if list item contains a link
            $link = $xpath->query('.//a', $li)->item(0);
            
            if ($link) {
                // Extract link data
                $items[] = [
                    'text' => html_entity_decode(trim($link->textContent), ENT_QUOTES | ENT_HTML5, 'UTF-8'),
                    'href' => $link->getAttribute('href'),
                    'target' => $link->getAttribute('target') ?: null,
                ];
            } else {
                // Plain text item
                $items[] = [
                    'text' => html_entity_decode(trim($li->textContent), ENT_QUOTES | ENT_HTML5, 'UTF-8'),
                    'href' => null,
                ];
            }
        }
        
        return [
            'type' => 'list',
            'ordered' => strtolower($node->nodeName) === 'ol',
            'items' => $items,
        ];
    }

    /**
     * Extract quote content
     *
     * @param \DOMNode $node DOM node
     * @return array Structured quote data
     */
    public static function extract_quote($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        
        // Find citation if present
        $cite = $xpath->query('.//cite', $node)->item(0);
        
        return [
            'type' => 'quote',
            'text' => trim($node->textContent),
            'citation' => $cite ? trim($cite->textContent) : null,
        ];
    }
    
    /**
     * Extract button content
     *
     * @param \DOMNode $node DOM node
     * @return array|null Structured button data or null
     */
    public static function extract_button($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        
        // Find <a> tag
        $link = $node->nodeName === 'a' ?
            $node :
            $xpath->query('.//a', $node)->item(0);
        
        if (!$link) {
            return null;
        }
        
        return [
            'type' => 'button',
            'text' => trim($link->textContent),
            'url' => $link->getAttribute('href'),
            'target' => $link->getAttribute('target'),
        ];
    }
    
    /**
     * Extract gallery content
     *
     * @param \DOMNode $node DOM node
     * @return array Structured gallery data
     */
    public static function extract_gallery($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        $images = [];
        
        $img_nodes = $xpath->query('.//img', $node);
        
        foreach ($img_nodes as $img) {
            $images[] = [
                'src' => $img->getAttribute('src'),
                'alt' => $img->getAttribute('alt'),
                'width' => $img->getAttribute('width'),
                'height' => $img->getAttribute('height'),
            ];
        }
        
        return [
            'type' => 'gallery',
            'images' => $images,
        ];
    }
    
    /**
     * Extract video content
     *
     * @param \DOMNode $node DOM node
     * @return array|null Structured video data or null
     */
    public static function extract_video($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        
        // Find video or iframe
        $video = $xpath->query('.//video', $node)->item(0);
        $iframe = $xpath->query('.//iframe', $node)->item(0);
        
        if ($video) {
            return [
                'type' => 'video',
                'src' => $video->getAttribute('src'),
                'poster' => $video->getAttribute('poster'),
            ];
        }
        
        if ($iframe) {
            return [
                'type' => 'video',
                'embed' => $iframe->getAttribute('src'),
            ];
        }
        
        return null;
    }
    
    /**
     * Extract group content
     *
     * @param \DOMNode $node DOM node
     * @return array Structured group data
     */
    public static function extract_group($node) {
        return [
            'type' => 'group',
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract columns content
     *
     * @param \DOMNode $node DOM node
     * @return array Structured columns data
     */
    public static function extract_columns($node) {
        return [
            'type' => 'columns',
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract separator content
     *
     * @param \DOMNode $node DOM node
     * @return array Structured separator data
     */
    public static function extract_separator($node) {
        return [
            'type' => 'separator',
            'style' => $node->getAttribute('class'),
        ];
    }
    
    /**
     * Extract buttons group content
     *
     * @param \DOMNode $node DOM node
     * @return array Structured buttons data
     */
    public static function extract_buttons($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        $buttons = [];
        
        $links = $xpath->query('.//a', $node);
        
        foreach ($links as $link) {
            $buttons[] = [
                'text' => trim($link->textContent),
                'url' => $link->getAttribute('href'),
                'target' => $link->getAttribute('target'),
            ];
        }
        
        return [
            'type' => 'buttons',
            'buttons' => $buttons,
        ];
    }
    
    /**
     * Extract table content
     *
     * @param \DOMNode $node DOM node
     * @return array|null Structured table data or null
     */
    public static function extract_table($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        
        // Find table element
        $table = $node->nodeName === 'table' ?
            $node :
            $xpath->query('.//table', $node)->item(0);
        
        if (!$table) {
            return null;
        }
        
        $rows = $xpath->query('.//tr', $table);
        $table_data = [];
        
        foreach ($rows as $row) {
            $cells = $xpath->query('.//td | .//th', $row);
            $row_data = [];
            
            foreach ($cells as $cell) {
                $row_data[] = trim($cell->textContent);
            }
            
            $table_data[] = $row_data;
        }
        
        return [
            'type' => 'table',
            'rows' => $table_data,
        ];
    }
    
    /**
     * Extract code content
     *
     * @param \DOMNode $node DOM node
     * @return array Structured code data
     */
    public static function extract_code($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        
        // Find code element
        $code = $xpath->query('.//code', $node)->item(0);
        $pre = $xpath->query('.//pre', $node)->item(0);
        
        $code_content = '';
        if ($code) {
            $code_content = $code->textContent;
        } elseif ($pre) {
            $code_content = $pre->textContent;
        } else {
            $code_content = $node->textContent;
        }
        
        // Try to detect language from class
        $language = null;
        if ($code) {
            $class = $code->getAttribute('class');
            if (preg_match('/language-(\w+)/', $class, $matches)) {
                $language = $matches[1];
            }
        }
        
        return [
            'type' => 'code',
            'code' => $code_content,
            'language' => $language,
        ];
    }
    
    // ==========================================
    // KADENCE BLOCKS - ALL 21 EXTRACTORS
    // ==========================================
    
    /**
     * Extract Kadence row layout
     *
     * @param \DOMNode $node DOM node
     * @return array Structured row layout data
     */
    public static function extract_rowlayout($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        $columns = $xpath->query('.//div[contains(@class, "wp-block-kadence-column")]', $node);
        
        $extracted_columns = [];
        foreach ($columns as $column) {
            $extracted_columns[] = [
                '_type' => 'block',
                'blockType' => 'kadence.column',
                'html' => $node->ownerDocument->saveHTML($column),
            ];
        }
        
        return [
            'type' => 'kadence_rowlayout',
            'columns' => $extracted_columns,
            'columnCount' => count($extracted_columns),
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence column
     *
     * @param \DOMNode $node DOM node
     * @return array Structured column data
     */
    public static function extract_column($node) {
        return [
            'type' => 'kadence_column',
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence advanced heading
     *
     * @param \DOMNode $node DOM node
     * @return array Structured heading data
     */
    public static function extract_advanced_heading($node) {
        $text = trim($node->textContent);
        $level = (int) str_replace('h', '', strtolower($node->nodeName));
        
        return [
            'type' => 'heading',
            'style' => 'h' . $level,
            'text' => $text,
            'level' => $level,
            'kadence' => true,
        ];
    }
    
    /**
     * Extract Kadence accordion - FIXED
     *
     * @param \DOMNode $node DOM node
     * @return array Structured accordion data
     */
    public static function extract_kadence_accordion($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        $items = [];
        
        // Kadence accordion structure: kt-accordion-pane (not kb!)
        $panes = $xpath->query('.//*[contains(@class, "kt-accordion-pane")]', $node);
        
        foreach ($panes as $pane) {
            // Title is in .kt-blocks-accordion-title (not kb-accordion-title!)
            $title_node = $xpath->query('.//*[contains(@class, "kt-blocks-accordion-title")]', $pane)->item(0);
            // Content is in .kt-accordion-panel-inner
            $content_node = $xpath->query('.//*[contains(@class, "kt-accordion-panel-inner")]', $pane)->item(0);
            
            $items[] = [
                'title' => $title_node ? trim($title_node->textContent) : '',
                'body' => $content_node ? $node->ownerDocument->saveHTML($content_node) : '',
            ];
        }
        
        // Fallback: HTML5 <details> structure
        if (empty($items)) {
            $details = $xpath->query('.//details', $node);
            
            foreach ($details as $detail) {
                $summary = $xpath->query('.//summary', $detail)->item(0);
                
                $content_html = '';
                foreach ($detail->childNodes as $child) {
                    if ($child->nodeName !== 'summary') {
                        $content_html .= $node->ownerDocument->saveHTML($child);
                    }
                }
                
                $items[] = [
                    'title' => $summary ? trim($summary->textContent) : '',
                    'body' => trim($content_html),
                ];
            }
        }
        
        return [
            'type' => 'accordion',
            'items' => $items,
        ];
    }
    
    /**
     * Extract Kadence tabs
     *
     * @param \DOMNode $node DOM node
     * @return array Structured tabs data
     */
    public static function extract_kadence_tabs($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        $title_items = $xpath->query('.//ul[contains(@class, "kt-tabs-title-list")]//li', $node);
        $tab_panels = $xpath->query('.//div[contains(@class, "wp-block-kadence-tab")]', $node);
        
        $tabs = [];
        foreach ($title_items as $index => $title_item) {
            $title_link = $xpath->query('.//span[contains(@class, "kt-title-text")]', $title_item)->item(0);
            $title = $title_link ? trim($title_link->textContent) : '';
            
            $content_html = '';
            if (isset($tab_panels[$index])) {
                $content_html = $node->ownerDocument->saveHTML($tab_panels[$index]);
            }
            
            $tabs[] = [
                'title' => $title,
                'body' => $content_html,
            ];
        }
        
        return [
            'type' => 'tabs',
            'items' => $tabs,
        ];
    }
    
    /**
     * Extract Kadence advanced gallery
     *
     * @param \DOMNode $node DOM node
     * @return array Structured gallery data
     */
    public static function extract_advanced_gallery($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        $images = [];
        
        $img_nodes = $xpath->query('.//img', $node);
        
        foreach ($img_nodes as $img) {
            $src = $img->getAttribute('src');
            $full_src = strpos($src, 'http') === 0 ? $src : home_url($src);
            
            $images[] = [
                'url' => $full_src,
                'alt' => $img->getAttribute('alt'),
                'width' => (int) $img->getAttribute('width'),
                'height' => (int) $img->getAttribute('height'),
            ];
        }
        
        return [
            'type' => 'gallery',
            'images' => $images,
            'kadence' => true,
        ];
    }
    
    /**
     * Extract Kadence lottie animation
     *
     * @param \DOMNode $node DOM node
     * @return array Structured lottie data
     */
    public static function extract_lottie($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        $player = $xpath->query('.//dotlottie-player', $node)->item(0);
        
        $animation_url = '';
        if ($player) {
            $animation_url = $player->getAttribute('src');
        }
        
        return [
            'type' => 'lottie',
            'animationUrl' => $animation_url,
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence video popup
     *
     * @param \DOMNode $node DOM node
     * @return array Structured video popup data
     */
    public static function extract_video_popup($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        $link = $xpath->query('.//a', $node)->item(0);
        
        $video_url = $link ? $link->getAttribute('href') : '';
        
        return [
            'type' => 'video_popup',
            'videoUrl' => $video_url,
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence vector (SVG)
     *
     * @param \DOMNode $node DOM node
     * @return array Structured vector data
     */
    public static function extract_vector($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        $svg = $xpath->query('.//svg', $node)->item(0);
        
        return [
            'type' => 'vector',
            'svg' => $svg ? $node->ownerDocument->saveHTML($svg) : '',
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence button
     *
     * @param \DOMNode $node DOM node
     * @return array|null Structured button data or null
     */
    public static function extract_kadence_button($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        
        // Find link element
        $link = $xpath->query('.//a', $node)->item(0);
        
        if (!$link) {
            return null;
        }
        
        // Extract Kadence-specific attributes
        $icon = $xpath->query('.//*[contains(@class, "kb-icon")]', $node)->item(0);
        
        return [
            'type' => 'button',
            'text' => trim($link->textContent),
            'url' => $link->getAttribute('href'),
            'target' => $link->getAttribute('target'),
            'icon' => $icon ? $icon->getAttribute('class') : null,
            'style' => $node->getAttribute('class'),
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence table of contents
     *
     * @param \DOMNode $node DOM node
     * @return array Structured TOC data
     */
    public static function extract_table_of_contents($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        $title_node = $xpath->query('.//span[contains(@class, "kb-table-of-contents-title")]', $node)->item(0);
        $title = $title_node ? trim($title_node->textContent) : '';
        
        $links = $xpath->query('.//a[contains(@class, "kb-table-of-contents__entry")]', $node);
        $items = [];
        
        foreach ($links as $link) {
            $items[] = [
                'text' => trim($link->textContent),
                'anchor' => $link->getAttribute('href'),
            ];
        }
        
        return [
            'type' => 'table_of_contents',
            'title' => $title,
            'items' => $items,
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence progress bar
     *
     * @param \DOMNode $node DOM node
     * @return array Structured progress bar data
     */
    public static function extract_kadence_progress_bar($node) {
        $label = $node->getAttribute('data-label');
        $percent = (int) $node->getAttribute('data-percent');
        
        return [
            'type' => 'progress_bar',
            'label' => $label,
            'percent' => $percent,
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence countdown
     *
     * @param \DOMNode $node DOM node
     * @return array Structured countdown data
     */
    public static function extract_kadence_countdown($node) {
        $data_id = $node->getAttribute('data-id');
        
        return [
            'type' => 'countdown',
            'countdownId' => $data_id,
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence count up
     *
     * @param \DOMNode $node DOM node
     * @return array Structured count up data
     */
    public static function extract_countup($node) {
        $start = (int) $node->getAttribute('data-start');
        $end = (int) $node->getAttribute('data-end');
        $duration = (float) $node->getAttribute('data-duration');
        $prefix = $node->getAttribute('data-prefix');
        $suffix = $node->getAttribute('data-suffix');
        
        return [
            'type' => 'countup',
            'start' => $start,
            'end' => $end,
            'duration' => $duration,
            'prefix' => $prefix,
            'suffix' => $suffix,
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence icon
     *
     * @param \DOMNode $node DOM node
     * @return array Structured icon data
     */
    public static function extract_icon($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        $svg = $xpath->query('.//svg', $node)->item(0);
        
        return [
            'type' => 'icon',
            'icon' => $svg ? $node->ownerDocument->saveHTML($svg) : '',
            'class' => $node->getAttribute('class'),
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence icon list
     *
     * @param \DOMNode $node DOM node
     * @return array Structured icon list data
     */
    public static function extract_kadence_icon_list($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        $items = [];
        
        $list_items = $xpath->query('.//li', $node);
        
        foreach ($list_items as $li) {
            $icon = $xpath->query('.//*[contains(@class, "kb-icon")]', $li)->item(0);
            $text = trim($li->textContent);
            
            $items[] = [
                'text' => $text,
                'icon' => $icon ? $icon->getAttribute('class') : null,
            ];
        }
        
        return [
            'type' => 'icon_list',
            'items' => $items,
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence info box
     *
     * @param \DOMNode $node DOM node
     * @return array Structured info box data
     */
    public static function extract_kadence_infobox($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        
        $title = $xpath->query('.//*[contains(@class, "kt-blocks-info-box-title")]', $node)->item(0);
        $text = $xpath->query('.//*[contains(@class, "kt-blocks-info-box-text")]', $node)->item(0);
        
        return [
            'type' => 'infobox',
            'title' => $title ? trim($title->textContent) : '',
            'text' => $text ? trim($text->textContent) : '',
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence spacer
     *
     * @param \DOMNode $node DOM node
     * @return array Structured spacer data
     */
    public static function extract_spacer($node) {
        $height = $node->getAttribute('data-height');
        
        return [
            'type' => 'spacer',
            'height' => $height,
            'divider' => strpos($node->getAttribute('class'), 'kb-divider') !== false ? 'line' : 'space',
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence show more
     *
     * @param \DOMNode $node DOM node
     * @return array Structured show more data
     */
    public static function extract_show_more($node) {
        return [
            'type' => 'show_more',
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence form
     *
     * @param \DOMNode $node DOM node
     * @return array Structured form data
     */
    public static function extract_form($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        $fields = $xpath->query('.//div[contains(@class, "kadence-blocks-form-field")]', $node);
        
        $form_fields = [];
        foreach ($fields as $field) {
            $label_node = $xpath->query('.//label', $field)->item(0);
            $input_node = $xpath->query('.//input | .//textarea', $field)->item(0);
            
            if ($input_node) {
                $form_fields[] = [
                    'label' => $label_node ? trim($label_node->textContent) : '',
                    'type' => $input_node->getAttribute('type') ?: $input_node->nodeName,
                    'required' => $input_node->hasAttribute('required'),
                ];
            }
        }
        
        return [
            'type' => 'form',
            'fields' => $form_fields,
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence posts
     *
     * @param \DOMNode $node DOM node
     * @return array Structured posts data
     */
    public static function extract_posts($node) {
        return [
            'type' => 'posts',
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence testimonials
     *
     * @param \DOMNode $node DOM node
     * @return array Structured testimonials data
     */
    public static function extract_testimonials($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        $items = $xpath->query('.//li[contains(@class, "kt-testimonial-grid-item")]', $node);
        
        $testimonials = [];
        foreach ($items as $item) {
            $title_node = $xpath->query('.//h2[contains(@class, "kt-testimonial-title")]', $item)->item(0);
            $content_node = $xpath->query('.//blockquote[contains(@class, "kt-testimonial-content")]', $item)->item(0);
            $name_node = $xpath->query('.//div[contains(@class, "kt-testimonial-name")]', $item)->item(0);
            $occupation_node = $xpath->query('.//div[contains(@class, "kt-testimonial-occupation")]', $item)->item(0);
            
            $testimonials[] = [
                'title' => $title_node ? trim($title_node->textContent) : '',
                'content' => $content_node ? trim($content_node->textContent) : '',
                'name' => $name_node ? trim($name_node->textContent) : '',
                'occupation' => $occupation_node ? trim($occupation_node->textContent) : '',
            ];
        }
        
        return [
            'type' => 'testimonials',
            'items' => $testimonials,
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence Google Maps
     *
     * @param \DOMNode $node DOM node
     * @return array Structured maps data
     */
    public static function extract_google_maps($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        $iframe = $xpath->query('.//iframe', $node)->item(0);
        
        return [
            'type' => 'google_maps',
            'src' => $iframe ? $iframe->getAttribute('src') : '',
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence search
     *
     * @param \DOMNode $node DOM node
     * @return array Structured search data
     */
    public static function extract_search($node) {
        return [
            'type' => 'search',
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence navigation
     *
     * @param \DOMNode $node DOM node
     * @return array Structured navigation data
     */
    public static function extract_navigation($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        $links = $xpath->query('.//a', $node);
        
        $menu_items = [];
        foreach ($links as $link) {
            $menu_items[] = [
                'text' => trim($link->textContent),
                'url' => $link->getAttribute('href'),
            ];
        }
        
        return [
            'type' => 'navigation',
            'items' => $menu_items,
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
    
    /**
     * Extract Kadence identity (logo/site title)
     *
     * @param \DOMNode $node DOM node
     * @return array Structured identity data
     */
    public static function extract_identity($node) {
        $xpath = new \DOMXPath($node->ownerDocument);
        
        $logo = $xpath->query('.//img', $node)->item(0);
        $title = $xpath->query('.//*[contains(@class, "site-title")]', $node)->item(0);
        
        return [
            'type' => 'identity',
            'logo' => $logo ? $logo->getAttribute('src') : null,
            'title' => $title ? trim($title->textContent) : '',
            'html' => $node->ownerDocument->saveHTML($node),
        ];
    }
}
