<?php
/**
 * Kadence Block Patterns - Complete Set
 *
 * Registers all 28 Kadence Blocks patterns with semantic type mapping.
 * Uses multiple selector strategies to handle theme variations.
 * 
 * v1.3.1: Added sanity_target declarations for all patterns
 *
 * @package STCWHeadlessAssistant
 * @since 2.0.0
 */

if (!defined('ABSPATH')) exit;

// Use fully qualified class names since this file has no namespace
use STCW\Headless\Engine\Detector\PatternRegistry;
use STCW\Headless\Engine\Extractor\KadenceExtractors;

// ==========================================
// LAYOUT BLOCKS (Container/Structure)
// ==========================================

PatternRegistry::register('kadence_rowlayout', [
    'selectors' => [
        '.wp-block-kadence-rowlayout',
        '.kb-row-layout-wrap',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_rowlayout'],
    'priority' => 5,
    'confidence' => 0.95,
    'description' => 'Kadence Row Layout (container with columns)',
    'sanity_target' => 'rawHtml',  // Layout container - preserve structure
    'sanity_note' => 'Kadence row layout - extract inner column content',
]);

PatternRegistry::register('kadence_column', [
    'selectors' => [
        '.wp-block-kadence-column',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_column'],
    'priority' => 4,
    'confidence' => 0.95,
    'description' => 'Kadence Column (single column within row)',
    'sanity_target' => 'rawHtml',  // Layout container - preserve structure
    'sanity_note' => 'Kadence column - extract inner content',
]);

// ==========================================
// TEXT & TYPOGRAPHY
// ==========================================

PatternRegistry::register('kadence_advanced_heading', [
    'selectors' => [
        '.wp-block-kadence-advancedheading',
        '.kt-adv-heading',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_advanced_heading'],
    'priority' => 6,
    'confidence' => 0.98,
    'description' => 'Kadence Advanced Heading with styling',
    // Note: Converts to standard Portable Text heading (built-in)
]);

// ==========================================
// INTERACTIVE BLOCKS
// ==========================================

PatternRegistry::register('kadence_accordion', [
    'selectors' => [
        '.wp-block-kadence-accordion',
        '.kb-accordion-wrap',
        '.kt-accordion-wrap',
        '[data-block-type="kadence/accordion"]',
        'details.kb-accordion-pane',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_accordion'],
    'priority' => 10,
    'confidence' => 0.95,
    'validators' => [
        'has_accordion_structure',
    ],
    'description' => 'Kadence accordion/FAQ blocks',
    'sanity_target' => 'accordionBlock',  // v1.3.1: Semantic type
]);

PatternRegistry::register('kadence_tabs', [
    'selectors' => [
        '.kb-tabs-wrap',
        '.wp-block-kadence-tabs',
        '.kt-tabs-wrap',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_tabs'],
    'priority' => 10,
    'confidence' => 0.95,
    'validators' => [
        'has_tabs_structure',
    ],
    'description' => 'Kadence tabs blocks',
    'sanity_target' => 'accordionBlock',  // v1.3.1: Tabs → accordionBlock
]);

// ==========================================
// MEDIA BLOCKS
// ==========================================

PatternRegistry::register('kadence_advanced_gallery', [
    'selectors' => [
        '.wp-block-kadence-advancedgallery',
        '.kb-gallery-wrap',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_advanced_gallery'],
    'priority' => 7,
    'confidence' => 0.95,
    'description' => 'Kadence Advanced Gallery',
    'sanity_target' => 'rawHtml',  // Complex gallery - needs custom handling
    'sanity_note' => 'Kadence advanced gallery - consider manual image import',
]);

PatternRegistry::register('kadence_image', [
    'selectors' => [
        '.wp-block-kadence-image',
        '.kb-image',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_image'],
    'priority' => 6,
    'confidence' => 0.95,
    'description' => 'Kadence Advanced Image',
    'sanity_target' => 'imageBlock',  // Use standard image handling
]);

PatternRegistry::register('kadence_lottie', [
    'selectors' => [
        '.wp-block-kadence-lottie',
        '.kb-lottie-container',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_lottie'],
    'priority' => 6,
    'confidence' => 0.95,
    'description' => 'Kadence Lottie Animation',
    'sanity_target' => 'rawHtml',  // Animation - needs special handling
    'sanity_note' => 'Kadence Lottie animation - requires Lottie library in frontend',
]);

PatternRegistry::register('kadence_video_popup', [
    'selectors' => [
        '.wp-block-kadence-videopopup',
        '.kb-video-popup',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_video_popup'],
    'priority' => 7,
    'confidence' => 0.95,
    'description' => 'Kadence Video Popup',
    'sanity_target' => 'rawHtml',  // Interactive video - preserve HTML
    'sanity_note' => 'Kadence video popup - requires modal/lightbox in frontend',
]);

PatternRegistry::register('kadence_vector', [
    'selectors' => [
        '.wp-block-kadence-vector',
        '.kb-vector-container',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_vector'],
    'priority' => 6,
    'confidence' => 0.95,
    'description' => 'Kadence Vector (SVG graphic)',
    'sanity_target' => 'rawHtml',  // SVG needs preservation
    'sanity_note' => 'Kadence vector graphic - SVG embedded',
]);

// ==========================================
// BUTTONS & CALLS TO ACTION
// ==========================================

PatternRegistry::register('kadence_button', [
    'extends' => 'button',
    'selectors' => [
        '.kb-button',
        '.kt-button',
        '.wp-block-kadence-advancedbutton',
        '.kb-btn',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_kadence_button'],
    'priority' => 10,
    'confidence' => 0.9,
    'description' => 'Kadence advanced button blocks',
    'sanity_target' => 'customButton',  // Maps to existing button schema
]);

// ==========================================
// DATA DISPLAY
// ==========================================

PatternRegistry::register('kadence_table', [
    'selectors' => [
        '.wp-block-kadence-table',
        '.kb-table-container',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_table'],
    'priority' => 7,
    'confidence' => 0.95,
    'description' => 'Kadence Table',
    'sanity_target' => 'tableBlock',  // v1.3.1: Semantic type
]);

PatternRegistry::register('kadence_table_of_contents', [
    'selectors' => [
        '.wp-block-kadence-tableofcontents',
        '.kb-table-of-content-nav',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_table_of_contents'],
    'priority' => 7,
    'confidence' => 0.95,
    'description' => 'Kadence Table of Contents',
    'sanity_target' => 'rawHtml',  // Navigation element - preserve
    'sanity_note' => 'Kadence table of contents - regenerate in frontend',
]);

// ==========================================
// PROGRESS & ANIMATIONS
// ==========================================

PatternRegistry::register('kadence_progress_bar', [
    'selectors' => [
        '.kb-progress-bar',
        '.wp-block-kadence-progressbar',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_kadence_progress_bar'],
    'priority' => 10,
    'confidence' => 0.95,
    'description' => 'Kadence progress bar blocks',
    'sanity_target' => 'rawHtml',  // Interactive widget
    'sanity_note' => 'Kadence progress bar - interactive component',
]);

PatternRegistry::register('kadence_countdown', [
    'selectors' => [
        '.kb-countdown-timer',
        '.wp-block-kadence-countdown',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_kadence_countdown'],
    'priority' => 10,
    'confidence' => 0.95,
    'description' => 'Kadence countdown timer blocks',
    'sanity_target' => 'rawHtml',  // Dynamic timer
    'sanity_note' => 'Kadence countdown timer - requires JavaScript',
]);

PatternRegistry::register('kadence_countup', [
    'selectors' => [
        '.wp-block-kadence-countup',
        '.kb-count-up',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_countup'],
    'priority' => 6,
    'confidence' => 0.95,
    'description' => 'Kadence Count Up animation',
    'sanity_target' => 'rawHtml',  // Animation
    'sanity_note' => 'Kadence count up - animated counter',
]);

// ==========================================
// ICONS & DECORATIVE
// ==========================================

PatternRegistry::register('kadence_icon', [
    'selectors' => [
        '.wp-block-kadence-icon',
        '.kb-icon',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_icon'],
    'priority' => 5,
    'confidence' => 0.90,
    'description' => 'Kadence Icon',
    'sanity_target' => 'rawHtml',  // Icon needs SVG preservation
    'sanity_note' => 'Kadence icon - may need icon library in frontend',
]);

PatternRegistry::register('kadence_icon_list', [
    'extends' => 'list',
    'selectors' => [
        '.kb-icon-list',
        '.wp-block-kadence-iconlist',
        '.kt-svg-icon-list',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_kadence_icon_list'],
    'priority' => 10,
    'confidence' => 0.9,
    'description' => 'Kadence icon list blocks',
    'sanity_target' => 'rawHtml',  // Icons need preservation
    'sanity_note' => 'Kadence icon list - custom icons require manual conversion',
]);

PatternRegistry::register('kadence_infobox', [
    'selectors' => [
        '.kb-info-box',
        '.wp-block-kadence-infobox',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_kadence_infobox'],
    'priority' => 10,
    'confidence' => 0.95,
    'description' => 'Kadence info box blocks',
    'sanity_target' => 'rawHtml',  // Complex styling
    'sanity_note' => 'Kadence info box - contains custom styling and icons',
]);

// ==========================================
// LAYOUT & SPACING
// ==========================================

PatternRegistry::register('kadence_spacer', [
    'selectors' => [
        '.wp-block-kadence-spacer',
        '.kb-block-spacer',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_spacer'],
    'priority' => 5,
    'confidence' => 0.95,
    'description' => 'Kadence Spacer/Divider',
    'sanity_target' => 'separator',  // Use standard separator
]);

PatternRegistry::register('kadence_show_more', [
    'selectors' => [
        '.wp-block-kadence-show-more',
        '.kb-show-more',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_show_more'],
    'priority' => 6,
    'confidence' => 0.90,
    'description' => 'Kadence Show More/Less',
    'sanity_target' => 'rawHtml',  // Interactive toggle
    'sanity_note' => 'Kadence show more - expand/collapse functionality',
]);

// ==========================================
// DYNAMIC CONTENT
// ==========================================

PatternRegistry::register('kadence_form', [
    'selectors' => [
        '.wp-block-kadence-form',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_form'],
    'priority' => 7,
    'confidence' => 0.95,
    'description' => 'Kadence Form with fields',
    'sanity_target' => 'rawHtml',  // Forms need special handling
    'sanity_note' => 'Kadence form - consider using headless form solution',
]);

PatternRegistry::register('kadence_posts', [
    'selectors' => [
        '.wp-block-kadence-posts',
        '.kb-posts-wrap',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_posts'],
    'priority' => 7,
    'confidence' => 0.90,
    'description' => 'Kadence Posts Query/Grid',
    'sanity_target' => 'rawHtml',  // Dynamic content
    'sanity_note' => 'Kadence posts query - rebuild with headless query',
]);

PatternRegistry::register('kadence_testimonials', [
    'selectors' => [
        '.wp-block-kadence-testimonials',
        '.kb-testimonials-wrap',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_testimonials'],
    'priority' => 7,
    'confidence' => 0.95,
    'description' => 'Kadence Testimonials/Reviews',
    'sanity_target' => 'rawHtml',  // Slider/carousel
    'sanity_note' => 'Kadence testimonials - consider structured testimonial type',
]);

// ==========================================
// UTILITY BLOCKS
// ==========================================

PatternRegistry::register('kadence_google_maps', [
    'selectors' => [
        '.wp-block-kadence-googlemaps',
        '.kb-google-maps-container',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_google_maps'],
    'priority' => 6,
    'confidence' => 0.95,
    'description' => 'Kadence Google Maps',
    'sanity_target' => 'rawHtml',  // Embedded map
    'sanity_note' => 'Kadence Google Maps - requires Maps API in frontend',
]);

PatternRegistry::register('kadence_search', [
    'selectors' => [
        '.wp-block-kadence-search',
        '.kb-search-wrap',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_search'],
    'priority' => 6,
    'confidence' => 0.95,
    'description' => 'Kadence Search',
    'sanity_target' => 'rawHtml',  // Search widget
    'sanity_note' => 'Kadence search - implement headless search',
]);

PatternRegistry::register('kadence_navigation', [
    'selectors' => [
        '.wp-block-kadence-navigation',
        '.kb-navigation-wrap',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_navigation'],
    'priority' => 7,
    'confidence' => 0.95,
    'description' => 'Kadence Navigation Menu',
    'sanity_target' => 'rawHtml',  // Navigation structure
    'sanity_note' => 'Kadence navigation - rebuild menu structure in headless',
]);

PatternRegistry::register('kadence_identity', [
    'selectors' => [
        '.wp-block-kadence-identity',
        '.kb-identity-wrap',
    ],
    'extractor' => [KadenceExtractors::class, 'extract_identity'],
    'priority' => 6,
    'confidence' => 0.95,
    'description' => 'Kadence Identity (Logo/Site Title)',
    'sanity_target' => 'rawHtml',  // Site branding
    'sanity_note' => 'Kadence identity - site logo/title element',
]);

/**
 * Allow themes/plugins to modify Kadence patterns
 *
 * @param array $patterns Current patterns
 */
do_action('stcw_headless_kadence_patterns_loaded');
