<?php
/**
 * Plugin Name: SSO JumpCloud - Enterprise SAML & SCIM
 * Plugin URI: https://airtonvancin.com/plugin/sso-connector-for-jumpcloud
 * Description: Securely connect WordPress with JumpCloud for Enterprise SSO via SAML 2.0 and automated user provisioning via SCIM.
 * Version: 1.1.6
 * Author: Airton Vancin Junior
 * Author URI: https://airtonvancin.com
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: sso-connector-for-jumpcloud
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) {
    exit;
}

require_once plugin_dir_path(__FILE__) . 'vendor/autoload.php';

class Jumpssco_JumpCloud_SSO_Connector
{

    private $saml_settings;

    /**
     * Parse Role Mapping string into an array.
     *
     * @param string $mapping_string
     * @return array
     */
    private function jumpssco_parse_role_mapping($mapping_string)
    {
        $mapping = array();
        $lines = explode("\n", str_replace("\r", "", $mapping_string));
        foreach ($lines as $line) {
            $parts = explode(":", $line);
            if (count($parts) === 2) {
                $mapping[trim($parts[0])] = trim($parts[1]);
            }
        }
        return $mapping;
    }

    public function __construct()
    {
        add_action('admin_menu', array($this, 'jumpssco_add_admin_menu'));
        add_action('admin_init', array($this, 'jumpssco_settings_init'));
        add_filter('login_message', array($this, 'jumpssco_add_sso_button_to_message'));
        // Intercept default authentication when SSO-only is enabled
        add_filter('authenticate', array($this, 'jumpssco_maybe_disable_default_login'), 1, 3);
        // Role-based redirects for both SSO and default login
        add_filter('login_redirect', array($this, 'jumpssco_role_based_login_redirect'), 10, 3);
        add_action('login_init', array($this, 'jumpssco_sso_login'), 1);
        add_action('init', array($this, 'jumpssco_acs'));
        add_action('login_enqueue_scripts', array($this, 'jumpssco_enqueue_styles'));
        add_action('wp_ajax_jumpssco_test_sso_config', array($this, 'jumpssco_test_sso_config'));
        add_action('admin_enqueue_scripts', array($this, 'jumpssco_enqueue_admin_scripts'));
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), array($this, 'jumpssco_add_action_links'));
        add_filter('plugin_row_meta', array($this, 'jumpcloud_sso_plugin_row_meta'), 10, 2);
        add_action('admin_init', array($this, 'jumpssco_auto_activate_license'));
        add_action('wp_ajax_jumpssco_check_subscription', array($this, 'jumpssco_check_subscription'));
        add_action('wp_ajax_jumpssco_revalidate_license', array($this, 'jumpssco_revalidate_license'));
        add_action('rest_api_init', array($this, 'jumpssco_register_scim_routes'));
        add_filter('wp_authenticate_user', array($this, 'jumpssco_check_user_active_status'), 10, 1);
    }

    public function jumpssco_add_action_links($links)
    {
        $settings_link = '<a href="' . admin_url('options-general.php?page=jumpssco-sso') . '">' . __('Settings', 'sso-connector-for-jumpcloud') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }



    public function jumpssco_enqueue_admin_scripts($hook)
    {
        if ('settings_page_jumpssco-sso' !== $hook) {
            return;
        }

        // Enqueue admin-specific CSS
        $style_path = plugin_dir_path(__FILE__) . 'assets/css/admin.css';
        wp_enqueue_style(
            'jumpssco-admin-style',
            plugin_dir_url(__FILE__) . 'assets/css/admin.css',
            array(),
            file_exists($style_path) ? filemtime($style_path) : null
        );

        wp_enqueue_script(
            'jumpssco-admin',
            plugin_dir_url(__FILE__) . 'assets/js/admin.js',
            array('jquery'),
            '1.0.9',
            true
        );

        wp_localize_script(
            'jumpssco-admin',
            'jumpssco_admin',
            array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('jumpssco_test_config_action'),
            )
        );

        // Enqueue Stripe Pricing Table script for the premium tab
        // phpcs:ignore WordPress.WP.EnqueuedResources.NonEnqueuedScript -- Properly enqueued here.
        wp_enqueue_script('stripe-pricing-table', 'https://js.stripe.com/v3/pricing-table.js', array(), '3.0', true);
    }

    /**
     * Check if the plugin is in premium mode.
     * This checks for a valid license ID and a cached activation status.
     * Periodically validates with the remote server using transients.
     *
     * @return bool
     */
    public function jumpssco_is_premium()
    {
        $options = get_option('jumpssco_sso_settings', array());

        // If no license_id or is_premium flag, definitely not premium
        if (empty($options['license_id']) || empty($options['is_premium'])) {
            return false;
        }

        // Check if we need to revalidate with server
        $this->jumpssco_maybe_validate_license();

        // Re-fetch options after potential validation update
        $options = get_option('jumpssco_sso_settings', array());
        return !empty($options['license_id']) && !empty($options['is_premium']);
    }

    /**
     * Periodically validate license with Supabase server.
     * Uses transients to cache the result and avoid excessive API calls.
     * Validates every 12 hours.
     */
    private function jumpssco_maybe_validate_license()
    {
        $options = get_option('jumpssco_sso_settings', array());
        $license_key = isset($options['license_id']) ? $options['license_id'] : '';

        // No license to validate
        if (empty($license_key)) {
            return;
        }

        // Check if we've validated recently (12 hours cache)
        $cache_key = 'jumpssco_license_valid_' . md5($license_key);
        $cached_result = get_transient($cache_key);

        if ($cached_result !== false) {
            // If cached as invalid, deactivate now
            if ($cached_result === 'invalid') {
                $this->jumpssco_deactivate_license();
            }
            return;
        }

        // Validate with Supabase server
        $domain = get_site_url();
        $response = wp_remote_post('https://whotqyuonbzsiniztife.supabase.co/functions/v1/license-manager', array(
            'body' => wp_json_encode(array(
                'action' => 'validate',
                'license_key' => $license_key,
                'domain' => $domain
            )),
            'headers' => array('Content-Type' => 'application/json'),
            'timeout' => 10
        ));

        // On network error, assume valid and try again later (1 hour)
        if (is_wp_error($response)) {
            set_transient($cache_key, 'valid', HOUR_IN_SECONDS);
            return;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($body['valid'])) {
            // License is valid - cache for 12 hours
            set_transient($cache_key, 'valid', 12 * HOUR_IN_SECONDS);
        } else {
            // License is invalid - deactivate and cache result
            set_transient($cache_key, 'invalid', 12 * HOUR_IN_SECONDS);
            $this->jumpssco_deactivate_license();
        }
    }

    /**
     * Deactivate the local license.
     * Called when server validation fails.
     */
    private function jumpssco_deactivate_license()
    {
        $options = get_option('jumpssco_sso_settings', array());
        $options['license_id'] = '';
        $options['is_premium'] = false;
        update_option('jumpssco_sso_settings', $options);
        wp_cache_delete('jumpssco_sso_settings', 'options');
    }

    /**
     * Auto-activate license from URL parameter (after Stripe checkout redirect)
     */
    public function jumpssco_auto_activate_license()
    {
        // Only run on settings page
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only check for settings page context.
        if (!isset($_GET['page']) || $_GET['page'] !== 'jumpssco-sso') {
            return;
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only check for activation parameters from external redirect.
        if (!isset($_GET['license_key']) || !isset($_GET['activated'])) {
            return;
        }

        // Security check - only allow for admin users
        if (!current_user_can('manage_options')) {
            return;
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- External redirect from Stripe.
        $license_key = isset($_GET['license_key']) ? sanitize_text_field(wp_unslash($_GET['license_key'])) : '';

        // Get current options
        $options = get_option('jumpssco_sso_settings', array());

        // Only auto-activate if not already activated
        if (empty($options['is_premium'])) {
            // Validate license with Supabase
            $domain = get_site_url();
            $response = wp_remote_post('https://whotqyuonbzsiniztife.supabase.co/functions/v1/license-manager', array(
                'body' => json_encode(array(
                    'action' => 'validate',
                    'license_key' => $license_key,
                    'domain' => $domain
                )),
                'headers' => array('Content-Type' => 'application/json'),
                'timeout' => 15
            ));

            if (!is_wp_error($response)) {
                $body = json_decode(wp_remote_retrieve_body($response), true);

                if (!empty($body['valid'])) {
                    // Activate license
                    $options['license_id'] = $license_key;
                    $options['is_premium'] = true;
                    update_option('jumpssco_sso_settings', $options);

                    // Redirect to remove license_key from URL
                    wp_safe_redirect(admin_url('options-general.php?page=jumpssco-sso&tab=premium&license_activated=true'));
                    exit;
                }
            }
        }


        // If already activated or validation failed, just redirect to clean URL
        // If already activated or validation failed, just redirect to clean URL
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only check for clean-up redirect.
        if (isset($_GET['license_key'])) {
            wp_safe_redirect(admin_url('options-general.php?page=jumpssco-sso&tab=premium'));
            exit;
        }
    }

    public function jumpssco_check_subscription()
    {
        check_ajax_referer('jumpssco_check_sub_nonce', 'security');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Unauthorized', 'sso-connector-for-jumpcloud')));
        }

        $current_user = wp_get_current_user();
        $email = $current_user->user_email;
        $domain = get_site_url();


        // Call Supabase Edge Function
        $response = wp_remote_post('https://whotqyuonbzsiniztife.supabase.co/functions/v1/license-manager', array(
            'body' => json_encode(array(
                'action' => 'activate-by-email',
                'email' => $email,
                'domain' => $domain
            )),
            'headers' => array('Content-Type' => 'application/json'),
            'timeout' => 15
        ));

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => __('Connection failed. Please try again.', 'sso-connector-for-jumpcloud')));
        }

        $body_raw = wp_remote_retrieve_body($response);

        $body = json_decode($body_raw, true);

        if (!empty($body['valid']) && !empty($body['license_key'])) {
            // Success! Activate license locally
            $options = get_option('jumpssco_sso_settings', array());
            $options['license_id'] = sanitize_text_field($body['license_key']);
            $options['is_premium'] = true;
            update_option('jumpssco_sso_settings', $options);
            wp_cache_delete('jumpssco_sso_settings', 'options');

            wp_send_json_success(array('message' => __('License activated successfully! Refreshing...', 'sso-connector-for-jumpcloud')));
        } else {
            // Error from server
            $msg = isset($body['message']) ? $body['message'] : __('No active subscription found.', 'sso-connector-for-jumpcloud');
            wp_send_json_error(array('message' => $msg));
        }
    }

    /**
     * AJAX handler to force immediate license revalidation.
     * Clears transient cache and validates license with server.
     */
    public function jumpssco_revalidate_license()
    {
        check_ajax_referer('jumpssco_revalidate_nonce', 'security');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Unauthorized', 'sso-connector-for-jumpcloud')));
        }

        $options = get_option('jumpssco_sso_settings', array());
        $license_key = isset($options['license_id']) ? $options['license_id'] : '';

        // If no license, nothing to validate
        if (empty($license_key)) {
            wp_send_json_error(array('message' => __('No license found to validate.', 'sso-connector-for-jumpcloud')));
        }

        // Clear the transient cache to force revalidation
        $cache_key = 'jumpssco_license_valid_' . md5($license_key);
        delete_transient($cache_key);

        // Validate with Supabase server
        $domain = get_site_url();
        $response = wp_remote_post('https://whotqyuonbzsiniztife.supabase.co/functions/v1/license-manager', array(
            'body' => wp_json_encode(array(
                'action' => 'validate',
                'license_key' => $license_key,
                'domain' => $domain
            )),
            'headers' => array('Content-Type' => 'application/json'),
            'timeout' => 15
        ));

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => __('Connection failed. Please try again.', 'sso-connector-for-jumpcloud')));
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($body['valid'])) {
            // License is valid - cache for 12 hours
            set_transient($cache_key, 'valid', 12 * HOUR_IN_SECONDS);
            wp_send_json_success(array('message' => __('License is valid! Refreshing...', 'sso-connector-for-jumpcloud')));
        } else {
            // License is invalid - deactivate
            set_transient($cache_key, 'invalid', 12 * HOUR_IN_SECONDS);
            $this->jumpssco_deactivate_license();
            wp_send_json_success(array('message' => __('License is no longer valid. Premium features deactivated.', 'sso-connector-for-jumpcloud'), 'deactivated' => true));
        }
    }

    public function jumpssco_test_sso_config()
    {
        check_ajax_referer('jumpssco_test_config_action', 'security');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Unauthorized access.', 'sso-connector-for-jumpcloud'));
            return;
        }

        $options = get_option('jumpssco_sso_settings', array());

        // Validate required fields
        if (empty($options['idp_entity_id'])) {
            wp_send_json_error(__('IDP Entity ID is required.', 'sso-connector-for-jumpcloud'));
            return;
        }

        if (empty($options['sso_url'])) {
            wp_send_json_error(__('SSO URL is required.', 'sso-connector-for-jumpcloud'));
            return;
        }

        if (empty($options['certificate'])) {
            wp_send_json_error(__('Certificate is required.', 'sso-connector-for-jumpcloud'));
            return;
        }

        // Validate certificate format
        $certificate = trim($options['certificate']);
        if (strpos($certificate, 'BEGIN CERTIFICATE') === false || strpos($certificate, 'END CERTIFICATE') === false) {
            wp_send_json_error(__('Invalid certificate format. Please include BEGIN and END certificate markers.', 'sso-connector-for-jumpcloud'));
            return;
        }

        try {
            $settings = $this->jumpssco_get_saml_settings();

            // Try to instantiate Auth object to validate configuration
            $auth = new OneLogin\Saml2\Auth($settings);

            // Try to instantiate Settings object for additional validation
            $settings_obj = new OneLogin\Saml2\Settings($settings);

            // Check if settings object was created successfully
            if ($settings_obj) {
                wp_send_json_success(__('✓ Configuration is valid and ready to use!', 'sso-connector-for-jumpcloud'));
            } else {
                wp_send_json_error(__('Failed to create SAML settings object.', 'sso-connector-for-jumpcloud'));
            }
        } catch (Exception $e) {
            wp_send_json_error(__('Configuration error: ', 'sso-connector-for-jumpcloud') . esc_html($e->getMessage()));
        }
    }


    public function jumpssco_enqueue_styles()
    {
        // Enqueue frontend CSS with version to prevent caching issues
        $style_path = plugin_dir_path(__FILE__) . 'assets/css/style.css';
        wp_enqueue_style(
            'jumpssco-style',
            plugin_dir_url(__FILE__) . 'assets/css/style.css',
            array(),
            file_exists($style_path) ? filemtime($style_path) : null
        );

        $options = get_option('jumpssco_sso_settings', array());

        // If 'Disable Default Login' is active, perform a JS redirect to the SSO endpoint.
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only check to prevent redirect loops.
        if (!empty($options['disable_default_login']) && empty($_GET['sso'])) {
            $sso_url = wp_nonce_url(add_query_arg('sso', 'true', wp_login_url()), 'jumpssco_sso');
            $redirect_script = "window.location.href = '" . esc_js($sso_url) . "';";
            wp_add_inline_script('jumpssco-style', $redirect_script);
        }

        // If premium 'disable default login' is active, hide default login form fields
        if (!empty($options['disable_default_login'])) {
            $css = ".login form#loginform,
            .login .forgetmenot,
            .login #nav,
            .login #backtoblog { display: none !important; }";
            wp_add_inline_style('jumpssco-style', $css);
        }
    }

    private function jumpssco_get_saml_settings()
    {
        if (isset($this->saml_settings)) {
            return $this->saml_settings;
        }

        $options = get_option('jumpssco_sso_settings');

        $this->saml_settings = array(
            'sp' => array(
                'entityId' => home_url('/'),
                'assertionConsumerService' => array(
                    'url' => home_url('/?acs'),
                ),
                'singleLogoutService' => array(
                    'url' => home_url('/?sls'),
                ),
                'NameIDFormat' => 'urn:oasis:names:tc:SAML:1.1:nameid-format:emailAddress',
            ),
            'idp' => array(
                'entityId' => isset($options['idp_entity_id']) ? $options['idp_entity_id'] : '',
                'singleSignOnService' => array(
                    'url' => isset($options['sso_url']) ? $options['sso_url'] : '',
                ),
                'singleLogoutService' => array(
                    'url' => isset($options['slo_url']) ? $options['slo_url'] : '',
                ),
                'x509cert' => isset($options['certificate']) ? $options['certificate'] : '',
            ),
        );

        return $this->saml_settings;
    }

    public function jumpssco_add_sso_button_to_message($message)
    {
        if (empty($message)) {
            $message = '';
        }
        $logo_url = plugin_dir_url(__FILE__) . 'assets/images/jumpcloud-logo.svg';
        $sso_url = wp_nonce_url(add_query_arg('sso', 'true', wp_login_url()), 'jumpssco_sso');
        $button_html = '<div class="jc-sso-button-container"><a href="' . esc_url($sso_url) . '" class="jc-sso-button"><img src="' . esc_url($logo_url) . '" alt="' . esc_attr__('JumpCloud Logo', 'sso-connector-for-jumpcloud') . '">' . esc_html__('Login with JumpCloud', 'sso-connector-for-jumpcloud') . '</a></div>';
        $separator_html = '<div class="jc-sso-separator"><span>' . esc_html__('OR', 'sso-connector-for-jumpcloud') . '</span></div>';
        return $message . $button_html . $separator_html;
    }

    public function jumpssco_sso_login()
    {
        if (!empty($_GET['sso'])) {
            // Sanitize and verify nonce for SSO request
            $nonce = isset($_REQUEST['_wpnonce'])
                ? sanitize_text_field(wp_unslash($_REQUEST['_wpnonce']))
                : '';
            if (empty($nonce) || !wp_verify_nonce($nonce, 'jumpssco_sso')) {
                wp_die(
                    esc_html__('Invalid SSO request.', 'sso-connector-for-jumpcloud'),
                    esc_html__('SSO Error', 'sso-connector-for-jumpcloud'),
                    array('back_link' => true)
                );
            }
            $options = get_option('jumpssco_sso_settings', array());
            if (empty($options['idp_entity_id']) || empty($options['sso_url']) || empty($options['certificate'])) {
                wp_die(
                    esc_html__('SSO is not configured. Please configure JumpCloud SSO settings in the admin.', 'sso-connector-for-jumpcloud'),
                    esc_html__('SSO Not Configured', 'sso-connector-for-jumpcloud'),
                    array('back_link' => true)
                );
            }
            try {
                $settings = $this->jumpssco_get_saml_settings();
                $auth = new OneLogin\Saml2\Auth($settings);
                $auth->login();
            } catch (Exception $e) {
                wp_die(
                    esc_html__('Unable to start SSO. Please check your configuration.', 'sso-connector-for-jumpcloud'),
                    esc_html__('SSO Error', 'sso-connector-for-jumpcloud'),
                    array('back_link' => true)
                );
            }
        }
    }
    /**
     * Prevent default WP authentication when SSO-only is enabled.
     *
     * @param WP_User|WP_Error|null $user     Authenticated user or error.
     * @param string                $username Submitted username.
     * @param string                $password Submitted password.
     * @return WP_User|WP_Error
     */
    public function jumpssco_maybe_disable_default_login($user, $username, $password)
    {
        $options = get_option('jumpssco_sso_settings', array());
        if (!empty($options['disable_default_login'])) {
            // Only allow SSO flow
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only check for SSO flag
            if (empty($_GET['sso'])) {
                return new WP_Error(
                    'sso_only',
                    __('Login disabled. Please use JumpCloud SSO.', 'sso-connector-for-jumpcloud')
                );
            }
        }
        return $user;
    }

    /**
     * Filter `wp_authenticate_user` to check for SCIM deactivation flag.
     *
     * @param WP_User|WP_Error $user The user object or WP_Error.
     * @return WP_User|WP_Error The user object if active, or WP_Error if deactivated.
     */
    public function jumpssco_check_user_active_status($user)
    {
        if (is_wp_error($user)) {
            return $user;
        }

        if ($this->jumpssco_is_premium() && get_user_meta($user->ID, 'jumpssco_scim_inactive', true)) {
            return new WP_Error(
                'user_deactivated',
                __('Your account has been deactivated. Please contact your administrator.', 'sso-connector-for-jumpcloud')
            );
        }

        return $user;
    }

    /**
     * Log SSO Event (Premium feature)
     *
     * @param string $email User email
     * @param string $status 'success' or 'failed'
     * @param string $message Details
     */
    private function jumpssco_log_sso_event($email, $status, $message)
    {
        if (!$this->jumpssco_is_premium()) {
            return;
        }

        $logs = get_option('jumpssco_audit_logs', array());

        $options = get_option('jumpssco_sso_settings', array());
        $retention_days = isset($options['log_retention']) ? (int) $options['log_retention'] : 30;

        $new_log = array(
            'time' => current_time('mysql'),
            'email' => $email,
            'status' => $status,
            'message' => $message,
            'ip' => isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : ''
        );

        array_unshift($logs, $new_log);

        // Perform retention cleanup
        if ($retention_days > 0) {
            $cutoff = strtotime("-" . $retention_days . " days");
            $logs = array_filter($logs, function ($log) use ($cutoff) {
                return strtotime($log['time']) > $cutoff;
            });
        }

        $logs = array_slice($logs, 0, 500); // Keep max 500 entries regardless of retention to prevent DB bloat

        update_option('jumpssco_audit_logs', $logs);
    }

    /**
     * Determine redirect URL based on user role.
     *
     * @param WP_User $user User object.
     * @return string Redirect URL.
     */
    public function jumpssco_get_post_login_redirect_url($user)
    {
        $options = get_option('jumpssco_sso_settings', array());
        if (!empty($user->roles) && is_array($user->roles)) {
            foreach ($user->roles as $role) {
                $key = 'redirect_' . $role;
                if (!empty($options[$key])) {
                    return esc_url_raw($options[$key]);
                }
            }
        }
        // Check for default redirect
        if (!empty($options['redirect_default'])) {
            return esc_url_raw($options['redirect_default']);
        }
        // Default to dashboard
        return admin_url();
    }
    /**
     * Filter login redirect to a role-based URL.
     *
     * @param string           $redirect_to           The redirect destination URL.
     * @param string           $requested_redirect_to The requested redirect destination URL passed as a parameter.
     * @param WP_User|WP_Error $user                  WP_User if login was successful, WP_Error otherwise.
     * @return string Modified redirect URL.
     */
    public function jumpssco_role_based_login_redirect($redirect_to, $requested_redirect_to, $user)
    {
        if ($user instanceof WP_User) {
            return $this->jumpssco_get_post_login_redirect_url($user);
        }
        return $redirect_to;
    }

    public function jumpssco_acs()
    {
        if (isset($_POST['SAMLResponse'])) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
            try {
                $settings = $this->jumpssco_get_saml_settings();
                $auth = new OneLogin\Saml2\Auth($settings);
                $auth->processResponse();

                // Check for errors
                $errors = $auth->getErrors();
                if (!empty($errors)) {
                    $error_reason = $auth->getLastErrorReason();
                    $this->jumpssco_log_sso_event($email ?? 'unknown', 'failed', 'SAML Error: ' . $error_reason);
                    wp_die(
                        esc_html__('SAML Error: ', 'sso-connector-for-jumpcloud') . esc_html($error_reason),
                        esc_html__('SSO Authentication Error', 'sso-connector-for-jumpcloud'),
                        array('back_link' => true)
                    );
                }

                if (!$auth->isAuthenticated()) {
                    $this->jumpssco_log_sso_event($email ?? 'unknown', 'failed', 'Authentication failed.');
                    wp_die(
                        esc_html__('Authentication failed. Please try again.', 'sso-connector-for-jumpcloud'),
                        esc_html__('SSO Authentication Failed', 'sso-connector-for-jumpcloud'),
                        array('back_link' => true)
                    );
                }

                $email = $auth->getNameId();
                $attributes = $auth->getAttributes();

                if (empty($email)) {
                    $this->jumpssco_log_sso_event('unknown', 'failed', 'No email address provided in SAML response.');
                    wp_die(
                        esc_html__('No email address provided in SAML response.', 'sso-connector-for-jumpcloud'),
                        esc_html__('SSO Error', 'sso-connector-for-jumpcloud'),
                        array('back_link' => true)
                    );
                }

                // Premium: Group-Based Access Control
                if ($this->jumpssco_is_premium()) {
                    $options = get_option('jumpssco_sso_settings', array()); // Re-fetch options to ensure 'allowed_groups' is available
                    $allowed_groups_str = isset($options['allowed_groups']) ? $options['allowed_groups'] : '';
                    if (!empty($allowed_groups_str)) {
                        $allowed_groups = array_map('trim', explode(',', $allowed_groups_str));
                        $user_groups = isset($attributes['groups']) ? $attributes['groups'] : array();

                        $has_access = false;
                        foreach ($user_groups as $group) {
                            if (in_array($group, $allowed_groups, true)) {
                                $has_access = true;
                                break;
                            }
                        }

                        if (!$has_access) {
                            $this->jumpssco_log_sso_event($email, 'failed', __('Access denied: Not in allowed groups.', 'sso-connector-for-jumpcloud'));
                            wp_die(
                                esc_html__('You do not have permission to access this site. Please contact your administrator if you believe this is an error.', 'sso-connector-for-jumpcloud'),
                                esc_html__('SSO Access Denied', 'sso-connector-for-jumpcloud'),
                                array('back_link' => true)
                            );
                        }
                    }
                }

                $user = get_user_by('email', $email);

                if ($user) {
                    // Premium: Check for SCIM deactivation
                    if ($this->jumpssco_is_premium() && get_user_meta($user->ID, 'jumpssco_scim_inactive', true)) {
                        $this->jumpssco_log_sso_event($email, 'failed', __('Access denied: Account deactivated via SCIM.', 'sso-connector-for-jumpcloud'));
                        wp_die(
                            esc_html__('Your account has been deactivated. Please contact your administrator.', 'sso-connector-for-jumpcloud'),
                            esc_html__('Access Denied', 'sso-connector-for-jumpcloud'),
                            array('back_link' => true)
                        );
                    }
                }

                if (!$user) {
                    $username = explode('@', $email)[0];
                    $password = wp_generate_password();
                    $user_id = wp_create_user($username, $password, $email);

                    if (is_wp_error($user_id)) {
                        $this->jumpssco_log_sso_event($email, 'failed', 'Error creating user: ' . $user_id->get_error_message());
                        wp_die(
                            esc_html__('Error creating user: ', 'sso-connector-for-jumpcloud') . esc_html($user_id->get_error_message()),
                            esc_html__('SSO User Creation Error', 'sso-connector-for-jumpcloud'),
                            array('back_link' => true)
                        );
                    }

                    $user = get_user_by('id', $user_id);
                    $user->set_role('subscriber');
                }

                // Premium Features: Role Mapping & Profile Sync
                if ($this->jumpssco_is_premium()) {
                    // 1. Profile Sync (First Name / Last Name)
                    $first_name = isset($attributes['firstname'][0]) ? $attributes['firstname'][0] : (isset($attributes['First Name'][0]) ? $attributes['First Name'][0] : '');
                    $last_name = isset($attributes['lastname'][0]) ? $attributes['lastname'][0] : (isset($attributes['Last Name'][0]) ? $attributes['Last Name'][0] : '');

                    if (!empty($first_name)) {
                        update_user_meta($user->ID, 'first_name', sanitize_text_field($first_name));
                    }
                    if (!empty($last_name)) {
                        update_user_meta($user->ID, 'last_name', sanitize_text_field($last_name));
                    }

                    // 2. Role Mapping
                    if (!empty($attributes['groups'])) {
                        $options = get_option('jumpssco_sso_settings', array());
                        $mapping_str = isset($options['role_mapping']) ? $options['role_mapping'] : '';
                        $mapping = $this->jumpssco_parse_role_mapping($mapping_str);

                        foreach ($attributes['groups'] as $group) {
                            if (isset($mapping[$group])) {
                                $user->set_role($mapping[$group]);
                                break; // Apply first matching role
                            }
                        }
                    }
                }

                wp_set_current_user($user->ID);
                if ($this->jumpssco_is_premium() && $user instanceof WP_User) {
                    $this->jumpssco_log_sso_event($email, 'success', __('Login successful.', 'sso-connector-for-jumpcloud'));
                }
                wp_set_auth_cookie($user->ID);

                // Redirect user based on role settings
                $redirect_url = $this->jumpssco_get_post_login_redirect_url($user);
                wp_safe_redirect($redirect_url);
                exit();
            } catch (Exception $e) {
                $this->jumpssco_log_sso_event($email ?? 'unknown', 'failed', 'SSO Processing Error: ' . $e->getMessage());
                wp_die(
                    esc_html__('SSO Error: ', 'sso-connector-for-jumpcloud') . esc_html($e->getMessage()),
                    esc_html__('SSO Processing Error', 'sso-connector-for-jumpcloud'),
                    array('back_link' => true)
                );
            }
        }
    }


    public function jumpssco_add_admin_menu()
    {
        add_options_page(
            __('SSO JumpCloud - Enterprise SAML & SCIM', 'sso-connector-for-jumpcloud'),
            __('SSO JumpCloud', 'sso-connector-for-jumpcloud'),
            'manage_options',
            'jumpssco-sso',
            array($this, 'jumpssco_options_page')
        );
    }

    public function jumpssco_settings_init()
    {
        register_setting(
            'jumpssco_sso_options',
            'jumpssco_sso_settings',
            array(
                'sanitize_callback' => array($this, 'jumpssco_sanitize_settings'),
                'default' => array(),
            )
        );
        // Determine if license is active (enables premium fields)
        $settings = get_option('jumpssco_sso_settings', array());
        $license_active = !empty($settings['license_id']);

        add_settings_section(
            'jumpssco_sso_section',
            __('Enterprise SAML & SCIM Settings', 'sso-connector-for-jumpcloud'),
            null,
            'jumpssco_sso_options'
        );

        // License key input (required to enable Premium features)
        add_settings_field(
            'jumpssco_idp_entity_id',
            __('IDP Entity ID (Required)', 'sso-connector-for-jumpcloud'),
            array($this, 'jumpssco_render_text_field'),
            'jumpssco_sso_options',
            'jumpssco_sso_section',
            array(
                'label_for' => 'jumpssco_idp_entity_id',
                'name' => 'jumpssco_sso_settings[idp_entity_id]',
                'placeholder' => __('e.g. https://sso.jumpcloud.com/saml2/your-app-id', 'sso-connector-for-jumpcloud'),
                'description' => __('The unique identifier for your Identity Provider (IdP), provided by JumpCloud. Also known as \'Issuer URL\'.', 'sso-connector-for-jumpcloud'),
            )
        );

        add_settings_field(
            'jumpssco_sso_sso_url',
            __('SSO URL (Required)', 'sso-connector-for-jumpcloud'),
            array($this, 'jumpssco_render_text_field'),
            'jumpssco_sso_options',
            'jumpssco_sso_section',
            array(
                'label_for' => 'jumpssco_sso_sso_url',
                'name' => 'jumpssco_sso_settings[sso_url]',
                'placeholder' => __('e.g. https://sso.jumpcloud.com/saml2/your-app-id', 'sso-connector-for-jumpcloud'),
                'description' => __('The URL to which WordPress will redirect the user for authentication. Provided by JumpCloud.', 'sso-connector-for-jumpcloud'),
            )
        );

        add_settings_field(
            'jumpssco_slo_url',
            __('SLO URL (Opcional)', 'sso-connector-for-jumpcloud'),
            array($this, 'jumpssco_render_text_field'),
            'jumpssco_sso_options',
            'jumpssco_sso_section',
            array(
                'label_for' => 'jumpssco_slo_url',
                'name' => 'jumpssco_sso_settings[slo_url]',
                'placeholder' => __('e.g. https://sso.jumpcloud.com/saml2/logout/your-app-id', 'sso-connector-for-jumpcloud'),
                'description' => __('(Optional) The URL to which WordPress will send a logout request to terminate the session at JumpCloud. If unsure, you can leave it blank.', 'sso-connector-for-jumpcloud'),
            )
        );

        add_settings_field(
            'jumpssco_certificate',
            __('Certificate (Required)', 'sso-connector-for-jumpcloud'),
            array($this, 'jumpssco_render_textarea_field'),
            'jumpssco_sso_options',
            'jumpssco_sso_section',
            array(
                'label_for' => 'jumpssco_certificate',
                'name' => 'jumpssco_sso_settings[certificate]',
                'placeholder' => __('-----BEGIN CERTIFICATE-----\nPaste your X.509 certificate here\n-----END CERTIFICATE-----', 'sso-connector-for-jumpcloud'),
                'description' => __('The X.509 certificate used to verify the signature of SAML responses sent by JumpCloud.', 'sso-connector-for-jumpcloud'),
            )
        );

        // Premium feature moved to General tab, but now enabled by default
        add_settings_field(
            'jumpssco_disable_default_login',
            __('Disable Default Login', 'sso-connector-for-jumpcloud'),
            array($this, 'jumpssco_render_checkbox_field'),
            'jumpssco_sso_options',
            'jumpssco_sso_section', // Moved to General section
            array(
                'label_for' => 'jumpssco_disable_default_login',
                'name' => 'jumpssco_sso_settings[disable_default_login]',
                'description' => __('Disables default username/password login. Only SSO will be allowed.', 'sso-connector-for-jumpcloud'),
            )
        );
        // Premium features section
        // Premium features section
        add_settings_section(
            'jumpssco_sso_section_premium',
            __('Premium Features', 'sso-connector-for-jumpcloud'),
            array($this, 'jumpssco_premium_section_description'),
            'jumpssco_sso_options'
        );

        // Subscription Status Display
        add_settings_field(
            'jumpssco_subscription_status',
            __('Subscription Status', 'sso-connector-for-jumpcloud'),
            array($this, 'jumpssco_render_subscription_status'),
            'jumpssco_sso_options',
            'jumpssco_sso_section_premium'
        );

        // Premium: Group-Based Access Control and Role Mapping are maintained in the main section

        // Premium: Group to Role Mapping
        add_settings_field(
            'jumpssco_role_mapping',
            __('Role Mapping (Groups → Roles)', 'sso-connector-for-jumpcloud'),
            array($this, 'jumpssco_render_textarea_field'),
            'jumpssco_sso_options',
            'jumpssco_sso_section_premium',
            array(
                'label_for' => 'jumpssco_role_mapping',
                'name' => 'jumpssco_sso_settings[role_mapping]',
                'placeholder' => "Admins:administrator\nEditors:editor",
                'description' => __('Map JumpCloud Attribute "Groups" to WordPress Roles. One per line (Format: GroupName:role_slug/name).', 'sso-connector-for-jumpcloud'),
                'disabled' => !$license_active,
            )
        );

        // Premium: Post-Login Redirects Section
        add_settings_section(
            'jumpssco_sso_section_redirects',
            __('Post-Login Redirects', 'sso-connector-for-jumpcloud'),
            array($this, 'jumpssco_redirect_section_description'),
            'jumpssco_sso_options'
        );

        $roles = array(
            'administrator' => __('Administrator', 'sso-connector-for-jumpcloud'),
            'editor' => __('Editor', 'sso-connector-for-jumpcloud'),
            'author' => __('Author', 'sso-connector-for-jumpcloud'),
            'contributor' => __('Contributor', 'sso-connector-for-jumpcloud'),
            'subscriber' => __('Subscriber', 'sso-connector-for-jumpcloud'),
            'default' => __('Default (All others)', 'sso-connector-for-jumpcloud'),
        );

        foreach ($roles as $role_slug => $role_name) {
            add_settings_field(
                'jumpssco_redirect_' . $role_slug,
                $role_name,
                array($this, 'jumpssco_render_redirect_field'),
                'jumpssco_sso_options',
                'jumpssco_sso_section_redirects',
                array('role' => $role_slug, 'disabled' => !$license_active)
            );
        }

        // Premium: Group-Based Access Control
        add_settings_field(
            'jumpssco_allowed_groups',
            __('Allowed Groups', 'sso-connector-for-jumpcloud'),
            array($this, 'jumpssco_render_text_field'),
            'jumpssco_sso_options',
            'jumpssco_sso_section_premium',
            array(
                'label_for' => 'jumpssco_allowed_groups',
                'name' => 'jumpssco_sso_settings[allowed_groups]',
                'placeholder' => __('e.g. IT-Dept, Managers', 'sso-connector-for-jumpcloud'),
                'description' => __('Comma-separated list of JumpCloud groups allowed to log in. Leave blank to allow all authenticated users.', 'sso-connector-for-jumpcloud'),
                'disabled' => !$license_active,
            )
        );

        // Premium: Audit Logs Section
        add_settings_section(
            'jumpssco_sso_section_audit',
            __('Audit Logs', 'sso-connector-for-jumpcloud'),
            array($this, 'jumpssco_audit_section_description'),
            'jumpssco_sso_options'
        );

        add_settings_field(
            'jumpssco_log_retention',
            __('Log Retention (Days)', 'sso-connector-for-jumpcloud'),
            array($this, 'jumpssco_render_text_field'),
            'jumpssco_sso_options',
            'jumpssco_sso_section_audit',
            array(
                'label_for' => 'jumpssco_log_retention',
                'name' => 'jumpssco_sso_settings[log_retention]',
                'type' => 'number',
                'placeholder' => '30',
                'description' => __('Number of days to keep audit logs. Enter 0 for unlimited.', 'sso-connector-for-jumpcloud'),
                'disabled' => !$license_active
            )
        );

        add_settings_field(
            'jumpssco_audit_logs_table',
            __('Recent Activity', 'sso-connector-for-jumpcloud'),
            array($this, 'jumpssco_render_audit_logs'),
            'jumpssco_sso_options',
            'jumpssco_sso_section_audit'
        );

        // Premium: SCIM Provisioning Section
        add_settings_section(
            'jumpssco_sso_section_scim',
            __('SCIM Provisioning (Premium)', 'sso-connector-for-jumpcloud'),
            array($this, 'jumpssco_scim_section_description'),
            'jumpssco_sso_options'
        );

        add_settings_field(
            'jumpssco_scim_enabled',
            __('Enable SCIM Provisioning', 'sso-connector-for-jumpcloud'),
            array($this, 'jumpssco_render_checkbox_field'),
            'jumpssco_sso_options',
            'jumpssco_sso_section_scim',
            array(
                'label_for' => 'jumpssco_scim_enabled',
                'name' => 'jumpssco_sso_settings[scim_enabled]',
                'description' => __('Allow JumpCloud to automatically create, update, and deactivate users.', 'sso-connector-for-jumpcloud'),
                'disabled' => !$license_active
            )
        );

        add_settings_field(
            'jumpssco_scim_token',
            __('SCIM Bearer Token', 'sso-connector-for-jumpcloud'),
            array($this, 'jumpssco_render_scim_token_field'),
            'jumpssco_sso_options',
            'jumpssco_sso_section_scim',
            array('disabled' => !$license_active)
        );
    }

    /**
     * Render the audit logs table
     */
    public function jumpssco_render_audit_logs()
    {
        $options = get_option('jumpssco_sso_settings', array());
        if (empty($options['is_premium'])) {
            echo '<p><i>' . esc_html__('Upgrade to Premium to view audit logs.', 'sso-connector-for-jumpcloud') . '</i></p>';
            return;
        }

        $logs = get_option('jumpssco_audit_logs', array());
        if (empty($logs)) {
            echo '<p>' . esc_html__('No recent login activity.', 'sso-connector-for-jumpcloud') . '</p>';
            return;
        }

        echo '<div class="jc-audit-logs-table-wrapper">';
        echo '<table class="wp-list-table widefat striped">';
        /* translators: 1: Time, 2: User, 3: Result, 4: Details, 5: IP */
        echo '<thead><tr><th>' . esc_html__('Time', 'sso-connector-for-jumpcloud') . '</th><th>' . esc_html__('User', 'sso-connector-for-jumpcloud') . '</th><th>' . esc_html__('Result', 'sso-connector-for-jumpcloud') . '</th><th>' . esc_html__('Details', 'sso-connector-for-jumpcloud') . '</th><th>' . esc_html__('IP', 'sso-connector-for-jumpcloud') . '</th></tr></thead>';
        echo '<tbody>';
        foreach ($logs as $log) {
            $status_color = ($log['status'] === 'success') ? '#46b450' : '#dc3232';
            echo '<tr>';
            echo '<td>' . esc_html($log['time']) . '</td>';
            echo '<td>' . esc_html($log['email']) . '</td>';
            echo '<td><span style="color:' . esc_attr($status_color) . '; font-weight:bold;">' . esc_html(strtoupper($log['status'])) . '</span></td>';
            echo '<td>' . esc_html($log['message']) . '</td>';
            echo '<td>' . esc_html($log['ip']) . '</td>';
            echo '</tr>';
        }
        echo '</tbody></table>';
        echo '</div>';
        echo '<p class="description">' . esc_html__('Showing last 20 events.', 'sso-connector-for-jumpcloud') . '</p>';

        $export_url = wp_nonce_url(add_query_arg('action', 'jumpssco_export_logs', admin_url('options-general.php?page=jumpssco-sso')), 'jumpssco_export_logs');
        echo '<p><a href="' . esc_url($export_url) . '" class="button">' . esc_html__('Export to CSV', 'sso-connector-for-jumpcloud') . '</a></p>';
    }

    /**
     * Section description for Audit Logs
     */
    public function jumpssco_audit_section_description()
    {
        // No extra description needed
        add_action('admin_init', array($this, 'jumpssco_handle_log_export'));
    }

    /**
     * Handle CSV export of audit logs
     */
    public function jumpssco_handle_log_export()
    {
        if (!isset($_GET['action']) || $_GET['action'] !== 'jumpssco_export_logs') {
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Unauthorized access.', 'sso-connector-for-jumpcloud'));
        }

        $nonce = isset($_GET['_wpnonce']) ? sanitize_text_field(wp_unslash($_GET['_wpnonce'])) : '';
        if (!wp_verify_nonce($nonce, 'jumpssco_export_logs')) {
            wp_die(esc_html__('Security check failed.', 'sso-connector-for-jumpcloud'));
        }

        if (!$this->jumpssco_is_premium()) {
            wp_die(esc_html__('Premium feature required.', 'sso-connector-for-jumpcloud'));
        }

        $logs = get_option('jumpssco_audit_logs', array());

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=jumpcloud-sso-logs-' . gmdate('Y-m-d') . '.csv');

        $output = fopen('php://output', 'w');
        fputcsv($output, array(
            esc_html__('Time', 'sso-connector-for-jumpcloud'),
            esc_html__('User', 'sso-connector-for-jumpcloud'),
            esc_html__('Status', 'sso-connector-for-jumpcloud'),
            esc_html__('Message', 'sso-connector-for-jumpcloud'),
            esc_html__('IP', 'sso-connector-for-jumpcloud')
        ));

        foreach ($logs as $log) {
            fputcsv($output, array($log['time'], $log['email'], strtoupper($log['status']), $log['message'], $log['ip']));
        }

        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Direct output stream closure.
        fclose($output);
        exit;
    }

    /**
     * Sanitize plugin settings input.
     *
     * @param array $input Raw input from settings form.
     * @return array Sanitized settings.
     */
    public function jumpssco_sanitize_settings($input)
    {
        $old_options = get_option('jumpssco_sso_settings', array());
        $sanitized = $old_options; // Initialise with old options to prevent wiping out fields from other tabs

        if (isset($input['idp_entity_id'])) {
            $sanitized['idp_entity_id'] = sanitize_text_field(wp_unslash($input['idp_entity_id']));
        }
        if (isset($input['sso_url'])) {
            $sanitized['sso_url'] = esc_url_raw(wp_unslash($input['sso_url']));
        }
        if (isset($input['slo_url'])) {
            $sanitized['slo_url'] = esc_url_raw(wp_unslash($input['slo_url']));
        }
        if (isset($input['certificate'])) {
            $sanitized['certificate'] = sanitize_textarea_field(wp_unslash($input['certificate']));
        }

        // Handle Checkboxes (they are only present if checked)
        // Only update if we are on the tab where they exist
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only check for tab navigation during sanitization.
        $current_tab = isset($_GET['tab']) ? sanitize_key(wp_unslash($_GET['tab'])) : 'general';
        if ($current_tab === 'general') {
            $sanitized['disable_default_login'] = !empty($input['disable_default_login']);
        }

        // License ID activation logic
        if (isset($input['license_id'])) {
            $new_license = sanitize_text_field(wp_unslash($input['license_id']));
            $old_license = isset($old_options['license_id']) ? $old_options['license_id'] : '';
            $old_is_premium = isset($old_options['is_premium']) ? $old_options['is_premium'] : false;

            $sanitized['license_id'] = $new_license;

            // Only validate via API if license changed
            if (!empty($new_license) && $new_license !== $old_license) {
                // Real Validation API Call
                $api_url = 'https://whotqyuonbzsiniztife.supabase.co/functions/v1/license-manager';
                $response = wp_remote_post($api_url, array(
                    'headers' => array('Content-Type' => 'application/json'),
                    'body' => json_encode(array(
                        'action' => 'validate',
                        'license_key' => $new_license,
                        'domain' => get_site_url()
                    )),
                    'timeout' => 15
                ));

                if (is_wp_error($response)) {
                    add_settings_error('jumpssco_sso_options', 'license_error', esc_html__('Could not connect to license server. Please try again later.', 'sso-connector-for-jumpcloud'), 'error');
                    $sanitized['is_premium'] = false;
                } else {
                    $body = wp_remote_retrieve_body($response);
                    $data = json_decode($body, true);

                    if (isset($data['valid']) && $data['valid'] === true) {
                        $sanitized['is_premium'] = true;
                        add_settings_error('jumpssco_sso_options', 'license_active', esc_html__('Premium features activated successfully!', 'sso-connector-for-jumpcloud'), 'updated');
                    } else {
                        $sanitized['is_premium'] = false;
                        $message = isset($data['message']) ? sanitize_text_field($data['message']) : esc_html__('Invalid license key.', 'sso-connector-for-jumpcloud');
                        add_settings_error('jumpssco_sso_options', 'license_invalid', $message, 'error');
                    }
                }

            } elseif (empty($new_license)) {
                $sanitized['is_premium'] = false;
            } else {
                // License didn't change, preserve current premium status (even if not in input)
                // If it IS in input (hidden field), take it, otherwise keep old
                if (isset($input['is_premium'])) {
                    $sanitized['is_premium'] = ($input['is_premium'] === '1' || $input['is_premium'] === true);
                } else {
                    $sanitized['is_premium'] = $old_is_premium;
                }
            }
        }

        // Premium Fields
        if (isset($input['role_mapping'])) {
            $sanitized['role_mapping'] = sanitize_textarea_field(wp_unslash($input['role_mapping']));
        }
        if (isset($input['allowed_groups'])) {
            $sanitized['allowed_groups'] = sanitize_text_field(wp_unslash($input['allowed_groups']));
        }
        if (isset($input['log_retention'])) {
            $sanitized['log_retention'] = absint(wp_unslash($input['log_retention']));
        }
        if (isset($input['scim_enabled'])) {
            $sanitized['scim_enabled'] = !empty($input['scim_enabled']);
        }
        if (isset($input['scim_token'])) {
            $sanitized['scim_token'] = sanitize_text_field(wp_unslash($input['scim_token']));
        }

        // Redirects
        $roles = array('administrator', 'editor', 'author', 'contributor', 'subscriber', 'default');
        foreach ($roles as $role) {
            $key = 'redirect_' . $role;
            if (isset($input[$key])) {
                $sanitized[$key] = esc_url_raw(wp_unslash($input[$key]));
            }
        }

        return $sanitized;
    }

    public function jumpssco_render_text_field($args)
    {
        $options = get_option('jumpssco_sso_settings');
        $name = substr($args['name'], strpos($args['name'], "[") + 1, -1);
        $value = isset($options[$name]) ? $options[$name] : '';
        $type = isset($args['type']) ? $args['type'] : 'text';
        ?>
        <input id="<?php echo esc_attr($args['label_for']); ?>" type="<?php echo esc_attr($type); ?>"
            name="<?php echo esc_attr($args['name']); ?>" value="<?php echo esc_attr($value); ?>" class="regular-text" <?php if (!empty($args['placeholder'])): ?> placeholder="<?php echo esc_attr($args['placeholder']); ?>" <?php endif; ?>         <?php disabled(!empty($args['disabled'])); ?>>
        <?php if (!empty($args['description'])): ?>
            <p class="description"><?php echo esc_html($args['description']); ?></p>
        <?php endif; ?>
    <?php
    }

    public function jumpssco_render_textarea_field($args)
    {
        $options = get_option('jumpssco_sso_settings');
        $name = substr($args['name'], strpos($args['name'], "[") + 1, -1);
        $value = isset($options[$name]) ? $options[$name] : '';
        ?>
        <textarea id="<?php echo esc_attr($args['label_for']); ?>" name="<?php echo esc_attr($args['name']); ?>" rows="10"
            cols="50" class="large-text" <?php if (!empty($args['placeholder'])): ?>
                placeholder="<?php echo esc_attr($args['placeholder']); ?>" <?php endif; ?>         <?php disabled(!empty($args['disabled'])); ?>><?php echo esc_textarea($value); ?></textarea>
        <?php if (!empty($args['description'])): ?>
            <p class="description"><?php echo esc_html($args['description']); ?></p>
        <?php endif; ?>
    <?php
    }
    /**
     * Render a checkbox field for settings.
     *
     * @param array $args Field arguments.
     */
    public function jumpssco_render_checkbox_field($args)
    {
        $options = get_option('jumpssco_sso_settings');
        $name = substr($args['name'], strpos($args['name'], "[") + 1, -1);
        $is_checked = !empty($options[$name]);
        $is_disabled = !empty($args['disabled']);
        printf(
            '<label><input type="checkbox" id="%1$s" name="%2$s" value="1" %3$s %4$s /> %5$s</label>',
            esc_attr($args['label_for']),
            esc_attr($args['name']),
            checked($is_checked, true, false),
            disabled($is_disabled, true, false),
            isset($args['description']) ? esc_html($args['description']) : ''
        );
    }

    /**
     * Render a redirect URL field for a specific role
     *
     * @param array $args Arguments (contains 'role')
     */
    public function jumpssco_render_redirect_field($args)
    {
        $role = isset($args['role']) ? $args['role'] : 'default';
        $options = get_option('jumpssco_sso_settings');
        $id = 'redirect_' . $role;
        $value = isset($options[$id]) ? $options[$id] : '';
        $is_disabled = !empty($args['disabled']);

        echo '<input type="url" id="' . esc_attr($id) . '" name="jumpssco_sso_settings[' . esc_attr($id) . ']" value="' . esc_attr($value) . '" class="regular-text" placeholder="' . esc_attr(home_url('/')) . '" ' . disabled($is_disabled, true, false) . '>';
        /* translators: %s: user role name */
        echo '<p class="description">' . sprintf(esc_html__('URL for %s', 'sso-connector-for-jumpcloud'), esc_html($role)) . '</p>';
    }

    /**
     * Section description for redirects
     */
    public function jumpssco_redirect_section_description()
    {
        echo '<p>' . esc_html__('Specify custom landing pages after a successful login based on user roles.', 'sso-connector-for-jumpcloud') . '</p>';
    }

    /**
     * Callback to render the Premium Features section description.
     */
    public function jumpssco_premium_section_description()
    {
        $options = get_option('jumpssco_sso_settings', array());
        if (empty($options['is_premium'])) {
            echo '<div class="notice notice-info inline"><p>' . esc_html__('Enter your License ID in the General tab to activate Premium Features.', 'sso-connector-for-jumpcloud') . '</p></div>';
        } else {
            echo '<div class="notice notice-success inline"><p>' . esc_html__('Premium License Active! Enjoy advanced features below.', 'sso-connector-for-jumpcloud') . '</p></div>';
            echo '<p>' . sprintf(
                /* translators: %s: attribute name */
                esc_html__('Note: To use Role Mapping, ensure your JumpCloud SAML App is configured to send an Attribute named "%s".', 'sso-connector-for-jumpcloud'),
                'groups'
            ) . '</p>';
        }
    }

    /**
     * Render subscription status with Stripe portal links
     */
    public function jumpssco_render_subscription_status()
    {
        $options = get_option('jumpssco_sso_settings', array());
        $is_premium = !empty($options['is_premium']);
        $license_key = isset($options['license_id']) ? $options['license_id'] : '';

        // Stripe Customer Portal URL
        $portal_url = 'https://billing.stripe.com/p/login/eVq3cu981byag9KgLm4Rq00';

        // Hidden field to preserve is_premium status when saving form
        echo '<input type="hidden" name="jumpssco_sso_settings[is_premium]" value="' . ($is_premium ? '1' : '0') . '">';

        echo '<div class="jc-subscription-status" style="padding: 15px; background: #f9f9f9; border-radius: 4px; border-left: 4px solid ' . ($is_premium ? '#46b450' : '#dba617') . ';">';

        if ($is_premium) {
            echo '<h4 style="margin-top: 0; color: #46b450;">✓ ' . esc_html__('Premium Active', 'sso-connector-for-jumpcloud') . '</h4>';
            echo '<p><strong>' . esc_html__('Status:', 'sso-connector-for-jumpcloud') . '</strong> ' . esc_html__('Active', 'sso-connector-for-jumpcloud') . '</p>';

            if (!empty($license_key)) {
                echo '<p><strong>' . esc_html__('License Key:', 'sso-connector-for-jumpcloud') . '</strong> <code>' . esc_html(substr($license_key, 0, 20)) . '...</code></p>';
            }

            echo '<p style="margin-bottom: 0;">';
            echo '<a href="' . esc_url($portal_url) . '" target="_blank" class="button button-secondary">';
            echo esc_html__('Manage Subscription', 'sso-connector-for-jumpcloud');
            echo '</a> ';
            echo '<button type="button" id="jumpssco-revalidate-btn" class="button button-secondary" style="margin-left: 5px;">';
            echo esc_html__('Revalidate License', 'sso-connector-for-jumpcloud');
            echo '</button>';
            echo '<span id="jumpssco-revalidate-msg" style="margin-left: 10px; font-weight: bold;"></span>';
            echo '</p>';
            ?>
            <script type="text/javascript">
                jQuery(document).ready(function ($) {
                    $('#jumpssco-revalidate-btn').on('click', function (e) {
                        e.preventDefault();
                        var $btn = $(this);
                        var $msg = $('#jumpssco-revalidate-msg');
                        $btn.prop('disabled', true).text('<?php echo esc_js(__('Validating...', 'sso-connector-for-jumpcloud')); ?>');
                        $msg.text('');
                        $.ajax({
                            url: ajaxurl,
                            method: 'POST',
                            data: {
                                action: 'jumpssco_revalidate_license',
                                security: '<?php echo esc_attr(wp_create_nonce("jumpssco_revalidate_nonce")); ?>'
                            },
                            success: function (response) {
                                if (response.success) {
                                    $msg.css('color', response.data.deactivated ? 'red' : 'green').text(response.data.message);
                                    setTimeout(function () { location.reload(); }, 1500);
                                } else {
                                    $btn.prop('disabled', false).text('<?php echo esc_js(__('Revalidate License', 'sso-connector-for-jumpcloud')); ?>');
                                    $msg.css('color', 'red').text(response.data.message || 'Error occurred');
                                }
                            },
                            error: function () {
                                $btn.prop('disabled', false).text('<?php echo esc_js(__('Revalidate License', 'sso-connector-for-jumpcloud')); ?>');
                                $msg.css('color', 'red').text('Connection failed.');
                            }
                        });
                    });
                });
            </script>
            <?php
        } else {
            echo '<h4 style="margin-top: 0; color: #dba617;">⚠ ' . esc_html__('Premium Not Activated', 'sso-connector-for-jumpcloud') . '</h4>';
            echo '<p>' . esc_html__('Unlock advanced features like Role Mapping, Audit Logs, and Post-Login Redirects.', 'sso-connector-for-jumpcloud') . '</p>';

            // Stripe Pricing Table
            $current_user = wp_get_current_user();
            $user_email = $current_user->user_email;

            echo '<div style="margin-top: 20px;">';
            echo '<stripe-pricing-table 
                    pricing-table-id="prctbl_1SgdbpAPQ6j8HbbHYnGHPCAo" 
                    publishable-key="pk_live_51SgCKJAPQ6j8HbbHNiXpa3zVnGJddWpAMWCIDnI1hoLLFh2XvxF8QF9S8egWuCFYP5N1g9d4RqB7FEi33olB4Otv00MmU3B4BK"
                    client-reference-id="' . esc_attr($user_email) . '"
                    customer-email="' . esc_attr($user_email) . '"
                  ></stripe-pricing-table>';
            echo '</div>';

            // Check Subscription Button
            echo '<div style="margin-top: 15px; border-top: 1px solid #eee; padding-top: 15px;">';
            echo '<p>' . esc_html__('Already subscribed but not active?', 'sso-connector-for-jumpcloud') . '</p>';
            echo '<button type="button" id="jumpssco-check-sub-btn" class="button button-secondary">' . esc_html__('Check Subscription Status', 'sso-connector-for-jumpcloud') . '</button>';
            echo '<span id="jumpssco-check-sub-msg" style="margin-left: 10px; font-weight: bold;"></span>';
            echo '</div>';

            ?>
            <script type="text/javascript">     jQuery(document).ready(function ($) {
                    $('#jumpssco-check-sub-btn').on('click', function (e) {
                        e.preventDefault(); var $btn = $(this); var $msg = $('#jumpssco-check-sub-msg');
                        $btn.prop('disabled', true).text('Checking...'); $msg.text('').removeClass('success-msg error-msg');
                        $.ajax({ url: ajaxurl, method: 'POST', data: { action: 'jumpssco_check_subscription', security: '<?php echo esc_attr(wp_create_nonce("jumpssco_check_sub_nonce")); ?>' }, success: function (response) { if (response.success) { $msg.css('color', 'green').text(response.data.message); setTimeout(function () { location.reload(); }, 1500); } else { $btn.prop('disabled', false).text('<?php echo esc_js(__('Check Subscription Status', 'sso-connector-for-jumpcloud')); ?>'); $msg.css('color', 'red').text(response.data.message || 'Error occurred'); } }, error: function () { $btn.prop('disabled', false).text('<?php echo esc_js(__('Check Subscription Status', 'sso-connector-for-jumpcloud')); ?>'); $msg.css('color', 'red').text('Connection failed.'); } });
                    });
                });
            </script>
            <?php
        }

        echo '</div>';
    }


    public function jumpssco_options_page()
    {
        ?>
        <div class="wrap jc-sso-settings">
            <style>
                .jc-main-layout {
                    display: grid;
                    grid-template-columns: 2fr 1fr;
                    gap: 30px;
                    align-items: start;
                    margin-top: 30px;
                    /* Removed margin top */
                }

                @media (max-width: 1000px) {
                    .jc-main-layout {
                        grid-template-columns: 1fr;
                    }
                }

                .jc-main-column {
                    background: #fff;
                    padding: 2px 20px 20px;
                    border: 1px solid #ccd0d4;
                    box-shadow: 0 1px 1px rgba(0, 0, 0, .04);
                    border-radius: 4px;
                }

                .jc-side-column {
                    display: flex;
                    flex-direction: column;
                    gap: 20px;
                    position: sticky;
                    top: 50px;
                    /* Offset for WordPress admin bar */
                }

                .jc-section-card {
                    background: #fff;
                    padding: 30px 20px 20px;
                    /* Increased padding top even more */
                    border: 1px solid #ccd0d4;
                    box-shadow: 0 1px 1px rgba(0, 0, 0, .04);
                    border-radius: 4px;
                }

                .jc-section-card h2,
                .jc-section-card h3 {
                    margin-top: 0;
                }


                .jc-sso-settings .form-table th {
                    width: 200px;
                    padding: 20px 10px 20px 0;
                }

                .jc-sso-settings p.description {
                    font-style: italic;
                    color: #666;
                    margin-top: 4px;
                }

                .jc-sso-settings h2.nav-tab-wrapper {
                    margin-bottom: 20px;
                    border-bottom: 1px solid #ccc;
                    padding-bottom: 0;
                }

                .jc-support-card {
                    text-align: center;
                    padding: 20px;
                    background: #f8f9fa;
                    border: 1px dashed #ccd0d4;
                }

                .jc-support-card img {
                    max-width: 150px;
                    height: auto;
                }

                .jc-audit-logs-table-wrapper {
                    max-height: 400px;
                    overflow-y: auto;
                    overflow-x: auto;
                    /* Enable horizontal scroll */
                    border: 1px solid #ccd0d4;
                    margin-bottom: 10px;
                }

                .jc-audit-logs-table-wrapper table {
                    border: none;
                    margin: 0;
                    min-width: 600px;
                    /* Ensure table doesn't get too squashed */
                    display: table !important;
                    table-layout: auto !important;
                }

                /* Force standard table structure and visibility */
                .jc-audit-logs-table-wrapper thead {
                    display: table-header-group !important;
                }

                .jc-audit-logs-table-wrapper tbody {
                    display: table-row-group !important;
                }

                .jc-audit-logs-table-wrapper tr {
                    display: table-row !important;
                }

                .jc-audit-logs-table-wrapper th,
                .jc-audit-logs-table-wrapper td {
                    display: table-cell !important;
                    width: auto !important;
                    padding: 8px 10px !important;
                    vertical-align: middle !important;
                }

                /* Disable WordPress responsive labels */
                .jc-audit-logs-table-wrapper td:before {
                    content: none !important;
                    display: none !important;
                }
            </style>

            <h1><?php esc_html_e('SSO JumpCloud - Enterprise SAML & SCIM', 'sso-connector-for-jumpcloud'); ?></h1>

            <div class="jc-main-layout">
                <div class="jc-main-column">
                    <form action="options.php" method="post">
                        <?php
                        // Tabs for General vs Premium settings
                        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only check for tab navigation
                        $current_tab = isset($_GET['tab']) ? sanitize_key(wp_unslash($_GET['tab'])) : 'general';
                        $base_url = remove_query_arg('tab');
                        $tabs = array(
                            'general' => esc_html__('General', 'sso-connector-for-jumpcloud'),
                            'premium' => esc_html__('Premium Features', 'sso-connector-for-jumpcloud'),
                            'scim' => esc_html__('SCIM Provisioning', 'sso-connector-for-jumpcloud'),
                        );
                        echo '<h2 class="nav-tab-wrapper">';
                        foreach ($tabs as $tab_key => $tab_label) {
                            $class = ($current_tab === $tab_key) ? ' nav-tab-active' : '';
                            $url = add_query_arg('tab', $tab_key, $base_url);
                            printf('<a href="%1$s" class="nav-tab%2$s">%3$s</a>', esc_url($url), esc_attr($class), esc_html($tab_label));
                        }
                        echo '</h2>';

                        // Output settings fields by tab
                        settings_fields('jumpssco_sso_options');

                        if ('premium' === $current_tab) {
                            // Premium Features Tab
                            echo '<h2>' . esc_html__('Premium Features', 'sso-connector-for-jumpcloud') . '</h2>';
                            $this->jumpssco_premium_section_description();

                            echo '<table class="form-table">';
                            do_settings_fields('jumpssco_sso_options', 'jumpssco_sso_section_premium');
                            echo '</table>';

                            echo '<hr><h3>' . esc_html__('Post-Login Redirects', 'sso-connector-for-jumpcloud') . '</h3>';
                            echo '<table class="form-table">';
                            do_settings_fields('jumpssco_sso_options', 'jumpssco_sso_section_redirects');
                            echo '</table>';

                            echo '<hr><h3>' . esc_html__('Audit Logs & Retention', 'sso-connector-for-jumpcloud') . '</h3>';
                            echo '<table class="form-table">';
                            do_settings_fields('jumpssco_sso_options', 'jumpssco_sso_section_audit');
                            echo '</table>';
                        } elseif ('scim' === $current_tab) {
                            // SCIM Tab
                            echo '<h2>' . esc_html__('SCIM Provisioning', 'sso-connector-for-jumpcloud') . '</h2>';
                            $this->jumpssco_scim_section_description();

                            echo '<table class="form-table">';
                            do_settings_fields('jumpssco_sso_options', 'jumpssco_sso_section_scim');
                            echo '</table>';
                        } else {
                            // General Settings Tab
                            echo '<table class="form-table">';
                            do_settings_fields('jumpssco_sso_options', 'jumpssco_sso_section');
                            echo '</table>';
                        }
                        submit_button();
                        ?>
                    </form>
                </div>

                <div class="jc-side-column">
                    <div class="jc-section-card">
                        <h2><?php esc_html_e('SP Metadata', 'sso-connector-for-jumpcloud'); ?></h2>
                        <p><?php esc_html_e('Configure your JumpCloud application with these details:', 'sso-connector-for-jumpcloud'); ?>
                        </p>

                        <div style="margin-bottom: 15px;">
                            <strong><?php esc_html_e('SP Entity ID', 'sso-connector-for-jumpcloud'); ?></strong><br>
                            <code><?php echo esc_url(home_url('/')); ?></code>
                        </div>

                        <div style="margin-bottom: 15px;">
                            <strong><?php esc_html_e('ACS URL', 'sso-connector-for-jumpcloud'); ?></strong><br>
                            <code><?php echo esc_url(home_url('/?acs')); ?></code>
                        </div>

                        <div style="margin-bottom: 15px;">
                            <strong><?php esc_html_e('SLS URL', 'sso-connector-for-jumpcloud'); ?></strong><br>
                            <code><?php echo esc_url(home_url('/?sls')); ?></code>
                        </div>

                        <div style="margin-bottom: 15px;">
                            <strong><?php esc_html_e('SCIM Base URL', 'sso-connector-for-jumpcloud'); ?></strong><br>
                            <code><?php echo esc_url(rest_url('jumpssco/v1/scim/v2')); ?></code>
                        </div>

                        <p>
                            <a href="#" id="test-jumpcloud-sso-config"
                                class="button button-secondary"><?php esc_html_e('Test Configuration', 'sso-connector-for-jumpcloud'); ?></a>
                        </p>
                        <div id="jumpssco-test-result-container" style="margin-top:10px;"></div>
                    </div>

                    <div class="jc-support-card">
                        <p><?php esc_html_e('Enjoying the plugin?', 'sso-connector-for-jumpcloud'); ?></p>
                        <a href="https://www.buymeacoffee.com/airton" target="_blank"
                            style="margin-bottom: 20px; display: block;">
                            <img src="<?php echo esc_url(plugin_dir_url(__FILE__) . 'assets/images/buy-me-a-coffee.png'); ?>"
                                alt="<?php echo esc_attr__('Buy Me A Coffee', 'sso-connector-for-jumpcloud'); ?>">
                        </a>
                        <hr style="margin: 20px 0; border: 0; border-top: 1px dashed #ccd0d4;">
                        <a href="https://www.producthunt.com/products/sso-connector-for-jumpcloud/reviews/new?utm_source=badge-product_review&utm_medium=badge&utm_source=badge-sso&#0045;connector&#0045;for&#0045;jumpcloud"
                            target="_blank"><img
                                src="https://api.producthunt.com/widgets/embed-image/v1/product_review.svg?product_id=1138970&theme=light"
                                alt="SSO&#0032;Connector&#0032;for&#0032;JumpCloud - The&#0032;simplest&#0032;way&#0032;to&#0032;connect&#0032;WordPress&#0032;to&#0032;JumpCloud | Product Hunt"
                                style="width: 250px; height: 54px;" width="250" height="54" /></a>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Section description for SCIM
     */
    public function jumpssco_scim_section_description()
    {
        $options = get_option('jumpssco_sso_settings', array());
        if (empty($options['is_premium'])) {
            echo '<div class="notice notice-info inline"><p>' . esc_html__('Upgrade to Premium to enable SCIM Provisioning.', 'sso-connector-for-jumpcloud') . '</p></div>';
        } else {
            echo '<p>' . esc_html__('SCIM (System for Cross-domain Identity Management) allows JumpCloud to automatically create, update, and deactivate users in your WordPress site.', 'sso-connector-for-jumpcloud') . '</p>';
        }
    }

    /**
     * Render SCIM Token field with regeneration button
     */
    public function jumpssco_render_scim_token_field($args)
    {
        $options = get_option('jumpssco_sso_settings');
        $token = isset($options['scim_token']) ? $options['scim_token'] : '';
        $is_disabled = !empty($args['disabled']);

        if (empty($token) && !$is_disabled) {
            try {
                $token = bin2hex(random_bytes(32));
            } catch (Exception $e) {
                $token = md5(uniqid(wp_generate_password(), true));
            }
        }

        echo '<div style="display: flex; align-items: center; gap: 10px;">';
        echo '<input type="text" id="jumpssco_scim_token" name="jumpssco_sso_settings[scim_token]" value="' . esc_attr($token) . '" class="regular-text" readonly style="background: #f0f0f1;">';
        if (!$is_disabled) {
            echo '<button type="button" id="jumpssco-regenerate-scim-token" class="button">' . esc_html__('Regenerate Token', 'sso-connector-for-jumpcloud') . '</button>';
        }
        echo '</div>';
        echo '<p class="description">' . esc_html__('Use this Bearer Token in JumpCloud to authenticate SCIM requests.', 'sso-connector-for-jumpcloud') . '</p>';
        ?>
        <script type="text/javascript">
            jQuery(document).ready(function ($) {
                $('#jumpssco-regenerate-scim-token').on('click', function (e) {
                    if (confirm('<?php echo esc_js(__('Are you sure you want to regenerate the SCIM token? The old token will stop working immediately.', 'sso-connector-for-jumpcloud')); ?>')) {
                        var newToken = '';
                        var chars = 'abcdef0123456789';
                        for (var i = 0; i < 64; i++) {
                            newToken += chars.charAt(Math.floor(Math.random() * chars.length));
                        }
                        $('#jumpssco_scim_token').val(newToken);
                    }
                });
            });
        </script>
        <?php
    }

    /**
     * Register SCIM 2.0 REST Routes
     */
    public function jumpssco_register_scim_routes()
    {
        register_rest_route('jumpssco/v1/scim/v2', '/Users', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'jumpssco_scim_get_users'),
                'permission_callback' => array($this, 'jumpssco_scim_authenticate'),
            ),
            array(
                'methods' => WP_REST_Server::CREATABLE,
                'callback' => array($this, 'jumpssco_scim_create_user'),
                'permission_callback' => array($this, 'jumpssco_scim_authenticate'),
            ),
        ));

        register_rest_route('jumpssco/v1/scim/v2', '/Users/(?P<id>[\d]+)', array(
            array(
                'methods' => WP_REST_Server::READABLE,
                'callback' => array($this, 'jumpssco_scim_get_user'),
                'permission_callback' => array($this, 'jumpssco_scim_authenticate'),
            ),
            array(
                'methods' => WP_REST_Server::EDITABLE, // PUT and PATCH
                'callback' => array($this, 'jumpssco_scim_update_user'),
                'permission_callback' => array($this, 'jumpssco_scim_authenticate'),
            ),
            array(
                'methods' => WP_REST_Server::DELETABLE,
                'callback' => array($this, 'jumpssco_scim_delete_user'),
                'permission_callback' => array($this, 'jumpssco_scim_authenticate'),
            ),
        ));
    }

    /**
     * Authenticate SCIM requests using Bearer Token
     */
    public function jumpssco_scim_authenticate($request)
    {
        if (!$this->jumpssco_is_premium()) {
            return new WP_Error('scim_forbidden', __('SCIM is a premium feature.', 'sso-connector-for-jumpcloud'), array('status' => 403));
        }

        $options = get_option('jumpssco_sso_settings', array());
        if (empty($options['scim_enabled'])) {
            return new WP_Error('scim_disabled', __('SCIM is disabled in settings.', 'sso-connector-for-jumpcloud'), array('status' => 403));
        }

        $auth_header = $request->get_header('Authorization');
        if (empty($auth_header) || strpos($auth_header, 'Bearer ') !== 0) {
            return new WP_Error('scim_unauthorized', __('Unauthorized: Bearer token missing.', 'sso-connector-for-jumpcloud'), array('status' => 401));
        }

        $token = substr($auth_header, 7);
        $saved_token = isset($options['scim_token']) ? $options['scim_token'] : '';

        if (empty($saved_token) || $token !== $saved_token) {
            return new WP_Error('scim_unauthorized', __('Unauthorized: Invalid token.', 'sso-connector-for-jumpcloud'), array('status' => 401));
        }

        return true;
    }

    /**
     * Helper to format WP User to SCIM User Resource
     */
    private function jumpssco_format_scim_user($user)
    {
        if (!$user instanceof WP_User) {
            $user = get_user_by('id', $user);
        }

        if (!$user) {
            return null;
        }

        // Check if user is inactive (WP doesn't have a native 'active' flag, we check if they are blocked by our logic or meta)
        // For simplicity, we assume they are active unless we've set a meta flag.
        $is_inactive = get_user_meta($user->ID, 'jumpssco_scim_inactive', true);

        return array(
            'schemas' => array('urn:ietf:params:scim:schemas:core:2.0:User'),
            'id' => (string) $user->ID,
            'userName' => $user->user_email,
            'name' => array(
                'givenName' => $user->first_name,
                'familyName' => $user->last_name,
                'formatted' => $user->display_name,
            ),
            'emails' => array(
                array(
                    'value' => $user->user_email,
                    'type' => 'work',
                    'primary' => true,
                ),
            ),
            'active' => !$is_inactive,
            'meta' => array(
                'resourceType' => 'User',
                'location' => rest_url('jumpssco/v1/scim/v2/Users/') . $user->ID,
            ),
        );
    }

    /**
     * GET /Users
     */
    public function jumpssco_scim_get_users($request)
    {
        $filter = $request->get_param('filter');
        $count = $request->get_param('count') ?: 20;
        $startIndex = $request->get_param('startIndex') ?: 1;

        $args = array(
            'number' => $count,
            'offset' => $startIndex - 1,
        );

        // Basic filter support (userName eq "..." or emails eq "...")
        if (!empty($filter)) {
            if (preg_match('/(?:userName|emails)\s+eq\s+"([^"]+)"/i', $filter, $matches)) {
                $args['search'] = $matches[1];
                $args['search_columns'] = array('user_email', 'user_login');
            }
        }

        $user_query = new WP_User_Query($args);
        $users = $user_query->get_results();
        $total = $user_query->get_total();

        $resources = array();
        foreach ($users as $user) {
            $resources[] = $this->jumpssco_format_scim_user($user);
        }

        return new WP_REST_Response(array(
            'schemas' => array('urn:ietf:params:scim:api:messages:2.0:ListResponse'),
            'totalResults' => $total,
            'startIndex' => (int) $startIndex,
            'itemsPerPage' => count($resources),
            'Resources' => $resources,
        ), 200);
    }

    /**
     * GET /Users/{id}
     */
    public function jumpssco_scim_get_user($request)
    {
        $id = $request['id'];
        $user = get_user_by('id', $id);

        if (!$user) {
            return new WP_Error('scim_not_found', __('User not found', 'sso-connector-for-jumpcloud'), array('status' => 404));
        }

        return new WP_REST_Response($this->jumpssco_format_scim_user($user), 200);
    }

    /**
     * POST /Users
     */
    public function jumpssco_scim_create_user($request)
    {
        $body = $request->get_json_params();
        $email = isset($body['userName']) ? $body['userName'] : '';

        if (empty($email) && !empty($body['emails'])) {
            foreach ($body['emails'] as $e) {
                if (!empty($e['primary'])) {
                    $email = $e['value'];
                    break;
                }
            }
        }

        if (empty($email)) {
            return new WP_Error('scim_invalid', __('Email (userName) is required.', 'sso-connector-for-jumpcloud'), array('status' => 400));
        }

        if (get_user_by('email', $email)) {
            return new WP_Error('scim_conflict', __('User already exists.', 'sso-connector-for-jumpcloud'), array('status' => 409));
        }

        $username = explode('@', $email)[0];
        // Ensure username is unique
        $base_username = $username;
        $counter = 1;
        while (username_exists($username)) {
            $username = $base_username . $counter;
            $counter++;
        }

        $password = wp_generate_password();
        $user_id = wp_create_user($username, $password, $email);

        if (is_wp_error($user_id)) {
            return new WP_Error('scim_error', $user_id->get_error_message(), array('status' => 500));
        }

        $user = get_user_by('id', $user_id);

        // Map names
        if (isset($body['name']['givenName'])) {
            update_user_meta($user_id, 'first_name', sanitize_text_field($body['name']['givenName']));
        }
        if (isset($body['name']['familyName'])) {
            update_user_meta($user_id, 'last_name', sanitize_text_field($body['name']['familyName']));
        }

        // Set default role
        $user->set_role('subscriber');

        // Handle Active flag
        if (isset($body['active']) && $body['active'] === false) {
            update_user_meta($user_id, 'jumpssco_scim_inactive', 1);
        }

        $this->jumpssco_log_sso_event($email, 'success', __('User created via SCIM.', 'sso-connector-for-jumpcloud'));

        return new WP_REST_Response($this->jumpssco_format_scim_user($user), 201);
    }

    /**
     * PUT/PATCH /Users/{id}
     */
    public function jumpssco_scim_update_user($request)
    {
        $id = $request['id'];
        $user = get_user_by('id', $id);

        if (!$user) {
            return new WP_Error('scim_not_found', __('User not found', 'sso-connector-for-jumpcloud'), array('status' => 404));
        }

        $body = $request->get_json_params();
        $method = $request->get_method();
        $user_data = array('ID' => $user->ID);
        $meta_data = array();
        $should_logout = false;

        if ($method === 'PATCH') {
            if (isset($body['Operations']) && is_array($body['Operations'])) {
                foreach ($body['Operations'] as $op) {
                    $path = isset($op['path']) ? $op['path'] : '';
                    $value = isset($op['value']) ? $op['value'] : null;
                    $operation = strtolower($op['op']);

                    if ($operation === 'replace' || $operation === 'add') {
                        if ($path === 'active' || (is_array($value) && isset($value['active']))) {
                            $active = is_array($value) ? (bool) $value['active'] : (bool) $value;
                            if ($active) {
                                delete_user_meta($user->ID, 'jumpssco_scim_inactive');
                            } else {
                                update_user_meta($user->ID, 'jumpssco_scim_inactive', 1);
                                $should_logout = true;
                            }
                        } elseif ($path === 'name.givenName' || $path === 'givenName') {
                            $meta_data['first_name'] = sanitize_text_field($value);
                        } elseif ($path === 'name.familyName' || $path === 'familyName') {
                            $meta_data['last_name'] = sanitize_text_field($value);
                        } elseif ($path === 'userName' || $path === 'emails[type eq "work"].value') {
                            $user_data['user_email'] = sanitize_email($value);
                        } elseif (empty($path) && is_array($value)) {
                            // Support for root value replace
                            if (isset($value['active'])) {
                                if ((bool) $value['active']) {
                                    delete_user_meta($user->ID, 'jumpssco_scim_inactive');
                                } else {
                                    update_user_meta($user->ID, 'jumpssco_scim_inactive', 1);
                                    $should_logout = true;
                                }
                            }
                            if (isset($value['name']['givenName'])) {
                                $meta_data['first_name'] = sanitize_text_field($value['name']['givenName']);
                            }
                            if (isset($value['name']['familyName'])) {
                                $meta_data['last_name'] = sanitize_text_field($value['name']['familyName']);
                            }
                            if (isset($value['userName'])) {
                                $user_data['user_email'] = sanitize_email($value['userName']);
                            }
                        }
                    }
                }
            }
        } else {
            // PUT (Full replace)
            if (isset($body['active'])) {
                if ($body['active']) {
                    delete_user_meta($user->ID, 'jumpssco_scim_inactive');
                } else {
                    update_user_meta($user->ID, 'jumpssco_scim_inactive', 1);
                    $should_logout = true;
                }
            }
            if (isset($body['userName'])) {
                $user_data['user_email'] = sanitize_email($body['userName']);
            }
            if (isset($body['name']['givenName'])) {
                $meta_data['first_name'] = sanitize_text_field($body['name']['givenName']);
            }
            if (isset($body['name']['familyName'])) {
                $meta_data['last_name'] = sanitize_text_field($body['name']['familyName']);
            }
            if (isset($body['displayName'])) {
                $user_data['display_name'] = sanitize_text_field($body['displayName']);
            }
        }

        // Apply changes
        if (count($user_data) > 1) {
            wp_update_user($user_data);
        }

        if (!empty($meta_data)) {
            foreach ($meta_data as $key => $val) {
                update_user_meta($user->ID, $key, $val);
            }
        }

        if ($should_logout) {
            wp_destroy_all_sessions($user->ID);
        }

        $this->jumpssco_log_sso_event($user->user_email, 'success', __('User updated via SCIM.', 'sso-connector-for-jumpcloud'));

        $updated_user = get_user_by('id', $user->ID);
        return new WP_REST_Response($this->jumpssco_format_scim_user($updated_user), 200);
    }

    /**
     * DELETE /Users/{id}
     */
    public function jumpssco_scim_delete_user($request)
    {
        $id = $request['id'];
        $user = get_user_by('id', $id);

        if (!$user) {
            return new WP_Error('scim_not_found', __('User not found', 'sso-connector-for-jumpcloud'), array('status' => 404));
        }

        // Instead of hard-deleting, we deactivate for safety, but the request says "remove alguém ... acesso cai na hora"
        // Deactivation is usually enough for "acesso cai".
        update_user_meta($user->ID, 'jumpssco_scim_inactive', 1);
        wp_destroy_all_sessions($user->ID);

        $this->jumpssco_log_sso_event($user->user_email, 'success', __('User deactivated via SCIM DELETE.', 'sso-connector-for-jumpcloud'));

        return new WP_REST_Response(null, 204);
    }

    public function jumpcloud_sso_plugin_row_meta($plugin_meta, $plugin_file)
    {
        if ($plugin_file !== 'sso-connector-for-jumpcloud/sso-connector-for-jumpcloud.php') {
            return $plugin_meta;
        }

        $plugin_meta[] = '<a href="https://www.buymeacoffee.com/airton" target="_blank">' . esc_html__('☕️ Buy me a coffee', 'sso-connector-for-jumpcloud') . '</a>';
        $plugin_meta[] = '<a href="https://airtonvancin.com/plugin/sso-connector-for-jumpcloud" target="_blank">' . esc_html__('Visit plugin page', 'sso-connector-for-jumpcloud') . '</a>';

        return $plugin_meta;
    }
}

new Jumpssco_JumpCloud_SSO_Connector();

