<?php
if (!defined('ABSPATH')) {
    exit;
}

class SSDF_Filters_Indexer
{

    private $table_name;

    public function __construct()
    {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'ssdf_filter_index';

        add_action('init', [$this, 'create_table']);
        add_action('save_post', [$this, 'index_product'], 10, 3);
        add_action('wp_ajax_ssdf_batch_index', [$this, 'ajax_batch_index']);
    }

    public function create_table()
    {
        if (get_option('ssdf_filter_db_version') === '1.0')
            return;

        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $sql = "CREATE TABLE $this->table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            post_id bigint(20) NOT NULL,
            filter_source varchar(50) NOT NULL,
            filter_value varchar(200) NOT NULL,
            filter_value_num double NULL,
            PRIMARY KEY  (id),
            KEY post_id (post_id),
            KEY filter_source (filter_source),
            KEY filter_value (filter_value)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        update_option('ssdf_filter_db_version', '1.0');
    }

    public function index_product($post_id, $post = null, $update = false)
    {
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE)
            return;
        if (wp_is_post_revision($post_id))
            return;
        if (get_post_status($post_id) !== 'publish')
            return;

        global $wpdb;

        // 1. Delete existing index for this product
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery
        $wpdb->delete($this->table_name, ['post_id' => $post_id]);

        // 2. NUCLEAR MODE: Get ALL Taxonomies, ignoring 'public' or 'show_ui' settings
        $taxonomies = get_object_taxonomies(get_post_type($post_id), 'names');

        // Exclude system taxonomies
        $exclude = ['nav_menu', 'link_category', 'post_format', 'wp_theme', 'wp_template_part_area'];

        foreach ($taxonomies as $tax_name) {
            if (in_array($tax_name, $exclude))
                continue;

            $terms = get_the_terms($post_id, $tax_name);

            if ($terms && !is_wp_error($terms)) {
                foreach ($terms as $term) {
                    // phpcs:ignore WordPress.DB.DirectDatabaseQuery
                    $wpdb->insert($this->table_name, [
                        'post_id' => $post_id,
                        'filter_source' => $tax_name,
                        'filter_value' => $term->slug,
                        'filter_value_num' => 0
                    ]);
                }
            }
        }

        // 3. Index Price
        if ('product' === get_post_type($post_id)) {
            $price = get_post_meta($post_id, '_price', true);
            if ('' !== $price) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery
                $wpdb->insert($this->table_name, [
                    'post_id' => $post_id,
                    'filter_source' => '_price',
                    'filter_value' => $price,
                    'filter_value_num' => (float) $price
                ]);
            }
        }
    }

    public function ajax_batch_index()
    {
        check_ajax_referer('ssdf_filter_nonce', 'security');
        if (!current_user_can('manage_options'))
            wp_send_json_error();

        $offset = isset($_POST['offset']) ? intval(wp_unslash($_POST['offset'])) : 0;
        $limit = 20;

        if ($offset === 0) {
            global $wpdb;
            // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $wpdb->query("TRUNCATE TABLE $this->table_name");
        }

        $products = get_posts([
            'post_type' => ['product', 'post'],
            'numberposts' => $limit,
            'offset' => $offset,
            'fields' => 'ids',
            'post_status' => 'publish'
        ]);

        if (empty($products)) {
            wp_send_json_success(['complete' => true]);
        }

        foreach ($products as $pid) {
            $this->index_product($pid);
        }

        wp_send_json_success([
            'complete' => false,
            'next_offset' => $offset + $limit,
            'count' => count($products)
        ]);
    }
}