<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Handles Frontend Rendering and Query Filtering
 */
class SSDF_Filters_Frontend
{

    private static $instance = null;

    public static function get_instance()
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function __construct()
    {
        add_action('wp_enqueue_scripts', [$this, 'enqueue_assets']);
        add_shortcode('ssdf_filter', [$this, 'render_shortcode']);
        add_shortcode('ssdf_active_filters', [$this, 'render_active_filters']);
        add_action('pre_get_posts', [$this, 'handle_filter_query']);
    }

    public function enqueue_assets()
    {
        if (is_admin() && !defined('SHOW_CT_BUILDER')) {
            return;
        }

        wp_enqueue_style('nouislider-css', SSDF_URL . 'assets/css/nouislider.min.css', [], '15.7.0');
        wp_enqueue_script('nouislider-js', SSDF_URL . 'assets/js/nouislider.min.js', [], '15.7.0', true);

        wp_enqueue_script('ssdf-front-js', SSDF_URL . 'assets/js/frontend.js', ['jquery', 'nouislider-js'], SSDF_VERSION, true);
        wp_enqueue_style('ssdf-front-css', SSDF_URL . 'assets/css/frontend.css', [], SSDF_VERSION);
    }

    public function render_shortcode($atts)
    {
        $atts = shortcode_atts(['id' => 0, 'repeater_id' => ''], $atts);
        return $this->generate_filter_html($atts['id'], $atts['repeater_id']);
    }

    public function render_active_filters($atts)
    {
        $atts = shortcode_atts(['repeater_id' => '', 'show_labels' => 'yes'], $atts);
        ob_start();
        ?>
        <div class="s-ssdf-active-chips" data-repeater-id="<?php echo esc_attr($atts['repeater_id']); ?>"
            data-show-labels="<?php echo esc_attr($atts['show_labels']); ?>">
        </div>
        <?php
        return ob_get_clean();
    }

    public function generate_filter_html($filter_id, $override_repeater_id = '')
    {
        if (!$filter_id)
            return '';
        $config = get_post_meta($filter_id, '_ssdf_filter_config', true);
        if (!$config)
            return '';

        $source = $config['source'];
        $type = $config['type'];
        $rep_id = !empty($override_repeater_id) ? $override_repeater_id : ($config['repeater_id'] ?? '');
        $all_label = $config['all_label'] ?? __('All', 'ss-dynamic-filter-for-oxygen');

        $source_label = $source;
        if (taxonomy_exists($source)) {
            $tax_obj = get_taxonomy($source);
            $source_label = $tax_obj->labels->name;
        } elseif ('_price' === $source) {
            $source_label = __('Price', 'ss-dynamic-filter-for-oxygen');
        } elseif ('_search' === $source) {
            $source_label = __('Search', 'ss-dynamic-filter-for-oxygen');
        }

        if ('_search' === $source) {
            $val = isset($_GET['search_kw']) ? sanitize_text_field(wp_unslash($_GET['search_kw'])) : ''; // phpcs:ignore
            $ph = !empty($all_label) && 'All' !== $all_label ? $all_label : __('Search...', 'ss-dynamic-filter-for-oxygen');
            return sprintf(
                '<div class="s-ssdf-filter-wrap type-search" data-source="_search" data-tax-label="%s" data-repeater-id="%s"><input type="text" class="s-ssdf-search-input" placeholder="%s" value="%s"></div>',
                esc_attr($source_label),
                esc_attr($rep_id),
                esc_attr($ph),
                esc_attr($val)
            );
        }

        if ('_price' === $source) {
            return $this->render_price_filter($source, $rep_id, $source_label, $type);
        }

        $args = ['taxonomy' => $source, 'hide_empty' => true, 'orderby' => $config['orderby'] ?? 'name', 'order' => $config['order'] ?? 'ASC'];
        if (!empty($config['include']))
            $args['include'] = $config['include'];

        if (!empty($config['exclude'])) {
            $args['exclude'] = $config['exclude']; // phpcs:ignore
        }

        if (!empty($config['parent']))
            $args['parent'] = $config['parent'];

        $terms = get_terms($args);
        if (is_wp_error($terms) || empty($terms))
            return '';

        $current_val = isset($_GET[$source]) ? sanitize_text_field(wp_unslash($_GET[$source])) : ''; // phpcs:ignore
        $current_arr = !empty($current_val) ? explode(',', $current_val) : [];

        ob_start();
        ?>
        <div class="s-ssdf-filter-wrap type-<?php echo esc_attr($type); ?>" data-source="<?php echo esc_attr($source); ?>"
            data-tax-label="<?php echo esc_attr($source_label); ?>" data-repeater-id="<?php echo esc_attr($rep_id); ?>">
            <?php
            if ('select' === $type) {
                echo '<select class="s-ssdf-filter-select">';
                if (!empty($all_label))
                    echo '<option value="">' . esc_html($all_label) . '</option>';
                foreach ($terms as $term) {
                    $sel = in_array($term->slug, $current_arr) ? 'selected' : '';
                    echo '<option value="' . esc_attr($term->slug) . '" ' . esc_attr($sel) . ' data-label="' . esc_attr($term->name) . '">' . esc_html($term->name) . '</option>';
                }
                echo '</select>';
            } elseif ('checkbox' === $type) {
                foreach ($terms as $term) {
                    $chk = in_array($term->slug, $current_arr) ? 'checked' : '';
                    echo '<label class="s-ssdf-input-label"><input type="checkbox" name="' . esc_attr($source) . '[]" value="' . esc_attr($term->slug) . '" data-label="' . esc_attr($term->name) . '" ' . esc_attr($chk) . '> ' . esc_html($term->name) . '</label>';
                }
            } else {
                if (!empty($all_label) && 'radio' !== $type) {
                    echo '<button class="s-ssdf-filter-btn ' . (empty($current_val) ? 'active' : '') . '" data-slug="">' . esc_html($all_label) . '</button>';
                }
                foreach ($terms as $term) {
                    $act = in_array($term->slug, $current_arr) ? 'active' : '';
                    $color = get_term_meta($term->term_id, '_ssdf_filter_color', true) ?: '#cccccc';
                    if ('color' === $type) {
                        echo '<button class="s-ssdf-swatch-btn ' . esc_attr($act) . '" data-slug="' . esc_attr($term->slug) . '" data-label="' . esc_attr($term->name) . '" title="' . esc_attr($term->name) . '"><span class="s-ssdf-swatch-color" style="background-color:' . esc_attr($color) . ';"></span></button>';
                    } elseif ('radio' === $type) {
                        $chk = in_array($term->slug, $current_arr) ? 'checked' : '';
                        echo '<label class="s-ssdf-input-label"><input type="radio" name="' . esc_attr($source) . '" value="' . esc_attr($term->slug) . '" data-label="' . esc_attr($term->name) . '" ' . esc_attr($chk) . '> ' . esc_html($term->name) . '</label>';
                    } else {
                        echo '<button class="s-ssdf-filter-btn ' . esc_attr($act) . '" data-slug="' . esc_attr($term->slug) . '" data-label="' . esc_attr($term->name) . '">' . esc_html($term->name) . '</button>';
                    }
                }
            }
            ?>
        </div>
        <?php
        return ob_get_clean();
    }

    private function render_price_filter($source, $rep_id, $label, $type = 'range')
    {
        global $wpdb;
        $table = $wpdb->prefix . 'ssdf_filter_index';

        $db_min = $wpdb->get_var("SELECT MIN(filter_value_num) FROM $table WHERE filter_source = '_price'") ?: 0; // phpcs:ignore
        $db_max = $wpdb->get_var("SELECT MAX(filter_value_num) FROM $table WHERE filter_source = '_price'") ?: 1000; // phpcs:ignore

        $min = isset($_GET['min_price']) ? sanitize_text_field(wp_unslash($_GET['min_price'])) : $db_min; // phpcs:ignore
        $max = isset($_GET['max_price']) ? sanitize_text_field(wp_unslash($_GET['max_price'])) : $db_max; // phpcs:ignore

        ob_start();
        ?>
        <div class="s-ssdf-filter-wrap type-<?php echo esc_attr($type); ?>" data-source="_price"
            data-tax-label="<?php echo esc_attr($label); ?>" data-repeater-id="<?php echo esc_attr($rep_id); ?>"
            data-min="<?php echo esc_attr($db_min); ?>" data-max="<?php echo esc_attr($db_max); ?>">

            <?php if ('slider' === $type): ?>
                <div class="s-ssdf-price-slider"></div>
                <div class="s-ssdf-price-range-labels">
                    <span class="price-min"><?php echo esc_html($min); ?></span> -
                    <span class="price-max"><?php echo esc_html($max); ?></span>
                </div>
                <input type="hidden" id="s-ssdf-min-price" value="<?php echo esc_attr($min); ?>">
                <input type="hidden" id="s-ssdf-max-price" value="<?php echo esc_attr($max); ?>">
            <?php else: ?>
                <input type="number" class="s-ssdf-price-input" id="s-ssdf-min-price"
                    placeholder="<?php esc_attr_e('Min', 'ss-dynamic-filter-for-oxygen'); ?>"
                    value="<?php echo esc_attr($min); ?>">
                <span class="s-ssdf-price-sep">-</span>
                <input type="number" class="s-ssdf-price-input" id="s-ssdf-max-price"
                    placeholder="<?php esc_attr_e('Max', 'ss-dynamic-filter-for-oxygen'); ?>"
                    value="<?php echo esc_attr($max); ?>">
                <button class="s-ssdf-apply-price"><?php esc_html_e('Apply', 'ss-dynamic-filter-for-oxygen'); ?></button>
            <?php endif; ?>
        </div>
        <?php
        return ob_get_clean();
    }

    public function handle_filter_query($query)
    {
        if (is_admin() || !$query->is_main_query())
            return;

        $filter_var = get_query_var('filter');
        if (!empty($filter_var)) {
            $filter_var = urldecode($filter_var);
            $segments = explode('/', $filter_var);

            // NUCLEAR FIX: Get ALL valid taxonomies, ignoring 'show_ui' settings
            $valid_taxonomies = get_taxonomies([], 'names');
            // Sort by length to avoid partial matches
            usort($valid_taxonomies, function ($a, $b) {
                return strlen($b) - strlen($a); });

            foreach ($segments as $segment) {
                if (strpos($segment, 'price-') === 0) {
                    $val = substr($segment, 6);
                    $prices = explode('-', $val);
                    $_GET['min_price'] = $prices[0] ?? 0;
                    $_GET['max_price'] = $prices[1] ?? 999999;
                    continue;
                }
                if (strpos($segment, 's-') === 0) {
                    $_GET['search_kw'] = substr($segment, 2);
                    continue;
                }

                foreach ($valid_taxonomies as $tax_name) {
                    $prefix = $tax_name . '-';
                    if (strpos($segment, $prefix) === 0) {
                        $val = substr($segment, strlen($prefix));
                        $_GET[$tax_name] = $val;
                        break;
                    }
                }
            }
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'ssdf_filter_index';
        $matched_ids = null;

        if (isset($_GET['search_kw']) && !empty($_GET['search_kw'])) { // phpcs:ignore
            $query->set('s', sanitize_text_field(wp_unslash($_GET['search_kw']))); // phpcs:ignore
        }

        // NUCLEAR FIX: Query logic also checks ALL taxonomies
        $taxonomies = get_taxonomies([], 'names');

        foreach ($taxonomies as $tax) {
            if (isset($_GET[$tax]) && !empty($_GET[$tax])) { // phpcs:ignore
                $slugs_raw = explode(',', sanitize_text_field(wp_unslash($_GET[$tax]))); // phpcs:ignore
                $slugs = array_map('sanitize_title', $slugs_raw);

                if (!empty($slugs)) {
                    $placeholders = implode(',', array_fill(0, count($slugs), '%s'));
                    $sql = "SELECT post_id FROM {$table_name} WHERE filter_source = %s AND filter_value IN ($placeholders)"; // phpcs:ignore
                    $params = array_merge([$tax], $slugs);
                    $results = $wpdb->get_col($wpdb->prepare($sql, $params)); // phpcs:ignore

                    if (is_null($matched_ids))
                        $matched_ids = $results;
                    else
                        $matched_ids = array_intersect($matched_ids, $results);
                }
            }
        }

        if (isset($_GET['min_price']) || isset($_GET['max_price'])) { // phpcs:ignore
            $min = !empty($_GET['min_price']) ? floatval($_GET['min_price']) : 0; // phpcs:ignore
            $max = !empty($_GET['max_price']) ? floatval($_GET['max_price']) : 9999999; // phpcs:ignore

            $sql = "SELECT post_id FROM {$table_name} WHERE filter_source = %s AND filter_value_num BETWEEN %f AND %f"; // phpcs:ignore
            $results = $wpdb->get_col($wpdb->prepare($sql, '_price', $min, $max)); // phpcs:ignore

            if (is_null($matched_ids))
                $matched_ids = $results;
            else
                $matched_ids = array_intersect($matched_ids, $results);
        }

        if (!is_null($matched_ids)) {
            if (empty($matched_ids))
                $query->set('post__in', [0]);
            else
                $query->set('post__in', $matched_ids);
            $query->set('tax_query', []);
            $query->set('meta_query', []);
        }
    }
}
