<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Handles the Admin Dashboard: CPT Registration, Metaboxes, and AJAX.
 */
class SSDF_Filters_Admin
{

    public function __construct()
    {
        add_action('init', [$this, 'register_cpt']);
        add_action('add_meta_boxes', [$this, 'add_metaboxes']);
        add_action('save_post', [$this, 'save_settings']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);

        // AJAX Handlers for Dynamic Dropdowns
        add_action('wp_ajax_ssdf_get_datasources', [$this, 'ajax_get_datasources']);
        add_action('wp_ajax_ssdf_get_terms', [$this, 'ajax_get_terms']);
    }

    /**
     * Register the "Filters" Custom Post Type
     */
    public function register_cpt()
    {
        $labels = [
            'name' => __('Filters', 'ss-dynamic-filter-for-oxygen'),
            'singular_name' => __('Filter', 'ss-dynamic-filter-for-oxygen'),
            'add_new' => __('Add New', 'ss-dynamic-filter-for-oxygen'),
            'add_new_item' => __('Add New Filter', 'ss-dynamic-filter-for-oxygen'),
            'edit_item' => __('Edit Filter', 'ss-dynamic-filter-for-oxygen'),
            'new_item' => __('New Filter', 'ss-dynamic-filter-for-oxygen'),
            'all_items' => __('All Filters', 'ss-dynamic-filter-for-oxygen'),
            'menu_name' => __('Filters', 'ss-dynamic-filter-for-oxygen'),
        ];

        $args = [
            'labels' => $labels,
            'public' => false, // Not queryable on frontend directly
            'show_ui' => true,  // Show in Admin
            'show_in_menu' => true,
            'supports' => ['title'],
            'menu_icon' => 'dashicons-filter',
            'capability_type' => 'post',
        ];

        register_post_type('ssdf_filter', $args);
    }

    /**
     * Add Meta Boxes
     */
    public function add_metaboxes()
    {
        add_meta_box('ssdf_filter_config', __('Filter Configuration', 'ss-dynamic-filter-for-oxygen'), [$this, 'render_config_box'], 'ssdf_filter', 'normal', 'high');
        add_meta_box('ssdf_filter_usage', __('Usage', 'ss-dynamic-filter-for-oxygen'), [$this, 'render_usage_box'], 'ssdf_filter', 'side', 'high');
    }

    /**
     * Enqueue Admin Scripts and Styles
     */
    public function enqueue_assets($hook)
    {
        global $post;

        if ('post.php' !== $hook && 'post-new.php' !== $hook) {
            return;
        }

        if ('ssdf_filter' !== $post->post_type) {
            return;
        }

        wp_enqueue_style('ssdf-admin-css', SSDF_URL . 'assets/css/admin.css', [], SSDF_VERSION);
        wp_enqueue_script('ssdf-admin-js', SSDF_URL . 'assets/js/admin.js', ['jquery'], SSDF_VERSION, true);

        wp_localize_script('ssdf-admin-js', 'SsdfAdmin', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('ssdf_filter_nonce')
        ]);
    }

    /**
     * AJAX: Get Datasources (Taxonomies/Price/Search)
     */
    public function ajax_get_datasources()
    {
        check_ajax_referer('ssdf_filter_nonce', 'security');

        $post_type = isset($_POST['post_type']) ? sanitize_text_field(wp_unslash($_POST['post_type'])) : '';
        $options = [];

        // Add Product Specific Sources
        if ('product' === $post_type && class_exists('WooCommerce')) {
            $options[] = ['value' => '_price', 'label' => __('Product Price', 'ss-dynamic-filter-for-oxygen')];
        }

        // Always allow Search
        $options[] = ['value' => '_search', 'label' => __('Search Input (Keywords)', 'ss-dynamic-filter-for-oxygen')];

        // Get Taxonomies
        $taxonomies = get_object_taxonomies($post_type, 'objects');
        foreach ($taxonomies as $tax) {
            if (in_array($tax->name, ['post_format', 'nav_menu', 'link_category'])) {
                continue;
            }
            $options[] = ['value' => $tax->name, 'label' => sprintf('%s (%s)', $tax->label, $tax->name)];
        }

        wp_send_json_success($options);
    }

    /**
     * AJAX: Get Terms for a specific Taxonomy
     */
    public function ajax_get_terms()
    {
        check_ajax_referer('ssdf_filter_nonce', 'security');

        $taxonomy = isset($_POST['taxonomy']) ? sanitize_text_field(wp_unslash($_POST['taxonomy'])) : '';

        // Skip term fetching for special types
        if (in_array($taxonomy, ['_price', '_rating', '_search'])) {
            wp_send_json_success([]);
        }

        $terms = get_terms(['taxonomy' => $taxonomy, 'hide_empty' => false]);
        $options = [];

        // Add "None" option
        $options[] = ['value' => '-1', 'label' => __('-- None --', 'ss-dynamic-filter-for-oxygen')];

        if (!is_wp_error($terms)) {
            foreach ($terms as $term) {
                $options[] = ['value' => $term->term_id, 'label' => sprintf('%s (ID: %d)', $term->name, $term->term_id)];
            }
        }

        wp_send_json_success($options);
    }

    /**
     * Render the Configuration Metabox
     */
    public function render_config_box($post)
    {
        $config = get_post_meta($post->ID, '_ssdf_filter_config', true);

        // Set Defaults
        $pt = $config['post_type'] ?? '';
        $source = $config['source'] ?? '';
        $type = $config['type'] ?? 'buttons';
        $rep_id = $config['repeater_id'] ?? '';
        $all_label = $config['all_label'] ?? __('All', 'ss-dynamic-filter-for-oxygen');

        $include = $config['include'] ?? [];
        $exclude = $config['exclude'] ?? [];
        $parent = $config['parent'] ?? '';
        $orderby = $config['orderby'] ?? 'name';
        $order = $config['order'] ?? 'ASC';

        $post_types = get_post_types(['public' => true], 'objects');
        ?>
        <div class="s-ssdf-admin-wrap">

            <div class="s-ssdf-section">
                <h3><?php esc_html_e('Data Source', 'ss-dynamic-filter-for-oxygen'); ?></h3>
                <div class="s-ssdf-row">
                    <div class="s-ssdf-field-group half">
                        <label><?php esc_html_e('Content Type', 'ss-dynamic-filter-for-oxygen'); ?></label>
                        <select name="ssdf_filter[post_type]" id="ssdf_filter_post_type" class="widefat">
                            <option value="">-- <?php esc_html_e('Select Post Type', 'ss-dynamic-filter-for-oxygen'); ?> --
                            </option>
                            <?php foreach ($post_types as $p): ?>
                                <option value="<?php echo esc_attr($p->name); ?>" <?php selected($pt, $p->name); ?>>
                                    <?php echo esc_html($p->label); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="s-ssdf-field-group half">
                        <label><?php esc_html_e('Filter By', 'ss-dynamic-filter-for-oxygen'); ?></label>
                        <select name="ssdf_filter[source]" id="ssdf_filter_source" class="widefat"
                            data-selected="<?php echo esc_attr($source); ?>">
                            <option value="">-- <?php esc_html_e('Select Post Type First', 'ss-dynamic-filter-for-oxygen'); ?>
                                --</option>
                        </select>
                    </div>
                </div>
            </div>

            <div class="s-ssdf-section">
                <h3><?php esc_html_e('Appearance', 'ss-dynamic-filter-for-oxygen'); ?></h3>
                <div class="s-ssdf-row">
                    <div class="s-ssdf-field-group half">
                        <label><?php esc_html_e('Filter Type', 'ss-dynamic-filter-for-oxygen'); ?></label>
                        <select name="ssdf_filter[type]" id="ssdf_filter_type" class="widefat">
                            <option value="buttons" <?php selected($type, 'buttons'); ?>>Buttons</option>
                            <option value="color" <?php selected($type, 'color'); ?>>Color Swatches</option>
                            <option value="radio" <?php selected($type, 'radio'); ?>>Radio Buttons</option>
                            <option value="checkbox" <?php selected($type, 'checkbox'); ?>>Checkboxes</option>
                            <option value="select" <?php selected($type, 'select'); ?>>Dropdown</option>
                            <option value="range" <?php selected($type, 'range'); ?>>Price Range (Inputs)</option>
                            <option value="slider" <?php selected($type, 'slider'); ?>>Price Slider (Dual Handle)</option>
                            <option value="search" <?php selected($type, 'search'); ?>>Search Input</option>
                        </select>
                    </div>
                    <div class="s-ssdf-field-group half">
                        <label><?php esc_html_e('Oxygen Repeater ID', 'ss-dynamic-filter-for-oxygen'); ?></label>
                        <input type="text" name="ssdf_filter[repeater_id]" value="<?php echo esc_attr($rep_id); ?>"
                            class="widefat" placeholder="_dynamic_list-27-96">
                    </div>
                </div>

                <div class="s-ssdf-field-group">
                    <label><?php esc_html_e('"All" Label / Search Placeholder', 'ss-dynamic-filter-for-oxygen'); ?></label>
                    <input type="text" name="ssdf_filter[all_label]" value="<?php echo esc_attr($all_label); ?>" class="widefat"
                        placeholder="e.g. All Products">
                    <p class="description">
                        <?php esc_html_e('Leave empty to hide the "All" button or default placeholder.', 'ss-dynamic-filter-for-oxygen'); ?>
                    </p>
                </div>
            </div>

            <div class="s-ssdf-section" style="border-left: 4px solid #2271b1; background: #f0f6fc;">
                <h3><?php esc_html_e('Performance Indexer', 'ss-dynamic-filter-for-oxygen'); ?></h3>
                <p class="description">
                    <?php esc_html_e('To ensure fast filtering, all products must be indexed. If you add new products or filters, click the button below.', 'ss-dynamic-filter-for-oxygen'); ?>
                </p>
                <div style="margin-top: 10px;">
                    <button id="ssdf-run-indexer"
                        class="button button-primary"><?php esc_html_e('Index All Products Now', 'ss-dynamic-filter-for-oxygen'); ?></button>
                    <span id="ssdf-index-status" style="margin-left:10px; font-weight:600; color:#2271b1;"></span>
                </div>
            </div>

            <div class="s-ssdf-advanced-wrapper">
                <div class="s-ssdf-advanced-toggle">
                    <span class="dashicons dashicons-arrow-right"></span>
                    <?php esc_html_e('Advanced Settings (Include/Exclude)', 'ss-dynamic-filter-for-oxygen'); ?>
                </div>

                <div class="s-ssdf-advanced-panel" style="display:none;">
                    <div class="s-ssdf-row">
                        <div class="s-ssdf-field-group half">
                            <label><?php esc_html_e('Include Terms', 'ss-dynamic-filter-for-oxygen'); ?></label>
                            <select name="ssdf_filter[include][]" id="ssdf_filter_include" class="widefat s-ssdf-term-select"
                                multiple style="height:100px;" data-selected='<?php echo esc_attr(wp_json_encode($include)); ?>'></select>
                        </div>
                        <div class="s-ssdf-field-group half">
                            <label><?php esc_html_e('Exclude Terms', 'ss-dynamic-filter-for-oxygen'); ?></label>
                            <select name="ssdf_filter[exclude][]" id="ssdf_filter_exclude" class="widefat s-ssdf-term-select"
                                multiple style="height:100px;" data-selected='<?php echo esc_attr(wp_json_encode($exclude)); ?>'></select>
                        </div>
                    </div>

                    <div class="s-ssdf-row">
                        <div class="s-ssdf-field-group half">
                            <label><?php esc_html_e('Parent Term', 'ss-dynamic-filter-for-oxygen'); ?></label>
                            <select name="ssdf_filter[parent]" id="ssdf_filter_parent" class="widefat s-ssdf-term-select"
                                data-selected="<?php echo esc_attr($parent); ?>">
                                <option value="">None</option>
                            </select>
                        </div>
                        <div class="s-ssdf-field-group half">
                            <label><?php esc_html_e('Order By', 'ss-dynamic-filter-for-oxygen'); ?></label>
                            <select name="ssdf_filter[orderby]" class="widefat">
                                <option value="name" <?php selected($orderby, 'name'); ?>>Name</option>
                                <option value="count" <?php selected($orderby, 'count'); ?>>Count</option>
                                <option value="slug" <?php selected($orderby, 'slug'); ?>>Slug</option>
                                <option value="id" <?php selected($orderby, 'id'); ?>>ID</option>
                            </select>
                        </div>
                    </div>

                    <div class="s-ssdf-field-group">
                        <label><?php esc_html_e('Order Direction', 'ss-dynamic-filter-for-oxygen'); ?></label>
                        <select name="ssdf_filter[order]" class="widefat" style="width: 150px;">
                            <option value="ASC" <?php selected($order, 'ASC'); ?>>ASC (A-Z)</option>
                            <option value="DESC" <?php selected($order, 'DESC'); ?>>DESC (Z-A)</option>
                        </select>
                    </div>
                </div>
            </div>
        </div>
        <?php
        wp_nonce_field('save_ssdf_filter', 'ssdf_filter_nonce_field');
    }

    /**
     * Save the Meta Data
     */
    public function save_settings($post_id)
    {
        if (!isset($_POST['ssdf_filter_nonce_field']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['ssdf_filter_nonce_field'])), 'save_ssdf_filter')) {
            return;
        }

        if (isset($_POST['ssdf_filter'])) {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            $data = wp_unslash($_POST['ssdf_filter']);

            // Helper to clean array inputs
            $clean_array = function ($arr) {
                if (!is_array($arr)) {
                    return [];
                }
                // Remove values <= 0 (handles our "-1" none option)
                return array_filter(array_map('intval', $arr), function ($v) {
                    return $v > 0;
                });
            };

            $clean = [
                'post_type' => sanitize_text_field($data['post_type']),
                'source' => sanitize_text_field($data['source']),
                'type' => sanitize_text_field($data['type']),
                'repeater_id' => sanitize_text_field($data['repeater_id']),
                'all_label' => sanitize_text_field($data['all_label']),
                'include' => $clean_array($data['include'] ?? []),
                // phpcs:ignore WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_exclude
                'exclude' => $clean_array($data['exclude'] ?? []),
                'parent' => sanitize_text_field($data['parent']),
                'orderby' => sanitize_text_field($data['orderby']),
                'order' => sanitize_text_field($data['order']),
            ];

            update_post_meta($post_id, '_ssdf_filter_config', $clean);
        }
    }

    /**
     * Shortcode Usage Box
     */
    public function render_usage_box($post)
    {
        echo '<p>' . esc_html__('Shortcode:', 'ss-dynamic-filter-for-oxygen') . '</p>';
        echo '<code style="display:block; padding:10px; background:#f0f0f1;">[ssdf_filter id="' . esc_attr($post->ID) . '"]</code>';
    }
}