(function ($) {
    $(document).ready(function () {

        let nameMap = {};

        function buildNameMap() {
            $('.s-ssdf-filter-wrap').each(function () {
                const slug = $(this).data('source');
                const name = $(this).data('tax-label');
                if (slug && name) nameMap[slug] = name;
            });
            $('[data-label]').each(function () {
                const slug = $(this).data('slug') || $(this).val();
                const label = $(this).data('label');
                if (slug && label) nameMap[slug] = label;
            });
        }

        // --- INIT SLIDERS ---
        function initSliders() {
            $('.s-ssdf-filter-wrap.type-slider').each(function () {
                const $wrap = $(this);
                const slider = $wrap.find('.s-ssdf-price-slider')[0];

                if (!slider || slider.noUiSlider) return;

                const min = parseFloat($wrap.data('min')) || 0;
                const max = parseFloat($wrap.data('max')) || 1000;

                const currentMin = parseFloat($wrap.find('#s-ssdf-min-price').val()) || min;
                const currentMax = parseFloat($wrap.find('#s-ssdf-max-price').val()) || max;

                noUiSlider.create(slider, {
                    start: [currentMin, currentMax],
                    connect: true,
                    range: { 'min': min, 'max': max },
                    step: 1
                });

                slider.noUiSlider.on('update', function (values) {
                    $wrap.find('.price-min').text(Math.round(values[0]));
                    $wrap.find('.price-max').text(Math.round(values[1]));
                    $wrap.find('#s-ssdf-min-price').val(Math.round(values[0]));
                    $wrap.find('#s-ssdf-max-price').val(Math.round(values[1]));
                });

                slider.noUiSlider.on('change', function () {
                    triggerFilter($wrap);
                });
            });
        }

        // --- TRIGGER FILTER ---
        function triggerFilter($wrapper) {
            const repeaterID = $wrapper.data('repeater-id');
            const $repeater = $('#' + repeaterID);

            if (!$repeater.length) return;

            let currentPath = window.location.pathname;
            const filterIndex = currentPath.indexOf('/filter/');
            if (filterIndex > -1) {
                currentPath = currentPath.substring(0, filterIndex);
            }
            currentPath = currentPath.replace(/\/$/, "");

            let prettyParams = [];
            let queryParams = new URLSearchParams();

            $('.s-ssdf-filter-wrap').each(function () {
                const $el = $(this);
                const source = $el.data('source');
                let val = '';

                if (source === '_search') {
                    val = $el.find('.s-ssdf-search-input').val();
                    if (val) {
                        prettyParams.push('s-' + val);
                        queryParams.set('search_kw', val);
                    }
                }
                else if (source === '_price') {
                    const min = $el.find('#s-ssdf-min-price').val() || 0;
                    const max = $el.find('#s-ssdf-max-price').val() || 999999;
                    const dbMin = $el.data('min') || 0;
                    const dbMax = $el.data('max') || 999999;

                    if (min != dbMin || max != dbMax) {
                        prettyParams.push('price-' + min + '-' + max);
                        queryParams.set('min_price', min);
                        queryParams.set('max_price', max);
                    }
                }
                else if ($el.find('input[type="checkbox"]').length > 0) {
                    let checked = [];
                    $el.find('input:checked').each(function () { checked.push($(this).val()); });
                    if (checked.length) {
                        let joined = checked.join(',');
                        prettyParams.push(source + '-' + joined);
                        queryParams.set(source, joined);
                    }
                }
                // --- FIX: IMPROVED RADIO LOGIC ---
                else if ($el.find('input[type="radio"]').length > 0) {
                    // Use .filter(':checked') for robustness
                    const $checked = $el.find('input[type="radio"]').filter(':checked');
                    if ($checked.length) {
                        val = $checked.val();
                        // Ignore empty values or 'all' placeholder
                        if (val && val !== '' && val !== 'all') {
                            prettyParams.push(source + '-' + val);
                            queryParams.set(source, val);
                        }
                    }
                }
                else if ($el.find('select').length > 0) {
                    val = $el.find('select').val();
                    if (val) {
                        prettyParams.push(source + '-' + val);
                        queryParams.set(source, val);
                    }
                }
                else {
                    // Standard Buttons / Swatches
                    val = $el.find('.active').data('slug');
                    if (val && val !== 'all') {
                        prettyParams.push(source + '-' + val);
                        queryParams.set(source, val);
                    }
                }
            });

            let prettyUrl = window.location.origin + currentPath;
            if (prettyParams.length > 0) {
                prettyUrl += '/filter/' + prettyParams.join('/') + '/';
            } else {
                prettyUrl += '/';
            }

            // Only push state if URL changed to avoid history spam
            if (window.location.href !== prettyUrl) {
                window.history.pushState({ path: prettyUrl }, '', prettyUrl);
            }

            let ajaxUrl = window.location.origin + currentPath + '/?' + queryParams.toString();

            fetchResults($repeater, ajaxUrl);
            renderActiveChips(queryParams);
        }

        // --- FETCH RESULTS ---
        function fetchResults($repeater, url, append = false) {
            $repeater.addClass('s-ssdf-repeater-loading');

            $.get(url, function (res) {
                const repeaterID = $repeater.attr('id');
                const $newContent = $(res).find('#' + repeaterID);
                const hasTextContent = $newContent.text().trim().length > 0;

                if ($newContent.length && hasTextContent) {
                    if (append) {
                        const $innerList = $newContent.find('.oxy-dynamic-list');
                        if ($innerList.length) {
                            $repeater.find('.oxy-dynamic-list').append($innerList.html());
                        } else {
                            $repeater.append($newContent.html());
                        }
                    } else {
                        $repeater.html($newContent.html());
                    }
                } else {
                    if (!append) {
                        $repeater.html('<div class="s-ssdf-no-results">No results found.</div>');
                    }
                }

                $repeater.removeClass('s-ssdf-repeater-loading');
                $('body').trigger('post-load');

                buildNameMap();
                initSliders();
            });
        }

        // --- RENDER ACTIVE CHIPS ---
        function renderActiveChips(params) {
            const $container = $('.s-ssdf-active-chips');
            if (!$container.length) return;

            const showLabels = $container.data('show-labels') === 'yes';
            $container.empty();
            let hasParams = false;

            params.forEach((val, key) => {
                if (key === 'paged') return;

                let values = val.split(',');

                values.forEach(singleVal => {
                    let taxName = nameMap[key] || key;
                    let termName = nameMap[singleVal] || singleVal;

                    if (key === 'search_kw') { taxName = 'Search'; termName = '"' + singleVal + '"'; }
                    if (key === 'min_price') { taxName = 'Min Price'; }
                    if (key === 'max_price') { taxName = 'Max Price'; }

                    let displayText = termName;
                    if (showLabels) displayText = `<strong>${taxName}:</strong> ${termName}`;

                    const $chip = $(`<span class="s-ssdf-chip">${displayText} <i class="dashicons dashicons-dismiss"></i></span>`);

                    $chip.appendTo($container).click(function () {
                        if (key === 'min_price' || key === 'max_price') {
                            let $sliderEl = $('.s-ssdf-filter-wrap.type-slider .s-ssdf-price-slider');
                            if ($sliderEl.length) {
                                $sliderEl[0].noUiSlider.reset();
                                let $wrap = $('.s-ssdf-filter-wrap.type-slider');
                                triggerFilter($wrap);
                            } else {
                                $('.s-ssdf-price-input').val('');
                                let $wrap = $('.s-ssdf-filter-wrap.type-price');
                                if ($wrap.length) triggerFilter($wrap);
                            }
                        } else {
                            // Handle removal of specific items
                            const $wrap = $(`.s-ssdf-filter-wrap[data-source="${key}"]`);
                            let $input = $wrap.find(`input[value="${singleVal}"]`);

                            // FIX: Include swatches and improve selector
                            let $btn = $wrap.find(`.s-ssdf-filter-btn[data-slug="${singleVal}"], .s-ssdf-swatch-btn[data-slug="${singleVal}"]`);

                            if ($input.length) {
                                // Uncheck
                                $input.prop('checked', false);
                                $input.trigger('change');
                            }
                            else if ($btn.length) {
                                // FIX: Simply deactivate the button and trigger filter
                                // We don't click() because that might re-toggle it on
                                $btn.removeClass('active');

                                // If there is an "All" button, activate it for visual consistency
                                $wrap.find('.s-ssdf-filter-btn[data-slug=""]').addClass('active');

                                triggerFilter($wrap);
                            }
                            else if (key === 'search_kw') {
                                $('.s-ssdf-search-input').val('').trigger('input');
                            }
                        }
                    });
                    hasParams = true;
                });
            });

            if (hasParams) {
                $('<button class="s-ssdf-clear-all">Clear All</button>').appendTo($container).click(function () {
                    let currentPath = window.location.pathname;
                    const filterIndex = currentPath.indexOf('/filter/');
                    let baseUrl = window.location.origin;
                    if (filterIndex > -1) {
                        baseUrl += currentPath.substring(0, filterIndex);
                    } else {
                        baseUrl += currentPath;
                    }
                    window.location.href = baseUrl;
                });
            }
        }

        // --- EVENTS ---
        let searchTimeout;
        $(document).on('input', '.s-ssdf-search-input', function () {
            clearTimeout(searchTimeout);
            const $wrap = $(this).closest('.s-ssdf-filter-wrap');
            searchTimeout = setTimeout(() => { triggerFilter($wrap); }, 500);
        });

        $(document).on('click', '.s-ssdf-load-more', function (e) {
            e.preventDefault();
            const $btn = $(this);
            const $repeater = $('#' + $btn.data('repeater-id'));
            let currentUrl = new URL(window.location.href);
            let page = parseInt(currentUrl.searchParams.get('paged') || 1);

            let ajaxUrl = currentUrl.href;
            if (ajaxUrl.indexOf('?') === -1) ajaxUrl += '?paged=' + (page + 1);
            else ajaxUrl += '&paged=' + (page + 1);

            fetchResults($repeater, ajaxUrl, true);
        });

        $(document).on('click', '.s-ssdf-filter-btn, .s-ssdf-swatch-btn, .s-ssdf-apply-price', function (e) {
            e.preventDefault();
            $(this).siblings().removeClass('active');
            $(this).addClass('active');
            triggerFilter($(this).closest('.s-ssdf-filter-wrap'));
        });

        // --- FIX: Separated Radio Event Listener for Robustness ---
        $(document).on('change', 'input[type="radio"]', function () {
            // Standard radios: ensure only this one is checked in the DOM if needed (browser handles this usually)
            // But we must trigger the filter immediately
            triggerFilter($(this).closest('.s-ssdf-filter-wrap'));
        });

        // Standard inputs (Checkbox, Select)
        $(document).on('change', 'input[type="checkbox"], .s-ssdf-filter-select', function () {
            triggerFilter($(this).closest('.s-ssdf-filter-wrap'));
        });

        // Init
        buildNameMap();
        initSliders();

        if ($('.s-ssdf-active-chips').length) {
            let params = new URLSearchParams(window.location.search);
            if (Array.from(params).length > 0) renderActiveChips(params);
        }
    });
})(jQuery);