(function ($, wp) {
  "use strict";

  if (!wp || !wp.media) {
    return;
  }

  var media = wp.media;

  var MODEL_OPTIONS = [
    {
      label: "FLUX.2 Dev (Cheapest - Around $0.01/image)",
      value: "runware:400@1",
      pricing: "https://runware.ai/pricing",
    },
    {
      label: "Nano Banana (Google - around $0.04/image)",
      value: "google:4@1",
      pricing: "https://runware.ai/pricing",
    },
    {
      label: "GPT Image 1.5 (OpenAI - around $0.04/image for Medium quality)",
      value: "openai:4@1",
      pricing: "https://runware.ai/pricing",
    },
    {
      label: "Midjourney V7 (Midjourney - around $0.10/four images)",
      value: "midjourney:3@1",
      pricing: "https://runware.ai/pricing",
    },
    {
      label: "Nano Banana 2 Pro (Google - expensive $0.138/image)",
      value: "google:4@2",
      pricing: "https://runware.ai/pricing",
    },
  ];

  var STORAGE_KEY = "srizon_ai_image_last_model";

  function getLastModel() {
    try {
      var saved = localStorage.getItem(STORAGE_KEY);
      if (
        saved &&
        MODEL_OPTIONS.some(function (m) {
          return m.value === saved;
        })
      ) {
        return saved;
      }
    } catch (e) {}
    return MODEL_OPTIONS[0].value;
  }

  function saveLastModel(model) {
    try {
      localStorage.setItem(STORAGE_KEY, model);
    } catch (e) {}
  }

  // Model-specific dimension presets
  // Sorted: Wide first (widest to narrowest), then Square, then Tall (narrowest to tallest)
  var MODEL_DIMENSIONS = {
    // FLUX.2 Dev - Min 64x64, max 4MP, multiples of 16
    "runware:400@1": [
      { label: "Ultra Wide 21:9 (1792x768)", value: "1792x768" },
      { label: "Wide 16:9 (1792x1024)", value: "1792x1024" },
      { label: "Landscape 3:2 (1536x1024)", value: "1536x1024" },
      { label: "Square 1:1 (1024x1024)", value: "1024x1024" },
      { label: "Portrait 2:3 (1024x1536)", value: "1024x1536" },
      { label: "Tall 9:16 (1024x1792)", value: "1024x1792" },
      { label: "Ultra Tall 9:21 (768x1792)", value: "768x1792" },
    ],
    // Nano Banana
    "google:4@1": [
      { label: "Ultra Wide (1536x672)", value: "1536x672" },
      { label: "Wide 16:9 (1344x768)", value: "1344x768" },
      { label: "Landscape 3:2 (1248x832)", value: "1248x832" },
      { label: "Landscape 4:3 (1184x864)", value: "1184x864" },
      { label: "Landscape 5:4 (1152x896)", value: "1152x896" },
      { label: "Square 1:1 (1024x1024)", value: "1024x1024" },
      { label: "Portrait 4:5 (896x1152)", value: "896x1152" },
      { label: "Portrait 3:4 (864x1184)", value: "864x1184" },
      { label: "Portrait 2:3 (832x1248)", value: "832x1248" },
      { label: "Tall 9:16 (768x1344)", value: "768x1344" },
    ],
    // Nano Banana 2 Pro
    "google:4@2": [
      { label: "Ultra Wide (1584x672)", value: "1584x672" },
      { label: "Wide 16:9 (1376x768)", value: "1376x768" },
      { label: "Landscape 3:2 (1264x848)", value: "1264x848" },
      { label: "Landscape 4:3 (1200x896)", value: "1200x896" },
      { label: "Landscape 5:4 (1152x928)", value: "1152x928" },
      { label: "Square 1:1 (1024x1024)", value: "1024x1024" },
      { label: "Square 2K (2048x2048)", value: "2048x2048" },
      { label: "Square 4K (4096x4096)", value: "4096x4096" },
      { label: "Portrait 4:5 (928x1152)", value: "928x1152" },
      { label: "Portrait 3:4 (896x1200)", value: "896x1200" },
      { label: "Portrait 2:3 (848x1264)", value: "848x1264" },
      { label: "Tall 9:16 (768x1376)", value: "768x1376" },
    ],
    // GPT Image 1.5
    "openai:4@1": [
      { label: "Landscape 3:2 (1536x1024)", value: "1536x1024" },
      { label: "Square 1:1 (1024x1024)", value: "1024x1024" },
      { label: "Portrait 2:3 (1024x1536)", value: "1024x1536" },
    ],
    // DALL-E 3
    "openai:2@3": [
      { label: "Wide 16:9 (1792x1024)", value: "1792x1024" },
      { label: "Square 1:1 (1024x1024)", value: "1024x1024" },
      { label: "Tall 9:16 (1024x1792)", value: "1024x1792" },
    ],
    // Midjourney V7
    "midjourney:3@1": [
      { label: "Cinematic 21:9 (1680x720)", value: "1680x720" },
      { label: "Wide 16:9 (1456x816)", value: "1456x816" },
      { label: "Landscape 3:2 (1344x896)", value: "1344x896" },
      { label: "Landscape 4:3 (1232x928)", value: "1232x928" },
      { label: "Square 1:1 (1024x1024)", value: "1024x1024" },
      { label: "Portrait 3:4 (928x1232)", value: "928x1232" },
      { label: "Portrait 2:3 (896x1344)", value: "896x1344" },
      { label: "Tall 9:16 (816x1456)", value: "816x1456" },
      { label: "Cinematic Vertical 9:21 (720x1680)", value: "720x1680" },
    ],
  };

  function getDimensionsForModel(model) {
    return MODEL_DIMENSIONS[model] || MODEL_DIMENSIONS["runware:400@1"];
  }

  function getDefaultDimensionForModel(model) {
    var dims = getDimensionsForModel(model);
    // Find Square 1:1 as default, or use first option
    for (var i = 0; i < dims.length; i++) {
      if (dims[i].value === "1024x1024") {
        return dims[i].value;
      }
    }
    return dims[0].value;
  }

  function buildDimensionOptions(model) {
    return getDimensionsForModel(model)
      .map(function (d) {
        return '<option value="' + d.value + '">' + d.label + "</option>";
      })
      .join("");
  }

  // Model parameter configurations by provider
  var MODEL_PARAMS = {
    runware: {
      supportsSteps: true,
      supportsCfg: true,
      supportsQuality: false,
      stepsDefault: 20,
      stepsMin: 1,
      stepsMax: 50,
      cfgDefault: 4,
      cfgMin: 1,
      cfgMax: 20,
    },
    civitai: {
      supportsSteps: true,
      supportsCfg: true,
      supportsQuality: false,
      stepsDefault: 30,
      stepsMin: 1,
      stepsMax: 50,
      cfgDefault: 7,
      cfgMin: 1,
      cfgMax: 20,
    },
    google: {
      supportsSteps: false,
      supportsCfg: false,
      supportsQuality: false,
    },
    openai: { supportsSteps: false, supportsCfg: false, supportsQuality: true },
    midjourney: {
      supportsSteps: false,
      supportsCfg: false,
      supportsQuality: false,
    },
  };

  var QUALITY_OPTIONS = [
    { label: "High", value: "high" },
    { label: "Medium", value: "medium" },
    { label: "Low", value: "low" },
    { label: "Auto", value: "auto" },
  ];

  function getProviderFromModel(model) {
    return model.split(":")[0];
  }

  function getModelParams(model) {
    var provider = getProviderFromModel(model);
    return (
      MODEL_PARAMS[provider] || { supportsSteps: false, supportsCfg: false }
    );
  }

  function isMidjourneyModel(model) {
    return getProviderFromModel(model) === "midjourney";
  }

  // Custom content view for AI Image tab
  media.view.SrizonAIImage = media.View.extend({
    tagName: "div",
    className: "srizon-ai-image-media-tab",

    events: {
      "click .srizon-generate-btn": "generateImage",
      "change .srizon-model-select": "onModelChange",
    },

    initialize: function () {
      this.isGenerating = false;
    },

    render: function () {
      var defaultModel = getLastModel();
      var modelOptions = MODEL_OPTIONS.map(function (m) {
        var selected = m.value === defaultModel ? " selected" : "";
        return (
          '<option value="' +
          m.value +
          '"' +
          selected +
          ">" +
          m.label +
          "</option>"
        );
      }).join("");

      var dimensionOptions = buildDimensionOptions(defaultModel);
      var defaultDimension = getDefaultDimensionForModel(defaultModel);

      var defaultParams = getModelParams(defaultModel);
      var defaultModelOption =
        MODEL_OPTIONS.find(function (m) {
          return m.value === defaultModel;
        }) || MODEL_OPTIONS[0];

      var html =
        '<div class="srizon-ai-image-form">' +
        "<h2>Generate Image with Srizon AI Image</h2>" +
        '<div class="srizon-form-field">' +
        '<label for="srizon-prompt">Image Prompt</label>' +
        '<textarea id="srizon-prompt" class="srizon-prompt" rows="4" placeholder="Describe the image you want to generate..."></textarea>' +
        "</div>" +
        '<div class="srizon-form-row">' +
        '<div class="srizon-form-field srizon-model-field">' +
        '<label for="srizon-model">Model</label>' +
        '<select id="srizon-model" class="srizon-model-select">' +
        modelOptions +
        "</select>" +
        "</div>" +
        '<a href="' +
        defaultModelOption.pricing +
        '" target="_blank" rel="noopener noreferrer" class="srizon-pricing-link">View Pricing ↗</a>' +
        "</div>" +
        '<div class="srizon-info-note srizon-midjourney-note" style="' +
        (isMidjourneyModel(defaultModel) ? "" : "display:none;") +
        '">' +
        "Note: Midjourney generates 4 images per request. All images will be saved to your Media Library." +
        "</div>" +
        '<div class="srizon-form-field">' +
        '<label for="srizon-dimensions">Image Dimensions</label>' +
        '<select id="srizon-dimensions" class="srizon-dimensions-select">' +
        dimensionOptions +
        "</select>" +
        '<span class="srizon-param-help">Higher dimensions will increase generation price.</span>' +
        "</div>" +
        '<div class="srizon-form-field srizon-advanced-params" style="' +
        (defaultParams.supportsSteps || defaultParams.supportsCfg
          ? ""
          : "display:none;") +
        '">' +
        '<div class="srizon-params-label">Advanced Parameters</div>' +
        '<div class="srizon-params-row">' +
        '<div class="srizon-param-input srizon-steps-input" style="' +
        (defaultParams.supportsSteps ? "" : "display:none;") +
        '">' +
        '<label for="srizon-steps">Steps</label>' +
        '<input type="number" id="srizon-steps" class="srizon-steps" value="' +
        (defaultParams.stepsDefault || 28) +
        '" min="' +
        (defaultParams.stepsMin || 1) +
        '" max="' +
        (defaultParams.stepsMax || 50) +
        '" step="1" />' +
        '<span class="srizon-param-hint">' +
        (defaultParams.stepsMin || 1) +
        "-" +
        (defaultParams.stepsMax || 50) +
        "</span>" +
        '<span class="srizon-param-help">Controls quality and refinement (how "finished" the image looks). Higher values increase price.</span>' +
        "</div>" +
        '<div class="srizon-param-input srizon-cfg-input" style="' +
        (defaultParams.supportsCfg ? "" : "display:none;") +
        '">' +
        '<label for="srizon-cfg">CFG Scale</label>' +
        '<input type="number" id="srizon-cfg" class="srizon-cfg" value="' +
        (defaultParams.cfgDefault || 7) +
        '" min="' +
        (defaultParams.cfgMin || 1) +
        '" max="' +
        (defaultParams.cfgMax || 20) +
        '" step="0.5" />' +
        '<span class="srizon-param-hint">' +
        (defaultParams.cfgMin || 1) +
        "-" +
        (defaultParams.cfgMax || 20) +
        "</span>" +
        '<span class="srizon-param-help">Controls creativity vs. obedience (how strictly it follows your prompt)</span>' +
        "</div>" +
        "</div>" +
        "</div>" +
        '<div class="srizon-form-field srizon-quality-params" style="' +
        (defaultParams.supportsQuality ? "" : "display:none;") +
        '">' +
        '<div class="srizon-params-label">Quality Settings</div>' +
        '<div class="srizon-quality-field">' +
        '<label for="srizon-quality">Quality</label>' +
        '<select id="srizon-quality" class="srizon-quality-select">' +
        QUALITY_OPTIONS.map(function (q) {
          return '<option value="' + q.value + '">' + q.label + "</option>";
        }).join("") +
        "</select>" +
        '<span class="srizon-param-hint">Higher quality takes longer and costs more.</span>' +
        "</div>" +
        "</div>" +
        '<div class="srizon-form-field">' +
        '<button type="button" class="button button-primary srizon-generate-btn">Generate Image</button>' +
        "</div>" +
        '<div class="srizon-error" style="display:none;"></div>' +
        '<div class="srizon-success" style="display:none;"></div>' +
        '<div class="srizon-loading" style="display:none;">' +
        '<span class="spinner is-active"></span> Generating image...' +
        "</div>" +
        "</div>";

      this.$el.html(html);
      return this;
    },

    onModelChange: function () {
      var selectedModel = this.$(".srizon-model-select").val();
      saveLastModel(selectedModel);
      var modelOption = MODEL_OPTIONS.find(function (m) {
        return m.value === selectedModel;
      });
      if (modelOption) {
        this.$(".srizon-pricing-link").attr("href", modelOption.pricing);
      }

      // Update dimension options for new model
      var newDimensionOptions = buildDimensionOptions(selectedModel);
      this.$(".srizon-dimensions-select").html(newDimensionOptions);
      this.$(".srizon-dimensions-select").val(
        getDefaultDimensionForModel(selectedModel)
      );

      // Show/hide Midjourney note
      if (isMidjourneyModel(selectedModel)) {
        this.$(".srizon-midjourney-note").show();
      } else {
        this.$(".srizon-midjourney-note").hide();
      }

      // Update advanced params visibility and values
      var params = getModelParams(selectedModel);
      if (params.supportsSteps || params.supportsCfg) {
        this.$(".srizon-advanced-params").show();
      } else {
        this.$(".srizon-advanced-params").hide();
      }

      if (params.supportsSteps) {
        this.$(".srizon-steps-input").show();
        this.$(".srizon-steps")
          .attr("min", params.stepsMin)
          .attr("max", params.stepsMax)
          .val(params.stepsDefault);
        this.$(".srizon-steps-input .srizon-param-hint").text(
          params.stepsMin + "-" + params.stepsMax
        );
      } else {
        this.$(".srizon-steps-input").hide();
      }

      if (params.supportsCfg) {
        this.$(".srizon-cfg-input").show();
        this.$(".srizon-cfg")
          .attr("min", params.cfgMin)
          .attr("max", params.cfgMax)
          .val(params.cfgDefault);
        this.$(".srizon-cfg-input .srizon-param-hint").text(
          params.cfgMin + "-" + params.cfgMax
        );
      } else {
        this.$(".srizon-cfg-input").hide();
      }

      // Show/hide quality params
      if (params.supportsQuality) {
        this.$(".srizon-quality-params").show();
      } else {
        this.$(".srizon-quality-params").hide();
      }
    },

    generateImage: function () {
      var self = this;
      var prompt = this.$(".srizon-prompt").val().trim();
      var model = this.$(".srizon-model-select").val();
      var dimensions = this.$(".srizon-dimensions-select").val();
      var modelParams = getModelParams(model);

      if (!prompt) {
        this.showError("Please enter a prompt.");
        return;
      }

      var parts = dimensions.split("x");
      var width = parseInt(parts[0], 10);
      var height = parseInt(parts[1], 10);

      var requestData = {
        prompt: prompt,
        width: width,
        height: height,
        model: model,
      };

      if (modelParams.supportsSteps) {
        var steps = parseInt(this.$(".srizon-steps").val(), 10);
        if (steps > 0) {
          requestData.steps = steps;
        }
      }

      if (modelParams.supportsCfg) {
        var cfgScale = parseFloat(this.$(".srizon-cfg").val());
        if (cfgScale > 0) {
          requestData.cfg_scale = cfgScale;
        }
      }

      if (modelParams.supportsQuality) {
        requestData.quality = this.$(".srizon-quality-select").val();
      }

      this.hideError();
      this.$(".srizon-success").hide();
      this.setLoading(true);

      wp.apiFetch({
        path: "/srizon-ai-image/v1/generate-image",
        method: "POST",
        data: requestData,
      })
        .then(function (response) {
          self.setLoading(false);
          if (response.success && response.data) {
            // Store response for success message
            self.controller._lastResponse = response;
            self.insertImage(response.data);
          } else {
            self.showError(response.message || "Failed to generate image.");
          }
        })
        .catch(function (err) {
          self.setLoading(false);
          self.showError(err.message || "An error occurred.");
        });
    },

    insertImage: function (imageData) {
      var self = this;
      var controller = this.controller;

      var attachment = new media.model.Attachment({
        id: imageData.attachment_id,
        url: imageData.url,
        alt: imageData.alt || "",
        title: imageData.alt || "",
        type: "image",
        subtype: "png",
      });

      // Fetch full attachment data
      attachment.fetch().then(function () {
        try {
          // Check if we're in the Media Library grid view (upload.php)
          if (controller.constructor === media.view.MediaFrame.Manage) {
            // Refresh the library to show the new image
            if (controller.state() && controller.state().get("library")) {
              controller.state().get("library").add(attachment, { at: 0 });
            }
            // Switch back to browse mode to see the new image
            if (controller.content && controller.content.mode) {
              controller.content.mode("browse");
            }
            // Show success message
            var successMsg =
              controller._lastResponse && controller._lastResponse.is_multi
                ? controller._lastResponse.all_images.length +
                  " images generated and added to Media Library!"
                : "Image generated and added to Media Library!";
            self.showSuccess(successMsg);
            // Clear the prompt
            self.$(".srizon-prompt").val("");
          } else {
            // Check if this is for featured image
            var isFeaturedImage =
              (controller.options &&
                controller.options.state === "featured-image") ||
              (controller.state() &&
                controller.state().id === "featured-image");

            if (isFeaturedImage) {
              // For featured image, switch to Media Library tab so user can select
              // Refresh the library to include the new image
              if (controller.state() && controller.state().get("library")) {
                controller.state().get("library").add(attachment, { at: 0 });
              }
              // Switch to the browse/library content
              if (controller.content && controller.content.mode) {
                controller.content.mode("browse");
              }
              // Show success message
              var featuredSuccessMsg =
                controller._lastResponse && controller._lastResponse.is_multi
                  ? controller._lastResponse.all_images.length +
                    " images generated! Select one from the Media Library to set as featured image."
                  : "Image generated! Select it from the Media Library to set as featured image.";
              self.showSuccess(featuredSuccessMsg);
              // Clear the prompt
              self.$(".srizon-prompt").val("");
            } else {
              // For other modal dialogs (insert, gallery, etc.)
              var state = controller.state();
              if (state) {
                var selection = state.get("selection");
                if (selection) {
                  selection.reset([attachment]);
                }

                // Trigger the insert by switching to the library and selecting
                var library = state.get("library");
                if (library) {
                  library.add(attachment, { at: 0 });
                }
              }

              // Close and trigger insert
              controller.close();
            }
          }
        } catch (e) {
          console.log("Srizon AI Image: Error updating media library view", e);
          // Still show success since images were generated
          var fallbackMsg =
            controller._lastResponse && controller._lastResponse.is_multi
              ? controller._lastResponse.all_images.length +
                " images generated! Refresh the page to see them in the Media Library."
              : "Image generated! Refresh the page to see it in the Media Library.";
          self.showSuccess(fallbackMsg);
          self.$(".srizon-prompt").val("");
        }
      });
    },

    showError: function (message) {
      this.$(".srizon-success").hide();
      this.$(".srizon-error").text(message).show();
    },

    hideError: function () {
      this.$(".srizon-error").hide();
    },

    showSuccess: function (message) {
      this.$(".srizon-error").hide();
      this.$(".srizon-success").text(message).show();
    },

    setLoading: function (loading) {
      this.isGenerating = loading;
      this.$(".srizon-generate-btn").prop("disabled", loading);
      this.$(".srizon-loading").toggle(loading);
    },
  });

  // Extend the media frame to add our tab
  var originalMediaFrameSelect = media.view.MediaFrame.Select;

  media.view.MediaFrame.Select = originalMediaFrameSelect.extend({
    browseRouter: function (routerView) {
      originalMediaFrameSelect.prototype.browseRouter.apply(this, arguments);

      routerView.set({
        "srizon-ai-image": {
          text: "Generate with AI",
          priority: 60,
        },
      });
    },

    bindHandlers: function () {
      originalMediaFrameSelect.prototype.bindHandlers.apply(this, arguments);

      this.on(
        "content:create:srizon-ai-image",
        this.createSrizonAIImageContent,
        this
      );
      this.on(
        "content:render:srizon-ai-image",
        this.renderSrizonAIImageContent,
        this
      );
    },

    createSrizonAIImageContent: function (contentRegion) {
      var view = new media.view.SrizonAIImage({
        controller: this,
      });

      contentRegion.view = view;
    },

    renderSrizonAIImageContent: function () {
      // Content is rendered in createSrizonAIImageContent
    },
  });

  // Also extend the Post/Featured Image frame
  var originalMediaFramePost = media.view.MediaFrame.Post;

  if (originalMediaFramePost) {
    media.view.MediaFrame.Post = originalMediaFramePost.extend({
      browseRouter: function (routerView) {
        originalMediaFramePost.prototype.browseRouter.apply(this, arguments);

        routerView.set({
          "srizon-ai-image": {
            text: "Generate with AI",
            priority: 60,
          },
        });
      },

      bindHandlers: function () {
        originalMediaFramePost.prototype.bindHandlers.apply(this, arguments);

        this.on(
          "content:create:srizon-ai-image",
          this.createSrizonAIImageContent,
          this
        );
        this.on(
          "content:render:srizon-ai-image",
          this.renderSrizonAIImageContent,
          this
        );
      },

      createSrizonAIImageContent: function (contentRegion) {
        var view = new media.view.SrizonAIImage({
          controller: this,
        });

        contentRegion.view = view;
      },

      renderSrizonAIImageContent: function () {
        // Content is rendered in createSrizonAIImageContent
      },
    });
  }

  // Add "Generate with AI" button to Media Library page (upload.php)
  $(document).ready(function () {
    // Check if we're on the upload.php page
    var $addNewButton = $(".page-title-action");
    if ($addNewButton.length && window.pagenow === "upload") {
      // Create the Generate with AI button
      var $generateBtn = $(
        '<a href="#" class="page-title-action srizon-ai-image-generate-btn">Generate with AI</a>'
      );

      // Insert after the Add New button
      $addNewButton.after($generateBtn);

      // Handle click
      $generateBtn.on("click", function (e) {
        e.preventDefault();
        openSrizonAIImageModal();
      });
    }
  });

  // Modal for upload.php page
  function openSrizonAIImageModal() {
    var defaultModel = getLastModel();
    var modelOptions = MODEL_OPTIONS.map(function (m) {
      var selected = m.value === defaultModel ? " selected" : "";
      return (
        '<option value="' +
        m.value +
        '"' +
        selected +
        ">" +
        m.label +
        "</option>"
      );
    }).join("");

    var dimensionOptions = buildDimensionOptions(defaultModel);

    var defaultParams = getModelParams(defaultModel);
    var defaultModelOption =
      MODEL_OPTIONS.find(function (m) {
        return m.value === defaultModel;
      }) || MODEL_OPTIONS[0];

    var modalHtml =
      '<div class="srizon-ai-image-modal-overlay">' +
      '<div class="srizon-ai-image-modal-dialog">' +
      '<div class="srizon-ai-image-modal-header">' +
      "<h2>Generate Image with Srizon AI Image</h2>" +
      '<button type="button" class="srizon-modal-close">&times;</button>' +
      "</div>" +
      '<div class="srizon-ai-image-modal-body">' +
      '<div class="srizon-ai-image-form">' +
      '<div class="srizon-form-field">' +
      '<label for="srizon-modal-prompt">Image Prompt</label>' +
      '<textarea id="srizon-modal-prompt" class="srizon-prompt" rows="4" placeholder="Describe the image you want to generate..."></textarea>' +
      "</div>" +
      '<div class="srizon-form-row">' +
      '<div class="srizon-form-field srizon-model-field">' +
      '<label for="srizon-modal-model">Model</label>' +
      '<select id="srizon-modal-model" class="srizon-model-select">' +
      modelOptions +
      "</select>" +
      "</div>" +
      '<a href="' +
      defaultModelOption.pricing +
      '" target="_blank" rel="noopener noreferrer" class="srizon-pricing-link">View Pricing ↗</a>' +
      "</div>" +
      '<div class="srizon-info-note srizon-midjourney-note" style="' +
      (isMidjourneyModel(defaultModel) ? "" : "display:none;") +
      '">' +
      "Note: Midjourney generates 4 images per request. All images will be saved to your Media Library." +
      "</div>" +
      '<div class="srizon-form-field">' +
      '<label for="srizon-modal-dimensions">Image Dimensions</label>' +
      '<select id="srizon-modal-dimensions" class="srizon-dimensions-select">' +
      dimensionOptions +
      "</select>" +
      '<span class="srizon-param-help">Higher dimensions will increase generation price.</span>' +
      "</div>" +
      '<div class="srizon-form-field srizon-advanced-params" style="' +
      (defaultParams.supportsSteps || defaultParams.supportsCfg
        ? ""
        : "display:none;") +
      '">' +
      '<div class="srizon-params-label">Advanced Parameters</div>' +
      '<div class="srizon-params-row">' +
      '<div class="srizon-param-input srizon-steps-input" style="' +
      (defaultParams.supportsSteps ? "" : "display:none;") +
      '">' +
      '<label for="srizon-modal-steps">Steps</label>' +
      '<input type="number" id="srizon-modal-steps" class="srizon-steps" value="' +
      (defaultParams.stepsDefault || 28) +
      '" min="' +
      (defaultParams.stepsMin || 1) +
      '" max="' +
      (defaultParams.stepsMax || 50) +
      '" step="1" />' +
      '<span class="srizon-param-hint">' +
      (defaultParams.stepsMin || 1) +
      "-" +
      (defaultParams.stepsMax || 50) +
      "</span>" +
      '<span class="srizon-param-help">Controls quality and refinement (how "finished" the image looks). Higher values increase price.</span>' +
      "</div>" +
      '<div class="srizon-param-input srizon-cfg-input" style="' +
      (defaultParams.supportsCfg ? "" : "display:none;") +
      '">' +
      '<label for="srizon-modal-cfg">CFG Scale</label>' +
      '<input type="number" id="srizon-modal-cfg" class="srizon-cfg" value="' +
      (defaultParams.cfgDefault || 7) +
      '" min="' +
      (defaultParams.cfgMin || 1) +
      '" max="' +
      (defaultParams.cfgMax || 20) +
      '" step="0.5" />' +
      '<span class="srizon-param-hint">' +
      (defaultParams.cfgMin || 1) +
      "-" +
      (defaultParams.cfgMax || 20) +
      "</span>" +
      '<span class="srizon-param-help">Controls creativity vs. obedience (how strictly it follows your prompt)</span>' +
      "</div>" +
      "</div>" +
      "</div>" +
      '<div class="srizon-form-field srizon-quality-params" style="' +
      (defaultParams.supportsQuality ? "" : "display:none;") +
      '">' +
      '<div class="srizon-params-label">Quality Settings</div>' +
      '<div class="srizon-quality-field">' +
      '<label for="srizon-modal-quality">Quality</label>' +
      '<select id="srizon-modal-quality" class="srizon-quality-select">' +
      QUALITY_OPTIONS.map(function (q) {
        return '<option value="' + q.value + '">' + q.label + "</option>";
      }).join("") +
      "</select>" +
      '<span class="srizon-param-hint">Higher quality takes longer and costs more.</span>' +
      "</div>" +
      "</div>" +
      '<div class="srizon-form-field">' +
      '<button type="button" class="button button-primary srizon-generate-btn">Generate Image</button>' +
      "</div>" +
      '<div class="srizon-error" style="display:none;"></div>' +
      '<div class="srizon-success" style="display:none;"></div>' +
      '<div class="srizon-loading" style="display:none;">' +
      '<span class="spinner is-active"></span> Generating image...' +
      "</div>" +
      "</div>" +
      "</div>" +
      "</div>" +
      "</div>";

    var $modal = $(modalHtml);
    $("body").append($modal);

    // Event handlers
    $modal.find(".srizon-modal-close").on("click", function () {
      $modal.remove();
    });

    $modal.find(".srizon-ai-image-modal-overlay").on("click", function (e) {
      if (e.target === this) {
        $modal.remove();
      }
    });

    $modal.find(".srizon-model-select").on("change", function () {
      var selectedModel = $(this).val();
      saveLastModel(selectedModel);
      var modelOption = MODEL_OPTIONS.find(function (m) {
        return m.value === selectedModel;
      });
      if (modelOption) {
        $modal.find(".srizon-pricing-link").attr("href", modelOption.pricing);
      }

      // Update dimension options for new model
      var newDimensionOptions = buildDimensionOptions(selectedModel);
      $modal.find(".srizon-dimensions-select").html(newDimensionOptions);
      $modal
        .find(".srizon-dimensions-select")
        .val(getDefaultDimensionForModel(selectedModel));

      // Show/hide Midjourney note
      if (isMidjourneyModel(selectedModel)) {
        $modal.find(".srizon-midjourney-note").show();
      } else {
        $modal.find(".srizon-midjourney-note").hide();
      }

      // Update advanced params visibility and values
      var params = getModelParams(selectedModel);
      if (params.supportsSteps || params.supportsCfg) {
        $modal.find(".srizon-advanced-params").show();
      } else {
        $modal.find(".srizon-advanced-params").hide();
      }

      if (params.supportsSteps) {
        $modal.find(".srizon-steps-input").show();
        $modal
          .find(".srizon-steps")
          .attr("min", params.stepsMin)
          .attr("max", params.stepsMax)
          .val(params.stepsDefault);
        $modal
          .find(".srizon-steps-input .srizon-param-hint")
          .text(params.stepsMin + "-" + params.stepsMax);
      } else {
        $modal.find(".srizon-steps-input").hide();
      }

      if (params.supportsCfg) {
        $modal.find(".srizon-cfg-input").show();
        $modal
          .find(".srizon-cfg")
          .attr("min", params.cfgMin)
          .attr("max", params.cfgMax)
          .val(params.cfgDefault);
        $modal
          .find(".srizon-cfg-input .srizon-param-hint")
          .text(params.cfgMin + "-" + params.cfgMax);
      } else {
        $modal.find(".srizon-cfg-input").hide();
      }

      // Show/hide quality params
      if (params.supportsQuality) {
        $modal.find(".srizon-quality-params").show();
      } else {
        $modal.find(".srizon-quality-params").hide();
      }
    });

    $modal.find(".srizon-generate-btn").on("click", function () {
      var prompt = $modal.find(".srizon-prompt").val().trim();
      var model = $modal.find(".srizon-model-select").val();
      var dimensions = $modal.find(".srizon-dimensions-select").val();
      var modelParams = getModelParams(model);

      if (!prompt) {
        $modal.find(".srizon-error").text("Please enter a prompt.").show();
        return;
      }

      var parts = dimensions.split("x");
      var width = parseInt(parts[0], 10);
      var height = parseInt(parts[1], 10);

      var requestData = {
        prompt: prompt,
        width: width,
        height: height,
        model: model,
      };

      if (modelParams.supportsSteps) {
        var steps = parseInt($modal.find(".srizon-steps").val(), 10);
        if (steps > 0) {
          requestData.steps = steps;
        }
      }

      if (modelParams.supportsCfg) {
        var cfgScale = parseFloat($modal.find(".srizon-cfg").val());
        if (cfgScale > 0) {
          requestData.cfg_scale = cfgScale;
        }
      }

      if (modelParams.supportsQuality) {
        requestData.quality = $modal.find(".srizon-quality-select").val();
      }

      $modal.find(".srizon-error").hide();
      $modal.find(".srizon-success").hide();
      $modal.find(".srizon-generate-btn").prop("disabled", true);
      $modal.find(".srizon-loading").show();

      wp.apiFetch({
        path: "/srizon-ai-image/v1/generate-image",
        method: "POST",
        data: requestData,
      })
        .then(function (response) {
          $modal.find(".srizon-generate-btn").prop("disabled", false);
          $modal.find(".srizon-loading").hide();

          if (response.success && response.data) {
            var successMsg = response.is_multi
              ? response.all_images.length +
                " images generated successfully! Refreshing library..."
              : "Image generated successfully! Refreshing library...";
            $modal.find(".srizon-success").text(successMsg).show();
            $modal.find(".srizon-prompt").val("");

            // Refresh the media library grid
            var refreshed = false;
            if (wp.media.frame && wp.media.frame.content) {
              try {
                var content = wp.media.frame.content.get();
                if (content && content.collection && content.collection.props) {
                  content.collection.props.set({ ignore: +new Date() });
                  refreshed = true;
                }
              } catch (e) {
                console.log(
                  "Srizon AI Image: Could not refresh library automatically",
                  e
                );
              }
            }
            if (!refreshed) {
              // Fallback: reload the page
              setTimeout(function () {
                window.location.reload();
              }, 1000);
            }
          } else {
            $modal
              .find(".srizon-error")
              .text(response.message || "Failed to generate image.")
              .show();
          }
        })
        .catch(function (err) {
          $modal.find(".srizon-generate-btn").prop("disabled", false);
          $modal.find(".srizon-loading").hide();
          $modal
            .find(".srizon-error")
            .text(err.message || "An error occurred.")
            .show();
        });
    });
  }

  console.log("Srizon AI Image: Media library integration loaded");
})(jQuery, window.wp);
