<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @package Srizon_AI_Image
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Srizon_AI_Image_Admin {

    /**
     * The ID of this plugin.
     *
     * @var string
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @var string
     */
    private $version;

    /**
     * Option name for the API key.
     *
     * @var string
     */
    const OPTION_API_KEY = 'srizon_ai_image_runware_api_key';

    /**
     * Initialize the class and set its properties.
     *
     * @param string $plugin_name The name of this plugin.
     * @param string $version     The version of this plugin.
     */
    public function __construct( $plugin_name, $version ) {
        $this->plugin_name = $plugin_name;
        $this->version     = $version;
    }

    /**
     * Register the stylesheets for the admin area.
     */
    public function enqueue_styles() {
        wp_enqueue_style(
            $this->plugin_name,
            SRIZON_AI_IMAGE_URL . 'admin/css/srizon-ai-image-admin.css',
            array(),
            $this->version,
            'all'
        );
    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @param string $hook_suffix The current admin page hook suffix.
     */
    public function enqueue_scripts( $hook_suffix ) {
        wp_enqueue_script(
            $this->plugin_name,
            SRIZON_AI_IMAGE_URL . 'admin/js/srizon-ai-image-admin.js',
            array( 'jquery' ),
            $this->version,
            false
        );

        // Add inline script for API key toggle on settings page.
        if ( 'settings_page_srizon-ai-image-settings' === $hook_suffix ) {
            $show_text = esc_js( __( 'Show', 'srizon-ai-image' ) );
            $hide_text = esc_js( __( 'Hide', 'srizon-ai-image' ) );
            $inline_script = "
                document.addEventListener('DOMContentLoaded', function() {
                    var toggleBtn = document.getElementById('srizon-toggle-api-key');
                    if (toggleBtn) {
                        toggleBtn.addEventListener('click', function() {
                            var input = document.getElementById('runware_api_key');
                            var button = this;
                            if (input.type === 'password') {
                                input.type = 'text';
                                button.textContent = '{$hide_text}';
                            } else {
                                input.type = 'password';
                                button.textContent = '{$show_text}';
                            }
                        });
                    }
                });
            ";
            wp_add_inline_script( $this->plugin_name, $inline_script );
        }
    }

    /**
     * Enqueue media library integration script.
     */
    public function enqueue_media_script() {
        wp_enqueue_script(
            $this->plugin_name . '-media',
            SRIZON_AI_IMAGE_URL . 'admin/js/srizon-ai-image-media.js',
            array( 'jquery', 'media-views', 'wp-api-fetch' ),
            $this->version,
            true
        );
    }

    /**
     * Enqueue background removal script.
     */
    public function enqueue_bg_removal_script() {
        wp_enqueue_script(
            $this->plugin_name . '-bg-removal',
            SRIZON_AI_IMAGE_URL . 'admin/js/srizon-ai-image-bg-removal.js',
            array( 'jquery', 'media-views' ),
            $this->version,
            true
        );

        wp_localize_script(
            $this->plugin_name . '-bg-removal',
            'srizonAiBgRemoval',
            array(
                'ajaxUrl'     => admin_url( 'admin-ajax.php' ),
                'nonce'       => wp_create_nonce( 'srizon_ai_bg_nonce' ),
                'buttonText'  => __( 'Remove Background', 'srizon-ai-image' ),
                'processing'  => __( 'Processing...', 'srizon-ai-image' ),
                'success'     => __( 'Background removed!', 'srizon-ai-image' ),
                'error'       => __( 'Error removing background.', 'srizon-ai-image' ),
                'noSelection' => __( 'Please select at least one image.', 'srizon-ai-image' ),
                'noApiKey'    => __( 'Please configure your API key first.', 'srizon-ai-image' ),
                'hasApiKey'   => self::is_api_key_configured(),
                'settingsUrl' => admin_url( 'options-general.php?page=srizon-ai-image-settings' ),
            )
        );
    }

    /**
     * Enqueue block editor assets.
     */
    public function enqueue_block_editor_assets() {
        wp_enqueue_script(
            $this->plugin_name . '-block-editor',
            SRIZON_AI_IMAGE_URL . 'admin/js/srizon-ai-image-block-editor.js',
            array(
                'wp-blocks',
                'wp-element',
                'wp-components',
                'wp-compose',
                'wp-hooks',
                'wp-i18n',
                'wp-block-editor',
                'wp-api-fetch',
            ),
            $this->version,
            true
        );

        wp_enqueue_style(
            $this->plugin_name . '-block-editor',
            SRIZON_AI_IMAGE_URL . 'admin/css/srizon-ai-image-admin.css',
            array( 'wp-components' ),
            $this->version
        );
    }

    /**
     * Add settings page to the admin menu.
     */
    public function add_settings_page() {
        add_options_page(
            __( 'Srizon AI Image Settings', 'srizon-ai-image' ),
            __( 'Srizon AI Image', 'srizon-ai-image' ),
            'manage_options',
            'srizon-ai-image-settings',
            array( $this, 'render_settings_page' )
        );
    }

    /**
     * Register plugin settings.
     */
    public function register_settings() {
        register_setting(
            'srizon_ai_image_settings',
            self::OPTION_API_KEY,
            array(
                'type'              => 'string',
                'sanitize_callback' => 'sanitize_text_field',
                'default'           => '',
            )
        );

        add_settings_section(
            'srizon_ai_image_api_section',
            __( 'API Configuration', 'srizon-ai-image' ),
            array( $this, 'render_api_section' ),
            'srizon-ai-image-settings'
        );

        add_settings_field(
            'runware_api_key',
            __( 'Runware API Key', 'srizon-ai-image' ),
            array( $this, 'render_api_key_field' ),
            'srizon-ai-image-settings',
            'srizon_ai_image_api_section'
        );
    }

    /**
     * Render the settings page.
     */
    public function render_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        ?>
        <div class="wrap">
            <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
            <form action="options.php" method="post">
                <?php
                settings_fields( 'srizon_ai_image_settings' );
                do_settings_sections( 'srizon-ai-image-settings' );
                submit_button( __( 'Save Settings', 'srizon-ai-image' ) );
                ?>
            </form>
        </div>
        <?php
    }

    /**
     * Render the API section description.
     */
    public function render_api_section() {
        echo '<p>' . esc_html__( 'Enter your Runware API key to enable the plugin functionality.', 'srizon-ai-image' ) . '</p>';
        echo '<p style="color: #666; font-style: italic;">' . esc_html__( 'Disclaimer: This plugin is not affiliated with, endorsed by, or sponsored by Runware. It simply integrates with the Runware API as a third-party service.', 'srizon-ai-image' ) . '</p>';
    }

    /**
     * Render the API key field.
     */
    public function render_api_key_field() {
        $api_key = get_option( self::OPTION_API_KEY, '' );
        ?>
        <input
            type="password"
            id="runware_api_key"
            name="<?php echo esc_attr( self::OPTION_API_KEY ); ?>"
            value="<?php echo esc_attr( $api_key ); ?>"
            class="regular-text"
        />
        <button type="button" id="srizon-toggle-api-key" class="button button-secondary">
            <?php esc_html_e( 'Show', 'srizon-ai-image' ); ?>
        </button>
        <p class="description">
            <?php esc_html_e( 'Your Runware API key is required for the plugin to function.', 'srizon-ai-image' ); ?>
        </p>
        <details class="srizon-ai-image-api-help">
            <summary><?php esc_html_e( 'How to get the API key', 'srizon-ai-image' ); ?></summary>
            <ol>
                <li><?php
					printf(
						/* translators: %1$s: opening link tag, %2$s: closing link tag */
						esc_html__( 'Go to %1$srunware.ai%2$s', 'srizon-ai-image' ),
						'<a href="https://runware.ai" target="_blank" rel="noopener noreferrer">',
						'</a>'
					);
				?></li>
                <li><?php esc_html_e( 'Create an account or sign in', 'srizon-ai-image' ); ?></li>
                <li><?php esc_html_e( 'Add funds to your account (required for image generation)', 'srizon-ai-image' ); ?></li>
                <li><?php esc_html_e( 'Navigate to the Dashboard', 'srizon-ai-image' ); ?></li>
                <li><?php esc_html_e( 'Copy your API key from the dashboard', 'srizon-ai-image' ); ?></li>
            </ol>
        </details>
        <?php
    }

    /**
     * Display admin notice if API key is not configured.
     */
    public function display_api_key_notice() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $api_key = get_option( self::OPTION_API_KEY, '' );

        if ( empty( $api_key ) ) {
            $settings_url = admin_url( 'options-general.php?page=srizon-ai-image-settings' );
            ?>
            <div class="notice notice-error">
                <p>
                    <strong><?php esc_html_e( 'Srizon AI Image:', 'srizon-ai-image' ); ?></strong>
                    <?php
                    printf(
                        /* translators: %1$s: opening link tag, %2$s: closing link tag */
                        esc_html__( 'Runware API key is required. Please %1$sconfigure it here%2$s.', 'srizon-ai-image' ),
                        '<a href="' . esc_url( $settings_url ) . '">',
                        '</a>'
                    );
                    ?>
                </p>
            </div>
            <?php
        }
    }

    /**
     * Get the Runware API key.
     *
     * @return string
     */
    public static function get_api_key() {
        return get_option( self::OPTION_API_KEY, '' );
    }

    /**
     * Check if API key is configured.
     *
     * @return bool
     */
    public static function is_api_key_configured() {
        return ! empty( self::get_api_key() );
    }
}
