<?php
/**
 * Uninstall SQMViews Plugin
 *
 * Removes all plugin data including:
 * - Database tables
 * - WordPress options
 * - Uploaded data files
 * - Drop-in endpoint file
 *
 * @package SQMViews
 * @since 1.0.0
 */

// If uninstall is not called from WordPress, exit.
if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
	exit;
}

// Load plugin dependencies.
if ( file_exists( __DIR__ . '/vendor/autoload.php' ) ) {
	require_once __DIR__ . '/vendor/autoload.php';
}

// Always load required files for uninstall.
require_once __DIR__ . '/src/includes/utils.php';
require_once __DIR__ . '/src/includes/activation.php';

/**
 * Delete all plugin data
 *
 * @since 1.0.0
 *
 * @return void
 */
function sqm_views_uninstall_cleanup(): void {
	try {
		// Call the main uninstall function.
		if ( function_exists( '\\SQMViews\\uninstall' ) ) {
			\SQMViews\uninstall();
		}

		// Remove data directories.
		sqm_views_remove_data_directories();

		/**
		 * Fires after all plugin data has been removed
		 *
		 * @since 1.0.0
		 */
		do_action( 'sqm_views_uninstalled' );
	} catch ( Exception $e ) {
		// Silently ignore errors to ensure uninstall completes.
		unset( $e );
	}
}

/**
 * Remove all data directories
 *
 * @since 1.0.0
 *
 * @return void
 */
function sqm_views_remove_data_directories(): void {
	$directories = array();

	// Build directory list from constants if they exist.
	if ( defined( 'SQMVIEWS_STALE_DIR' ) ) {
		$directories[] = SQMVIEWS_STALE_DIR;
	}
	if ( defined( 'SQMVIEWS_ARCHIVE_DIR' ) ) {
		$directories[] = SQMVIEWS_ARCHIVE_DIR;
	}
	if ( defined( 'SQMVIEWS_PROCESSED_DIR' ) ) {
		$directories[] = SQMVIEWS_PROCESSED_DIR;
	}
	if ( defined( 'SQMVIEWS_RAW_DIR' ) ) {
		$directories[] = SQMVIEWS_RAW_DIR;
	}
	if ( defined( 'SQMVIEWS_DIR' ) ) {
		$directories[] = SQMVIEWS_DIR;
	}

	foreach ( $directories as $dir ) {
		if ( file_exists( $dir ) && is_dir( $dir ) ) {
			sqm_views_recursive_remove_directory( $dir );
		}
	}
}

/**
 * Recursively remove a directory and its contents
 *
 * @since 1.0.0
 *
 * @param string $dir Directory path.
 * @return bool True on success, false on failure
 */
function sqm_views_recursive_remove_directory( $dir ) {
	if ( ! is_dir( $dir ) ) {
		return false;
	}

	// Initialize WP_Filesystem.
	global $wp_filesystem;
	if ( ! function_exists( 'WP_Filesystem' ) ) {
		require_once ABSPATH . 'wp-admin/includes/file.php';
	}

	// Initialize filesystem with direct method (no credentials needed for uninstall).
	if ( ! WP_Filesystem() ) {
		// Fallback to direct PHP functions if WP_Filesystem fails.
		$files = array_diff( scandir( $dir ), array( '.', '..' ) );

		foreach ( $files as $file ) {
			$path = $dir . '/' . $file;

			if ( is_dir( $path ) ) {
				sqm_views_recursive_remove_directory( $path );
			} else {
				wp_delete_file( $path );
			}
		}
		return true;
	}

	// Use WP_Filesystem to remove directory recursively.
	return $wp_filesystem->rmdir( $dir, true );
}

// Run the cleanup.
sqm_views_uninstall_cleanup();
