<?php
/**
 * SQM Views utility functions.
 *
 * DO NOT include any other files here, this file should be able to loaded
 * before any other files and before WordPress is loaded.
 *
 * @package SQMViews
 */

namespace SQMViews;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use SodiumException;

define( 'SQMVIEWS', true );

/**
 * Plugin build global version constant
 *
 * WARNING: Do NOT edit this version manually!
 * Use the bump-version.sh script to update versions across all files:
 *   ./bump-version.sh patch|minor|major
 *
 * This constant is used globally across all build artifacts (PHP, JS, TS).
 *
 * @since 1.0.0
 */
define( 'SQMVIEWS_BUILD_GLOBAL_VERSION', '1.1.9' );

define( 'SQMVIEWS_NAME', 'sqm-views' );

define( 'SQMVIEWS_FILE_ROTATION_INTERVAL', 60 );
define( 'SQMVIEWS_FILE_ROTATION_INTERVAL_DAILY', 60 * 60 * 24 );
define( 'SQMVIEWS_FILE_COOLDOWN_PERIOD', 60 );
define( 'SQMVIEWS_REQUEST_LENGTH_HARD_LIMIT', 1024 * 10 );

define( 'SQMVIEWS_SESSION_INACTIVITY_TIMEOUT', 30 * 60 /* 30 minutes */ );
define( 'SQMVIEWS_SESSION_PING_INTERVAL', 60  /* 60 seconds */ );
// phpcs:disable Squiz.PHP.CommentedOutCode.Found, Squiz.Commenting.InlineComment.InvalidEndChar
// define( 'SQMVIEWS_SESSION_INACTIVITY_TIMEOUT', 60  /* 60 seconds */ );
// define( 'SQMVIEWS_SESSION_PING_INTERVAL', 10  /* 10 seconds */ );

// define( 'SQMVIEWS_USE_USER_TIME', false ); // testing only
// phpcs:enable Squiz.PHP.CommentedOutCode.Found, Squiz.Commenting.InlineComment.InvalidEndChar
if ( ! defined( 'SQMVIEWS_USE_USER_TIME' ) ) {
	$sqm_views_env_value = getenv( 'SQMVIEWS_USE_USER_TIME' );
	define( 'SQMVIEWS_USE_USER_TIME', false !== $sqm_views_env_value ? filter_var( $sqm_views_env_value, FILTER_VALIDATE_BOOLEAN ) : false );
}

// We can not always use WordPress functions and constants here, because it might not be defined in case of a drop-in installation.
if ( ! defined( 'SQMVIEWS_WP_UPLOADS' ) ) {
	if ( function_exists( '\\wp_upload_dir' ) ) {
		$sqm_views_upload_dir = \wp_upload_dir();
		define( 'SQMVIEWS_WP_UPLOADS', $sqm_views_upload_dir['basedir'] );
	} elseif ( false !== getenv( 'SQMVIEWS_WP_UPLOADS' ) ) {
		// Allow override via environment variable (useful for Docker/CI environments).
		define( 'SQMVIEWS_WP_UPLOADS', getenv( 'SQMVIEWS_WP_UPLOADS' ) );
	}
	// 'SQMVIEWS_WP_UPLOADS' might still be empty if we bootstrapped from the drop-in script (sqm-views-pages.php).
}

define( 'SQMVIEWS_DIR', SQMVIEWS_WP_UPLOADS . '/sqm-views' );
/*-*/
define( 'SQMVIEWS_RAW_DIR', SQMVIEWS_DIR . '/raw' );
/*-*/
define( 'SQMVIEWS_PROCESSED_DIR', SQMVIEWS_DIR . '/processed' );
/*-*/
define( 'SQMVIEWS_ARCHIVE_DIR', SQMVIEWS_DIR . '/archive' );
/*-*/
define( 'SQMVIEWS_STALE_DIR', SQMVIEWS_DIR . '/stale' );


define( 'SQMVIEWS_OPTION_LAST_RUN', 'sqm_views_last_run' );
define( 'SQMVIEWS_OPTION_TRACKABLE_POST_TYPES', 'sqm_views_trackable_post_types' );
define( 'SQMVIEWS_OPTION_TRACKABLE_TAXONOMIES', 'sqm_views_trackable_taxonomies' );
define( 'SQMVIEWS_OPTION_DATA_TAXONOMIES', 'sqm_views_data_taxonomies' );
define( 'SQMVIEWS_OPTION_INLINE_JS', 'sqm_views_use_inline_js' );
define( 'SQMVIEWS_OPTION_MIN_JS', 'sqm_views_use_min_js' );
define( 'SQMVIEWS_OPTION_ENCRYPTION_KEY', 'sqm_views_encryption_key' );
define( 'SQMVIEWS_OPTION_CRON_INTERVAL', 'sqm_views_cron_interval' );
define( 'SQMVIEWS_OPTION_HANDLER', 'sqm_views_handler' );

define( 'SQMVIEWS_HANDLERS_FAST', '/sqm-views-pages.php' );
define( 'SQMVIEWS_HANDLERS_SLOW', '/wp-json/sqm-views/v1/track' );

define( 'SQMVIEWS_EVENTS_INIT', 'init' );
define( 'SQMVIEWS_EVENTS_PING', 'ping' );
define( 'SQMVIEWS_EVENTS_EXIT', 'exit' );
define( 'SQMVIEWS_EVENTS_TIMEOUT', 'timeout' );
define( 'SQMVIEWS_EVENTS_LOST', 'lost' );
define( 'SQMVIEWS_EVENTS_TEST', 'test' );

define( 'SQMVIEWS_TRACK_EVENTS_PAGEVIEW_ID', 1 );
define( 'SQMVIEWS_TRACK_EVENTS_PAGEVIEW_LABEL', 'pageview' );


/**
 * Encrypt data and encode as base64 payload.
 *
 * @param mixed  $data Data to encrypt.
 * @param string $key Encryption key.
 *
 * @return string|null Encrypted payload or null on failure.
 */
function sqm_views_data2payload( $data, $key ): ?string {
	$nonce     = random_bytes( SODIUM_CRYPTO_SECRETBOX_NONCEBYTES );
	$data_text = wp_json_encode( $data );
	try {
		$encrypted_result = sodium_crypto_secretbox( $data_text, $nonce, $key );
	} catch ( SodiumException $e ) {
		Logger::log_error(
			'Encryption failed in sqm_views_data2payload',
			array(
				'error' => $e->getMessage(),
				'code'  => $e->getCode(),
				'data'  => $data,
			)
		);

		return null;
	}
	// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode -- Used for legitimate encryption encoding.
	$payload = base64_encode( $nonce . $encrypted_result );

	return $payload;
}

/**
 * Decrypt base64-encoded payload back to data.
 *
 * @param string $payload Base64-encoded encrypted payload.
 * @param string $key Encryption key.
 *
 * @return array<string, mixed>|null Decrypted data or null on failure.
 */
function sqm_views_payload2data( $payload, $key ): ?array {
	if ( ! $payload || ! $key ) {
		return null;
	}

	// Type validation: payload must be a string.
	if ( ! is_string( $payload ) ) {
		Logger::log_error(
			'Invalid payload type in sqm_views_payload2data - expected string',
			array(
				'actual_type'  => gettype( $payload ),
				'payload_dump' => wp_json_encode( $payload ),
				'has_key'      => (bool) $key,
			)
		);

		return null;
	}

	// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode -- Used for legitimate decryption decoding.
	$raw_nonce_encrypted_text = base64_decode( $payload );
	$nonce                    = substr( $raw_nonce_encrypted_text, 0, SODIUM_CRYPTO_SECRETBOX_NONCEBYTES );
	$encrypted_result         = substr( $raw_nonce_encrypted_text, SODIUM_CRYPTO_SECRETBOX_NONCEBYTES );
	try {
		$decrypted_result = sodium_crypto_secretbox_open( $encrypted_result, $nonce, $key );
	} catch ( SodiumException $e ) {
		Logger::log_error(
			'Decryption failed in sqm_views_payload2data',
			array(
				'error'        => $e->getMessage(),
				'code'         => $e->getCode(),
				'payload_size' => strlen( $payload ),
				'has_key'      => (bool) $key,
			)
		);

		return null;
	}
	$data = json_decode( $decrypted_result, true );

	return $data;
}

/**
 * Prepare a directory with security files.
 *
 * @param string $dir Directory path to prepare.
 *
 * @return void
 */
function sqm_views_prepare_dir( $dir ): void {
	if ( ! file_exists( $dir ) ) {
		sqm_load_wp_core();

		// Use WordPress filesystem API.
		require_once ABSPATH . 'wp-admin/includes/file.php';
		WP_Filesystem();
		global $wp_filesystem;

		// Create directory using WordPress API.
		if ( ! $wp_filesystem->is_dir( $dir ) ) {
			$wp_filesystem->mkdir( $dir, FS_CHMOD_DIR );
		}

		// Create .htaccess file to prevent direct access.
		$htaccess_content = "Order Deny,Allow\nDeny from all";
		$htaccess_path    = trailingslashit( $dir ) . '.htaccess';
		$wp_filesystem->put_contents( $htaccess_path, $htaccess_content, FS_CHMOD_FILE );

		// Create empty index.html for additional protection.
		$index_path = trailingslashit( $dir ) . 'index.html';
		$wp_filesystem->put_contents( $index_path, '', FS_CHMOD_FILE );
	}
}

/**
 * Get the current stat filename based on time and rotation interval.
 *
 * @param int|null $time Timestamp (defaults to current time).
 * @param string   $suffix Filename suffix.
 * @param int      $shift Period shift.
 * @param int      $interval Rotation interval in seconds.
 *
 * @return string Filename.
 */
function sqm_views_get_current_stat_filename( $time = null, $suffix = '', $shift = 0, $interval = SQMVIEWS_FILE_ROTATION_INTERVAL ): string {
	if ( ! $time ) {
		$time = time();
	}

	return gmdate( 'Y-m-d\TH:i', intval( floor( ( $time + $shift * $interval ) / $interval ) * $interval ) ) . "$suffix.jsonl";
}

/**
 * Extract period timestamp from a stat filename.
 *
 * @param string $filename Stat filename.
 *
 * @return string Period timestamp.
 */
function sqm_views_file_get_period( $filename ) {
	$tstamp = explode( '.', basename( $filename, '.jsonl' ) )[0];

	return $tstamp;
}

/**
 * Generate a unique ID with a prefix.
 *
 * @param string $prefix ID prefix.
 *
 * @return string Unique ID.
 */
function generateUniqueId( string $prefix ): string { // phpcs:ignore WordPress.NamingConventions.ValidFunctionName.FunctionNameInvalid
	// Current time in milliseconds, converted to base 36.
	$time_hex = base_convert( (string) floor( microtime( true ) * 1000 ), 10, 36 );

	// Alphanumeric alphabet.
	$alphabet        = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_@';
	$alphabet_length = strlen( $alphabet );

	// Generate 10 random characters using a cryptographically secure RNG.
	$rand_part = '';
	for ( $i = 0; $i < 10; $i++ ) {
		$idx        = random_int( 0, $alphabet_length - 1 );
		$rand_part .= $alphabet[ $idx ];
	}

	return $prefix . $time_hex . '-' . $rand_part;
}

/**
 * Safely load WordPress core on demand.
 *
 * This function provides a safe way to bootstrap WordPress when needed,
 * typically for accessing WordPress functions like wp_delete_file().
 * If WordPress is already loaded, it does nothing.
 *
 * Performance note: Loading WordPress core adds significant overhead (~50-100ms).
 * Only call this function when WordPress functionality is absolutely required.
 *
 * @return void
 */
function sqm_load_wp_core(): void {
	// Check if WordPress is already loaded by testing for a core constant.
	if ( defined( 'ABSPATH' ) && function_exists( 'wp_delete_file' ) ) {
		// WordPress is already loaded, nothing to do.
		return;
	}

	// Walk up the directory tree to find WordPress root.
	$path = __DIR__;
	while ( ! file_exists( $path . '/wp-load.php' ) ) {
		$parent = dirname( $path );
		if ( $path === $parent || '/' === $parent ) {
			// Reached root without finding wp-load.php.
			Logger::log_error(
				'WordPress installation not found. Cannot load WordPress core.'
			);

			return;
		}
		$path = $parent;
	}

	// Load WordPress core.
	require_once $path . '/wp-load.php';
}
