<?php
/**
 * SQM Views tracker functions.
 *
 * @package SQMViews
 */

namespace SQMViews;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Add tracking code to the head of trackable pages
 *
 * @since 1.0.0
 */
function sqm_views_tracker(): void {
	// Get tracking information based on current page type.
	$tracking_data = sqm_views_get_tracking_data();

	// Allow plugins to prevent tracking on specific pages.
	if ( ! $tracking_data || ! apply_filters( 'sqm_views_should_track', true, $tracking_data ) ) {
		return;
	}

	// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode -- Used for legitimate decryption.
	$key = base64_decode( get_option( SQMVIEWS_OPTION_ENCRYPTION_KEY ) );
	// Allow filtering of encryption key (use with caution!).
	$key = apply_filters( 'sqm_views_encryption_key', $key );

	if ( ! $key ) {
		Logger::log_error(
			'Encryption key not set, tracker disabled',
			array(
				'option_name' => SQMVIEWS_OPTION_ENCRYPTION_KEY,
				'solution'    => 'Reinstall the plugin to regenerate the encryption key',
				'page_url'    => isset( $_SERVER['REQUEST_URI'] ) ? wp_unslash( sanitize_url( wp_unslash( $_SERVER['REQUEST_URI'] ) ) ) : '',
			)
		);
		return;
	}

	$tracker = sqm_views_data2payload( $tracking_data, $key );

	// Output meta tags and enqueue script.
	$endpoint       = 'no';
	$saved_endpoint = get_option( SQMVIEWS_OPTION_HANDLER, false );
	if ( SQMVIEWS_HANDLERS_FAST === $saved_endpoint ) {
		$endpoint = 'fast';
	}
	if ( SQMVIEWS_HANDLERS_SLOW === $saved_endpoint ) {
		$endpoint = 'api';
	}

	// Allow filtering of tracking endpoint.
	$endpoint = apply_filters( 'sqm_views_tracker_endpoint', $endpoint, $saved_endpoint );

	// Allow filtering of JS settings.
	$use_min_js    = apply_filters( 'sqm_views_minified_js', get_option( SQMVIEWS_OPTION_MIN_JS, true ) );
	$min_js_suffix = $use_min_js ? '.min' : '';

	// Build tracker configuration.
	$tracker_config = array(
		'endpoint'     => $endpoint,
		'use_minified' => $use_min_js,
		'use_inline'   => get_option( SQMVIEWS_OPTION_INLINE_JS, false ),
	);

	/**
	 * Fires before tracker output
	 *
	 * @since 1.0.0
	 *
	 * @param array $tracking_data  The tracking data array
	 * @param array $tracker_config The tracker configuration
	 */
	do_action( 'sqm_views_before_tracker_output', $tracking_data, $tracker_config );

	// Add meta tags with content type and ID for the tracker to use.
	echo "\n" . '<meta name="sqm-views-tracker" tracker="' . esc_attr( $endpoint ) . '" content="' . esc_attr( $tracker ) . '">';

	// Allow developers to see tracking data in HTML comments (can be disabled for production).
	if ( apply_filters( 'sqm_views_show_debug_comment', WP_DEBUG ) ) {
		echo "\n" . '<!--' . wp_json_encode( $tracking_data ) . '-->' . "\n";
	}

	// Build script path and URL.
	$script_path = plugin_dir_path( SQMVIEWS_PLUGIN_FILE ) . 'artifacts/sqm-views-pages-' . SQMVIEWS_BUILD_GLOBAL_VERSION . "$min_js_suffix.js";
	$script_path = apply_filters( 'sqm_views_tracker_script_path', $script_path, $use_min_js );
	$script_url  = plugins_url( 'artifacts/sqm-views-pages-' . SQMVIEWS_BUILD_GLOBAL_VERSION . "$min_js_suffix.js", SQMVIEWS_PLUGIN_FILE );
	$script_url  = apply_filters( 'sqm_views_tracker_script_url', $script_url, $use_min_js );

	if ( file_exists( $script_path ) ) {
		$use_inline_js = apply_filters( 'sqm_views_inline_js', $tracker_config['use_inline'] );

		if ( $use_inline_js ) {
			// Inline the JavaScript using wp_add_inline_script.
			// First enqueue a dummy handle, then add inline script.
			wp_register_script( 'sqm-views-tracker', '', array(), SQMVIEWS_BUILD_GLOBAL_VERSION, false );
			wp_enqueue_script( 'sqm-views-tracker' );

			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents -- Reading plugin js file to inline.
			$content = file_get_contents( $script_path );
			wp_add_inline_script( 'sqm-views-tracker', $content );
		} else {
			// Enqueue external script with async loading.
			wp_enqueue_script(
				'sqm-views-tracker',
				$script_url,
				array(),
				SQMVIEWS_BUILD_GLOBAL_VERSION,
				array(
					'in_footer' => false,
					'strategy'  => 'async',
				)
			);
		}
	}

	/**
	 * Fires after tracker output
	 *
	 * @since 1.0.0
	 *
	 * @param array $tracking_data The tracking data array
	 */
	do_action( 'sqm_views_after_tracker_output', $tracking_data );
}

/**
 * Determine what content is being viewed and return tracking data
 *
 * @since 1.0.0
 *
 * @return array<string, mixed> Tracking data with 'type' and 'id', or empty array if not trackable
 */
function sqm_views_get_tracking_data(): array {
	$tracking_data = array();

	// Get the list of post types we want to track.
	$trackable_post_types = get_option( SQMVIEWS_OPTION_TRACKABLE_POST_TYPES, array( 'post', 'page' ) );
	$trackable_post_types = apply_filters( 'sqm_views_trackable_post_types', $trackable_post_types );

	$trackable_taxonomies = get_option( SQMVIEWS_OPTION_TRACKABLE_TAXONOMIES, array( 'category', 'post_tag' ) );
	$trackable_taxonomies = apply_filters( 'sqm_views_trackable_taxonomies', $trackable_taxonomies );

	$data_taxonomies = get_option( SQMVIEWS_OPTION_DATA_TAXONOMIES, array( 'category', 'post_tag' ) );
	$data_taxonomies = apply_filters( 'sqm_views_data_taxonomies', $data_taxonomies );

	// Check if we're on a singular post/page.
	if ( count( $trackable_post_types ) && is_singular( $trackable_post_types ) ) {
		$post  = get_post();
		$terms = count( $data_taxonomies ) ? wp_get_post_terms( $post->ID, $data_taxonomies, array( 'fields' => 'ids' ) ) : array();
		if ( $post ) {
			$tracking_data += array(
				'event' => SQMVIEWS_TRACK_EVENTS_PAGEVIEW_ID,
				'group' => 'content',
				'type'  => $post->post_type,
				'id'    => $post->ID,
				'terms' => $terms,
			);
		}
	} elseif ( count( $trackable_taxonomies ) && ( is_category() || is_tag() || is_tax() ) ) {
		// Check for category archive.
		$term = get_queried_object();
		if ( in_array( $term->taxonomy, $trackable_taxonomies, true ) ) {
			$tracking_data += array(
				'event' => SQMVIEWS_TRACK_EVENTS_PAGEVIEW_ID,
				'group' => 'taxonomy',
				'type'  => $term->taxonomy,
				'id'    => $term->term_id,
			);
		}
	} elseif ( is_author() ) {
		// Check for author archive.
		$author         = get_queried_object();
		$tracking_data += array(
			'event' => SQMVIEWS_TRACK_EVENTS_PAGEVIEW_ID,
			'group' => 'browse',
			'type'  => 'author',
			'id'    => $author->ID,
		);
	} elseif ( is_date() ) {
		// Check for date archives.
		if ( is_year() ) {
			$tracking_data += array(
				'event' => SQMVIEWS_TRACK_EVENTS_PAGEVIEW_ID,
				'group' => 'browse',
				'type'  => 'archive:year',
				'id'    => 0,
				'altid' => get_the_date( 'Y' ),
			);
		} elseif ( is_month() ) {
			$tracking_data += array(
				'event' => SQMVIEWS_TRACK_EVENTS_PAGEVIEW_ID,
				'group' => 'browse',
				'type'  => 'archive:month',
				'id'    => 0,
				'altid' => get_the_date( 'Y-m' ),
			);
		} elseif ( is_day() ) {
			$tracking_data += array(
				'event' => SQMVIEWS_TRACK_EVENTS_PAGEVIEW_ID,
				'group' => 'browse',
				'type'  => 'archive:day',
				'id'    => 0,
				'altid' => get_the_date( 'Y-m-d' ),
			);
		}
	} elseif ( is_home() || is_front_page() ) {
		// Check for home page.
		$tracking_data += array(
			'event' => SQMVIEWS_TRACK_EVENTS_PAGEVIEW_ID,
			'group' => 'special',
			'type'  => 'page',
			'id'    => 0,
			'altid' => 'home',
		);
	} elseif ( is_search() ) {
		// Check for search results.
		$tracking_data += array(
			'event' => SQMVIEWS_TRACK_EVENTS_PAGEVIEW_ID,
			'group' => 'special',
			'type'  => 'search',
			'id'    => 0,
			// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.urlencode_urlencode -- Required for URL encoding search query.
			'altid' => urlencode( get_search_query() ),
		);
	}

	/**
	 * Filters the tracking data before encryption
	 *
	 * Allows developers to add custom metadata or modify tracking behavior
	 *
	 * @since 1.0.0
	 *
	 * @param array $tracking_data The tracking data array
	 */
	return apply_filters( 'sqm_views_tracking_data', $tracking_data );
}
