<?php
/**
 * REST API endpoints for the plugin.
 *
 * @package SQMViews
 */

namespace SQMViews;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

require_once __DIR__ . '/charts-api.php';

/**
 * Register REST API routes
 *
 * @since 1.0.0
 *
 * @return void
 */
function register_rest_routes() {
	/**
	 * Filters the required capability for accessing REST API charts endpoint
	 *
	 * @since 1.0.0
	 *
	 * @param string $capability Required capability (default: 'manage_options')
	 */
	$charts_capability = apply_filters( 'sqm_views_rest_permissions', 'manage_options' );

	/**
	 * Filters the chart query parameters allowed in REST API
	 *
	 * @since 1.0.0
	 *
	 * @param array $filters Array of filter definitions
	 */
	$chart_filters = apply_filters( 'sqm_views_rest_chart_filters', SQMVIEWS_CHART_FILTERS );

	register_rest_route(
		'sqm-views/v1',
		'/track',
		array(
			'methods'             => 'POST',
			'callback'            => '\SQMViews\track_pageview',
			'permission_callback' => '__return_true', // Public pageview endpoint accessible for all visitors.
		)
	);

	register_rest_route(
		'sqm-views/v1',
		'/charts',
		array(
			'methods'             => 'GET',
			'callback'            => '\\SQMViews\\get_charts_data',
			'permission_callback' => function () use ( $charts_capability ) {
				return current_user_can( $charts_capability );
			},
			'args'                => $chart_filters,
		)
	);
}
add_action( 'rest_api_init', '\SQMViews\register_rest_routes' );

/**
 * Process a pageview tracking request
 *
 * @param mixed $request Full data about the request.
 * @return \WP_REST_Response|\WP_Error Response object.
 *
 * phpcs:ignore Squiz.Commenting.FunctionComment.IncorrectTypeHint -- Generic syntax for PHPStan
 */
function track_pageview( mixed $request ) {
	$processor = new SQMViewsRapidProcessor();

	// Get the request body from WordPress REST API request.
	$request_body = $request->get_body();

	// Get the response from processor.
	$response = $processor->process( $request_body );

	// Convert ProcessorResponse to WordPress REST API response.
	if ( $response->is_error() ) {
		return new \WP_Error(
			$response->get_error_code(),
			$response->get_error_message(),
			array( 'status' => $response->get_status_code() )
		);
	}

	return new \WP_REST_Response( $response->get_data(), $response->get_status_code() );
}
