<?php
/**
 * Charts API - Security Hardened Implementation
 *
 * This file implements the REST API endpoint for chart data with comprehensive
 * security measures against SQL injection, XSS, and other vulnerabilities.
 *
 * SECURITY REVIEW NOTES:
 * ======================
 *
 * 1. SQL INJECTION PREVENTION:
 *    - Column names validated through strict whitelist (METRIC_COLUMN_MAP)
 *    - All user inputs passed through $wpdb->prepare() with proper placeholders
 *    - Integer parameters cast with intval()
 *    - Date parameters validated with regex before use
 *    - No dynamic SQL construction from user input
 *
 * 2. INPUT VALIDATION:
 *    - startDate/endDate: Regex validated (YYYY-MM-DD format only)
 *    - metric: Whitelist validated against predefined columns
 *    - granularity: Sanitized with sanitize_key()
 *    - eid/tid: Integer cast with intval()
 *    - limit: Integer cast with intval()
 *
 * 3. OUTPUT SANITIZATION:
 *    - All database values cast to appropriate types (float, string)
 *    - Text passed through WordPress i18n functions
 *    - No raw user input echoed back to client
 *
 * @package SQMViews
 * @since 1.0.0
 */

namespace SQMViews;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

define(
	'SQMVIEWS_CHART_FILTERS',
	array(
		'startDate'   => array(
			'required' => false,
			'type'     => 'string',
		),
		'endDate'     => array(
			'required' => false,
			'type'     => 'string',
		),
		'eid'         => array(
			'required' => false,
			'type'     => 'integer',
		),
		'tid'         => array(
			'required' => false,
			'type'     => 'integer',
		),
		'metric'      => array(
			'required' => false,
			'type'     => 'string',
			'default'  => 'count',
		),
		'granularity' => array(
			'required' => false,
			'type'     => 'string',
			'default'  => 'daily',
		),
		'limit'       => array(
			'required' => false,
			'type'     => 'integer',
			'default'  => 10,
		),
	)
);

/**
 * Metric column whitelist mapping
 *
 * SECURITY: This whitelist prevents SQL injection by ensuring only valid
 * column names are used in queries. Any metric not in this list defaults to 'count'.
 *
 * @since 1.0.5
 */
define(
	'SQMVIEWS_METRIC_COLUMN_MAP',
	array(
		'count'     => 'count',
		'on_page'   => 'on_page',
		'active'    => 'active',
		'high_freq' => 'high_freq',
		'low_freq'  => 'low_freq',
	)
);

/**
 * Validate and sanitize metric parameter
 *
 * SECURITY: Uses strict whitelist validation. Any invalid metric defaults to 'count'.
 * This prevents SQL injection through column name manipulation.
 *
 * @since 1.0.5
 *
 * @param string $metric User-provided metric name.
 * @return string Validated column name (guaranteed to be in whitelist).
 */
function sqm_views_validate_metric( $metric ) {
	// SECURITY: sanitize_key() removes any special characters.
	$metric = sanitize_key( $metric );

	// SECURITY: Whitelist validation - only predefined columns allowed.
	$metric_map = SQMVIEWS_METRIC_COLUMN_MAP;
	if ( ! isset( $metric_map[ $metric ] ) ) {
		$metric = 'count'; // Safe default.
	}

	return $metric_map[ $metric ];
}

/**
 * Charts datasource endpoint
 *
 * GET /wp-json/sqm-views/v1/charts?startDate=YYYY-MM-DD&endDate=YYYY-MM-DD[&eid=...][&tid=...][&metric=count][&granularity=daily|by_content]
 *
 * SECURITY MEASURES APPLIED:
 * ==========================
 * 1. All parameters validated before use
 * 2. SQL injection prevented through $wpdb->prepare() with proper placeholders
 * 3. Column names validated through strict whitelist
 * 4. Integer parameters cast with intval()
 * 5. String parameters sanitized with WordPress functions
 * 6. Output values type-cast to prevent XSS
 * 7. Cache keys sanitized to prevent cache poisoning
 *
 * Returns:
 * {
 *   series: [{ name: string, values: [{ x: string, y: number }] }],
 *   xLabel: "Date",
 *   yLabel: "Views"
 * }
 *
 * @since 1.0.0
 *
 * @param mixed $request Request object.
 * @return \WP_REST_Response Response object with chart data.
 *
 * phpcs:ignore Squiz.Commenting.FunctionComment.IncorrectTypeHint -- Generic syntax for PHPStan
 */
function get_charts_data( mixed $request ) {
	global $wpdb;

	// =========================================================================
	// STEP 1: VALIDATE AND SANITIZE ALL INPUT PARAMETERS
	// =========================================================================

	// SECURITY: sanitize_text_field() prevents XSS and removes unwanted characters.
	$start = sanitize_text_field( $request->get_param( 'startDate' ) );
	$end   = sanitize_text_field( $request->get_param( 'endDate' ) );

	// SECURITY: Strict regex validation - only YYYY-MM-DD format allowed.
	// This prevents SQL injection through date manipulation.
	if ( ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $start ) || ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $end ) ) {
		return new \WP_REST_Response(
			array( 'error' => __( 'Invalid date format. Use YYYY-MM-DD', 'sqm-views' ) ),
			400
		);
	}

	// SECURITY: Integer parameters - intval() ensures only integers.
	// NULL values preserved for optional parameters.
	$eid = $request->get_param( 'eid' );
	$tid = $request->get_param( 'tid' );
	if ( null !== $eid && '' !== $eid ) {
		$eid = intval( $eid );
	}
	if ( null !== $tid && '' !== $tid ) {
		$tid = intval( $tid );
	}

	// SECURITY: Metric validated through whitelist function.
	$metric_param = $request->get_param( 'metric' );
	$metric_param = $metric_param ?? 'count';
	$col          = sqm_views_validate_metric( $metric_param );
	$metric       = array_search( $col, SQMVIEWS_METRIC_COLUMN_MAP, true ); // For display purposes.

	// SECURITY: sanitize_key() allows only alphanumeric, dashes, underscores.
	$granularity_param = $request->get_param( 'granularity' );
	$granularity       = sanitize_key( $granularity_param ?? 'daily' );

	// SECURITY: Integer cast with intval(), bounded to reasonable range.
	$limit_param = $request->get_param( 'limit' );
	$limit       = intval( $limit_param ? $limit_param : 25 );
	$limit       = max( 1, min( 100, $limit ) ); // Limit between 1 and 100.

	// =========================================================================
	// STEP 2: CHECK CACHE (using sanitized parameters)
	// =========================================================================

	// Build cache key based on all query parameters.
	// SECURITY: All parameters already sanitized above.
	$cache_key = sprintf(
		'sqm_views_chart_%s_%s_%s_%s_%s_%s_%d',
		$granularity,
		$metric,
		$start,
		$end,
		null !== $eid ? $eid : 'all',
		null !== $tid ? $tid : 'all',
		$limit
	);

	// Try to get cached data.
	$cached_data = wp_cache_get( $cache_key, 'sqm_views_charts' );
	if ( false !== $cached_data ) {
		return new \WP_REST_Response( $cached_data, 200 );
	}

	// =========================================================================
	// STEP 3: BUILD AND EXECUTE QUERY
	// =========================================================================

	// Prepare query parameters with descriptive names.
	$column_name = $col;
	$table_name  = "{$wpdb->prefix}sqm_views_daily";
	$start_date  = $start;
	$end_date    = $end;

	// Event ID filtering: when eid is provided, filter by it; otherwise allow all.
	if ( null !== $eid && '' !== $eid ) {
		$eid_value     = intval( $eid );
		$eid_condition = 0; // Forces the eid = $eid_value condition to be evaluated.
	} else {
		$eid_value     = 0;
		$eid_condition = 1; // Makes the condition always true (0 OR 1).
	}

	// Trackable ID filtering: when tid is provided, filter by it; otherwise allow all.
	if ( null !== $tid && '' !== $tid ) {
		$tid_value     = intval( $tid );
		$tid_condition = 0; // Forces the tid = $tid_value condition to be evaluated.
	} else {
		$tid_value     = 0;
		$tid_condition = 1; // Makes the condition always true (0 OR 1).
	}

	if ( 'by_content' === $granularity ) {
		// Return per-tid time series for the chosen metric.

		$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- caching applied above using $cache_key
			$wpdb->prepare(
				/* @lang MySQL */                'SELECT `tid`, `date`, SUM( %i ) AS v
				FROM %i
				WHERE
				    (`date` BETWEEN %s AND %s)
				AND (`eid` = %d OR %d)
				AND (`tid` = %d OR %d)

				GROUP BY `tid`, `date`
				ORDER BY `tid` ASC, `date` ASC',
				$column_name,
				$table_name,
				$start_date,
				$end_date,
				$eid_value,
				$eid_condition,
				$tid_value,
				$tid_condition
			),
			ARRAY_A
		);

		// Group rows by tid into series.
		$series_map = array();
		foreach ( $rows as $r ) {
			// SECURITY: Cast to string to prevent type juggling attacks.
			$key = strval( $r['tid'] );
			if ( ! isset( $series_map[ $key ] ) ) {
				$series_map[ $key ] = array();
			}
			// SECURITY: Cast values to proper types for JSON output.
			$series_map[ $key ][] = array(
				'x' => $r['date'],        // String (date from database).
				'y' => (float) $r['v'],   // Float (numeric value).
			);
		}

		// Optionally limit to top N tids by total metric.
		$totals = array();
		foreach ( $series_map as $k => $vals ) {
			$totals[ $k ] = array_reduce(
				$vals,
				function ( $acc, $p ) {
					return $acc + (float) $p['y'];
				},
				0.0
			);
		}
		arsort( $totals );
		$top_keys = array_slice( array_keys( $totals ), 0, $limit );

		$series = array();
		foreach ( $top_keys as $k ) {
			$series[] = array(
				'name'   => 'tid ' . $k,  // SECURITY: Integer key converted to string.
				'values' => $series_map[ $k ],
			);
		}

		$response_data = array(
			'series' => $series,
			'xLabel' => __( 'Date', 'sqm-views' ),
			'yLabel' => ucfirst( str_replace( '_', ' ', $metric ) ),
		);
	} else {
		// Daily totals (default granularity).

		// See $cache_key, we can safely ignore WordPress.DB.DirectDatabaseQuery.DirectQuery here.
		$rows = $wpdb->get_results( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
			$wpdb->prepare(
				/* @lang MySQL */                'SELECT `date`, SUM( %i ) AS v
				FROM %i
				WHERE
				    (`date` BETWEEN %s AND %s)
				AND (`eid` = %d OR %d)
				AND (`tid` = %d OR %d)

				GROUP BY `date`
				ORDER BY `date` ASC',
				$column_name,
				$table_name,
				$start_date,
				$end_date,
				$eid_value,
				$eid_condition,
				$tid_value,
				$tid_condition
			),
			ARRAY_A
		);

		// Transform results.
		$values = array_map(
			function ( $r ) {
				// SECURITY: Cast values to proper types for JSON output.
				return array(
					'x' => $r['date'],        // String (date from database).
					'y' => (float) $r['v'],   // Float (numeric value).
				);
			},
			$rows
		);

		$response_data = array(
			'series' => array(
				array(
					'name'   => __( 'Total', 'sqm-views' ),
					'values' => $values,
				),
			),
			'xLabel' => __( 'Date', 'sqm-views' ),
			'yLabel' => ucfirst( str_replace( '_', ' ', $metric ) ),
		);
	}

	// =========================================================================
	// STEP 4: APPLY FILTERS AND CACHE RESULT
	// =========================================================================

	/**
	 * Filters chart data before returning in REST API response
	 *
	 * @since 1.0.0
	 *
	 * @param array            $response_data Chart data array with series, xLabel, yLabel
	 * @param \WP_REST_Request $request       Original request object
	 */
	$response_data = apply_filters( 'sqm_views_rest_chart_data', $response_data, $request );

	// Cache the result for 5 minutes (300 seconds).
	// Allow filtering of cache expiration time.
	$cache_expiration = apply_filters( 'sqm_views_chart_cache_expiration', 300 );
	wp_cache_set( $cache_key, $response_data, 'sqm_views_charts', $cache_expiration );

	return new \WP_REST_Response( $response_data, 200 );
}

/**
 * Invalidates all chart data caches
 *
 * Should be called after statistics processing completes to ensure fresh data
 *
 * @since 1.0.4
 *
 * @return void
 */
function sqm_views_invalidate_chart_cache() {
	// WordPress doesn't support wildcard cache deletion by default,
	// so we use wp_cache_flush_group if available (requires persistent cache like Redis/Memcached).
	// Otherwise, individual cache keys are cleared on next request when TTL expires.

	if ( function_exists( 'wp_cache_flush_group' ) ) {
		wp_cache_flush_group( 'sqm_views_charts' );
	}

	/**
	 * Fires after chart cache is invalidated
	 *
	 * Allows custom cache implementations to clear their caches
	 *
	 * @since 1.0.4
	 */
	do_action( 'sqm_views_chart_cache_invalidated' );
}
