<?php
/**
 * Plugin activation and deactivation handlers.
 *
 * @package SQMViews
 */

namespace SQMViews;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

define( 'SQMVIEWS_REST_TIMEOUT', 5 );

/**
 * Handles plugin activation tasks
 *
 * @return void
 */
function activate(): void {
	// Check PHP version.
	if ( version_compare( PHP_VERSION, '7.4', '<' ) ) {
		wp_die(
		/* translators: %s: current PHP version */
			sprintf( esc_html__( 'SQMViews requires PHP 7.4 or higher. You are running PHP %s.', 'sqm-views' ), PHP_VERSION ),
			esc_html__( 'Plugin Activation Error', 'sqm-views' ),
			array( 'back_link' => true )
		);
	}

	// Check WordPress version.
	if ( version_compare( get_bloginfo( 'version' ), '5.0', '<' ) ) {
		wp_die(
			esc_html__( 'SQMViews requires WordPress 6.0 or higher.', 'sqm-views' ),
			esc_html__( 'Plugin Activation Error', 'sqm-views' ),
			array( 'back_link' => true )
		);
	}

	// Check required extensions.
	if ( ! extension_loaded( 'sodium' ) ) {
		wp_die(
			esc_html__( 'SQMViews requires the Sodium PHP extension for encryption.', 'sqm-views' ),
			esc_html__( 'Missing PHP Extension', 'sqm-views' ),
			array( 'back_link' => true )
		);
	}

	// Create database tables.
	create_tables();

	// Check endpoint status and log result.
	$endpoint_status = sqm_views_check_endpoint();
	Logger::log_debug(
		'Plugin activated - endpoint check result',
		array(
			'endpoint_status' => $endpoint_status,
		)
	);

	// Set default options with filter support.
	$default_settings = apply_filters(
		'sqm_views_default_settings',
		array(
			SQMVIEWS_OPTION_LAST_RUN             => 0,
			SQMVIEWS_OPTION_TRACKABLE_POST_TYPES => array( 'post', 'page' ),
			SQMVIEWS_OPTION_TRACKABLE_TAXONOMIES => array( 'category', 'post_tag' ),
			SQMVIEWS_OPTION_DATA_TAXONOMIES      => array( 'category', 'post_tag' ),
			SQMVIEWS_OPTION_INLINE_JS            => 1,
			SQMVIEWS_OPTION_MIN_JS               => 1,
			// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode -- Encoding encryption key for storage.
			SQMVIEWS_OPTION_ENCRYPTION_KEY       => base64_encode( sodium_crypto_secretbox_keygen() ),
		)
	);

	foreach ( $default_settings as $option_name => $default_value ) {
		add_option( $option_name, $default_value );
	}

	// Allow filtering of data directory.
	$data_dir = apply_filters( 'sqm_views_data_directory', SQMVIEWS_DIR );

	sqm_views_prepare_dir( $data_dir );
	sqm_views_prepare_dir( SQMVIEWS_RAW_DIR );
	sqm_views_prepare_dir( SQMVIEWS_PROCESSED_DIR );
	sqm_views_prepare_dir( SQMVIEWS_ARCHIVE_DIR );
	sqm_views_prepare_dir( SQMVIEWS_STALE_DIR );

	/**
	 * Fires after plugin activation is complete
	 *
	 * @since 1.0.0
	 */
	do_action( 'sqm_views_activated' );
}

/**
 * Creates required database tables
 *
 * @return void
 */
function create_tables() {
	global $wpdb;
	require_once ABSPATH . 'wp-admin/includes/upgrade.php';
	$prefix = $wpdb->prefix;

	$pageview_event_id = SQMVIEWS_TRACK_EVENTS_PAGEVIEW_ID;

	$create_tables_results = array();

	$create_tables_results[] = dbDelta(
		$wpdb->prepare(
			// phpcs:ignore Generic.Commenting.DocComment.MissingShort -- IDE language hint.
			/** @lang MySQL */            '
    CREATE TABLE IF NOT EXISTS %i (
	`tid` BIGINT UNSIGNED NOT NULL AUTO_INCREMENT UNIQUE,
	`tgroup` VARCHAR(20) NOT NULL,
	`ttype` VARCHAR(20) NOT NULL,
	`wpid` BIGINT UNSIGNED NOT NULL DEFAULT 0,
	`altid` VARCHAR(50),
	PRIMARY KEY(`tid`));',
			"{$prefix}sqm_views_trackables"
		)
	);

	$create_tables_results[] = dbDelta(
		$wpdb->prepare(
			// phpcs:ignore Generic.Commenting.DocComment.MissingShort -- IDE language hint.
			/** @lang MySQL */            '
    CREATE TABLE IF NOT EXISTS %i (
	`gid` VARCHAR(50) NOT NULL,
	`tid` BIGINT UNSIGNED NOT NULL,
	`eid` BIGINT UNSIGNED NOT NULL,
	`on_page` DECIMAL(18,2) NOT NULL,
	`active` DECIMAL(18,2) NOT NULL,
	`high_freq` BIGINT UNSIGNED NOT NULL,
	`low_freq` BIGINT UNSIGNED NOT NULL,
	`count` BIGINT UNSIGNED NOT NULL,
	`user_moment` DATETIME NOT NULL,
	`user_timezone` VARCHAR(50) NOT NULL,
	`exit` VARCHAR(20) NOT NULL,
	`utc_moment` DATETIME NOT NULL,
	PRIMARY KEY(`gid`)
);',
			"{$prefix}sqm_views_records"
		)
	);
	$create_tables_results[] = dbDelta(
		$wpdb->prepare(
			// phpcs:ignore Generic.Commenting.DocComment.MissingShort -- IDE language hint.
			/** @lang MySQL */            '
    CREATE TABLE IF NOT EXISTS %i (
	`date` DATE NOT NULL,
	`eid` BIGINT UNSIGNED NOT NULL,
	`tid` BIGINT UNSIGNED NOT NULL,
	`on_page` DECIMAL(18,2) NOT NULL,
	`active` DECIMAL(18,2) NOT NULL,
	`high_freq` BIGINT UNSIGNED NOT NULL,
	`low_freq` BIGINT UNSIGNED NOT NULL,
	`count` BIGINT UNSIGNED NOT NULL,
	PRIMARY KEY(`date`, `eid`, `tid`)
);',
			"{$prefix}sqm_views_daily"
		)
	);

	$create_tables_results[] = dbDelta(
		$wpdb->prepare(
			// phpcs:ignore Generic.Commenting.DocComment.MissingShort -- IDE language hint.
			/** @lang MySQL */            '
    CREATE TABLE IF NOT EXISTS %i (
	`eid` BIGINT UNSIGNED NOT NULL AUTO_INCREMENT UNIQUE,
	`name` VARCHAR(20) NOT NULL UNIQUE,
	PRIMARY KEY(`eid`)
);',
			"{$prefix}sqm_views_events"
		)
	);

	foreach ( $create_tables_results as $res ) {
		if ( count( $res ) ) {
			Logger::log_debug(
				'Database table creation/update results',
				array(
					'results' => $res,
				)
			);
		}
	}

	// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange -- Schema operations during plugin activation.
	$index_results = array();

	// Suppress database errors during index/constraint creation to allow safe re-activation.
	// This prevents "Duplicate key name" and "Duplicate foreign key constraint" errors from
	// causing activation failure when indexes/constraints already exist.
	$suppress_errors = $wpdb->suppress_errors( true );

	// Index creation - errors are suppressed to allow safe re-activation.
	// If indexes already exist, the queries will fail silently (expected behavior).
	$index_results[] = $wpdb->query(
		$wpdb->prepare(
			// phpcs:ignore Generic.Commenting.DocComment.MissingShort -- IDE language hint.
			/** @lang MySQL */
			'CREATE INDEX `sqm_views_trackables_index_0` ON %i (`tgroup`);',
			"{$prefix}sqm_views_trackables"
		)
	);

	$index_results[] = $wpdb->query(
		$wpdb->prepare(
			// phpcs:ignore Generic.Commenting.DocComment.MissingShort -- IDE language hint.
			/** @lang MySQL */
			'CREATE INDEX `sqm_views_trackables_index_1` ON %i (`tgroup`, `ttype`, `wpid`, `altid`);',
			"{$prefix}sqm_views_trackables"
		)
	);

	$index_results[] = $wpdb->query(
		$wpdb->prepare(
			// phpcs:ignore Generic.Commenting.DocComment.MissingShort -- IDE language hint.
			/** @lang MySQL */
			'CREATE INDEX `sqm_views_records_index_0` ON %i (`tid`);',
			"{$prefix}sqm_views_records"
		)
	);

	$index_results[] = $wpdb->query(
		$wpdb->prepare(
			// phpcs:ignore Generic.Commenting.DocComment.MissingShort -- IDE language hint.
			/** @lang MySQL */
			'CREATE INDEX `sqm_views_records_index_1` ON %i (`gid`);',
			"{$prefix}sqm_views_records"
		)
	);

	$index_results[] = $wpdb->query(
		$wpdb->prepare(
			// phpcs:ignore Generic.Commenting.DocComment.MissingShort -- IDE language hint.
			/** @lang MySQL */
			'CREATE INDEX `sqm_views_daily_index_0` ON %i (`date`, `eid`, `tid`);',
			"{$prefix}sqm_views_daily"
		)
	);

	$index_results[] = $wpdb->query(
		$wpdb->prepare(
			// phpcs:ignore Generic.Commenting.DocComment.MissingShort -- IDE language hint.
			/** @lang MySQL */
			'ALTER TABLE %i ADD FOREIGN KEY(`tid`) REFERENCES %i(`tid`) ON UPDATE NO ACTION ON DELETE NO ACTION;',
			"{$prefix}sqm_views_records",
			"{$prefix}sqm_views_trackables"
		)
	);

	$index_results[] = $wpdb->query(
		$wpdb->prepare(
			// phpcs:ignore Generic.Commenting.DocComment.MissingShort -- IDE language hint.
			/** @lang MySQL */
			'ALTER TABLE %i ADD FOREIGN KEY(`eid`) REFERENCES %i(`eid`) ON UPDATE NO ACTION ON DELETE NO ACTION;',
			"{$prefix}sqm_views_records",
			"{$prefix}sqm_views_events"
		)
	);

	$index_results[] = $wpdb->query(
		$wpdb->prepare(
			// phpcs:ignore Generic.Commenting.DocComment.MissingShort -- IDE language hint.
			/** @lang MySQL */
			'ALTER TABLE %i ADD FOREIGN KEY(`eid`) REFERENCES %i(`eid`) ON UPDATE NO ACTION ON DELETE NO ACTION;',
			"{$prefix}sqm_views_daily",
			"{$prefix}sqm_views_events"
		)
	);

	$index_results[] = $wpdb->query(
		$wpdb->prepare(
			// phpcs:ignore Generic.Commenting.DocComment.MissingShort -- IDE language hint.
			/** @lang MySQL */
			'ALTER TABLE %i ADD FOREIGN KEY(`tid`) REFERENCES %i(`tid`) ON UPDATE NO ACTION ON DELETE NO ACTION;',
			"{$prefix}sqm_views_daily",
			"{$prefix}sqm_views_trackables"
		)
	);

	$index_results[] = $wpdb->query(
		$wpdb->prepare(
			// phpcs:ignore Generic.Commenting.DocComment.MissingShort -- IDE language hint.
			/** @lang MySQL */
			'INSERT IGNORE INTO %i (`eid`, `name`) VALUES (%d, %s);',
			"{$prefix}sqm_views_events",
			$pageview_event_id,
			'pageview'
		)
	);

	// Restore error suppression to previous state.
	$wpdb->suppress_errors( $suppress_errors );
	// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange

	foreach ( $index_results as $idx => $result ) {
		if ( false === $result ) {
			// Log silently - these errors are expected on re-activation when indexes/constraints exist.
			Logger::log_debug(
				'Index/constraint creation returned false (may already exist)',
				array(
					'index' => $idx,
				)
			);
		}
	}

	add_option( SQMVIEWS_NAME, SQMVIEWS_BUILD_GLOBAL_VERSION );
}

/**
 * Checks if plugin version has changed and runs upgrade routines if needed
 *
 * Should be called on every admin page load to handle upgrades
 *
 * @return void
 */
function sqm_views_check_version() {
	$installed_version = get_option( SQMVIEWS_NAME, '0.0.0' );
	$current_version   = SQMVIEWS_BUILD_GLOBAL_VERSION;

	// Version hasn't changed, no upgrade needed.
	if ( version_compare( $installed_version, $current_version, '==' ) ) {
		return;
	}

	// Run upgrade routine.
	sqm_views_upgrade( $installed_version, $current_version );

	// Update stored version.
	update_option( SQMVIEWS_NAME, $current_version );

	/**
	 * Fires after plugin version upgrade is complete
	 *
	 * @param string $installed_version The previous installed version
	 * @param string $current_version The new current version
	 *
	 * @since 1.0.3
	 */
	do_action( 'sqm_views_upgraded', $installed_version, $current_version );

	Logger::log_debug(
		'Plugin version upgraded',
		array(
			'from' => $installed_version,
			'to'   => $current_version,
		)
	);
}

/**
 * Handles version-specific upgrade routines
 *
 * Add version-specific migration logic here as needed
 *
 * @param string $from_version Previous installed version.
 * @param string $to_version   New current version.
 *
 * @return void
 */
function sqm_views_upgrade( $from_version, $to_version ) {
	/**
	 * Allows plugins/themes to hook into the upgrade process
	 *
	 * @param string $from_version Previous installed version
	 * @param string $to_version New current version
	 *
	 * @since 1.0.3
	 */
	do_action( 'sqm_views_before_upgrade', $from_version, $to_version );

	// Example: Upgrade from versions before 1.0.0.
	if ( version_compare( $from_version, '1.0.0', '<' ) ) {
		// Placeholder for pre-1.0.0 migrations.
		do_action( 'sqm_views_migrate_to_1_0_0', $from_version, $to_version );
	}

	// Example: Upgrade from 1.0.x to 1.1.0.
	if ( version_compare( $from_version, '1.1.0', '<' ) && version_compare( $to_version, '1.1.0', '>=' ) ) {
		// Placeholder for 1.1.0 migrations.
		do_action( 'sqm_views_migrate_to_1_1_0', $from_version, $to_version );
	}

	// Ensure database tables are up to date.
	// This is safe to run multiple times as dbDelta only makes necessary changes.
	create_tables();

	/**
	 * Allows plugins/themes to hook after the upgrade process
	 *
	 * @param string $from_version Previous installed version
	 * @param string $to_version New current version
	 *
	 * @since 1.0.3
	 */
	do_action( 'sqm_views_after_upgrade', $from_version, $to_version );
}

/**
 * Handles plugin deactivation tasks
 *
 * @return void
 */
function deactivate() {
}

/**
 * Handles plugin uninstallation tasks
 *
 * @return void
 */
function uninstall() {
	drop_tables();
	delete_option( SQMVIEWS_OPTION_LAST_RUN );
	delete_option( SQMVIEWS_OPTION_TRACKABLE_POST_TYPES );
	delete_option( SQMVIEWS_OPTION_TRACKABLE_TAXONOMIES );
	delete_option( SQMVIEWS_OPTION_DATA_TAXONOMIES );
	delete_option( SQMVIEWS_OPTION_INLINE_JS );
	delete_option( SQMVIEWS_OPTION_MIN_JS );
	delete_option( SQMVIEWS_OPTION_HANDLER );
	delete_option( SQMVIEWS_OPTION_CRON_INTERVAL );
	delete_option( SQMVIEWS_OPTION_ENCRYPTION_KEY );
	delete_option( SQMVIEWS_NAME );
}

/**
 * Drops all plugin database tables
 *
 * @return void
 */
function drop_tables() {
	global $wpdb;
	require_once ABSPATH . 'wp-admin/includes/upgrade.php';
	$prefix = $wpdb->prefix;

	// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, Generic.Commenting.DocComment.MissingShort -- Schema operations during plugin uninstall.
	$wpdb->query( $wpdb->prepare( /** @lang MySQL */ 'DROP TABLE IF EXISTS %i', "{$prefix}sqm_views_daily" ) );
	$wpdb->query( $wpdb->prepare( /** @lang MySQL */ 'DROP TABLE IF EXISTS %i', "{$prefix}sqm_views_trackables" ) );
	$wpdb->query( $wpdb->prepare( /** @lang MySQL */ 'DROP TABLE IF EXISTS %i', "{$prefix}sqm_views_records" ) );
	$wpdb->query( $wpdb->prepare( /** @lang MySQL */ 'DROP TABLE IF EXISTS %i', "{$prefix}sqm_views_events" ) );
	// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, Generic.Commenting.DocComment.MissingShort
}

/**
 * Checks if REST API endpoint is operational.
 *
 * @param string $url The URL to check.
 *
 * @return array{success: bool, message: string, data?: array<string, mixed>} Array with 'success' boolean, 'message' string, and optional 'data'.
 */
function sqm_views_check_rest_endpoint( string $url ): array {
	$response = wp_remote_post(
		$url,
		array(
			'method'  => 'POST',
			'timeout' => SQMVIEWS_REST_TIMEOUT,
			'headers' => array(
				'Content-Type' => 'application/json',
			),
			'body'    => wp_json_encode( array( 't' => 'test' ) ),
		)
	);

	if ( is_wp_error( $response ) ) {
		return array(
			'success' => false,
			'message' => $response->get_error_message(),
		);
	}

	$body = wp_remote_retrieve_body( $response );
	$json = json_decode( $body, true );

	if ( JSON_ERROR_NONE !== json_last_error() ) {
		return array(
			'success' => false,
			/* translators: %s: JSON error message */
			'message' => sprintf( __( 'Invalid JSON response: %s', 'sqm-views' ), json_last_error_msg() ),
		);
	}

	$status_code = wp_remote_retrieve_response_code( $response );
	if ( 200 !== $status_code ) {
		return array(
			'success' => false,
			/* translators: %d: HTTP status code */
			'message' => sprintf( __( 'Unexpected status code: %d', 'sqm-views' ), $status_code ),
		);
	}

	return array(
		'success' => true,
		'data'    => $json,
		'message' => '',
	);
}

/**
 * Checks all available endpoints and returns the best one.
 *
 * @return string The handler status and message.
 */
function sqm_views_check_endpoint() {
	$handler             = null;
	$message             = __( '❌ We are not able to reach any endpoint. Please check your server configuration and try again.', 'sqm-views' );
	$endpoint_check_slow = sqm_views_check_rest_endpoint( get_rest_url( null, 'sqm-views/v1/track' ) );
	if ( $endpoint_check_slow && $endpoint_check_slow['success'] && 'ok' === $endpoint_check_slow['data']['test']['status'] ) {
		$handler = SQMVIEWS_HANDLERS_SLOW;
		$message = __( '⚠️ You are using WordPress API endpoint, which is slower than the recommended one. Please configure the drop-in sqm-views-pages.php to improve performance.', 'sqm-views' );
	}

	$endpoint_check_fast = sqm_views_check_rest_endpoint( site_url( '/sqm-views-pages.php' ) );
	if ( $endpoint_check_fast && $endpoint_check_fast['success'] && 'ok' === $endpoint_check_fast['data']['test']['status'] ) {
		$handler = SQMVIEWS_HANDLERS_FAST;
		$message = __( '✅ You are using the fastest endpoint. Nice job!', 'sqm-views' );
	}
	if ( $handler ) {
		update_option( SQMVIEWS_OPTION_HANDLER, $handler );
	}

	return $handler . ' <br> ' . $message;
}
