<?php
/**
 * SQMViews Settings Page class file.
 *
 * @package SQMViews
 */

namespace SQMViews;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WP Views Settings Page.
 *
 * Handles all admin settings pages for the SQMViews plugin.
 *
 * @package SQMViews
 * @since 1.0.0
 */
class SQMViewsSettings {

	/**
	 * Allowed granularity values.
	 *
	 * @var string[]
	 */
	private const ALLOWED_GRANULARITY = array( 'hourly', 'daily', 'weekly', 'monthly' );

	/**
	 * Allowed metric values.
	 *
	 * @var string[]
	 */
	private const ALLOWED_METRIC = array( 'count', 'duration', 'unique' );

	/**
	 * Register the plugin settings page.
	 *
	 * @return void
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'register_settings_page' ) );
		add_action( 'admin_init', array( $this, 'handle_manual_run' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
		add_action( 'wp_ajax_sqm_views_process_batch', array( $this, 'ajax_process_batch' ) );
	}

	/**
	 * Register the settings page and submenus.
	 *
	 * @return void
	 */
	public function register_settings_page(): void {
		add_menu_page(
			esc_html__( 'SQMViews Tracking', 'sqm-views' ),
			esc_html__( 'SQMViews', 'sqm-views' ),
			'manage_options',
			'sqm-views-settings',
			'__return_empty_string',
			'dashicons-chart-area'
		);

		// Configuration array for submenu pages.
		$submenu_pages = array(
			array(
				'parent_slug' => 'sqm-views-settings',
				'page_title'  => esc_html__( 'Tracking Settings', 'sqm-views' ),
				'menu_title'  => esc_html__( 'Tracking', 'sqm-views' ),
				'capability'  => 'manage_options',
				'menu_slug'   => 'sqm-views-settings',
				'callback'    => array( $this, 'render_config_tracking_page' ),
			),
			array(
				'parent_slug' => 'sqm-views-settings',
				'page_title'  => esc_html__( 'Processing Settings', 'sqm-views' ),
				'menu_title'  => esc_html__( 'Processing', 'sqm-views' ),
				'capability'  => 'manage_options',
				'menu_slug'   => 'sqm-views-process',
				'callback'    => array( $this, 'render_config_processing_page' ),
			),
			array(
				'parent_slug' => 'sqm-views-settings',
				'page_title'  => esc_html__( 'Statistics', 'sqm-views' ),
				'menu_title'  => esc_html__( 'Statistics', 'sqm-views' ),
				'capability'  => 'manage_options',
				'menu_slug'   => 'sqm-views-stats',
				'callback'    => array( $this, 'render_statistics_page' ),
			),
		);

		// Loop through configuration and add submenu pages.
		foreach ( $submenu_pages as $page ) {
			add_submenu_page(
				$page['parent_slug'],
				$page['page_title'],
				$page['menu_title'],
				$page['capability'],
				$page['menu_slug'],
				$page['callback']
			);
		}
	}

	/**
	 * Enqueue admin scripts and styles.
	 *
	 * @param string $hook Current admin page hook.
	 * @return void
	 */
	public function enqueue_admin_scripts( $hook ) {
		// Only load on our processing page.
		if ( 'sqmviews_page_sqm-views-process' !== $hook ) {
			return;
		}

		// Enqueue the processing script.
		wp_enqueue_script(
			'sqm-views-processing',
			plugins_url( 'assets/admin/processing.js', SQMVIEWS_PLUGIN_FILE ),
			array( 'jquery' ),
			SQMVIEWS_BUILD_GLOBAL_VERSION,
			true
		);

		// Get stats for initial file count.
		$stats = $this->get_basic_stats();

		// Localize script with PHP data and translations.
		wp_localize_script(
			'sqm-views-processing',
			'sqmViewsProcessing',
			array(
				'nonce'            => wp_create_nonce( 'sqm_views_process' ),
				'initialFileCount' => $stats['total_files'] ?? 0,
				'i18n'             => array(
					'confirmMessage' => __( 'This will process all raw files. Continue?', 'sqm-views' ),
					'starting'       => __( 'Starting processing...', 'sqm-views' ),
					'processing'     => __( 'Processing...', 'sqm-views' ),
					'complete'       => __( 'All files processed successfully!', 'sqm-views' ),
					'failed'         => __( 'Processing failed.', 'sqm-views' ),
					'error'          => __( 'An error occurred:', 'sqm-views' ),
					'locked'         => __( 'Another processing task is already running. Please wait and try again.', 'sqm-views' ),
					'completeStatus' => __( 'Complete!', 'sqm-views' ),
					'errorStatus'    => __( 'Error', 'sqm-views' ),
				),
			)
		);
	}

	/**
	 * Plugin settings page callback.
	 *
	 * @return void
	 */
	public function render_config_tracking_page() {
		// Check user capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Save settings if form was submitted.
		if ( isset( $_POST['sqm_views_submit'] ) ) {
			check_admin_referer( 'sqm_views_settings' );

			// Get available post types.
			$available_post_types = get_post_types( array( 'public' => true ), 'names' );

			// Get and sanitize selected post types from form.
			$submitted_post_types = isset( $_POST['sqm_views_post_types'] ) ?
					array_map( 'sanitize_key', (array) $_POST['sqm_views_post_types'] ) : array();
			$trackable_post_types = array_intersect( $submitted_post_types, $available_post_types );

			// Get and sanitize selected taxonomies from form (for tracking pageviews).
			$available_taxonomies = get_taxonomies( array( 'public' => true ), 'names' );
			$submitted_taxonomies = isset( $_POST['sqm_views_taxonomies'] ) ?
					array_map( 'sanitize_key', (array) $_POST['sqm_views_taxonomies'] ) : array();
			$trackable_taxonomies = array_intersect( $submitted_taxonomies, $available_taxonomies );

			// Get and sanitize selected taxonomies for data collection.
			$submitted_data_taxonomies = isset( $_POST[ SQMVIEWS_OPTION_DATA_TAXONOMIES ] ) ?
					array_map( 'sanitize_key', (array) $_POST[ SQMVIEWS_OPTION_DATA_TAXONOMIES ] ) : array();
			$data_taxonomies           = array_intersect( $submitted_data_taxonomies, $available_taxonomies );

			// Get and sanitize JS loading preference.
			$use_inline_js = isset( $_POST[ SQMVIEWS_OPTION_INLINE_JS ] ) ? true : false;
			$use_min_js    = isset( $_POST[ SQMVIEWS_OPTION_MIN_JS ] ) ? true : false;

			// Save settings.
			update_option( SQMVIEWS_OPTION_TRACKABLE_POST_TYPES, $trackable_post_types );
			update_option( SQMVIEWS_OPTION_TRACKABLE_TAXONOMIES, $trackable_taxonomies );
			update_option( SQMVIEWS_OPTION_DATA_TAXONOMIES, $data_taxonomies );
			update_option( SQMVIEWS_OPTION_INLINE_JS, $use_inline_js );
			update_option( SQMVIEWS_OPTION_MIN_JS, $use_min_js );

			echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Settings saved.', 'sqm-views' ) . '</p></div>';
		}

		// Get current settings.
		$trackable_post_types = get_option( SQMVIEWS_OPTION_TRACKABLE_POST_TYPES, array( 'post', 'page' ) );
		$trackable_taxonomies = get_option( SQMVIEWS_OPTION_TRACKABLE_TAXONOMIES, array( 'category', 'post_tag' ) );
		$data_taxonomies      = get_option( SQMVIEWS_OPTION_DATA_TAXONOMIES, array( 'category', 'post_tag' ) );
		$use_inline_js        = get_option( SQMVIEWS_OPTION_INLINE_JS, true );
		$use_min_js           = get_option( SQMVIEWS_OPTION_MIN_JS, true );

		// Get all public post types and taxonomies.
		$available_post_types = get_post_types( array( 'public' => true ), 'objects' );
		$available_taxonomies = get_taxonomies( array( 'public' => true ), 'objects' );

		// Display settings form.
		?>
			<div class="wrap">
				<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
				<form method="post" action="<?php echo esc_url( admin_url( 'admin.php?page=sqm-views-settings' ) ); ?>">
				<?php wp_nonce_field( 'sqm_views_settings' ); ?>

					<h2><?php esc_html_e( 'Trackable Content', 'sqm-views' ); ?></h2>
					<p><?php esc_html_e( 'Select which content types should be tracked for pageviews:', 'sqm-views' ); ?></p>

					<h3><?php esc_html_e( 'Post Types', 'sqm-views' ); ?></h3>
				<?php foreach ( $available_post_types as $post_type ) : ?>
						<label>
							<input type="checkbox" name="sqm_views_post_types[]"
									value="<?php echo esc_attr( $post_type->name ); ?>"
									<?php checked( in_array( $post_type->name, $trackable_post_types, true ) ); ?>>
							<?php echo esc_html( $post_type->labels->name ); ?>
						</label><br>
					<?php endforeach; ?>

					<h3><?php esc_html_e( 'Taxonomies', 'sqm-views' ); ?></h3>
					<p><?php esc_html_e( 'Select taxonomies to track pageviews for (taxonomy archive pages):', 'sqm-views' ); ?></p>
				<?php foreach ( $available_taxonomies as $taxonomy ) : ?>
						<label>
							<input type="checkbox" name="sqm_views_taxonomies[]"
									value="<?php echo esc_attr( $taxonomy->name ); ?>"
									<?php checked( in_array( $taxonomy->name, $trackable_taxonomies, true ) ); ?>>
							<?php echo esc_html( $taxonomy->labels->name ); ?>
						</label><br>
					<?php endforeach; ?>

					<h2><?php esc_html_e( 'Trackable Data', 'sqm-views' ); ?></h2>
					<p><?php esc_html_e( 'Select which taxonomies data should be collected and stored with individual pageviews:', 'sqm-views' ); ?></p>

					<h3><?php esc_html_e( 'Taxonomies Data', 'sqm-views' ); ?></h3>
					<p><?php esc_html_e( 'Collect following taxonomies data for each pageview:', 'sqm-views' ); ?></p>
				<?php foreach ( $available_taxonomies as $taxonomy ) : ?>
						<label>
							<input type="checkbox" name="sqm_views_data_taxonomies[]"
									value="<?php echo esc_attr( $taxonomy->name ); ?>"
									<?php checked( in_array( $taxonomy->name, $data_taxonomies, true ) ); ?>>
							<?php echo esc_html( $taxonomy->labels->name ); ?>
						</label><br>
					<?php endforeach; ?>

					<h2><?php esc_html_e( 'JavaScript Loading', 'sqm-views' ); ?></h2>
					<label>
						<input type="checkbox" name="sqm_views_use_inline_js"
								value="1" <?php checked( $use_inline_js ); ?>>
						<?php esc_html_e( 'Use inline JavaScript tracking code.', 'sqm-views' ); ?>
					</label>
					<br>
					<label>
						<input type="checkbox" name="sqm_views_use_min_js"
								value="1" <?php checked( $use_min_js ); ?>>
						<?php esc_html_e( 'Use minified JavaScript version.', 'sqm-views' ); ?>
					</label>

					<p>
						<input type="submit" name="sqm_views_submit" class="button button-primary"
								value="<?php esc_attr_e( 'Save Settings', 'sqm-views' ); ?>">
					</p>
				</form>
			</div>
			<?php
	}

	/**
	 * Handle manual processing run.
	 *
	 * @return void
	 */
	public function handle_manual_run() {
		if ( isset( $_GET['page'] ) && 'sqm-views-process' === sanitize_key( wp_unslash( $_GET['page'] ) ) &&
			isset( $_GET['action'] ) && 'run_now' === sanitize_key( wp_unslash( $_GET['action'] ) ) &&
			isset( $_GET['_wpnonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'sqm_views_run_now' ) ) {

			// Check user permissions.
			if ( ! current_user_can( 'manage_options' ) ) {
				wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'sqm-views' ) );
			}

			// Run the processing function directly.
			$result = sqm_views_process_statistics( false );

			// Redirect with result message.
			$redirect_url = admin_url( 'admin.php?page=sqm-views-process' );
			if ( $result['success'] ) {
				$redirect_url = add_query_arg( 'message', 'success', $redirect_url );
				$redirect_url = add_query_arg( 'processed', $result['message'], $redirect_url );
			} else {
				$redirect_url = add_query_arg( 'message', 'error', $redirect_url );
				$redirect_url = add_query_arg( 'error_msg', rawurlencode( $result['error'] ), $redirect_url );
			}

			// Add nonce for message verification.
			$redirect_url = add_query_arg( 'message_nonce', wp_create_nonce( 'sqm_views_message' ), $redirect_url );

			wp_safe_redirect( $redirect_url );
			exit;
		}
	}

	/**
	 * Admin page callback for processing settings.
	 *
	 * @return void
	 */
	public function render_config_processing_page(): void {

		// Check user permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'sqm-views' ) );
		}

		// Show messages.
		$this->show_admin_messages();

		// Get cron information.
		$cron             = SQMViewsCron::get_instance();
		$next_run         = $cron->get_next_run_time();
		$current_interval = get_option( SQMVIEWS_OPTION_CRON_INTERVAL, 'hourly' );
		$last_run         = wp_date( 'Y-m-d H:i:s', get_option( SQMVIEWS_OPTION_LAST_RUN, 0 ) );

		// Get some basic stats.
		$stats = $this->get_basic_stats();

		?>
		<div class="wrap">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>

			<div class="card">
				<h2><?php esc_html_e( 'Current Status', 'sqm-views' ); ?></h2>
				<table class="form-table">
					<tr>
						<th scope="row"><?php esc_html_e( 'Cron Interval', 'sqm-views' ); ?></th>
						<td><?php echo esc_html( ucfirst( str_replace( '_', ' ', $current_interval ) ) ); ?></td>
					</tr>
					<tr>
						<th scope="row"><?php esc_html_e( 'Last Run', 'sqm-views' ); ?></th>
						<td><?php echo esc_html( $last_run ); ?></td>
					</tr>
					<tr>
						<th scope="row"><?php esc_html_e( 'Next Scheduled Run', 'sqm-views' ); ?></th>
						<td><?php echo esc_html( $next_run ); ?></td>
					</tr>
					<?php if ( $stats ) : ?>
						<tr>
							<th scope="row"><?php esc_html_e( 'Total Files', 'sqm-views' ); ?></th>
							<td><?php echo esc_html( number_format( $stats['total_files'] ) ); ?></td>
						</tr>
						<tr>
							<th scope="row"><?php esc_html_e( 'Total Events (estimate)', 'sqm-views' ); ?></th>
							<td><?php echo esc_html( number_format( $stats['total_records'] ) ); ?></td>
						</tr>
						<tr>
							<th scope="row"><?php esc_html_e( "Today's Files", 'sqm-views' ); ?></th>
							<td><?php echo esc_html( number_format( $stats['todays_files'] ) ); ?></td>
						</tr>
						<tr>
							<th scope="row"><?php esc_html_e( "Today's Events (estimate)", 'sqm-views' ); ?></th>
							<td><?php echo esc_html( number_format( $stats['today_records'] ) ); ?></td>
						</tr>
					<?php endif; ?>
					<tr>
						<th scope="row"><?php esc_html_e( 'Active Endpoint', 'sqm-views' ); ?></th>
						<td><?php echo wp_kses_post( sqm_views_check_endpoint() ); ?></td>
					</tr>
				</table>
			</div>

			<div class="card">
				<h2><?php esc_html_e( 'Manual Processing', 'sqm-views' ); ?></h2>
				<p><?php esc_html_e( 'Click the button below to run the statistics processing function immediately. Processing will continue until all files are processed.', 'sqm-views' ); ?></p>

				<p>
					<button type="button" id="sqm-views-run-processing" class="button button-primary">
						<?php esc_html_e( 'Run Processing Now', 'sqm-views' ); ?>
					</button>
				</p>

				<div id="sqm-views-progress-container" style="display:none; margin-top: 15px;">
					<p id="sqm-views-progress-status">
						<?php esc_html_e( 'Initializing...', 'sqm-views' ); ?>
					</p>
					<div style="background: #f0f0f0; border: 1px solid #ddd; border-radius: 3px; height: 24px; overflow: hidden;">
						<div id="sqm-views-progress-bar" style="background: #0073aa; height: 100%; width: 0%; transition: width 0.3s;">
						</div>
					</div>
					<p id="sqm-views-progress-message" style="margin-top: 10px; font-style: italic;"></p>
				</div>

				<p class="description">
					<?php esc_html_e( 'Processing runs in batches to avoid timeouts. The progress bar will update as files are processed.', 'sqm-views' ); ?>
				</p>
			</div>

			<div class="card">
				<h2><?php esc_html_e( 'Cron Settings', 'sqm-views' ); ?></h2>
				<form method="post" action="options.php">
					<?php
					settings_fields( 'sqm-views' );
					do_settings_sections( 'sqm-views' );
					submit_button( __( 'Save Cron Settings', 'sqm-views' ) );
					?>
				</form>
			</div>
		</div>
		<?php
	}

	/**
	 * Show admin messages.
	 *
	 * @return void
	 */
	private function show_admin_messages() {
		// Check if message parameter exists and is from a trusted source.
		if ( isset( $_GET['message'] ) && isset( $_GET['message_nonce'] ) ) {
			// Verify nonce before processing message parameters.
			$nonce = sanitize_text_field( wp_unslash( $_GET['message_nonce'] ) );
			if ( ! wp_verify_nonce( $nonce, 'sqm_views_message' ) ) {
				return;
			}

			$message = sanitize_key( wp_unslash( $_GET['message'] ) );

			if ( 'success' === $message ) {
				$processed_message = isset( $_GET['processed'] ) ? sanitize_text_field( wp_unslash( $_GET['processed'] ) ) : __( 'Processing completed.', 'sqm-views' );
				echo '<div class="notice notice-success is-dismissible">';
				echo '<p><strong>' . esc_html__( 'Success!', 'sqm-views' ) . '</strong> ' . esc_html( $processed_message ) . '</p>';
				echo '</div>';
			} elseif ( 'error' === $message ) {
				$error_msg = isset( $_GET['error_msg'] ) ? sanitize_text_field( wp_unslash( $_GET['error_msg'] ) ) : __( 'Unknown error occurred.', 'sqm-views' );
				echo '<div class="notice notice-error is-dismissible">';
				echo '<p><strong>' . esc_html__( 'Error!', 'sqm-views' ) . '</strong> ' . esc_html( $error_msg ) . '</p>';
				echo '</div>';
			}
		}
	}

	/**
	 * Get basic statistics for display.
	 *
	 * @return array<string, mixed>|null Statistics array or null on error.
	 */
	private function get_basic_stats() {
		try {
			// Initialize counters.
			$total_files   = 0;
			$todays_files  = 0;
			$total_records = 0;
			$today_records = 0;

			// Get today's date in Y-m-d format.
			$today = gmdate( 'Y-m-d' );

			// Check if the raw directory exists.
			if ( defined( 'SQMVIEWS_RAW_DIR' ) && is_dir( SQMVIEWS_RAW_DIR ) ) {
				// Get all files in the raw directory.
				$files = glob( SQMVIEWS_RAW_DIR . '/*.raw.jsonl' );

				if ( false !== $files && count( $files ) > 0 ) {
					$total_files = count( $files );
					$total_size  = 0;
					$todays_size = 0;

					// Calculate total and today's file sizes.
					foreach ( $files as $file ) {
						$file_size = filesize( $file );
						if ( false !== $file_size ) {
							$total_size += $file_size;

							$filename = basename( $file );
							if ( str_starts_with( $filename, $today ) ) {
								++$todays_files;
								$todays_size += $file_size;
							}
						}
					}

					// Sample up to 20 files to calculate records per byte ratio.
					$sample_size   = min( 20, $total_files );
					$sampled_files = array_rand( array_flip( $files ), $sample_size );

					// Ensure $sampled_files is always an array.
					if ( ! is_array( $sampled_files ) ) {
						$sampled_files = array( $sampled_files );
					}

					$total_lines_in_sample = 0;
					$total_bytes_in_sample = 0;

					foreach ( $sampled_files as $sample_file ) {
						if ( file_exists( $sample_file ) && is_readable( $sample_file ) ) {
							$file_size = filesize( $sample_file );
							if ( false !== $file_size ) {
								$total_bytes_in_sample += $file_size;

								// Count lines in the file efficiently.
								$line_count = 0;
								// The file could be quite large, so reading the file line by line is necessary to avoid loading it entirely into memory.
								// Using WP_Filesystem methods here is not practical.
								$handle = fopen( $sample_file, 'r' ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen
								if ( $handle ) {
									while ( ! feof( $handle ) ) {
										$line = fgets( $handle );
										if ( false !== $line ) {
											++$line_count;
										}
									}
									fclose( $handle ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
								}
								$total_lines_in_sample += $line_count;
							}
						}
					}

					// Calculate records per byte ratio and estimate totals.
					if ( $total_bytes_in_sample > 0 ) {
						$records_per_byte = $total_lines_in_sample / $total_bytes_in_sample;

						// Estimate total and today's records based on file sizes.
						$total_records = (int) round( $total_size * $records_per_byte );
						$today_records = (int) round( $todays_size * $records_per_byte );
					}
				}
			}

			return array(
				'total_records' => $total_records,
				'today_records' => $today_records,
				'total_files'   => $total_files,
				'todays_files'  => $todays_files,
			);
		} catch ( \Exception $e ) {
			return null;
		}
	}

	/**
	 * Render the statistics page.
	 *
	 * @return void
	 */
	public function render_statistics_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Allow filtering of JS settings.
		$use_min_js  = apply_filters( 'sqm_views_dashboard_minified_js', get_option( SQMVIEWS_OPTION_MIN_JS, true ) );
		$file_suffix = $use_min_js ? '.min' : '';

		// Build dashboard script path and URL.
		$script_path = plugin_dir_path( SQMVIEWS_PLUGIN_FILE ) . 'artifacts/dashboard-' . SQMVIEWS_BUILD_GLOBAL_VERSION . ".bundle{$file_suffix}.js";
		$script_path = apply_filters( 'sqm_views_dashboard_script_path', $script_path, $use_min_js );
		$script_url  = plugins_url( 'artifacts/dashboard-' . SQMVIEWS_BUILD_GLOBAL_VERSION . ".bundle{$file_suffix}.js", SQMVIEWS_PLUGIN_FILE );
		$script_url  = apply_filters( 'sqm_views_dashboard_script_url', $script_url, $use_min_js );

		// Enqueue scripts.
		if ( file_exists( $script_path ) ) {
			$dependencies = array();

			// Non-minified version requires D3.js as external dependency.
			if ( ! $use_min_js ) {
				// Load local D3 libraries bundle for debug version.
				$libs_script_path = plugin_dir_path( SQMVIEWS_PLUGIN_FILE ) . 'artifacts/dashboard-' . SQMVIEWS_BUILD_GLOBAL_VERSION . '.bundle.libs.js';
				$libs_script_url  = plugins_url( 'artifacts/dashboard-' . SQMVIEWS_BUILD_GLOBAL_VERSION . '.bundle.libs.js', SQMVIEWS_PLUGIN_FILE );

				if ( file_exists( $libs_script_path ) ) {
					// Enqueue external libraries script.
					wp_enqueue_script(
						'sqm-views-dashboard-libs',
						$libs_script_url,
						array(),
						SQMVIEWS_BUILD_GLOBAL_VERSION,
						true
					);
					$dependencies[] = 'sqm-views-dashboard-libs';
				} else {
					echo '<div class="notice notice-warning"><p>' . esc_html__( 'Dashboard libraries script not found. Please rebuild the plugin assets.', 'sqm-views' ) . '</p></div>';
				}
			}

			// Enqueue external dashboard script.
			wp_enqueue_script(
				'sqm-views-dashboard',
				$script_url,
				$dependencies,
				SQMVIEWS_BUILD_GLOBAL_VERSION,
				true
			);

			// Get default filter values (can be customized per WordPress context).
			$default_filters          = $this->sanitize_dashboard_filters( $this->get_default_dashboard_filters() );
			$default_filters['nonce'] = wp_create_nonce( 'wp_rest' );

			// Add initialization script.
			$init_script = sprintf(
				'window.__sqm_wp_pageviews_dashboard_init_with_controls?.("dashboard-controls", "chart", %s);',
				wp_json_encode( $default_filters )
			);
			wp_add_inline_script( 'sqm-views-dashboard', $init_script );
		} else {
			echo '<div class="notice notice-error"><p>' . esc_html__( 'Dashboard script not found. Please rebuild the plugin assets.', 'sqm-views' ) . '</p></div>';
		}

		?>
		<div class="wrap">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>

			<!-- Dashboard controls will be rendered by JavaScript -->
			<div id="dashboard-controls"></div>

			<!-- Chart container -->
			<div id="dashboard-chart">
				<div id="chart"></div>
			</div>
		</div>
		<?php
	}

	/**
	 * Get dashboard filter defaults with dynamic date values.
	 *
	 * @return array<string, string> Full default filter values.
	 */
	private function get_filter_constants() {
		return array(
			'startDate'   => gmdate( 'Y-m-d', strtotime( '-1 month' ) ),
			'endDate'     => gmdate( 'Y-m-d' ),
			'granularity' => 'daily',
			'metric'      => 'count',
		);
	}

	/**
	 * Get default dashboard filters - can be customized based on WordPress context.
	 *
	 * @return array<string, string> Default filter values (unfiltered).
	 */
	private function get_default_dashboard_filters() {
		// Allow filtering via WordPress hooks.
		return apply_filters( 'sqm_views_dashboard_default_filters', $this->get_filter_constants() );
	}

	/**
	 * Sanitize dashboard filter values.
	 *
	 * Validates and sanitizes filter values after apply_filters to prevent
	 * malicious input from third-party code.
	 *
	 * @param mixed $filters Filter values to sanitize.
	 * @return array<string, string> Sanitized filter values with safe defaults for invalid input.
	 */
	private function sanitize_dashboard_filters( $filters ) {
		$defaults = $this->get_filter_constants();

		// Ensure input is an array.
		if ( ! is_array( $filters ) ) {
			return $defaults;
		}

		return array(
			'startDate'   => $this->sanitize_date( $filters['startDate'] ?? '', $defaults['startDate'] ),
			'endDate'     => $this->sanitize_date( $filters['endDate'] ?? '', $defaults['endDate'] ),
			'granularity' => in_array( $filters['granularity'] ?? '', self::ALLOWED_GRANULARITY, true )
				? sanitize_key( $filters['granularity'] )
				: $defaults['granularity'],
			'metric'      => in_array( $filters['metric'] ?? '', self::ALLOWED_METRIC, true )
				? sanitize_key( $filters['metric'] )
				: $defaults['metric'],
		);
	}

	/**
	 * Sanitize a date string to Y-m-d format.
	 *
	 * @param string $date    Date string to sanitize.
	 * @param string $default_date Default value if invalid.
	 *
	 * @return string Sanitized date in Y-m-d format.
	 */
	private function sanitize_date( $date, $default_date ) {
		if ( ! is_string( $date ) || '' === $date ) {
			return $default_date;
		}

		// Validate date format (Y-m-d).
		$parsed = \DateTime::createFromFormat( 'Y-m-d', $date );
		if ( false === $parsed || $parsed->format( 'Y-m-d' ) !== $date ) {
			return $default_date;
		}

		return $date;
	}

	/**
	 * AJAX handler for batch processing.
	 *
	 * @return void
	 */
	public function ajax_process_batch() {
		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'sqm-views' ) ) );
		}

		// Verify nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'sqm_views_process' ) ) {
			wp_send_json_error( array( 'message' => __( 'Invalid nonce.', 'sqm-views' ) ) );
		}

		// Run the processing function directly.
		$result = sqm_views_process_statistics( false );

		// Check if there are still files to process.
		$remaining_files = 0;
		if ( defined( 'SQMVIEWS_RAW_DIR' ) && is_dir( SQMVIEWS_RAW_DIR ) ) {
			$files = glob( SQMVIEWS_RAW_DIR . '/*.raw.jsonl' );
			if ( false !== $files ) {
				$remaining_files = count( $files );
			}
		}

		if ( $result['success'] ) {
			wp_send_json_success(
				array(
					'message'         => $result['message'],
					'remaining_files' => $remaining_files,
					'completed'       => 0 === $remaining_files,
					'locked'          => isset( $result['locked'] ) ? $result['locked'] : false,
				)
			);
		} else {
			wp_send_json_error(
				array(
					'message'         => $result['error'],
					'remaining_files' => $remaining_files,
				)
			);
		}
	}
}
