<?php
/**
 * SQMViewsRecord class file.
 *
 * @package SQMViews
 */

namespace SQMViews;

/**
 * Represents a single pageview record with accumulated events.
 *
 * @property string $gid           Global ID.
 * @property string $tid           Trackable ID.
 * @property string $eid           Event ID.
 * @property string $user_moment   User's local timestamp.
 * @property string $user_timezone User's timezone.
 * @property string $utc_moment    UTC timestamp.
 * @property string $url           Page URL.
 * @property string $referrer      Referrer URL.
 * @property float  $on_page       Time on page in seconds.
 * @property float  $active        Active time in seconds.
 * @property int    $high_freq     High frequency activity count.
 * @property int    $low_freq      Low frequency activity count.
 * @property array<string, mixed>  $payload       Event payload data.
 * @property string $exit          Exit event type.
 * @property array<string, mixed>  $trackable     Trackable data.
 * @property string $ip            User IP address.
 */
class SQMViewsRecord {

	/**
	 * Log of events for this record.
	 *
	 * @var SQMViewsEvent[]
	 */
	private array $log = array();

	/**
	 * Hashes of logged events for deduplication.
	 *
	 * @var string[]
	 */
	private array $log_hashes = array();

	/**
	 * The start (init) event for this record.
	 *
	 * @var SQMViewsEvent|null
	 */
	public SQMViewsEvent|null $start_event = null;

	/**
	 * The end (exit/timeout) event for this record.
	 *
	 * @var SQMViewsEvent|null
	 */
	public SQMViewsEvent|null $end_event = null;

	/**
	 * The last ping event received.
	 *
	 * @var SQMViewsEvent|null
	 */
	public SQMViewsEvent|null $last_ping = null;

	/**
	 * Record fields data.
	 *
	 * @var array<string, mixed>
	 */
	private array $fields = array(
		'gid'           => '',
		'tid'           => '',
		'eid'           => '',

		'user_moment'   => '',
		'user_timezone' => '',
		'utc_moment'    => '',

		'url'           => '',
		'referrer'      => '',
		'on_page'       => 0,
		'active'        => 0,
		'high_freq'     => 0,
		'low_freq'      => 0,
		'payload'       => array(),
		'exit'          => '',

		'trackable'     => array(),
	);

	/**
	 * Magic getter for record fields.
	 *
	 * @param string $name Field name.
	 * @return mixed Field value or null.
	 */
	public function __get( $name ) {
		return $this->fields[ $name ] ?? null;
	}

	/**
	 * Magic setter for record fields.
	 *
	 * @param string $name  Field name.
	 * @param mixed  $value Field value.
	 */
	public function __set( $name, $value ) {
		if ( array_key_exists( $name, $this->fields ) ) {
			$this->fields[ $name ] = $value;
		}
	}

	/**
	 * Accumulate an event into this record.
	 *
	 * @param array<string, mixed> $data Event data.
	 * @param string               $raw  Raw event string.
	 * @return void
	 */
	public function accumulate( array $data, string $raw ): void {
		$event = new SQMViewsEvent( $data, $raw );
		if ( in_array( $event->get_hash(), $this->log_hashes, true ) ) {
			return;
		}
		$this->log[]        = $event;
		$this->log_hashes[] = $event->get_hash();
		usort(
			$this->log,
			function ( $a, $b ) {
				return $a->ts <=> $b->ts;
			}
		);

		if ( SQMVIEWS_EVENTS_INIT === $event->t ) {
			$this->start_event = $event;
			$this->process();
		}
		if ( SQMVIEWS_EVENTS_PING === $event->t ) {
			if ( ! $this->last_ping || $this->last_ping->ts < $event->ts ) {
				$this->last_ping = $event;
			}
		}
		if ( SQMVIEWS_EVENTS_EXIT === $event->t ) {
			$this->end_event = $event;
		}
		if ( SQMVIEWS_EVENTS_TIMEOUT === $event->t && ! $this->end_event ) {
			$this->end_event = $event;
		}

		if ( $this->is_complete() ) {
			$this->process();
		}
	}

	/**
	 * Check if record is complete (has both start and end events).
	 *
	 * @return bool True if complete.
	 */
	public function is_complete(): bool {
		if ( $this->start_event && $this->end_event ) {
			return true;
		}

		return false;
	}

	/**
	 * Check if record is valid (nonces match).
	 *
	 * @return bool True if valid.
	 */
	public function is_valid(): bool {
		if ( $this->start_event->server_nonce !== $this->end_event->user_nonce ) {
			return false;
		}

		return true;
	}

	/**
	 * Check if record is stale (inactive for too long).
	 *
	 * @return bool True if stale.
	 */
	public function is_staled() {
		if ( ! $this->start_event && ! $this->end_event ) {
			return true;
		}
		if (
			! $this->start_event &&
			$this->end_event &&
			$this->end_event->ts < time() - SQMVIEWS_SESSION_INACTIVITY_TIMEOUT
		) {
			return true;
		}
		if ( ! $this->end_event ) {
			if ( ! $this->last_ping && $this->start_event->ts < time() - SQMVIEWS_SESSION_INACTIVITY_TIMEOUT ) {
				return true;
			}
			if ( $this->last_ping && $this->last_ping->ts < time() - SQMVIEWS_SESSION_INACTIVITY_TIMEOUT ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Check if record can be force completed.
	 *
	 * @return bool True if can be force completed.
	 */
	public function can_force_complete(): bool {
		if ( ! $this->start_event ) {
			return false;
		}
		$inactive_time = time() - SQMVIEWS_SESSION_INACTIVITY_TIMEOUT;

		return ( ! $this->last_ping && $this->start_event->ts < $inactive_time ) ||
				( $this->last_ping && $this->last_ping->ts < $inactive_time );
	}

	/**
	 * Force complete the record using the last available event.
	 *
	 * @return void
	 */
	public function force_complete() {
		if ( $this->end_event ) {
			return;
		}
		if ( $this->last_ping ) {
			$this->end_event = $this->last_ping;
		}
		if ( $this->start_event ) {
			$this->end_event             = $this->start_event;
			$this->end_event->user_nonce = $this->end_event->server_nonce;
		}
		$this->process();
	}

	/**
	 * Processes the event log and calculates metrics.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function process(): void {
		$this->gid      = $this->start_event->gid;
		$this->url      = $this->start_event->user_url;
		$this->referrer = $this->start_event->user_referrer;

		$this->ip = $this->start_event->server_ip;

		if ( $this->end_event ) {
			$this->on_page   = round( $this->end_event->ts - $this->start_event->ts, 2 );
			$this->active    = $this->end_event->user_active_time;
			$this->high_freq = $this->end_event->user_activity_high_frequency;
			$this->low_freq  = $this->end_event->user_activity_low_frequency;
			$this->exit      = $this->end_event->t;
		}

		$this->user_moment   = $this->start_event->user_timestamp;
		$this->user_timezone = $this->start_event->user_timezone;
		$this->utc_moment    = gmdate( 'c', strtotime( $this->start_event->server_date ) );

		$this->payload = $this->start_event->payload;
		if ( $this->payload ) {
			$this->trackable = array(
				'tgroup' => ( isset( $this->payload['group'] ) ? $this->payload['group'] : '' ),
				'ttype'  => ( isset( $this->payload['type'] ) ? $this->payload['type'] : '' ),
				'wpid'   => ( isset( $this->payload['id'] ) ? $this->payload['id'] : 0 ),
				'altid'  => ( isset( $this->payload['altid'] ) ? $this->payload['altid'] : null ),
			);
			$this->eid       = $this->payload['event'] ?? 0;
		}

		/**
		 * Filters the calculated metrics for a record
		 *
		 * Allows developers to modify or add custom metrics
		 *
		 * @since 1.0.0
		 *
		 * @param array           $metrics Calculated metrics (on_page, active, high_freq, low_freq, exit)
		 * @param SQMViewsRecord   $record  The record being processed
		 */
		$metrics = apply_filters(
			'sqm_views_calculated_metrics',
			array(
				'on_page'   => $this->on_page,
				'active'    => $this->active,
				'high_freq' => $this->high_freq,
				'low_freq'  => $this->low_freq,
				'exit'      => $this->exit,
			),
			$this
		);

		// Apply filtered metrics back to the record.
		$this->on_page   = $metrics['on_page'];
		$this->active    = $metrics['active'];
		$this->high_freq = $metrics['high_freq'];
		$this->low_freq  = $metrics['low_freq'];
		$this->exit      = $metrics['exit'];
	}

	/**
	 * Get the event log.
	 *
	 * @return SQMViewsEvent[] Array of events.
	 */
	public function get_log() {
		return $this->log;
	}

	/**
	 * Get the timestamp of the last activity.
	 *
	 * @return int|null Timestamp or null.
	 */
	public function get_last_activity_timestamp() {
		$last_event = end( $this->log );
		return $last_event ? $last_event->ts : null;
	}

	/**
	 * Get processed fields as JSON.
	 *
	 * @return string JSON string.
	 */
	public function get_processed() {
		return wp_json_encode( $this->fields );
	}

	/**
	 * Get raw event data.
	 *
	 * @return string[] Array of raw event strings.
	 */
	public function get_raw_events(): array {
		return array_map(
			function ( $event ) {
				return $event->get_raw();
			},
			$this->get_log()
		);
	}
}
