<?php
/**
 * SQMViewsRapidProcessor class file.
 *
 * @package SQMViews
 */

namespace SQMViews;

require_once __DIR__ . '/ProcessorResponse.php';
require_once __DIR__ . '/RequestValidator.php';

/**
 * Fast processor for tracking requests.
 */
class SQMViewsRapidProcessor {

	/**
	 * Constructor.
	 */
	public function __construct() {
	}

	/**
	 * Send CORS and content type headers.
	 *
	 * @return void
	 */
	public function send_headers(): void {
		header( 'Access-Control-Allow-Origin: *' );
		header( 'Access-Control-Allow-Methods: GET, POST' );
		header( 'Content-Type: application/json' );
	}

	/**
	 * Process a tracking request.
	 *
	 * @param string $request_body JSON request body.
	 * @return ProcessorResponse
	 */
	public function process( $request_body ): ProcessorResponse {
		$content_length  = isset( $_SERVER['CONTENT_LENGTH'] ) ? intval( $_SERVER['CONTENT_LENGTH'] ) : null;
		$remote_addr     = isset( $_SERVER['REMOTE_ADDR'] ) ? \filter_input( INPUT_SERVER, 'REMOTE_ADDR', FILTER_VALIDATE_IP ) : 'unknown';
		$http_user_agent = isset( $_SERVER['HTTP_USER_AGENT'] ) ? \filter_input( INPUT_SERVER, 'HTTP_USER_AGENT', FILTER_SANITIZE_FULL_SPECIAL_CHARS ) : 'unknown';

		if ( ! empty( $content_length ) && $content_length > SQMVIEWS_REQUEST_LENGTH_HARD_LIMIT ) {
			Logger::log_warning(
				'Request too large, rejecting',
				array(
					'content_length' => $content_length,
					'limit'          => SQMVIEWS_REQUEST_LENGTH_HARD_LIMIT,
					'remote_addr'    => $remote_addr,
					'user_agent'     => $http_user_agent,
				)
			);
			return ProcessorResponse::error(
				'invalid_request_too_large',
				'Request payload too large',
				400
			);
		}

		if ( ! $request_body || 0 === strlen( $request_body ) ) {
			Logger::log_warning(
				'Request body empty',
				array(
					'content_length' => $content_length,
					'actual_size'    => strlen( (string) $request_body ),
					'limit'          => SQMVIEWS_REQUEST_LENGTH_HARD_LIMIT,
					'remote_addr'    => $remote_addr,
				)
			);
			return ProcessorResponse::error(
				'invalid_request_body_empty',
				'Request body empty',
				400
			);
		}

		if ( strlen( $request_body ) > SQMVIEWS_REQUEST_LENGTH_HARD_LIMIT ) {
			Logger::log_warning(
				'Request body too large, refuse to parse JSON',
				array(
					'content_length' => $content_length,
					'actual_size'    => strlen( $request_body ),
					'limit'          => SQMVIEWS_REQUEST_LENGTH_HARD_LIMIT,
					'remote_addr'    => $remote_addr,
				)
			);
			return ProcessorResponse::error(
				'invalid_body_too_large',
				'Request body too large',
				400
			);
		}

		$data         = array(
			't'   => '',
			'gid' => '',
			'ts'  => '',
		);
		$data['user'] = json_decode( $request_body, true );
		if ( empty( $data['user'] ) ) {
			Logger::log_warning(
				'Invalid JSON in request body',
				array(
					'json_error'  => json_last_error_msg(),
					'body_length' => strlen( $request_body ),
					'remote_addr' => $remote_addr,
				)
			);
			return ProcessorResponse::error(
				'invalid_json',
				'Invalid JSON in request body',
				400
			);
		}
		$data['server'] = array();

		if ( empty( $data['user']['t'] ) ) {
			Logger::log_warning(
				'Request validation failed, type missing',
				array(
					'error_code'    => 'missing_type',
					'error_message' => 'Request validation failed, type missing',
					'event_type'    => 'unknown',
					'remote_addr'   => $remote_addr,
				)
			);
			return ProcessorResponse::error(
				'invalid_event_missing_type',
				'Request validation failed, type missing',
				400
			);
		}
		// Validate request structure against schema.
		if ( ! empty( $data['user']['t'] ) ) {
			$validation_result = RequestValidator::validate( $data['user'], 'request_' . $data['user']['t'] );
			if ( ! $validation_result->is_valid() ) {
				Logger::log_warning(
					'Request validation failed',
					array(
						'error_code'    => $validation_result->get_error_code(),
						'error_message' => $validation_result->get_error_message(),
						'event_type'    => $data['user']['t'] ?? 'unknown',
						'remote_addr'   => $remote_addr,
					)
				);
				return ProcessorResponse::error(
					'\SQMViews\RequestValidator::errorCode:' . $validation_result->get_error_code(),
					$validation_result->get_error_message(),
					400
				);
			}
		}

		$data['t'] = $data['user']['t'];
		unset( $data['user']['t'] );

		if ( 'test' === $data['t'] ) {
			// Skip diagnostics and directory creation on fast endpoint for security reasons.
			// phpcs:ignore Squiz.Commenting.InlineComment.InvalidEndChar
			// @phpstan-ignore booleanAnd.rightAlwaysTrue (constant may or may not be defined at runtime)
			if ( defined( 'SQMVIEWS_FAST_ENDPOINT' ) && SQMVIEWS_FAST_ENDPOINT ) {
				return ProcessorResponse::success(
					array(
						'ok'   => 'ok',
						'test' => array(
							'version'   => '1.0.0',
							'timestamp' => time(),
							'status'    => 'ok',
							'endpoint'  => 'fast',
							// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode -- Encoding nonce for response.
							'nonce'     => base64_encode( random_bytes( SODIUM_CRYPTO_SECRETBOX_NONCEBYTES ) ),
						),
					)
				);
			}

			sqm_views_prepare_dir( SQMVIEWS_DIR );
			sqm_views_prepare_dir( SQMVIEWS_RAW_DIR );
			sqm_views_prepare_dir( SQMVIEWS_PROCESSED_DIR );
			sqm_views_prepare_dir( SQMVIEWS_ARCHIVE_DIR );
			sqm_views_prepare_dir( SQMVIEWS_STALE_DIR );

			list( $issues, $tested ) = $this->run_diagnostics();

			return ProcessorResponse::success(
				array(
					'ok'   => 'ok',
					'test' => array(
						'version'   => '1.0.0',
						'timestamp' => time(),
						'status'    => count( $issues ) ? 'fail' : 'ok',
						'endpoint'  => 'api',
						'issues'    => $issues,
						'tested'    => $tested,
						// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode -- Encoding nonce for response.
						'nonce'     => base64_encode( random_bytes( SODIUM_CRYPTO_SECRETBOX_NONCEBYTES ) ),
					),
				)
			);
		}

		$nonce = null;
		if ( 'init' === $data['t'] ) {
			if ( empty( $data['user']['payload'] ) ) {
				Logger::log_warning(
					'Init request missing payload field',
					array(
						'remote_addr' => $remote_addr ?? 'unknown',
					)
				);

				return ProcessorResponse::error(
					'invalid_event_missing_payload',
					'Init request missing required payload field',
					400
				);
			}
			// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode -- Encoding nonce for response.
			$nonce                   = base64_encode( random_bytes( SODIUM_CRYPTO_SECRETBOX_NONCEBYTES ) );
			$data['server']['nonce'] = $nonce;
			$data['gid']             = generateUniqueId( 'r-' );
		}
		if ( 'init' !== $data['t'] ) {
			if ( empty( $data['user']['gid'] ) || empty( $data['user']['nonce'] ) ) {
				Logger::log_warning(
					'Request missing required gid or nonce',
					array(
						'action'      => $data['t'],
						'has_gid'     => ! empty( $data['user']['gid'] ),
						'has_nonce'   => ! empty( $data['user']['nonce'] ),
						'remote_addr' => $remote_addr ?? 'unknown',
					)
				);
				return ProcessorResponse::error(
					'invalid_event_missing_required_fields',
					'Missing required gid or nonce',
					400
				);
			}
			$data['gid'] = $data['user']['gid'];
			unset( $data['user']['gid'] );
		}

		$data['server']['ip']              = $remote_addr ?? 'unknown';
		$data['server']['HTTP_USER_AGENT'] = $http_user_agent ?? 'unknown';
		if ( defined( 'SQMVIEWS_USE_USER_TIME' ) && SQMVIEWS_USE_USER_TIME ) {
			$user_datetime          = new \DateTime( $data['user']['localTime']['time'], new \DateTimeZone( $data['user']['localTime']['tz'] ) );
			$data['server']['date'] = $user_datetime->format( 'c' );
			$user_datetime->setTimezone( new \DateTimeZone( 'UTC' ) );
			$data['ts'] = floatval( $user_datetime->format( 'U.u' ) );
		} else {
			$data['ts']             = microtime( true );
			$data['server']['date'] = gmdate( 'c' );
		}

		sqm_views_prepare_dir( SQMVIEWS_DIR );
		sqm_views_prepare_dir( SQMVIEWS_RAW_DIR );

		$filename = sqm_views_get_current_stat_filename( intval( $data['ts'] ), '.raw', 0 );

		$filepath = realpath( SQMVIEWS_RAW_DIR ) . '/' . $filename;

		$event_validation_result = RequestValidator::validate( $data, 'event' );
		if ( ! $event_validation_result->is_valid() ) {
			Logger::log_warning(
				'Event validation failed',
				array(
					'error_code'    => $event_validation_result->get_error_code(),
					'error_message' => $event_validation_result->get_error_message(),
					'event_type'    => $data['t'],
					'remote_addr'   => $remote_addr,
				)
			);
			return ProcessorResponse::error(
				'\SQMViews\RequestValidator::errorCode:' . $event_validation_result->get_error_code(),
				$event_validation_result->get_error_message(),
				400
			);
		}

		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents, WordPress.WP.AlternativeFunctions.json_encode_json_encode -- Direct file write for performance; json_encode for raw data.
		file_put_contents( $filepath, json_encode( $data ) . "\n", FILE_APPEND | LOCK_EX );

		$response = array(
			'ok' => 'ok',
		);
		if ( 'init' === $data['t'] ) {
			/**
			 * Filters the session timeout period in seconds.
			 *
			 * Only available when using WordPress REST API endpoint.
			 * Not available when using fast endpoint (sqm-views-pages.php).
			 *
			 * @since 1.0.0
			 *
			 * @param int $timeout Session inactivity timeout in seconds.
			 */
			// @phpstan-ignore booleanNot.alwaysFalse (constant may or may not be defined at runtime)
			$session_timeout = ( ! defined( 'SQMVIEWS_FAST_ENDPOINT' ) || ! SQMVIEWS_FAST_ENDPOINT )
				? apply_filters( 'sqm_views_session_timeout', SQMVIEWS_SESSION_INACTIVITY_TIMEOUT )
				: SQMVIEWS_SESSION_INACTIVITY_TIMEOUT;

			/**
			 * Filters the ping interval in seconds.
			 *
			 * Only available when using WordPress REST API endpoint.
			 * Not available when using fast endpoint (sqm-views-pages.php).
			 *
			 * @since 1.0.0
			 *
			 * @param int $interval Ping interval in seconds.
			 */
			// @phpstan-ignore booleanNot.alwaysFalse (constant may or may not be defined at runtime)
			$ping_interval = ( ! defined( 'SQMVIEWS_FAST_ENDPOINT' ) || ! SQMVIEWS_FAST_ENDPOINT )
				? apply_filters( 'sqm_views_ping_interval', SQMVIEWS_SESSION_PING_INTERVAL )
				: SQMVIEWS_SESSION_PING_INTERVAL;

			$response['config'] = array(
				'session_timeout' => $session_timeout,
				'ping_interval'   => $ping_interval,
			);
			$response['gid']    = $data['gid'];
			$response['nonce']  = $nonce;
		}

		return ProcessorResponse::success( $response );
	}

	/**
	 * Run diagnostics on storage directories.
	 *
	 * @return array{0: string[], 1: string[]} Array containing issues and tested items.
	 */
	public function run_diagnostics(): array {
		sqm_views_prepare_dir( SQMVIEWS_RAW_DIR );
		sqm_views_prepare_dir( SQMVIEWS_PROCESSED_DIR );

		$issues = array();
		$tested = array();

		$directories = array(
			array( SQMVIEWS_RAW_DIR, 'raw pageviews directory' ),
			array( SQMVIEWS_PROCESSED_DIR, 'processed pageviews directory' ),
			array( SQMVIEWS_ARCHIVE_DIR, 'archive pageviews directory' ),
			array( SQMVIEWS_STALE_DIR, 'incomplete pageviews directory' ),
		);

		$tmpfilename = sqm_views_get_current_stat_filename( null, '.test', 0 );

		foreach ( $directories as list( $dir, $label ) ) {
			$tmpfilepath                     = realpath( $dir ) . '/' . $tmpfilename;
			list( $dir_issues, $dir_tested ) = $this->diagnostics_directory( $tmpfilepath, $label );
			$issues                          = array_merge( $issues, $dir_issues );
			$tested                          = array_merge( $tested, $dir_tested );
		}

		return array( $issues, $tested );
	}

	/**
	 * Run diagnostics on a specific directory.
	 *
	 * @param string $tmpfilepath Path to temporary test file.
	 * @param string $tag         Directory label for reporting.
	 * @return array{0: string[], 1: string[]} Array containing issues and tested items.
	 *
	 * @throws \Random\RandomException If random_bytes fails.
	 */
	public function diagnostics_directory( string $tmpfilepath, string $tag ): array {
		sqm_load_wp_core();

		$issues = array();
		$tested = array();

		if ( dirname( $tmpfilepath ) ) {
			$tested[] = "$tag exists";
		} else {
			$issues[] = "$tag does not exist";
		}

		if ( file_exists( $tmpfilepath ) ) {
			$issues[] = "$tag, issue with directory";
		}
		$rnd = random_bytes( 10 );
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Direct file write for diagnostics.
		file_put_contents( $tmpfilepath, $rnd );
		if ( file_exists( $tmpfilepath ) ) {
			$tested[] = "$tag is writable";
		}
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents -- Reading local test file.
		if ( file_get_contents( $tmpfilepath ) !== $rnd ) {
			$issues[] = "$tag, issue reading directory data";
		} else {
			$tested[] = "$tag, test file written successfully";
		}
		wp_delete_file( $tmpfilepath );
		if ( file_exists( $tmpfilepath ) ) {
			$issues[] = "$tag, can't delete data";
		} else {
			$tested[] = "$tag, test file deleted successfully";
		}

		return array( $issues, $tested );
	}
}
