<?php
/**
 * SQMViewsEvent class file.
 *
 * @package SQMViews
 */

namespace SQMViews;

/**
 * Represents a single pageview event.
 */
class SQMViewsEvent {

	/**
	 * Raw event data.
	 *
	 * @var string|null
	 */
	private $raw = null;

	/**
	 * Global ID.
	 *
	 * @var string|null
	 */
	public $gid;

	/**
	 * Event type.
	 *
	 * @var string|null
	 */
	public $t;

	/**
	 * Timestamp.
	 *
	 * @var int|null
	 */
	public $ts;

	/**
	 * User URL.
	 *
	 * @var string|null
	 */
	public $user_url;

	/**
	 * User referrer.
	 *
	 * @var string|null
	 */
	public $user_referrer;

	/**
	 * User timestamp.
	 *
	 * @var string|null
	 */
	public $user_timestamp;

	/**
	 * User timezone.
	 *
	 * @var string|null
	 */
	public $user_timezone;

	/**
	 * User nonce.
	 *
	 * @var string|null
	 */
	public $user_nonce;

	/**
	 * User high frequency activity count.
	 *
	 * @var int
	 */
	public $user_activity_high_frequency;

	/**
	 * User low frequency activity count.
	 *
	 * @var int
	 */
	public $user_activity_low_frequency;

	/**
	 * User active time.
	 *
	 * @var float
	 */
	public $user_active_time;

	/**
	 * Server IP.
	 *
	 * @var string|null
	 */
	public $server_ip;

	/**
	 * Server user agent.
	 *
	 * @var string|null
	 */
	public $server_user_agent;

	/**
	 * Server nonce.
	 *
	 * @var string|null
	 */
	public $server_nonce;

	/**
	 * Server date.
	 *
	 * @var string|null
	 */
	public $server_date;

	/**
	 * Payload data.
	 *
	 * @var array<string, mixed>|null
	 */
	public $payload;

	/**
	 * Constructor.
	 *
	 * @param array<string, mixed> $data Event data array.
	 * @param string|null          $raw  Raw event string.
	 */
	public function __construct( array $data, $raw = null ) {
		$this->raw = $raw;

		$this->gid                          = $data['gid'] ?? null;
		$this->t                            = $data['t'] ?? null;
		$this->user_url                     = $data['user']['url'] ?? null;
		$this->user_referrer                = $data['user']['referrer'] ?? null;
		$this->user_timestamp               = $data['user']['localTime']['time'] ?? null;
		$this->user_timezone                = $data['user']['localTime']['tz'] ?? null;
		$this->user_nonce                   = $data['user']['nonce'] ?? null;
		$this->user_activity_high_frequency = $data['user']['activityCounts']['highFrequencyEvents'] ?? 0;
		$this->user_activity_low_frequency  = $data['user']['activityCounts']['lowFrequencyEvents'] ?? 0;
		$this->user_active_time             = $data['user']['activityCounts']['activeTime'] ?? 0;

		$this->server_ip         = $data['server']['ip'] ?? null;
		$this->server_user_agent = $data['server']['HTTP_USER_AGENT'] ?? null;
		$this->server_nonce      = $data['server']['nonce'] ?? null;
		$this->ts                = $data['ts'] ?? null;
		$this->server_date       = $data['server']['date'] ?? null;

		$this->payload = $data['payload'] ?? null;
	}

	/**
	 * Get raw event string.
	 *
	 * @return string|null Raw event string.
	 */
	public function get_raw() {
		return $this->raw;
	}

	/**
	 * Get event hash.
	 *
	 * @return string Event hash string.
	 */
	public function get_hash() {
		$gid = is_scalar( $this->gid ) ? $this->gid : 'invalid-gid';
		$t   = is_scalar( $this->t ) ? $this->t : 'invalid-t';
		$ts  = is_scalar( $this->ts ) ? $this->ts : 'invalid-ts';
		return $gid . '-' . $t . '-' . $ts;
	}
}
