<?php
/**
 * SQMViewsCron class file.
 *
 * @package SQMViews
 */

namespace SQMViews;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * WordPress Cron Integration.
 */
class SQMViewsCron {

	const CRON_HOOK = 'sqm_views_process_statistics_cron';

	/**
	 * Singleton instance.
	 *
	 * @var SQMViewsCron|null
	 */
	private static ?SQMViewsCron $instance = null;

	/**
	 * Constructor.
	 */
	private function __construct() {
		// Hook into WordPress cron.
		add_action( self::CRON_HOOK, array( $this, 'run_scheduled_processing' ) );

		// Add custom cron intervals if needed.
		add_filter( 'cron_schedules', array( $this, 'add_custom_cron_intervals' ) );

		// Add admin settings for cron management.
		add_action( 'admin_init', array( $this, 'register_cron_settings' ) );
	}

	/**
	 * Add custom cron intervals.
	 *
	 * @param array<string, array{interval: int, display: string}> $schedules Existing cron schedules.
	 * @return array<string, array{interval: int, display: string}> Modified cron schedules.
	 */
	public function add_custom_cron_intervals( array $schedules ): array {
		$schedules['every_15_minutes'] = array(
			'interval' => 15 * 60,
			'display'  => __( 'Every 15 Minutes', 'sqm-views' ),
		);

		$schedules['every_6_hours'] = array(
			'interval' => 6 * 60 * 60,
			'display'  => __( 'Every 6 Hours', 'sqm-views' ),
		);

		return $schedules;
	}

	/**
	 * Schedule the cron job.
	 *
	 * @return void
	 */
	public function schedule_cron(): void {
		if ( ! wp_next_scheduled( self::CRON_HOOK ) ) {
			$interval = get_option( SQMVIEWS_OPTION_CRON_INTERVAL, 'hourly' );
			wp_schedule_event( time(), $interval, self::CRON_HOOK );
			Logger::get_instance()->info( "Scheduled cron job with interval: {$interval}" );
		}
	}

	/**
	 * Unschedule the cron job.
	 *
	 * @return void
	 */
	public function unschedule_cron(): void {
		$timestamp = wp_next_scheduled( self::CRON_HOOK );
		if ( $timestamp ) {
			wp_unschedule_event( $timestamp, self::CRON_HOOK );
			Logger::get_instance()->info( 'Unscheduled cron job' );
		}
	}

	/**
	 * Reschedule cron with new interval.
	 *
	 * @param string $new_interval The new cron interval.
	 * @return void
	 */
	public function reschedule_cron( string $new_interval ): void {
		$this->unschedule_cron();
		wp_schedule_event( time(), $new_interval, self::CRON_HOOK );
		Logger::get_instance()->info( "Rescheduled cron job with interval: {$new_interval}" );
	}

	/**
	 * Run the scheduled processing.
	 *
	 * @return void
	 */
	public function run_scheduled_processing(): void {
		Logger::get_instance()->info( 'Running scheduled statistics processing' );
		$result = sqm_views_process_statistics( false );

		if ( ! $result['success'] ) {
			Logger::get_instance()->error( 'Cron error: ' . $result['error'] );
			Logger::get_instance()->debug( wp_json_encode( Logger::get_instance()->get_buffer() ) );
		}
	}

	/**
	 * Register admin settings for cron management.
	 *
	 * @return void
	 */
	public function register_cron_settings(): void {
		add_settings_section(
			'sqm_views_cron_section',
			__( 'Cron Settings', 'sqm-views' ),
			'__return_empty_string',
			'sqm-views'
		);

		add_settings_field(
			SQMVIEWS_OPTION_CRON_INTERVAL,
			__( 'Processing Interval', 'sqm-views' ),
			array( $this, 'cron_interval_callback' ),
			'sqm-views',
			'sqm_views_cron_section'
		);

		register_setting(
			'sqm-views',
			SQMVIEWS_OPTION_CRON_INTERVAL,
			array(
				'type'              => 'string',
				'sanitize_callback' => array( $this, 'sanitize_cron_interval' ),
			)
		);
	}

	/**
	 * Cron interval setting callback.
	 *
	 * @return void
	 */
	public function cron_interval_callback(): void {
		$current_interval = get_option( SQMVIEWS_OPTION_CRON_INTERVAL, 'hourly' );
		$intervals        = array(
			'every_15_minutes' => __( 'Every 15 Minutes', 'sqm-views' ),
			'hourly'           => __( 'Hourly', 'sqm-views' ),
			'every_6_hours'    => __( 'Every 6 Hours', 'sqm-views' ),
			'daily'            => __( 'Daily', 'sqm-views' ),
			'weekly'           => __( 'Weekly', 'sqm-views' ),
		);

		echo '<select name="sqm_views_cron_interval">';
		foreach ( $intervals as $key => $label ) {
			$selected = selected( $current_interval, $key, false );
			echo '<option value="' . esc_attr( $key ) . '" ' . esc_attr( $selected ) . '>' . esc_html( $label ) . '</option>';
		}
		echo '</select>';
		echo '<p class="description">' . esc_html__( 'How often should the statistics processing run?', 'sqm-views' ) . '</p>';
	}

	/**
	 * Sanitize cron interval setting.
	 *
	 * @param string $value The cron interval value to sanitize.
	 * @return string Sanitized cron interval.
	 */
	public function sanitize_cron_interval( $value ) {
		$valid_intervals = array( 'every_15_minutes', 'hourly', 'every_6_hours', 'daily', 'weekly' );

		if ( in_array( $value, $valid_intervals, true ) ) {
			// Reschedule if interval changed.
			$current_interval = get_option( SQMVIEWS_OPTION_CRON_INTERVAL, 'hourly' );
			if ( $value !== $current_interval ) {
				$this->reschedule_cron( $value );
			}
			return $value;
		}

		// Default fallback.
		return 'hourly';
	}

	/**
	 * Get next scheduled run time.
	 *
	 * @return string Formatted date or 'Not scheduled'.
	 */
	public function get_next_run_time() {
		$timestamp = wp_next_scheduled( self::CRON_HOOK );
		return $timestamp ? wp_date( 'Y-m-d H:i:s', $timestamp ) : __( 'Not scheduled', 'sqm-views' );
	}

	/**
	 * Get singleton instance.
	 *
	 * @return self
	 */
	public static function get_instance(): self {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Prevent cloning.
	 */
	private function __clone() {
	}

	/**
	 * Prevent unserialize.
	 */
	public function __wakeup() {
	}
}
