<?php
/**
 * RequestValidator class file.
 *
 * @package SQMViews
 */

namespace SQMViews;

// phpcs:disable Generic.Files.OneObjectStructurePerFile.MultipleFound -- ValidationResult is a small helper class tightly coupled to RequestValidator.

/**
 * Lightweight JSON request validator.
 *
 * Validates incoming tracking requests against declarative schemas.
 * Performance target: < 1ms per validation.
 *
 * Features:
 * - Type checking (string, integer, float, array, object)
 * - Required field validation
 * - Pattern matching (regex)
 * - Range validation (min/max)
 * - Nested object validation
 * - Enum validation
 * - Custom validators (URL, timezone, ISO 8601)
 */
class RequestValidator {
	/**
	 * Cached list of valid IANA timezones for fast lookup (keys are timezone names, values are indexes).
	 *
	 * @var array<string, int>|null
	 */
	private static ?array $valid_timezones = null;

	/**
	 * Validate request data against schema for specific document type.
	 *
	 * @param array<string, mixed> $data       Request data.
	 * @param string               $event_type Document type (request_init, request_ping, request_exit, request_timeout, request_test).
	 * @return ValidationResult
	 */
	public static function validate( array $data, string $event_type ): ValidationResult {
		$schema = DocumentSchemas::get_schema( $event_type );

		if ( null === $schema ) {
			return ValidationResult::error( 'unknown_event_type', "Unknown event type: $event_type" );
		}

		return self::validate_against_schema( $data, $schema, '' );
	}

	/**
	 * Validate data against schema recursively.
	 *
	 * @param mixed                $data   Data to validate.
	 * @param array<string, mixed> $schema Schema definition.
	 * @param string               $path   Current field path (for error messages).
	 * @return ValidationResult
	 */
	private static function validate_against_schema( $data, array $schema, string $path ): ValidationResult {
		// First, check for unknown fields (fields not in schema).
		$allowed_fields  = array_keys( $schema );
		$provided_fields = array_keys( $data );
		$unknown_fields  = array_diff( $provided_fields, $allowed_fields );

		if ( ! empty( $unknown_fields ) ) {
			$unknown_field = reset( $unknown_fields );
			$field_path    = $path ? "$path.$unknown_field" : $unknown_field;
			return ValidationResult::error(
				'unknown_field',
				"Unknown field: $field_path"
			);
		}

		// Then validate known fields.
		foreach ( $schema as $field_name => $rules ) {
			$field_path  = $path ? "$path.$field_name" : $field_name;
			$field_value = $data[ $field_name ] ?? null;

			// Check required fields.
			if ( ! isset( $data[ $field_name ] ) ) {
				if ( $rules['required'] ?? false ) {
					return ValidationResult::error(
						'missing_required_field',
						"Missing required field: $field_path"
					);
				}
				// Optional field not present - skip validation.
				continue;
			}

			// Check nullable fields.
			if ( ( $rules['nullable'] ?? false ) && '' === $field_value ) {
				// Empty string is allowed for nullable fields (like referrer).
				continue;
			}

			// Validate field type.
			$type_check = self::validate_type( $field_value, $rules['type'], $field_path );
			if ( ! $type_check->is_valid() ) {
				return $type_check;
			}

			// Validate enum values.
			if ( isset( $rules['enum'] ) ) {
				if ( ! in_array( $field_value, $rules['enum'], true ) ) {
					$allowed = implode( ', ', $rules['enum'] );
					return ValidationResult::error(
						'invalid_enum_value',
						"Invalid value for $field_path. Must be one of: $allowed"
					);
				}
			}

			// Validate string patterns.
			if ( isset( $rules['pattern'] ) && is_string( $field_value ) ) {
				if ( ! preg_match( $rules['pattern'], $field_value ) ) {
					return ValidationResult::error(
						'pattern_mismatch',
						"Field $field_path does not match required pattern"
					);
				}
			}

			// Validate string length.
			if ( is_string( $field_value ) ) {
				if ( isset( $rules['min_length'] ) && strlen( $field_value ) < $rules['min_length'] ) {
					return ValidationResult::error(
						'string_too_short',
						"Field $field_path is too short (min: {$rules['min_length']})"
					);
				}
				if ( isset( $rules['max_length'] ) && strlen( $field_value ) > $rules['max_length'] ) {
					return ValidationResult::error(
						'string_too_long',
						"Field $field_path is too long (max: {$rules['max_length']})"
					);
				}
			}

			// Validate numeric ranges.
			if ( is_numeric( $field_value ) ) {
				if ( isset( $rules['min'] ) && $field_value < $rules['min'] ) {
					return ValidationResult::error(
						'value_too_small',
						"Field $field_path is too small (min: {$rules['min']})"
					);
				}
				if ( isset( $rules['max'] ) && $field_value > $rules['max'] ) {
					return ValidationResult::error(
						'value_too_large',
						"Field $field_path is too large (max: {$rules['max']})"
					);
				}
			}

			// Recursively validate nested objects.
			if ( isset( $rules['fields'] ) && is_array( $field_value ) ) {
				$nested_result = self::validate_against_schema( $field_value, $rules['fields'], $field_path );
				if ( ! $nested_result->is_valid() ) {
					return $nested_result;
				}
			}

			// Custom validators.
			if ( 'url' === $field_name ) {
				$url_check = self::validate_url( $field_value, $field_path );
				if ( ! $url_check->is_valid() ) {
					return $url_check;
				}
			}

			if ( 'localTime.tz' === $field_path ) {
				$tz_check = self::validate_timezone( $field_value, $field_path );
				if ( ! $tz_check->is_valid() ) {
					return $tz_check;
				}
			}

			if ( 'localTime.time' === $field_path ) {
				$time_check = self::validate_iso8601( $field_value, $field_path );
				if ( ! $time_check->is_valid() ) {
					return $time_check;
				}
			}
		}

		return ValidationResult::success();
	}

	/**
	 * Validate field type.
	 *
	 * @param mixed  $value         Value to check.
	 * @param string $expected_type Expected type (string, integer, float, number, array, object).
	 * @param string $field_path    Field path for error messages.
	 * @return ValidationResult
	 */
	private static function validate_type( $value, string $expected_type, string $field_path ): ValidationResult {
		$actual_type = gettype( $value );

		switch ( $expected_type ) {
			case 'string':
				if ( ! is_string( $value ) ) {
					return ValidationResult::error(
						'invalid_type',
						"Field $field_path must be string, got $actual_type"
					);
				}
				break;

			case 'integer':
				if ( ! is_int( $value ) ) {
					return ValidationResult::error(
						'invalid_type',
						"Field $field_path must be integer, got $actual_type"
					);
				}
				break;

			case 'float':
				if ( ! is_float( $value ) ) {
					return ValidationResult::error(
						'invalid_type',
						"Field $field_path must be float, got $actual_type"
					);
				}
				break;

			case 'number':
				// Accept both integer and float.
				if ( ! is_numeric( $value ) ) {
					return ValidationResult::error(
						'invalid_type',
						"Field $field_path must be number, got $actual_type"
					);
				}
				break;

			case 'array':
				if ( ! is_array( $value ) || self::is_associative_array( $value ) ) {
					return ValidationResult::error(
						'invalid_type',
						"Field $field_path must be array, got $actual_type"
					);
				}
				break;

			case 'object':
				if ( ! is_array( $value ) || ! self::is_associative_array( $value ) ) {
					return ValidationResult::error(
						'invalid_type',
						"Field $field_path must be object, got $actual_type"
					);
				}
				break;

			default:
				return ValidationResult::error(
					'unknown_type',
					"Unknown type in schema: $expected_type"
				);
		}

		return ValidationResult::success();
	}

	/**
	 * Check if array is associative (object-like).
	 *
	 * @param array<mixed> $arr Array to check.
	 * @return bool True if associative.
	 */
	private static function is_associative_array( array $arr ): bool {
		if ( array() === $arr ) {
			return false;
		}
		return array_keys( $arr ) !== range( 0, count( $arr ) - 1 );
	}

	/**
	 * Validate URL format.
	 *
	 * Uses lightweight regex validation instead of parse_url for better performance.
	 * Validates that URL has proper http/https scheme and hostname.
	 *
	 * @param string $url        URL to validate.
	 * @param string $field_path Field path for error messages.
	 * @return ValidationResult
	 */
	private static function validate_url( string $url, string $field_path ): ValidationResult {
		// Basic length check.
		if ( strlen( $url ) > 2048 ) {
			return ValidationResult::error(
				'invalid_url',
				"Field $field_path is too long to be a valid URL"
			);
		}

		// Check for http or https scheme at the start.
		if ( ! preg_match( '/^https?:\/\//i', $url ) ) {
			return ValidationResult::error(
				'invalid_url_scheme',
				"Field $field_path must use http or https scheme"
			);
		}

		// Extract and validate hostname.
		// Pattern: http(s)://hostname/path.
		// This regex captures the hostname part after the scheme.
		if ( ! preg_match( '/^https?:\/\/([a-zA-Z0-9]([a-zA-Z0-9\-]{0,61}[a-zA-Z0-9])?(\.[a-zA-Z0-9]([a-zA-Z0-9\-]{0,61}[a-zA-Z0-9])?)*)(:[0-9]{1,5})?(\/|$)/i', $url, $matches ) ) {
			return ValidationResult::error(
				'invalid_url',
				"Field $field_path must include a valid hostname"
			);
		}

		return ValidationResult::success();
	}

	/**
	 * Validate IANA timezone.
	 *
	 * @param string $timezone   Timezone to validate.
	 * @param string $field_path Field path for error messages.
	 * @return ValidationResult
	 */
	private static function validate_timezone( string $timezone, string $field_path ): ValidationResult {
		// Lazy load timezone list (only once).
		if ( null === self::$valid_timezones ) {
			self::$valid_timezones = array_flip( timezone_identifiers_list() );
		}

		if ( ! isset( self::$valid_timezones[ $timezone ] ) ) {
			return ValidationResult::error(
				'invalid_timezone',
				"Field $field_path is not a valid IANA timezone"
			);
		}

		return ValidationResult::success();
	}

	/**
	 * Validate ISO 8601 datetime with milliseconds.
	 *
	 * Format: YYYY-MM-DDTHH:mm:ss.SSS
	 *
	 * @param string $datetime   Datetime to validate.
	 * @param string $field_path Field path for error messages.
	 * @return ValidationResult
	 */
	private static function validate_iso8601( string $datetime, string $field_path ): ValidationResult {
		// Try to parse the datetime.
		$dt = \DateTime::createFromFormat( 'Y-m-d\TH:i:s.u', $datetime );

		if ( false === $dt ) {
			return ValidationResult::error(
				'invalid_datetime',
				"Field $field_path is not a valid ISO 8601 datetime with milliseconds"
			);
		}

		// Additional validation: check if the parsed date matches input.
		// (catches invalid dates like 2025-02-30).
		$formatted = $dt->format( 'Y-m-d\TH:i:s.v' );
		if ( $formatted !== $datetime ) {
			return ValidationResult::error(
				'invalid_datetime',
				"Field $field_path contains invalid date/time values"
			);
		}

		return ValidationResult::success();
	}
}

/**
 * Validation result object.
 *
 * Immutable result of validation operation.
 */
class ValidationResult {

	/**
	 * Whether validation passed.
	 *
	 * @var bool
	 */
	private bool $valid;

	/**
	 * Error code if validation failed.
	 *
	 * @var string|null
	 */
	private ?string $error_code;

	/**
	 * Error message if validation failed.
	 *
	 * @var string|null
	 */
	private ?string $error_message;

	/**
	 * Constructor.
	 *
	 * @param bool        $valid         Whether validation passed.
	 * @param string|null $error_code    Error code.
	 * @param string|null $error_message Error message.
	 */
	private function __construct( bool $valid, ?string $error_code = null, ?string $error_message = null ) {
		$this->valid         = $valid;
		$this->error_code    = $error_code;
		$this->error_message = $error_message;
	}

	/**
	 * Create success result.
	 *
	 * @return self
	 */
	public static function success(): self {
		return new self( true );
	}

	/**
	 * Create error result.
	 *
	 * @param string $error_code    Error code.
	 * @param string $error_message Error message.
	 * @return self
	 */
	public static function error( string $error_code, string $error_message ): self {
		return new self( false, $error_code, $error_message );
	}

	/**
	 * Check if validation passed.
	 *
	 * @return bool
	 */
	public function is_valid(): bool {
		return $this->valid;
	}

	/**
	 * Get error code.
	 *
	 * @return string|null
	 */
	public function get_error_code(): ?string {
		return $this->error_code;
	}

	/**
	 * Get error message.
	 *
	 * @return string|null
	 */
	public function get_error_message(): ?string {
		return $this->error_message;
	}

	/**
	 * Convert to array.
	 *
	 * @return array<string, mixed>
	 */
	public function to_array(): array {
		if ( $this->valid ) {
			return array( 'valid' => true );
		}

		return array(
			'valid'   => false,
			'error'   => $this->error_code,
			'message' => $this->error_message,
		);
	}
}
