<?php
/**
 * ProcessorResponse class file.
 *
 * @package SQMViews
 */

namespace SQMViews;

/**
 * Response object for SQMViewsRapidProcessor.
 *
 * Provides a consistent interface for success and error responses.
 * Does not contain any WordPress dependencies or HTTP-related functions.
 * The caller is responsible for converting this to the appropriate response format.
 */
class ProcessorResponse {

	/**
	 * Whether this response represents an error.
	 *
	 * @var bool
	 */
	private bool $is_error;

	/**
	 * HTTP status code for the response.
	 *
	 * @var int
	 */
	private int $status_code;

	/**
	 * Response data array.
	 *
	 * @var array<string, mixed>
	 */
	private array $data;

	/**
	 * Error code if this is an error response.
	 *
	 * @var string|null
	 */
	private ?string $error_code;

	/**
	 * Error message if this is an error response.
	 *
	 * @var string|null
	 */
	private ?string $error_message;

	/**
	 * Constructor.
	 *
	 * @param bool                 $is_error      Whether this is an error response.
	 * @param int                  $status_code   HTTP status code.
	 * @param array<string, mixed> $data          Response data.
	 * @param string|null          $error_code    Error code.
	 * @param string|null          $error_message Error message.
	 */
	private function __construct(
		bool $is_error,
		int $status_code,
		array $data = array(),
		?string $error_code = null,
		?string $error_message = null
	) {
		$this->is_error      = $is_error;
		$this->status_code   = $status_code;
		$this->data          = $data;
		$this->error_code    = $error_code;
		$this->error_message = $error_message;
	}

	/**
	 * Create a success response.
	 *
	 * @param array<string, mixed> $data        Response data.
	 * @param int                  $status_code HTTP status code (default: 200).
	 * @return self
	 */
	public static function success( array $data = array(), int $status_code = 200 ): self {
		return new self( false, $status_code, $data );
	}

	/**
	 * Create an error response.
	 *
	 * @param string $code        Error code.
	 * @param string $message     Error message.
	 * @param int    $status_code HTTP status code (default: 400).
	 * @return self
	 */
	public static function error( string $code, string $message, int $status_code = 400 ): self {
		return new self( true, $status_code, array(), $code, $message );
	}

	/**
	 * Check if response is an error.
	 *
	 * @return bool
	 */
	public function is_error(): bool {
		return $this->is_error;
	}

	/**
	 * Check if response is successful.
	 *
	 * @return bool
	 */
	public function is_success(): bool {
		return ! $this->is_error;
	}

	/**
	 * Get HTTP status code.
	 *
	 * @return int
	 */
	public function get_status_code(): int {
		return $this->status_code;
	}

	/**
	 * Get response data.
	 *
	 * @return array<string, mixed>
	 */
	public function get_data(): array {
		return $this->data;
	}

	/**
	 * Get error code (null if not an error).
	 *
	 * @return string|null
	 */
	public function get_error_code(): ?string {
		return $this->error_code;
	}

	/**
	 * Get error message (null if not an error).
	 *
	 * @return string|null
	 */
	public function get_error_message(): ?string {
		return $this->error_message;
	}
}
