<?php
/**
 * Logger class file.
 *
 * @package SQMViews
 */

namespace SQMViews;

/**
 * Enhanced Logger class for SQMViews plugin.
 *
 * Provides structured logging with multiple levels, context support,
 * and WordPress integration.
 */
class Logger {

	// Log levels (PSR-3 inspired).
	const EMERGENCY = 'emergency';
	const ALERT     = 'alert';
	const CRITICAL  = 'critical';
	const ERROR     = 'error';
	const WARNING   = 'warning';
	const NOTICE    = 'notice';
	const INFO      = 'info';
	const DEBUG     = 'debug';

	/**
	 * Log buffer.
	 *
	 * @var array<int, array{timestamp: string, level: string, message: string, context: array<string, mixed>}>
	 */
	private $buffer = array();

	/**
	 * Singleton instance.
	 *
	 * @var Logger|null
	 */
	private static $instance = null;

	/**
	 * Verbose mode flag.
	 *
	 * @var bool
	 */
	private $verbose = false;

	/**
	 * Current log level.
	 *
	 * @var string
	 */
	private $log_level = self::CRITICAL;

	/**
	 * Log handler callback.
	 *
	 * @var callable
	 */
	private $do_log;

	/**
	 * Error handler callback.
	 *
	 * @var callable
	 */
	private $do_error;

	/**
	 * Warning handler callback.
	 *
	 * @var callable
	 */
	private $do_warning;

	/**
	 * Debug handler callback.
	 *
	 * @var callable
	 */
	private $do_debug;

	/**
	 * Log level hierarchy for filtering.
	 *
	 * @var array<string, int>
	 */
	private static $level_hierarchy = array(
		self::EMERGENCY => 0,
		self::ALERT     => 1,
		self::CRITICAL  => 2,
		self::ERROR     => 3,
		self::WARNING   => 4,
		self::NOTICE    => 5,
		self::INFO      => 6,
		self::DEBUG     => 7,
	);

	/**
	 * Constructor.
	 */
	private function __construct() {
		$this->setup_log_handlers();

		// Set log level from WordPress debug settings.
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			$this->log_level = self::DEBUG;
		}
	}

	/**
	 * Wrapper for error_log function to standardize logging.
	 *
	 * This method provides a standardized way to handle logging through WordPress' error_log function.
	 * Currently disabled for development purposes.
	 *
	 * @param mixed $msg The message to log. Can be any type that error_log accepts.
	 *
	 * @return void
	 * @phpstan-ignore void.pure (Intentional debug code)
	 */
	private function error_log_wrapper( mixed $msg ): void {
		// phpcs:ignore Squiz.PHP.CommentedOutCode.Found -- Intentional debug code.
		// error_log($msg);\\.
		unset( $msg );
	}

	/**
	 * Setup log handlers based on environment.
	 */
	private function setup_log_handlers(): void {
		if ( defined( 'WP_CLI' ) && WP_CLI ) {
			// WP-CLI environment.
			$this->do_log     = function ( $message ): void {
				\WP_CLI::line( $message );
			};
			$this->do_error   = function ( $message ): void {
				\WP_CLI::error( $message, false ); // False = don't exit.
			};
			$this->do_warning = function ( $message ): void {
				\WP_CLI::warning( $message );
			};
			$this->do_debug   = function ( $message ): void {
				\WP_CLI::debug( $message );
			};
		} else {
			// Standard WordPress environment.
			$this->do_log     = function ( $message ): void {
				$this->error_log_wrapper( '[SQMViews] ' . $message );
				unset( $message );
			};
			$this->do_error   = function ( $message ): void {
				$this->error_log_wrapper( '[SQMViews] [ERROR] ' . $message );
				unset( $message );
			};
			$this->do_warning = function ( $message ): void {
				$this->error_log_wrapper( '[SQMViews] [WARNING] ' . $message );
				unset( $message );
			};
			$this->do_debug   = function ( $message ): void {
				if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
					$this->error_log_wrapper( '[SQMViews] [DEBUG] ' . $message );
				}
				unset( $message );
			};
		}
	}

	/**
	 * Get singleton instance.
	 *
	 * @return Logger
	 */
	public static function get_instance(): Logger {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Generic log method.
	 *
	 * @param string               $level   Log level.
	 * @param string               $message Log message.
	 * @param array<string, mixed> $context Context data.
	 */
	public function log( $level, $message, array $context = array() ): void {
		if ( ! $this->should_log( $level ) ) {
			return;
		}

		$formatted_message = $this->format_message( $message, $context );
		$log_entry         = $this->create_log_entry( $level, $formatted_message, $context );

		$this->buffer[] = $log_entry;

		if ( $this->verbose ) {
			$this->write_log( $level, $formatted_message );
		}
	}

	/**
	 * Emergency: system is unusable.
	 *
	 * @param string               $message Log message.
	 * @param array<string, mixed> $context Context data.
	 */
	public function emergency( $message, array $context = array() ): void {
		$this->log( self::EMERGENCY, $message, $context );
	}

	/**
	 * Alert: action must be taken immediately.
	 *
	 * @param string               $message Log message.
	 * @param array<string, mixed> $context Context data.
	 */
	public function alert( $message, array $context = array() ): void {
		$this->log( self::ALERT, $message, $context );
	}

	/**
	 * Critical: critical conditions.
	 *
	 * @param string               $message Log message.
	 * @param array<string, mixed> $context Context data.
	 */
	public function critical( $message, array $context = array() ): void {
		$this->log( self::CRITICAL, $message, $context );
	}

	/**
	 * Error: error conditions.
	 *
	 * @param string               $message Log message.
	 * @param array<string, mixed> $context Context data.
	 */
	public function error( $message, array $context = array() ): void {
		$this->log( self::ERROR, $message, $context );
	}

	/**
	 * Warning: warning conditions.
	 *
	 * @param string               $message Log message.
	 * @param array<string, mixed> $context Context data.
	 */
	public function warning( $message, array $context = array() ): void {
		$this->log( self::WARNING, $message, $context );
	}

	/**
	 * Notice: normal but significant condition.
	 *
	 * @param string               $message Log message.
	 * @param array<string, mixed> $context Context data.
	 */
	public function notice( $message, array $context = array() ): void {
		$this->log( self::NOTICE, $message, $context );
	}

	/**
	 * Info: informational messages.
	 *
	 * @param string               $message Log message.
	 * @param array<string, mixed> $context Context data.
	 */
	public function info( $message, array $context = array() ): void {
		$this->log( self::INFO, $message, $context );
	}

	/**
	 * Debug: debug-level messages.
	 *
	 * @param string               $message Log message.
	 * @param array<string, mixed> $context Context data.
	 */
	public function debug( $message, array $context = array() ): void {
		$this->log( self::DEBUG, $message, $context );
	}

	/**
	 * Format message with context placeholders.
	 *
	 * @param string               $message Log message.
	 * @param array<string, mixed> $context Context data.
	 * @return string Formatted message.
	 */
	private function format_message( $message, array $context ): string {
		if ( empty( $context ) ) {
			return $message;
		}

		// Replace placeholders in message.
		$formatted = $message;
		foreach ( $context as $key => $value ) {
			if ( is_scalar( $value ) || ( is_object( $value ) && method_exists( $value, '__toString' ) ) ) {
				$formatted = str_replace( '{' . $key . '}', $value, $formatted );
			}
		}

		return $formatted;
	}

	/**
	 * Create structured log entry.
	 *
	 * @param string               $level   Log level.
	 * @param string               $message Log message.
	 * @param array<string, mixed> $context Context data.
	 * @return array{timestamp: string, level: string, message: string, context: array<string, mixed>} Log entry.
	 */
	private function create_log_entry( string $level, string $message, array $context ): array {
		return array(
			'timestamp' => gmdate( 'Y-m-d H:i:s' ),
			'level'     => strtoupper( $level ),
			'message'   => $message,
			'context'   => $context,
		);
	}

	/**
	 * Write log based on level.
	 *
	 * @param string $level   Log level.
	 * @param string $message Log message.
	 */
	private function write_log( $level, $message ): void {
		switch ( $level ) {
			case self::EMERGENCY:
			case self::ALERT:
			case self::CRITICAL:
			case self::ERROR:
				( $this->do_error )( $message );
				break;
			case self::WARNING:
			case self::NOTICE:
				( $this->do_warning )( $message );
				break;
			case self::DEBUG:
				( $this->do_debug )( $message );
				break;
			default:
				( $this->do_log )( $message );
				break;
		}
	}

	/**
	 * Check if message should be logged based on level.
	 *
	 * @param string $level Log level.
	 * @return bool Whether to log.
	 */
	private function should_log( $level ): bool {
		if ( $this->is_verbose() ) {
			return true;
		}
		if ( ! isset( self::$level_hierarchy[ $level ] ) ) {
			// Log unknown levels by default.
			return true;
		}

		$current_level_value = self::$level_hierarchy[ $this->log_level ] ?? 7;
		$message_level_value = self::$level_hierarchy[ $level ];

		return $message_level_value <= $current_level_value;
	}

	/**
	 * Set minimum log level.
	 *
	 * @param string $level Log level.
	 */
	public function set_log_level( $level ): void {
		if ( isset( self::$level_hierarchy[ $level ] ) ) {
			$this->log_level = $level;
		}
	}

	/**
	 * Get current log level.
	 *
	 * @return string Current log level.
	 */
	public function get_log_level(): string {
		return $this->log_level;
	}

	/**
	 * Get log buffer.
	 *
	 * @return array<int, array{timestamp: string, level: string, message: string, context: array<string, mixed>}> Log buffer.
	 */
	public function get_buffer(): array {
		return $this->buffer;
	}

	/**
	 * Clear log buffer.
	 */
	public function clear_buffer(): void {
		$this->buffer = array();
	}

	/**
	 * Set verbose mode.
	 *
	 * @param bool $verbose Verbose mode flag.
	 */
	public function set_verbose( bool $verbose ): void {
		$this->verbose = $verbose;
	}

	/**
	 * Check if verbose mode is enabled.
	 *
	 * @return bool Verbose mode flag.
	 */
	public function is_verbose(): bool {
		return $this->verbose;
	}

	/**
	 * Static convenience method for info logging.
	 *
	 * @param string               $message Log message.
	 * @param array<string, mixed> $context Context data.
	 */
	public static function log_info( $message, array $context = array() ): void {
		self::get_instance()->info( $message, $context );
	}

	/**
	 * Static convenience method for error logging.
	 *
	 * @param string               $message Log message.
	 * @param array<string, mixed> $context Context data.
	 */
	public static function log_error( $message, array $context = array() ): void {
		self::get_instance()->error( $message, $context );
	}

	/**
	 * Static convenience method for warning logging.
	 *
	 * @param string               $message Log message.
	 * @param array<string, mixed> $context Context data.
	 */
	public static function log_warning( $message, array $context = array() ): void {
		self::get_instance()->warning( $message, $context );
	}

	/**
	 * Static convenience method for debug logging.
	 *
	 * @param string               $message Log message.
	 * @param array<string, mixed> $context Context data.
	 */
	public static function log_debug( $message, array $context = array() ): void {
		self::get_instance()->debug( $message, $context );
	}
}
