<?php
/**
 * DocumentSchemas class file.
 *
 * @package SQMViews
 */

namespace SQMViews;

/**
 * Declarative schemas for all document types in the system.
 *
 * This class defines three types of documents:
 * 1. Request - JSON sent by client (user browser)
 * 2. Event - JSON saved to raw files (enriched with server data)
 * 3. Record - JSON saved to processed files (aggregated pageview data)
 *
 * Each schema defines:
 * - Field type (string, integer, float, number, array, object)
 * - Required/optional status
 * - Validation patterns (regex, min/max values)
 * - Nested object structures
 * - Field descriptions
 */
class DocumentSchemas {

	// =========================================================================
	// REQUEST SCHEMAS (from client)
	// =========================================================================

	/**
	 * Common fields in all request types
	 */
	private const REQUEST_COMMON_FIELDS = array(
		't'         => array(
			'type'        => 'string',
			'required'    => true,
			'enum'        => array( 'init', 'ping', 'exit', 'timeout', 'test' ),
			'description' => 'Event type',
		),
		'url'       => array(
			'type'        => 'string',
			'required'    => true,
			'pattern'     => '/^https?:\/\/.+/',
			'description' => 'Page URL',
		),
		'localTime' => array(
			'type'        => 'object',
			'required'    => true,
			'description' => 'Client local time with timezone',
			'fields'      => array(
				'time' => array(
					'type'        => 'string',
					'required'    => true,
					'pattern'     => '/^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}\.\d{3}$/',
					'description' => 'ISO 8601 with milliseconds',
				),
				'tz'   => array(
					'type'        => 'string',
					'required'    => true,
					'min_length'  => 3,
					'max_length'  => 100,
					'description' => 'IANA timezone',
				),
			),
		),
		'referrer'  => array(
			'type'        => 'string',
			'required'    => true,
			'nullable'    => true,
			'description' => 'HTTP referrer (can be empty)',
		),
		'client'    => array(
			'type'        => 'string',
			'required'    => true,
			'pattern'     => '/^\d+\.\d+\.\d+$/',
			'description' => 'Client version (e.g., "1.0.0")',
		),
	);

	/**
	 * Activity counts structure (used in PING, EXIT, TIMEOUT requests)
	 */
	private const REQUEST_ACTIVITY_COUNTS = array(
		'type'        => 'object',
		'required'    => true,
		'description' => 'User activity metrics',
		'fields'      => array(
			'highFrequencyEvents' => array(
				'type'        => 'integer',
				'required'    => true,
				'min'         => 0,
				'description' => 'Count of high-frequency events (mouse moves, scrolls)',
			),
			'lowFrequencyEvents'  => array(
				'type'        => 'integer',
				'required'    => true,
				'min'         => 0,
				'description' => 'Count of low-frequency events (clicks, keypresses)',
			),
			'activeTime'          => array(
				'type'        => 'number',
				'required'    => true,
				'min'         => 0,
				'description' => 'Active time in seconds (can be float)',
			),
		),
	);

	/**
	 * Get INIT request schema.
	 *
	 * @return array<string, mixed>
	 */
	public static function get_request_init_schema(): array {
		return array_merge(
			self::REQUEST_COMMON_FIELDS,
			array(
				'payload' => array(
					'type'        => 'string',
					'required'    => true,
					'min_length'  => 1,
					'description' => 'Base64 encrypted payload with tracking data',
				),
			)
		);
	}

	/**
	 * Get PING/EXIT/TIMEOUT request schema.
	 *
	 * @return array<string, mixed>
	 */
	public static function get_request_ping_schema(): array {
		return array_merge(
			self::REQUEST_COMMON_FIELDS,
			array(
				'gid'            => array(
					'type'        => 'string',
					'required'    => true,
					'pattern'     => '/^r-[a-zA-Z0-9@_]+-[a-zA-Z0-9@_]{10}$/',
					'description' => 'Global ID (format: r-{timestamp}-{random10})',
				),
				'nonce'          => array(
					'type'        => 'string',
					'required'    => true,
					'min_length'  => 24,
					'max_length'  => 64,
					'description' => 'Base64 encoded nonce',
				),
				'payload'        => array(
					'type'        => 'string',
					'required'    => false,
					'description' => 'Optional encrypted payload',
				),
				'activityCounts' => self::REQUEST_ACTIVITY_COUNTS,
			)
		);
	}

	/**
	 * Get TEST request schema.
	 *
	 * @return array<string, mixed>
	 */
	public static function get_request_test_schema(): array {
		return array(
			't' => self::REQUEST_COMMON_FIELDS['t'],
		);
	}

	// =========================================================================
	// EVENT SCHEMAS (saved to raw files)
	// =========================================================================

	/**
	 * Get Event schema (saved to .raw.jsonl files).
	 *
	 * Event = Request + server-side data
	 *
	 * @return array<string, mixed>
	 */
	public static function get_event_schema(): array {
		return array(
			't'      => array(
				'type'        => 'string',
				'required'    => true,
				'enum'        => array( 'init', 'ping', 'exit', 'timeout', 'test' ),
				'description' => 'Event type',
			),
			'gid'    => array(
				'type'        => 'string',
				'required'    => true,
				'pattern'     => '/^r-[a-zA-Z0-9@_]+-[a-zA-Z0-9@_]{10}$/',
				'description' => 'Global ID (generated for INIT, from request for others)',
			),
			'ts'     => array(
				'type'        => 'number',
				'required'    => true,
				'description' => 'Server timestamp (Unix timestamp with microseconds)',
			),
			'user'   => array(
				'type'        => 'object',
				'required'    => true,
				'description' => 'User (client) data',
				'fields'      => array(
					'url'            => array(
						'type'        => 'string',
						'required'    => true,
						'description' => 'Page URL',
					),
					'referrer'       => array(
						'type'        => 'string',
						'required'    => true,
						'nullable'    => true,
						'description' => 'HTTP referrer',
					),
					'localTime'      => array(
						'type'        => 'object',
						'required'    => true,
						'description' => 'Client local time',
						'fields'      => array(
							'time' => array(
								'type'        => 'string',
								'required'    => true,
								'description' => 'ISO 8601 timestamp',
							),
							'tz'   => array(
								'type'        => 'string',
								'required'    => true,
								'description' => 'IANA timezone',
							),
						),
					),
					'client'         => array(
						'type'        => 'string',
						'required'    => false,
						'description' => 'Client version',
					),
					'nonce'          => array(
						'type'        => 'string',
						'required'    => false,
						'description' => 'Client nonce (for PING/EXIT/TIMEOUT)',
					),
					'activityCounts' => array(
						'type'        => 'object',
						'required'    => false,
						'description' => 'Activity metrics (for PING/EXIT/TIMEOUT)',
						'fields'      => array(
							'highFrequencyEvents' => array(
								'type'        => 'integer',
								'required'    => true,
								'description' => 'High-frequency event count',
							),
							'lowFrequencyEvents'  => array(
								'type'        => 'integer',
								'required'    => true,
								'description' => 'Low-frequency event count',
							),
							'activeTime'          => array(
								'type'        => 'number',
								'required'    => true,
								'description' => 'Active time in seconds',
							),
						),
					),
					'payload'        => array(
						'type'        => 'string',
						'required'    => false,
						'description' => 'Encrypted payload (for INIT, optional for others)',
					),
				),
			),
			'server' => array(
				'type'        => 'object',
				'required'    => true,
				'description' => 'Server-side data',
				'fields'      => array(
					'ip'              => array(
						'type'        => 'string',
						'required'    => true,
						'description' => 'Client IP address',
					),
					'HTTP_USER_AGENT' => array(
						'type'        => 'string',
						'required'    => true,
						'description' => 'User agent string',
					),
					'nonce'           => array(
						'type'        => 'string',
						'required'    => false,
						'description' => 'Server-generated nonce (for INIT)',
					),
					'date'            => array(
						'type'        => 'string',
						'required'    => true,
						'description' => 'Server date (ISO 8601)',
					),
				),
			),
		);
	}

	// =========================================================================
	// RECORD SCHEMAS (saved to processed files and database)
	// =========================================================================

	/**
	 * Get Record schema (saved to .processed.jsonl files and database).
	 *
	 * Record = aggregated pageview data from multiple events
	 *
	 * @return array<string, mixed>
	 */
	public static function get_record_schema(): array {
		return array(
			'gid'           => array(
				'type'        => 'string',
				'required'    => true,
				'description' => 'Global ID',
			),
			'tid'           => array(
				'type'        => 'string',
				'required'    => false,
				'description' => 'Trackable ID (derived from payload)',
			),
			'eid'           => array(
				'type'        => 'integer',
				'required'    => false,
				'description' => 'Event ID (from payload)',
			),
			'user_moment'   => array(
				'type'        => 'string',
				'required'    => true,
				'description' => 'User local timestamp (ISO 8601)',
			),
			'user_timezone' => array(
				'type'        => 'string',
				'required'    => true,
				'description' => 'User timezone (IANA)',
			),
			'utc_moment'    => array(
				'type'        => 'string',
				'required'    => true,
				'description' => 'UTC timestamp (ISO 8601)',
			),
			'url'           => array(
				'type'        => 'string',
				'required'    => true,
				'description' => 'Page URL',
			),
			'referrer'      => array(
				'type'        => 'string',
				'required'    => true,
				'nullable'    => true,
				'description' => 'HTTP referrer',
			),
			'on_page'       => array(
				'type'        => 'number',
				'required'    => true,
				'description' => 'Time on page in seconds',
			),
			'active'        => array(
				'type'        => 'number',
				'required'    => true,
				'description' => 'Active time in seconds',
			),
			'high_freq'     => array(
				'type'        => 'integer',
				'required'    => true,
				'description' => 'High-frequency event count',
			),
			'low_freq'      => array(
				'type'        => 'integer',
				'required'    => true,
				'description' => 'Low-frequency event count',
			),
			'payload'       => array(
				'type'        => 'object',
				'required'    => false,
				'description' => 'Decrypted payload data',
				'fields'      => array(
					'event' => array(
						'type'        => 'integer',
						'required'    => false,
						'description' => 'Event ID',
					),
					'group' => array(
						'type'        => 'string',
						'required'    => false,
						'description' => 'Trackable group (e.g., "content")',
					),
					'type'  => array(
						'type'        => 'string',
						'required'    => false,
						'description' => 'Trackable type (e.g., "post")',
					),
					'id'    => array(
						'type'        => 'integer',
						'required'    => false,
						'description' => 'WordPress post ID',
					),
					'altid' => array(
						'type'        => 'string',
						'required'    => false,
						'nullable'    => true,
						'description' => 'Alternative ID',
					),
				),
			),
			'exit'          => array(
				'type'        => 'string',
				'required'    => true,
				'enum'        => array( 'exit', 'timeout', 'ping' ),
				'description' => 'Exit type',
			),
			'trackable'     => array(
				'type'        => 'object',
				'required'    => false,
				'description' => 'Processed trackable data',
				'fields'      => array(
					'tgroup' => array(
						'type'        => 'string',
						'required'    => false,
						'description' => 'Trackable group',
					),
					'ttype'  => array(
						'type'        => 'string',
						'required'    => false,
						'description' => 'Trackable type',
					),
					'wpid'   => array(
						'type'        => 'integer',
						'required'    => false,
						'description' => 'WordPress ID',
					),
					'altid'  => array(
						'type'        => 'string',
						'required'    => false,
						'nullable'    => true,
						'description' => 'Alternative ID',
					),
				),
			),
		);
	}

	// =========================================================================
	// HELPER METHODS
	// =========================================================================

	/**
	 * Get schema for specific document type.
	 *
	 * @param string $type Document type: request_init, request_ping, event, record.
	 * @return array<string, mixed>|null Schema definition or null if unknown type.
	 */
	public static function get_schema( string $type ): ?array {
		switch ( $type ) {
			case 'request_' . SQMVIEWS_EVENTS_INIT:
				return self::get_request_init_schema();
			case 'request_' . SQMVIEWS_EVENTS_PING:
			case 'request_' . SQMVIEWS_EVENTS_EXIT:
			case 'request_' . SQMVIEWS_EVENTS_TIMEOUT:
				return self::get_request_ping_schema();
			case 'request_' . SQMVIEWS_EVENTS_TEST:
				return self::get_request_test_schema();
			case 'event':
				return self::get_event_schema();
			case 'record':
				return self::get_record_schema();
			default:
				return null;
		}
	}

	/**
	 * Get all available schema types.
	 *
	 * @return string[] List of schema type names.
	 */
	public static function get_available_types(): array {
		return array(
			'request_' . SQMVIEWS_EVENTS_INIT,
			'request_' . SQMVIEWS_EVENTS_PING,
			'request_' . SQMVIEWS_EVENTS_EXIT,
			'request_' . SQMVIEWS_EVENTS_TIMEOUT,
			'request_' . SQMVIEWS_EVENTS_TEST,
			'event',
			'record',
		);
	}
}
