<?php
/**
 * Statistics processing functions file.
 *
 * @package SQMViews
 */

namespace SQMViews;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

require_once __DIR__ . '/src/RequestValidator.php';
require_once __DIR__ . '/src/DocumentSchemas.php';

// phpcs:disable Universal.Files.SeparateFunctionsFromOO.Mixed -- Functions and class are tightly coupled in this processing module.

/**
 * Parameters class for statistics processing.
 *
 * Holds configuration and callbacks for the statistics processor.
 */
class SQMViewsStatProcessorParams {
	/**
	 * Unique run identifier.
	 *
	 * @var string
	 */
	public $run_id;

	/**
	 * Generator for cold files to process.
	 *
	 * @var \Generator
	 */
	public $cold_files_generator;

	/**
	 * Generator for already processed files.
	 *
	 * @var \Generator
	 */
	public $processed_files_generator;

	/**
	 * Callback to process complete records.
	 *
	 * @var callable
	 */
	public $complete_records_processor;

	/**
	 * Callback to process incomplete records.
	 *
	 * @var callable
	 */
	public $incomplete_records_processor;

	/**
	 * Callback to process invalid records.
	 *
	 * @var callable
	 */
	public $invalid_records_processor;

	/**
	 * Callback to write malformed events.
	 *
	 * @var callable
	 */
	public $malformed_event_writer;


	/**
	 * Constructor.
	 *
	 * @param string     $run_id                       Unique run identifier.
	 * @param \Generator $cold_files_generator         Generator for cold files.
	 * @param \Generator $processed_files_generator    Generator for processed files.
	 * @param callable   $complete_records_processor   Callback for complete records.
	 * @param callable   $incomplete_records_processor Callback for incomplete records.
	 * @param callable   $invalid_records_processor    Callback for invalid records.
	 * @param callable   $malformed_event_writer       Callback for malformed events.
	 */
	public function __construct(
		$run_id,
		$cold_files_generator,
		$processed_files_generator,
		$complete_records_processor,
		$incomplete_records_processor,
		$invalid_records_processor,
		$malformed_event_writer,
	) {
		$this->run_id                    = $run_id;
		$this->cold_files_generator      = $cold_files_generator;
		$this->processed_files_generator = $processed_files_generator;

		$this->complete_records_processor   = $complete_records_processor;
		$this->incomplete_records_processor = $incomplete_records_processor;
		$this->invalid_records_processor    = $invalid_records_processor;
		$this->malformed_event_writer       = $malformed_event_writer;
	}
}

/**
 * Main function that can be called by both cron and WP-CLI.
 * Add this to your main plugin file or a separate functions file.
 *
 * @since 1.0.0
 *
 * @param bool $verbose Enable verbose logging.
 * @return array<string, mixed> Processing result with success status and message.
 */
function sqm_views_process_statistics( $verbose ) {
	$old_verbosity = Logger::get_instance()->is_verbose();
	\SQMViews\Logger::get_instance()->set_verbose( $verbose );

	/**
	 * Fires before statistics processing begins
	 *
	 * @since 1.0.0
	 *
	 * @param bool $verbose Whether verbose logging is enabled
	 */
	do_action( 'sqm_views_before_processing', $verbose );

	sqm_views_prepare_dir( SQMVIEWS_DIR );
	sqm_views_prepare_dir( SQMVIEWS_RAW_DIR );
	sqm_views_prepare_dir( SQMVIEWS_PROCESSED_DIR );
	sqm_views_prepare_dir( SQMVIEWS_ARCHIVE_DIR );
	sqm_views_prepare_dir( SQMVIEWS_STALE_DIR );

	// Check if another process is already running (lock mechanism).
	$lock_timeout    = 60; // Seconds.
	$processed_files = glob( realpath( SQMVIEWS_PROCESSED_DIR ) . '/*.processed.jsonl' );
	if ( false !== $processed_files ) {
		$current_time = time();
		foreach ( $processed_files as $file ) {
			$file_mtime = filemtime( $file );
			if ( false !== $file_mtime && ( $current_time - $file_mtime ) < $lock_timeout ) {
				Logger::get_instance()->info( 'Another processing task is already running. Exiting.' );
				\SQMViews\Logger::get_instance()->set_verbose( $old_verbosity );
				return array(
					'success' => true,
					'message' => 'Skipped: Another processing task is already running.',
					'locked'  => true,
				);
			}
		}
	}

	$run_id                  = generateUniqueId( 'cron-' );
	$processed_records_path  = realpath( SQMVIEWS_PROCESSED_DIR ) . '/' . "$run_id.processed.jsonl";
	$incomplete_records_path = realpath( SQMVIEWS_RAW_DIR ) . '/' . "$run_id.incomplete.jsonl";
	$invalid_records_path    = realpath( SQMVIEWS_ARCHIVE_DIR ) . '/' . "$run_id.invalid.jsonl";
	$stale_records_path      = realpath( SQMVIEWS_STALE_DIR ) . '/' . ( explode( 'T', sqm_views_get_current_stat_filename() )[0] ) . '.stale.jsonl';
	$malformed_records_path  = realpath( SQMVIEWS_STALE_DIR ) . '/' . ( explode( 'T', sqm_views_get_current_stat_filename() )[0] ) . '.malformed.jsonl';

	/**
	 * Generator for incomplete files.
	 *
	 * @return \Generator Yields paths to incomplete files.
	 * @phpstan-ignore function.inner (inner named functions work at runtime, PHPStan limitation)
	 */
	function incomplete_files_generator() {
		foreach ( glob( realpath( SQMVIEWS_RAW_DIR ) . '/*.incomplete.jsonl' ) as $path ) {
			yield $path;
		}
	}

	/**
	 * Generator for cold files (incomplete + raw).
	 *
	 * @return \Generator Yields paths to cold files.
	 * @phpstan-ignore function.inner (inner named functions work at runtime, PHPStan limitation)
	 */
	function cold_files_generator() {
		yield from incomplete_files_generator(); // @phpstan-ignore function.notFound (inner function)
		$files = glob( realpath( SQMVIEWS_RAW_DIR ) . '/*.raw.jsonl' );
		sort( $files );
		if ( $files ) {
			$hot_period =
			strtotime( sqm_views_file_get_period( sqm_views_get_current_stat_filename( null, '.raw', 0 ) ) ) - SQMVIEWS_FILE_COOLDOWN_PERIOD;
			foreach ( $files as $path ) {
				$file_period = strtotime( sqm_views_file_get_period( $path ) );
				if ( $file_period && $hot_period && $file_period < $hot_period ) {
					yield $path;
				}
			}
		}
	}

	/**
	 * Generator for processed files (leftovers).
	 *
	 * @return \Generator Yields paths to processed files.
	 * @phpstan-ignore function.inner (inner named functions work at runtime, PHPStan limitation)
	 */
	function processed_files_generator() {
		$leftovers = glob( realpath( SQMVIEWS_PROCESSED_DIR ) . '/*.processed.jsonl' );
		sort( $leftovers );
		foreach ( $leftovers as $leftover ) {
			yield $leftover;
		}
	}

	$complete_records_processor = function ( $record ) use ( $processed_records_path ) {
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Performance-critical batch processing.
		file_put_contents( $processed_records_path, $record->get_processed() . "\n", FILE_APPEND | LOCK_EX );
	};

	$incomplete_records_processor = function ( $record, $reference_time = null ) use ( $incomplete_records_path, $stale_records_path ) {
		$data = implode( "\n", $record->get_raw_events() ) . "\n";
		// Use reference time (from file timestamp) if provided, otherwise use current time.
		$current_time = null !== $reference_time ? $reference_time : time();

		// Check if record is still active (not stale) based on reference time.
		if ( $record->get_last_activity_timestamp() + SQMVIEWS_SESSION_INACTIVITY_TIMEOUT >= $current_time ) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Performance-critical batch processing.
			file_put_contents( $incomplete_records_path, $data, FILE_APPEND | LOCK_EX );
		} else {
			// Record is truly stale - write to stale file.
			if ( function_exists( 'gzopen' ) ) {
				$gz = gzopen( $stale_records_path . '.gz', file_exists( $stale_records_path . '.gz' ) ? 'a9' : 'w9' );
				if ( $gz ) {
					gzwrite( $gz, $data );
					gzclose( $gz );
					return;
				}
			}
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Performance-critical batch processing.
			file_put_contents( $stale_records_path, $data, FILE_APPEND | LOCK_EX );
		}
	};

	$invalid_records_processor = function ( $record ) use ( $invalid_records_path ) {
		$data = implode( "\n", $record->get_raw_events() ) . "\n";
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Performance-critical batch processing.
		file_put_contents( $invalid_records_path, $data, FILE_APPEND | LOCK_EX );
	};

	$malformed_event_writer = function ( $line ) use ( $malformed_records_path ) {
		if ( function_exists( 'gzopen' ) ) {
			$gz = gzopen( $malformed_records_path . '.gz', file_exists( $malformed_records_path . '.gz' ) ? 'a9' : 'w9' );
			if ( $gz ) {
				gzwrite( $gz, $line . "\n" );
				gzclose( $gz );
				return;
			}
		}
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Performance-critical batch processing.
		file_put_contents( $malformed_records_path, $line . "\n", FILE_APPEND | LOCK_EX );
	};

	$params = new SQMViewsStatProcessorParams(
		$run_id,
		cold_files_generator(), // @phpstan-ignore function.notFound (inner function)
		processed_files_generator(), // @phpstan-ignore function.notFound (inner function)
		$complete_records_processor,
		$incomplete_records_processor,
		$invalid_records_processor,
		$malformed_event_writer,
	);

	$sqm_views_process_statistics_runner = sqm_views_process_statistics_runner( $params );
	\SQMViews\Logger::get_instance()->set_verbose( $old_verbosity );
	sqm_views_compact_archive();
	sqm_views_daily_aggregation();

	// Invalidate chart cache after processing new data.
	if ( function_exists( '\SQMViews\sqm_views_invalidate_chart_cache' ) ) {
		\SQMViews\sqm_views_invalidate_chart_cache();
	}

	/**
	 * Fires after statistics processing completes
	 *
	 * @since 1.0.0
	 *
	 * @param array $result Processing result with success status and message
	 */
	do_action( 'sqm_views_after_processing', $sqm_views_process_statistics_runner );

	return $sqm_views_process_statistics_runner;
}

/**
 * Runner function for statistics processing.
 *
 * @since 1.0.0
 *
 * @param SQMViewsStatProcessorParams $params Processing parameters.
 * @return array<string, mixed> Processing result with success status and message.
 */
function sqm_views_process_statistics_runner( SQMViewsStatProcessorParams $params ) {
	$logger = Logger::get_instance();
	try {
		$logger->info( 'Starting statistics processing' );

		// Execution timeout handling.
		$max_execution_time = ini_get( 'max_execution_time' );
		$start_time         = time();
		$timeout_threshold  = 0; // 0 means no timeout check.
		$timeout_reached    = false;

		if ( $max_execution_time > 0 ) {
			// Use 80% of max execution time as threshold (20% buffer).
			$timeout_threshold = $max_execution_time * 0.8;
			$logger->info( 'Execution timeout threshold: ' . $timeout_threshold . ' seconds' );
		} else {
			$logger->info( 'No execution timeout limit (unlimited execution time)' );
		}

		// Memory limit handling.
		$memory_limit_str = ini_get( 'memory_limit' );
		$memory_threshold = 0; // 0 means no memory check.
		$memory_reached   = false;

		if ( $memory_limit_str && '-1' !== $memory_limit_str ) {
			// Convert memory_limit to bytes.
			$memory_limit = sqm_views_parse_memory_limit( $memory_limit_str );
			if ( $memory_limit > 0 ) {
				// Use 80% of memory limit as threshold (20% buffer).
				$memory_threshold = $memory_limit * 0.8;
				$logger->info( 'Memory threshold: ' . round( $memory_threshold / 1024 / 1024, 2 ) . ' MB' );
			}
		} else {
			$logger->info( 'No memory limit (unlimited memory)' );
		}

		$complete_records   = array();
		$incomplete_records = array();
		$invalid_records    = array();

		$processed_files     = array();
		$last_file_timestamp = null;

		foreach ( $params->cold_files_generator as $path ) {
			$records = sqm_views_process_cold_file( $path, $incomplete_records, $params->malformed_event_writer );
			if ( false === $records ) {
				$logger->error( 'Failed to process file: ' . $path );
				continue;
			}
			$processed_files[]  = $path;
			$incomplete_records = array();

			// Extract timestamp from filename for use in stale detection.
			$file_period         = sqm_views_file_get_period( $path );
			$last_file_timestamp = strtotime( str_replace( 'T', ' ', $file_period ) );

			/**
			 * Record being processed.
			 *
			 * @var SQMViewsRecord $record
			 */
			foreach ( $records as $record ) {
				/**
				 * Fires after a raw record is processed
				 *
				 * @since 1.0.0
				 *
				 * @param SQMViewsRecord $record The processed record
				 */
				do_action( 'sqm_views_record_processed', $record );

				if ( ! $record->is_complete() ) {
					$incomplete_records[ $record->gid ] = $record;
				} elseif ( ! $record->is_valid() ) {
						$invalid_records[ $record->gid ] = $record;
				} else {
					$complete_records[ $record->gid ] = $record;
				}
			}

			foreach ( $complete_records as $record ) {
				( $params->complete_records_processor )( $record );
			}
			$complete_records = array();

			foreach ( $invalid_records as $record ) {
				( $params->invalid_records_processor )( $record );
			}
			$invalid_records = array();

			// Check if we're approaching execution timeout.
			if ( $timeout_threshold > 0 ) {
				$elapsed = time() - $start_time;
				if ( $elapsed >= $timeout_threshold ) {
					$logger->warning(
						'Approaching execution timeout, breaking from file processing loop',
						array(
							'elapsed'          => $elapsed,
							'threshold'        => $timeout_threshold,
							'files_processed'  => count( $processed_files ),
							'incomplete_count' => count( $incomplete_records ),
						)
					);
					$timeout_reached = true;
					break;
				}
			}

			// Check if we're approaching memory limit.
			if ( $memory_threshold > 0 ) {
				$current_memory = memory_get_usage( true );
				if ( $current_memory >= $memory_threshold ) {
					$logger->warning(
						'Approaching memory limit, breaking from file processing loop',
						array(
							'current_memory'   => round( $current_memory / 1024 / 1024, 2 ) . ' MB',
							'threshold'        => round( $memory_threshold / 1024 / 1024, 2 ) . ' MB',
							'files_processed'  => count( $processed_files ),
							'incomplete_count' => count( $incomplete_records ),
						)
					);
					$memory_reached = true;
					break;
				}
			}
		}

		foreach ( $incomplete_records as $record ) {
			// Use last processed file's timestamp as reference time for historical data.
			( $params->incomplete_records_processor )( $record, $last_file_timestamp );
		}

		foreach ( $processed_files as $path ) {
			archive_file( $path );
		}

		foreach ( $params->processed_files_generator as $leftovers_path ) {
			if ( false !== process_records_file_to_database( $leftovers_path ) ) {
				// phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged -- Intentional cleanup, file may already be deleted.
				@wp_delete_file( $leftovers_path );
			}
		}

		$processed_files_count = count( $processed_files );
		if ( $processed_files_count ) {
			$basenames = array_map( 'basename', $processed_files );
			if ( $processed_files_count > 10 ) {
				$first_five = array_slice( $basenames, 0, 5 );
				$last_five  = array_slice( $basenames, -5 );
				$files_list = implode( ', ', $first_five ) . ', ..., ' . implode( ', ', $last_five );
			} else {
				$files_list = implode( ', ', $basenames );
			}
			$logger->info( 'Files to process: ' . $files_list );
		} else {
			$logger->info( 'No files to process.' );
		}

		if ( $timeout_reached ) {
			$logger->info( 'Processing stopped due to timeout. Remaining files will be processed in next run.' );
		}

		if ( $memory_reached ) {
			$logger->info( 'Processing stopped due to memory limit. Remaining files will be processed in next run.' );
		}

		update_option( SQMVIEWS_OPTION_LAST_RUN, time() );

		$success_message = $timeout_reached
			? sprintf(
				/* translators: %d: number of files processed */
				__( 'Partially processed %d files (stopped due to timeout)', 'sqm-views' ),
				$processed_files_count
			)
			: ( $memory_reached
				? sprintf(
					/* translators: %d: number of files processed */
					__( 'Partially processed %d files (stopped due to memory limit)', 'sqm-views' ),
					$processed_files_count
				)
				: sprintf(
					/* translators: %d: number of files processed */
					__( 'Successfully processed %d files', 'sqm-views' ),
					$processed_files_count
				) );

		return array(
			'success' => true,
			'message' => $success_message,
		);
	} catch ( \Exception $e ) {
		$logger->error( 'sqm_views_process_statistics error: ' . $e->getMessage() );

		return array(
			'success' => false,
			'error'   => $e->getMessage(),
		);
	}
}

/**
 * Archives a processed file to the archive directory.
 *
 * @since 1.0.0
 *
 * @param string $file2archive Path to the file to archive.
 * @return void
 */
function archive_file( $file2archive ) {
	$logger = \SQMViews\Logger::get_instance();
	if ( file_exists( $file2archive ) ) {
		// Use WordPress filesystem API.
		require_once ABSPATH . 'wp-admin/includes/file.php';
		WP_Filesystem();
		global $wp_filesystem;

		$destination = trailingslashit( SQMVIEWS_ARCHIVE_DIR ) . basename( $file2archive );
		if ( ! $wp_filesystem->move( $file2archive, $destination, true ) ) {
			$logger->error( 'Failed to move file to archive: ' . $file2archive );
		}
	}
}

/**
 * Aggregates daily statistics from records.
 *
 * @since 1.0.0
 *
 * @return bool True on success, false on failure.
 */
function sqm_views_daily_aggregation() {
	global $wpdb;
	$logger = \SQMViews\Logger::get_instance();

	try {
		// We can not cache this query as data was just updated by the cron.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Aggregation query that cannot be cached.
		$result = $wpdb->query(
			// phpcs:ignore Generic.WhiteSpace.ScopeIndent.IncorrectExact -- SQL string formatting.
			/* @lang MySQL */
			"REPLACE INTO {$wpdb->prefix}sqm_views_daily (
				`date`,
				`eid`,
				`tid`,
				`on_page`,
				`active`,
				`high_freq`,
				`low_freq`,
				`count`
			)
			SELECT
				DATE(utc_moment) as `date`,
				eid,
				tid,
				AVG(on_page) as `on_page`,
				AVG(active) as `active`,
				AVG(high_freq) as `high_freq`,
				AVG(low_freq) as `low_freq`,
				SUM(count) as `count`
			FROM {$wpdb->prefix}sqm_views_records
			GROUP BY
				DATE(utc_moment),
				eid,
				tid"
		);

		if ( false === $result ) {
			$logger->error( 'Failed to aggregate daily statistics: ' . $wpdb->last_error );

			return false;
		}

		/**
		 * Fires after daily aggregation completes
		 *
		 * @since 1.0.0
		 *
		 * @param int|bool $result Number of rows affected or false on failure
		 */
		do_action( 'sqm_views_daily_aggregated', $result );

		return true;
	} catch ( \Exception $e ) {
		$logger->error( 'Error in daily aggregation: ' . $e->getMessage() );

		return false;
	}
}

/**
 * Compacts archive files by grouping them by date.
 *
 * @since 1.0.0
 *
 * @return void
 */
function sqm_views_compact_archive() {
	$logger        = \SQMViews\Logger::get_instance();
	$files         = glob( realpath( SQMVIEWS_ARCHIVE_DIR ) . '/*T*.raw.jsonl' );
	$grouped_files = array();

	foreach ( $files as $file ) {
		$date = explode( 'T', basename( $file ) )[0];
		if ( ! isset( $grouped_files[ $date ] ) ) {
			$grouped_files[ $date ] = array();
		}
		$grouped_files[ $date ][] = $file;
	}

	foreach ( $grouped_files as $date => $date_files ) {
		$target_filename = "$date.raw.jsonl";
		$target_file     = realpath( SQMVIEWS_ARCHIVE_DIR ) . "/$target_filename";
		foreach ( $date_files as $source_file ) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents -- Local file, not remote URL.
			$content = file_get_contents( $source_file );
			if ( false === $content ) {
				$logger->error( 'Failed to read file: ' . $source_file );
				continue;
			}
			if ( function_exists( 'gzopen' ) ) {
				$gz = gzopen( $target_file . '.gz', file_exists( $target_file . '.gz' ) ? 'a9' : 'w9' );
				if ( $gz ) {
					gzwrite( $gz, $content );
					gzclose( $gz );
					// phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged -- Intentional cleanup, file may already be deleted.
					@wp_delete_file( $source_file );
				} else {
					$logger->error( 'Failed to write compressed file: ' . $target_file . '.gz' );
				}
				continue;
			}
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents -- Performance-critical batch processing.
			if ( false !== file_put_contents( $target_file, $content, file_exists( $target_file ) ? FILE_APPEND | LOCK_EX : LOCK_EX ) ) {
				// phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged -- Intentional cleanup, file may already be deleted.
				@wp_delete_file( $source_file );
				continue;
			}
			$logger->error( 'Failed to bundle archive file: ' . $target_file . ' to ' . $source_file );
		}
	}
}


/**
 * Processes records from a processed file into the database.
 *
 * @since 1.0.0
 *
 * @param string $processed_path Path to the processed records file.
 * @return bool True on success, false on failure.
 */
function process_records_file_to_database( string $processed_path ): bool {
	global $wpdb;
	$prefix = $wpdb->prefix;
	$logger = \SQMViews\Logger::get_instance();

	// Process records from processed path file.
	if ( file_exists( $processed_path ) ) {
		// The file could be quite large, so reading the file line by line is necessary to avoid loading it entirely into memory.
		// Using WP_Filesystem methods here is not practical.
		$handle = fopen( $processed_path, 'r' ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen
		if ( $handle ) {
			try {
				// phpcs:ignore Generic.CodeAnalysis.AssignmentInCondition.FoundInWhileCondition -- Standard pattern for reading lines.
				while ( ( $line = fgets( $handle ) ) !== false ) {
					$line = trim( $line );
					if ( empty( $line ) ) {
						continue;
					}

					$record_data = json_decode( $line, true );
					if ( null === $record_data ) {
						$logger->error( 'Failed to decode JSON from processed record: ' . $line );
						continue;
					}

					// Record has to be updated because there could be more data from previously unprocessed file.
					$wpdb->replace( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
						"{$prefix}sqm_views_records",
						array(
							'tid'           => prepare_trackable( $record_data['trackable'] ?? null ),
							'eid'           => $record_data['eid'],

							'gid'           => $record_data['gid'],
							'on_page'       => $record_data['on_page'],
							'active'        => $record_data['active'],
							'high_freq'     => $record_data['high_freq'],
							'low_freq'      => $record_data['low_freq'],
							'count'         => 1,
							'user_moment'   => $record_data['user_moment'],
							'user_timezone' => $record_data['user_timezone'],
							'utc_moment'    => $record_data['utc_moment'],
							'exit'          => $record_data['exit'],
						)
					);
				}
			} catch ( \Exception $e ) {
				$logger->error( 'Error processing records from processed path file: ' . $e->getMessage() );

				return false;
			} finally {
				// Because of potential file size, using WP_Filesystem methods here is not practical.
				fclose( $handle ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose
			}

			return true;
		} else {
			$logger->error( 'Failed to open processed file: ' . $processed_path );

			return false;
		}
	}

	return false;
}

/**
 * Prepares a trackable record in the database.
 *
 * @since 1.0.0
 *
 * @param mixed $trackable Trackable data array or null.
 * @return int|null Trackable ID or null on failure.
 */
function prepare_trackable( mixed $trackable ) {
	global $wpdb;
	$logger = \SQMViews\Logger::get_instance();
	if ( ! is_array( $trackable ) ) {
		return null;
	}

	if ( ! isset( $trackable['tgroup'] ) || ! isset( $trackable['ttype'] ) || ! isset( $trackable['wpid'] ) ) {
		return null;
	}

	$tgroup = sanitize_text_field( $trackable['tgroup'] );
	$ttype  = sanitize_text_field( $trackable['ttype'] );
	$wpid   = intval( $trackable['wpid'] );
	$altid  = isset( $trackable['altid'] ) ? sanitize_text_field( $trackable['altid'] ) : null;

	$cache_key   = 'sqm_trackable_' . md5( $tgroup . ':' . $ttype . ':' . $wpid . ':' . $altid );
	$cache_group = 'sqm_views_trackables';

	// Try WordPress object cache first.
	$cached_id = wp_cache_get( $cache_key, $cache_group );
	if ( false !== $cached_id ) {
		return (int) $cached_id;
	}

	try {
		// Check if record exists in database.
		$existing_id = $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table with object caching.
			$wpdb->prepare(
				"SELECT tid FROM {$wpdb->prefix}sqm_views_trackables
				WHERE tgroup = %s AND ttype = %s AND wpid = %d AND (altid = %s OR altid IS NULL)",
				$tgroup,
				$ttype,
				$wpid,
				$altid
			)
		);

		if ( $existing_id ) {
			wp_cache_set( $cache_key, (int) $existing_id, $cache_group, HOUR_IN_SECONDS );
			return (int) $existing_id;
		}

		// Insert new record.
		$result = $wpdb->insert( // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom plugin table.
			"{$wpdb->prefix}sqm_views_trackables",
			array(
				'tgroup' => $tgroup,
				'ttype'  => $ttype,
				'wpid'   => $wpid,
				'altid'  => $altid,
			),
			array( '%s', '%s', '%d', '%s' )
		);

		if ( false === $result ) {
			$logger->error( 'Failed to insert trackable record: ' . $wpdb->last_error );

			return null;
		}

		$new_id = (int) $wpdb->insert_id;
		wp_cache_set( $cache_key, $new_id, $cache_group, HOUR_IN_SECONDS );

		return $new_id;
	} catch ( \Exception $e ) {
		$logger->error( 'Error in prepare_trackable: ' . $e->getMessage() );

		return null;
	}
}

/**
 * Processes a cold file containing raw tracking events.
 *
 * @since 1.0.0
 *
 * @param string                        $path                   Path to the file to process.
 * @param array<string, SQMViewsRecord> $records                Existing records to accumulate into.
 * @param callable|null                 $malformed_event_writer Callback to write malformed events (optional).
 * @return array<string, SQMViewsRecord>|false Array of SQMViewsRecord objects or false on failure.
 */
function sqm_views_process_cold_file( string $path, $records = array(), $malformed_event_writer = null ) {
	static $key = null;
	if ( ! $key ) {
		// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode -- Decoding stored encryption key, not obfuscation.
		$key = base64_decode( get_option( SQMVIEWS_OPTION_ENCRYPTION_KEY ) );
	}
	$logger = \SQMViews\Logger::get_instance();

	// Use WordPress filesystem API.
	require_once ABSPATH . 'wp-admin/includes/file.php';
	WP_Filesystem();
	global $wp_filesystem;

	$content = $wp_filesystem->get_contents( $path );
	if ( false === $content ) {
		return false;
	}

	$line_number = 0;
	try {
		$lines = explode( "\n", $content );
		foreach ( $lines as $line ) {
			++$line_number;
			$line = trim( $line );
			if ( empty( $line ) ) {
				continue;
			}

			// Isolate each event processing to prevent one malformed event from crashing the entire pipeline.
			try {
				$data = json_decode( $line, true );
				if ( ! $data ) {
					$logger->warning(
						'Skipping malformed JSON line during processing',
						array(
							'file'        => basename( $path ),
							'line_number' => $line_number,
							'json_error'  => json_last_error_msg(),
							'raw_line'    => substr( $line, 0, 200 ),
						)
					);

					// Save malformed JSON using the callback writer.
					if ( $malformed_event_writer && is_callable( $malformed_event_writer ) ) {
						$malformed_event_writer( $line );
					}

					continue;
				}

				// Validate event structure against schema.
				$event_validation = RequestValidator::validate( $data, 'event' );
				if ( ! $event_validation->is_valid() ) {
					$logger->warning(
						'Event schema validation failed - saving as malformed event ' .
						$event_validation->get_error_code() . ' ' .
						$event_validation->get_error_message(),
						array(
							'file'          => basename( $path ),
							'line_number'   => $line_number,
							'error_code'    => $event_validation->get_error_code(),
							'error_message' => $event_validation->get_error_message(),
							'event_type'    => $data['t'] ?? 'unknown',
							'gid'           => $data['gid'] ?? 'unknown',
							'raw_line'      => substr( $line, 0, 200 ),
						)
					);

					// Save event with invalid structure.
					if ( $malformed_event_writer && is_callable( $malformed_event_writer ) ) {
						$malformed_event_writer( $line );
					}

					continue;
				}

				// Attempt payload decryption.
				// Note: Schema validation above already ensures payload is a string type if present.
				$data['payload'] = sqm_views_payload2data( $data['user']['payload'] ?? null, $key );

				/**
				 * Filters a raw record before processing
				 *
				 * Allows developers to modify or skip records during processing
				 *
				 * @since 1.0.0
				 *
				 * @param array|null $data Raw record data, or null to skip this record
				 */
				$data = apply_filters( 'sqm_views_raw_record', $data );

				if ( $data ) {
					// Validate required fields before processing.
					if ( ! isset( $data['gid'] ) || empty( $data['gid'] ) ) {
						$logger->warning(
							'Missing or empty gid field - saving as malformed event',
							array(
								'file'        => basename( $path ),
								'line_number' => $line_number,
								'raw_line'    => substr( $line, 0, 200 ),
							)
						);

						if ( $malformed_event_writer && is_callable( $malformed_event_writer ) ) {
							$malformed_event_writer( $line );
						}

						continue;
					}

					if ( ! isset( $records[ $data['gid'] ] ) ) {
						$records[ $data['gid'] ] = new SQMViewsRecord();
					}
					$records[ $data['gid'] ]->accumulate( $data, $line );
				}
			} catch ( \Exception $event_error ) {
				// Log the error but continue processing other events.
				$logger->error(
					'Error processing individual event - skipping and continuing',
					array(
						'file'        => basename( $path ),
						'line_number' => $line_number,
						'error'       => $event_error->getMessage(),
						'trace'       => $event_error->getTraceAsString(),
						'raw_line'    => substr( $line, 0, 200 ),
					)
				);

				// Save malformed event using the callback writer.
				if ( $malformed_event_writer && is_callable( $malformed_event_writer ) ) {
					$malformed_event_writer( $line );
				}

				// Continue to next event instead of halting.
				continue;
			}
		}
	} catch ( \Exception $e ) {
		$logger->error( 'Critical error processing cold file: ' . $e->getMessage() );

		return false;
	}
	return $records;
}

/**
 * Parse memory_limit string to bytes.
 *
 * @since 1.0.0
 *
 * @param string $memory_limit Memory limit string (e.g., '128M', '1G', '512K').
 * @return int Memory limit in bytes, or 0 if invalid.
 */
function sqm_views_parse_memory_limit( string $memory_limit ): int {
	$memory_limit = trim( $memory_limit );
	$last_char    = strtoupper( substr( $memory_limit, -1 ) );
	$value        = (int) substr( $memory_limit, 0, -1 );

	switch ( $last_char ) {
		case 'G':
			$value *= 1024 * 1024 * 1024;
			break;
		case 'M':
			$value *= 1024 * 1024;
			break;
		case 'K':
			$value *= 1024;
			break;
		default:
			// If no unit specified, assume bytes.
			$value = (int) $memory_limit;
			break;
	}

	return $value;
}
