/**
 * SQM Views - Admin Processing Page
 * Handles batch processing of raw pageview files via AJAX
 *
 * @package SQMViews
 * @since 1.1.4
 */

(function($) {
	'use strict';

	// Wait for DOM ready
	$(document).ready(function() {
		let initialFileCount = 0;
		let isProcessing = false;

		// Circuit breaker: detect if processing is stuck
		let lastRemainingFiles = null;
		let stuckCount = 0;
		const MAX_STUCK_ATTEMPTS = 3; // Stop after 3 consecutive unchanged responses

		// Handle click on "Run Processing Now" button
		$('#sqm-views-run-processing').on('click', function() {
			if (isProcessing) {
				return;
			}

			if (!confirm(sqmViewsProcessing.i18n.confirmMessage)) {
				return;
			}

			isProcessing = true;
			initialFileCount = parseInt(sqmViewsProcessing.initialFileCount, 10);

			// Reset circuit breaker
			lastRemainingFiles = null;
			stuckCount = 0;

			// Disable button and show progress container
			$(this).prop('disabled', true);
			$('#sqm-views-progress-container').show();
			$('#sqm-views-progress-status').text(sqmViewsProcessing.i18n.starting);
			$('#sqm-views-progress-bar').css('width', '0%');
			$('#sqm-views-progress-message').text('');

			// Start processing
			processBatch();
		});

		/**
		 * Process a single batch via AJAX
		 */
		function processBatch() {
			$.ajax({
				url: ajaxurl,
				type: 'POST',
				data: {
					action: 'sqm_views_process_batch',
					nonce: sqmViewsProcessing.nonce
				},
				success: function(response) {
					console.log('AJAX Response:', response);

					if (response.success) {
						const data = response.data;
						console.log('Data:', data);

						// Check if another process is running (locked)
						if (data.locked) {
							handleCompletion(false, sqmViewsProcessing.i18n.locked);
							return;
						}

						const remainingFiles = data.remaining_files || 0;
						const processedFiles = initialFileCount - remainingFiles;
						const percentComplete = initialFileCount > 0
							? Math.round((processedFiles / initialFileCount) * 100)
							: 100;

						console.log('Progress:', {
							initialFileCount: initialFileCount,
							remainingFiles: remainingFiles,
							processedFiles: processedFiles,
							percentComplete: percentComplete,
							message: data.message
						});

						// Circuit breaker: check if file count has changed
						if (lastRemainingFiles !== null && lastRemainingFiles === remainingFiles) {
							stuckCount++;
							console.warn('Circuit breaker: No progress detected', {
								remainingFiles: remainingFiles,
								stuckCount: stuckCount,
								maxAttempts: MAX_STUCK_ATTEMPTS
							});

							if (stuckCount >= MAX_STUCK_ATTEMPTS) {
								handleCompletion(
									false,
									'Processing stopped: No files have been processed in the last batch. ' +
									'This may be due to files with invalid timestamps or other issues. ' +
									'Check the server logs for more details.'
								);
								return;
							}
						} else {
							// Files count changed, reset stuck counter
							stuckCount = 0;
						}
						lastRemainingFiles = remainingFiles;

						// Update progress bar
						const $progressBar = $('#sqm-views-progress-bar');
						const $progressStatus = $('#sqm-views-progress-status');
						const $progressMessage = $('#sqm-views-progress-message');

						console.log('Updating DOM elements:', {
							progressBarExists: $progressBar.length > 0,
							progressStatusExists: $progressStatus.length > 0,
							progressMessageExists: $progressMessage.length > 0
						});

						$progressBar.css('width', percentComplete + '%');
						$progressStatus.text(sqmViewsProcessing.i18n.processing + ' ' + percentComplete + '%');
						$progressMessage.text(data.message);

						// Check if processing is complete
						if (data.completed) {
							handleCompletion(true, sqmViewsProcessing.i18n.complete);
						} else {
							// Continue processing
							setTimeout(processBatch, 500);
						}
					} else {
						handleCompletion(false, response.data.message || sqmViewsProcessing.i18n.failed);
					}
				},
				error: function(xhr, status, error) {
					handleCompletion(false, sqmViewsProcessing.i18n.error + ' ' + error);
				}
			});
		}

		/**
		 * Handle completion (success or error)
		 */
		function handleCompletion(success, message) {
			isProcessing = false;
			$('#sqm-views-run-processing').prop('disabled', false);

			if (success) {
				$('#sqm-views-progress-bar').css('width', '100%').css('background', '#46b450');
				$('#sqm-views-progress-status').text(sqmViewsProcessing.i18n.completeStatus);
			} else {
				$('#sqm-views-progress-bar').css('background', '#dc3232');
				$('#sqm-views-progress-status').text(sqmViewsProcessing.i18n.errorStatus);
			}

			$('#sqm-views-progress-message').text(message);
		}
	});

})(jQuery);
