/******/ (() => { // webpackBootstrap
/******/ 	"use strict";
/******/ 	var __webpack_modules__ = ({

/***/ "./src/activity-tracker.ts":
/***/ ((__unused_webpack_module, __webpack_exports__, __webpack_require__) => {

__webpack_require__.r(__webpack_exports__);
 __webpack_require__.d(__webpack_exports__, {
   ActivityCounter: () => ( ActivityCounter),
   HIGH_FREQUENCY_EVENTS: () => ( HIGH_FREQUENCY_EVENTS),
   LOW_FREQUENCY_EVENTS: () => ( LOW_FREQUENCY_EVENTS)
 });
 var _utils__WEBPACK_IMPORTED_MODULE_0__ = __webpack_require__("./src/utils.ts");
/**
 * Activity tracking - monitors user interactions
 */

const HIGH_FREQUENCY_EVENTS = ['mousemove', 'scroll', 'touchmove'];
const LOW_FREQUENCY_EVENTS = ['mousedown', 'keypress', 'touchstart', 'click'];
/**
 * Activity counter class
 */
class ActivityCounter {
    constructor() {
        this.counts = {
            highFrequency: 0,
            lowFrequency: 0,
        };
    }
    /**
     * Reset all counters
     */
    reset() {
        this.counts = {
            highFrequency: 0,
            lowFrequency: 0,
        };
        window.SQMViews.activeTime = 0;
        (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)('Activity counters reset');
    }
    /**
     * Increment counter based on event type
     */
    increment(eventType) {
        if (HIGH_FREQUENCY_EVENTS.includes(eventType)) {
            this.counts.highFrequency++;
        }
        else if (LOW_FREQUENCY_EVENTS.includes(eventType)) {
            this.counts.lowFrequency++;
        }
    }
    /**
     * Get current counts
     */
    getCounts() {
        return { ...this.counts };
    }
    /**
     * Get high frequency count
     */
    getHighFrequency() {
        return this.counts.highFrequency;
    }
    /**
     * Get low frequency count
     */
    getLowFrequency() {
        return this.counts.lowFrequency;
    }
}

/***/ }),

/***/ "./src/api-client.ts":
/***/ ((__unused_webpack_module, __webpack_exports__, __webpack_require__) => {

__webpack_require__.r(__webpack_exports__);
 __webpack_require__.d(__webpack_exports__, {
   ApiClient: () => ( ApiClient)
 });
 var _utils__WEBPACK_IMPORTED_MODULE_0__ = __webpack_require__("./src/utils.ts");
/**
 * API client for sending tracking data
 */

class ApiClient {
    constructor(activityCounter, onInitResponse) {
        this.activityCounter = activityCounter;
        this.onInitResponse = onInitResponse;
    }
    /**
     * Send tracking request to server
     */
    async sendTrackingRequest(type) {
        (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)(`${type} tracking request sending with ID: ${window.SQMViews.gid || 'pending'}`);
        const body = {
            t: type,
            url: window.location.href,
            referrer: document.referrer,
            localTime: (0,_utils__WEBPACK_IMPORTED_MODULE_0__.getUserLocalTime)(),
            client: window.SQMViews.CONFIG.client,
        };
        if (window.SQMViews.gid) {
            body.gid = window.SQMViews.gid;
        }
        let useBeacon = true;
        if (type !== 'init') {
            body.nonce = window.SQMViews.nonce || undefined;
        }
        if (type === 'init') {
            useBeacon = false;
            const payload = this.getPayload();
            if (!payload) {
                return;
            }
            body.payload = payload;
            this.activityCounter.reset();
        }
        else if (type === 'exit' || type === 'timeout') {
            const activeTime = this.calculateActiveTime();
            body.activityCounts = {
                highFrequencyEvents: this.activityCounter.getHighFrequency(),
                lowFrequencyEvents: this.activityCounter.getLowFrequency(),
                activeTime,
            };
            this.activityCounter.reset();
            (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)(`Sending activity counts - High: ${body.activityCounts.highFrequencyEvents}, ` +
                `Low: ${body.activityCounts.lowFrequencyEvents}, Time: ${activeTime}`);
        }
        else if (type === 'ping') {
            const activeTime = this.calculateActiveTime();
            body.activityCounts = {
                highFrequencyEvents: this.activityCounter.getHighFrequency(),
                lowFrequencyEvents: this.activityCounter.getLowFrequency(),
                activeTime,
            };
            (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)(`Sending ping event, activity counts - High: ${body.activityCounts.highFrequencyEvents}, ` +
                `Low: ${body.activityCounts.lowFrequencyEvents}, Time: ${activeTime}`);
        }
        const data = JSON.stringify(body);
        if (useBeacon && navigator.sendBeacon) {
            const queued = navigator.sendBeacon(window.SQMViews.handler, data);
            (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)(`sendBeacon to ${window.SQMViews.handler} => ${queued ? 'queued' : 'failed'}`);
        }
        else {
            // Regular fetch for init and other events
            const fetchOptions = {
                method: 'POST',
                keepalive: true,
                headers: { 'Content-Type': 'application/json' },
                body: data,
            };
            try {
                const response = await fetch(window.SQMViews.handler, fetchOptions);
                const initResponse = await response.json();
                if ((initResponse === null || initResponse === void 0 ? void 0 : initResponse.config) && typeof initResponse.config === 'object') {
                    window.SQMViews.CONFIG = { ...window.SQMViews.CONFIG, ...initResponse.config };
                    window.SQMViews.gid = initResponse.gid || null;
                    window.SQMViews.nonce = initResponse.nonce || null;
                    (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)(`${type} server config: ${JSON.stringify(initResponse)}`);
                    (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)(`${type} combined config: ${JSON.stringify(window.SQMViews.CONFIG)}`);
                    (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)(`${type} window.SQMViews.gid: ${window.SQMViews.gid}`);
                    (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)(`${type} window.SQMViews.nonce: ${window.SQMViews.nonce}`);
                    if (this.onInitResponse) {
                        this.onInitResponse(initResponse);
                    }
                }
            }
            catch (error) {
                (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)(`Tracking error (all handlers failed): ${error}`, 'error');
            }
        }
    }
    /**
     * Calculate total active time
     */
    calculateActiveTime() {
        return Math.round((window.SQMViews.active
            ? window.SQMViews.activeTime + (performance.now() - (window.SQMViews.activeStartTime || 0))
            : window.SQMViews.activeTime) + Number.EPSILON) / 1000;
    }
    /**
     * Get payload from meta tag
     */
    getPayload() {
        var _a;
        return ((_a = document.querySelector('meta[name="sqm-views-tracker"]')) === null || _a === void 0 ? void 0 : _a.getAttribute('content')) || null;
    }
}

/***/ }),

/***/ "./src/session-manager.ts":
/***/ ((__unused_webpack_module, __webpack_exports__, __webpack_require__) => {

__webpack_require__.r(__webpack_exports__);
 __webpack_require__.d(__webpack_exports__, {
   SessionManager: () => ( SessionManager)
 });
 var _utils__WEBPACK_IMPORTED_MODULE_0__ = __webpack_require__("./src/utils.ts");
/**
 * Session management - handles user active/idle states and timeouts
 */

class SessionManager {
    constructor(onExit, onPing) {
        this.onExit = onExit;
        this.onPing = onPing;
        this._pageviewCloseByTimeoutTimer = null;
        this._signalUserIdleTimer = null;
        this._exitSent = false;
    }
    /**
     * Signal that user is active
     */
    signalUserActive(eventType = null) {
        if (this._exitSent) {
            return;
        }
        if (!window.SQMViews.active) {
            (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)('User is active');
            window.SQMViews.active = true;
            window.SQMViews.activeStartTime = performance.now();
            if (!window.SQMViews._ping_interval_reference) {
                this.resetPingHandler();
            }
        }
        // Reset timer detecting session timeout
        if (this._pageviewCloseByTimeoutTimer !== null) {
            clearTimeout(this._pageviewCloseByTimeoutTimer);
        }
        this._pageviewCloseByTimeoutTimer = window.setTimeout(() => this.sendExitEvent('timeout'), window.SQMViews.CONFIG.session_timeout * 1000);
        // Reset idle timer
        if (this._signalUserIdleTimer !== null) {
            clearTimeout(this._signalUserIdleTimer);
        }
        this._signalUserIdleTimer = window.setTimeout(() => this.signalUserIdle(), window.SQMViews.CONFIG.idle_timeout * 1000);
    }
    /**
     * Signal that user is idle
     */
    signalUserIdle() {
        if (!window.SQMViews.active) {
            return;
        }
        (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)('User idle detected');
        window.SQMViews.active = false;
        if (window.SQMViews.activeStartTime !== null) {
            const activeTime = performance.now() - window.SQMViews.activeStartTime;
            window.SQMViews.activeTime += activeTime;
            (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)(`Active time window: ${Math.round(activeTime)}ms, Total: ${Math.round(window.SQMViews.activeTime)}ms`);
            window.SQMViews.activeStartTime = null;
        }
    }
    /**
     * Send exit event
     */
    sendExitEvent(type = 'exit') {
        if (!this._exitSent) {
            (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)(`Sending ${type} event`);
            this._exitSent = true;
            this.signalUserIdle();
            this.onExit(type);
            window.SQMViews.gid = null;
            window.SQMViews.nonce = null;
        }
    }
    /**
     * Send ping event
     */
    sendPingEvent() {
        if (!this._exitSent) {
            (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)('Sending ping event');
            this.onPing();
        }
    }
    /**
     * Reset ping interval handler
     */
    resetPingHandler() {
        if (window.SQMViews._ping_interval_reference) {
            clearInterval(window.SQMViews._ping_interval_reference);
        }
        if (window.SQMViews.CONFIG.ping_interval && !this._exitSent) {
            window.SQMViews._ping_interval_reference = window.setInterval(() => this.sendPingEvent(), window.SQMViews.CONFIG.ping_interval * 1000);
            (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)(`Ping interval set to ${window.SQMViews.CONFIG.ping_interval} seconds`);
        }
    }
    /**
     * Reset exit flag (for SPA navigation)
     */
    resetExitFlag() {
        this._exitSent = false;
    }
    /**
     * Check if exit was sent
     */
    isExitSent() {
        return this._exitSent;
    }
}

/***/ }),

/***/ "./src/tracker.ts":
/***/ ((__unused_webpack_module, __webpack_exports__, __webpack_require__) => {

__webpack_require__.r(__webpack_exports__);
 __webpack_require__.d(__webpack_exports__, {
   SQMViewsTracker: () => ( SQMViewsTracker)
 });
 var _utils__WEBPACK_IMPORTED_MODULE_0__ = __webpack_require__("./src/utils.ts");
 var _activity_tracker__WEBPACK_IMPORTED_MODULE_1__ = __webpack_require__("./src/activity-tracker.ts");
 var _session_manager__WEBPACK_IMPORTED_MODULE_2__ = __webpack_require__("./src/session-manager.ts");
 var _api_client__WEBPACK_IMPORTED_MODULE_3__ = __webpack_require__("./src/api-client.ts");
/**
 * Main tracker class - coordinates all tracking functionality
 */

/**
 * Plugin build global version constant
 *
 * WARNING: Do NOT edit this version manually!
 * Use the bump-version.sh script in ../sqm-views/ to update versions:
 *   cd ../sqm-views && ./bump-version.sh patch|minor|major
 *
 * This constant is used globally across all build artifacts (PHP, JS, TS).
 * This version MUST match SQMVIEWS_BUILD_GLOBAL_VERSION in other files.
 */
const SQMVIEWS_BUILD_GLOBAL_VERSION = '1.1.8';
class SQMViewsTracker {
    constructor() {
        // Initialize global state
        this.initializeGlobalState();
        // Get handler from meta tag
        const tracker = (0,_utils__WEBPACK_IMPORTED_MODULE_0__.getHandler)();
        window.SQMViews.handler =
            window.SQMViews.CONFIG.handlers[tracker] ||
                window.SQMViews.CONFIG.handlers.api;
        this.currentUrl = window.location.href;
        // Initialize components
        this.activityCounter = new _activity_tracker__WEBPACK_IMPORTED_MODULE_1__.ActivityCounter();
        this.sessionManager = new _session_manager__WEBPACK_IMPORTED_MODULE_2__.SessionManager((type) => this.handleExit(type), () => this.handlePing());
        this.apiClient = new _api_client__WEBPACK_IMPORTED_MODULE_3__.ApiClient(this.activityCounter, (response) => {
            if (response) {
                this.sessionManager.signalUserActive();
                this.sessionManager.resetPingHandler();
            }
        });
        // Start tracking
        this.start();
    }
    /**
     * Initialize global SQMViews object
     */
    initializeGlobalState() {
        window.SQMViews = window.SQMViews || {};
        window.SQMViews.gid = window.SQMViews.gid || null;
        window.SQMViews.nonce = window.SQMViews.nonce || null;
        window.SQMViews.active = window.SQMViews.active || false;
        window.SQMViews.activeStartTime = window.SQMViews.activeStartTime || null;
        window.SQMViews.activeTime = window.SQMViews.activeTime || 0;
        // Create a base configuration object with default values
        const defaultConfig = {
            session_timeout: 30 * 60,
            idle_timeout: 10,
            ping_interval: 60,
            handler: null,
            handlers: {
                fast: '/sqm-views-pages.php',
                api: '/wp-json/sqm-views/v1/track',
            },
            client: SQMVIEWS_BUILD_GLOBAL_VERSION,
        };
        // Merge with existing configuration if any
        window.SQMViews.CONFIG = {
            ...defaultConfig,
            ...window.SQMViews.CONFIG,
        };
    }
    /**
     * Start tracking
     */
    start() {
        (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)('Initializing page view tracking');
        // Setup event listeners
        this.setupPageLifecycleEvents();
        this.setupActivityEvents();
        this.setupSPANavigation();
        // Send initial page view
        this.apiClient.sendTrackingRequest('init');
        // Start session management
        this.sessionManager.signalUserActive();
        this.sessionManager.resetPingHandler();
    }
    /**
     * Setup page lifecycle event listeners
     */
    setupPageLifecycleEvents() {
        // Handle page unload (page close, refresh, navigation)
        window.addEventListener('beforeunload', () => this.sessionManager.sendExitEvent('exit'));
        window.addEventListener('unload', () => this.sessionManager.sendExitEvent('exit'));
        // Handle page visibility changes (tab switching, minimizing)
        document.addEventListener('visibilitychange', () => {
            (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)(`Page visibility changed to ${document.visibilityState}`);
            if (document.visibilityState === 'hidden') {
                this.sessionManager.signalUserIdle();
            }
            if (document.visibilityState === 'visible') {
                this.sessionManager.signalUserActive();
            }
        });
        // For traditional page navigation
        window.addEventListener('pagehide', () => this.sessionManager.signalUserIdle());
    }
    /**
     * Setup activity event listeners
     */
    setupActivityEvents() {
        const activityEvents = [..._activity_tracker__WEBPACK_IMPORTED_MODULE_1__.HIGH_FREQUENCY_EVENTS, ..._activity_tracker__WEBPACK_IMPORTED_MODULE_1__.LOW_FREQUENCY_EVENTS];
        activityEvents.forEach((event) => {
            document.addEventListener(event, () => {
                this.activityCounter.increment(event);
                this.sessionManager.signalUserActive(event);
            }, { passive: true });
        });
    }
    /**
     * Setup SPA navigation handling
     */
    setupSPANavigation() {
        // Monitor URL changes for single-page applications
        const originalPushState = history.pushState;
        const originalReplaceState = history.replaceState;
        history.pushState = (...args) => {
            originalPushState.apply(history, args);
            this.handleUrlChange();
        };
        history.replaceState = (...args) => {
            originalReplaceState.apply(history, args);
            this.handleUrlChange();
        };
        window.addEventListener('popstate', () => this.handleUrlChange());
    }
    /**
     * Handle URL changes (for SPAs)
     */
    handleUrlChange() {
        if (window.location.href !== this.currentUrl) {
            (0,_utils__WEBPACK_IMPORTED_MODULE_0__.debugLog)(`URL changed from ${this.currentUrl} to ${window.location.href}`);
            this.sessionManager.sendExitEvent('exit');
            this.currentUrl = window.location.href;
            // Reset exit flag for new page
            this.sessionManager.resetExitFlag();
            // Send new init event for the new page
            setTimeout(() => {
                this.apiClient.sendTrackingRequest('init');
            }, 100);
        }
    }
    /**
     * Handle exit event
     */
    handleExit(type) {
        this.apiClient.sendTrackingRequest(type);
    }
    /**
     * Handle ping event
     */
    handlePing() {
        this.apiClient.sendTrackingRequest('ping');
    }
}

/***/ }),

/***/ "./src/utils.ts":
/***/ ((__unused_webpack_module, __webpack_exports__, __webpack_require__) => {

__webpack_require__.r(__webpack_exports__);
 __webpack_require__.d(__webpack_exports__, {
   compareVersions: () => ( compareVersions),
   debugLog: () => ( debugLog),
   getHandler: () => ( getHandler),
   getPayload: () => ( getPayload),
   getUserLocalTime: () => ( getUserLocalTime),
   isVersionGte: () => ( isVersionGte)
 });
/**
 * Utility functions for SQMViews tracker
 */
/**
 * Debug logging with timestamp and session ID
 */
function debugLog(message, level = 'info') {
    var _a;
    const prefix = '[SQMViews] ' + (((_a = window.SQMViews) === null || _a === void 0 ? void 0 : _a.gid) || 'no-gid') + ' ';
    const time = getUserLocalTime().time;
    if (level === 'error') {
        console.error(`${prefix} ${time} - ${message}`);
    }
    else {
        console.log(`${prefix} ${time} - ${message}`);
    }
}
/**
 * Get user's local time with timezone information
 */
function getUserLocalTime(now = new Date(), withMilliseconds = true) {
    const tz = Intl.DateTimeFormat().resolvedOptions().timeZone;
    const shifted = new Date(now.getTime() - now.getTimezoneOffset() * 60000)
        .toISOString()
        .replace('Z', '');
    const localTime = withMilliseconds
        ? shifted.slice(0, 23)
        : shifted.slice(0, 19);
    return {
        tz,
        time: localTime,
    };
}
/**
 * Get encrypted payload from meta tag
 */
function getPayload() {
    var _a;
    return ((_a = document.querySelector('meta[name="sqm-views-tracker"]')) === null || _a === void 0 ? void 0 : _a.getAttribute('content')) || null;
}
/**
 * Get handler type from meta tag
 */
function getHandler() {
    var _a;
    return ((_a = document.querySelector('meta[name="sqm-views-tracker"]')) === null || _a === void 0 ? void 0 : _a.getAttribute('tracker')) || null;
}
/**
 * Compare two semantic version strings
 * @param version1 - First version string (e.g., "1.2.3")
 * @param version2 - Second version string (e.g., "1.0.0")
 * @returns -1 if version1 < version2, 0 if equal, 1 if version1 > version2
 */
function compareVersions(version1, version2) {
    const v1parts = version1.split('.').map(Number);
    const v2parts = version2.split('.').map(Number);
    for (let i = 0; i < Math.max(v1parts.length, v2parts.length); i++) {
        const v1 = v1parts[i] || 0;
        const v2 = v2parts[i] || 0;
        if (v1 > v2)
            return 1;
        if (v1 < v2)
            return -1;
    }
    return 0;
}
/**
 * Check if version1 is greater than or equal to version2
 * @param version1 - Version to check
 * @param version2 - Minimum required version
 * @returns true if version1 >= version2
 */
function isVersionGte(version1, version2) {
    return compareVersions(version1, version2) >= 0;
}

/***/ })

/******/ 	});
/************************************************************************/
/******/ 	// The module cache
/******/ 	var __webpack_module_cache__ = {};
/******/ 	
/******/ 	// The require function
/******/ 	function __webpack_require__(moduleId) {
/******/ 		// Check if module is in cache
/******/ 		var cachedModule = __webpack_module_cache__[moduleId];
/******/ 		if (cachedModule !== undefined) {
/******/ 			return cachedModule.exports;
/******/ 		}
/******/ 		// Create a new module (and put it into the cache)
/******/ 		var module = __webpack_module_cache__[moduleId] = {
/******/ 			// no module.id needed
/******/ 			// no module.loaded needed
/******/ 			exports: {}
/******/ 		};
/******/ 	
/******/ 		// Execute the module function
/******/ 		__webpack_modules__[moduleId](module, module.exports, __webpack_require__);
/******/ 	
/******/ 		// Return the exports of the module
/******/ 		return module.exports;
/******/ 	}
/******/ 	
/************************************************************************/
/******/ 	
/******/ 	(() => {
/******/ 		// define getter functions for harmony exports
/******/ 		__webpack_require__.d = (exports, definition) => {
/******/ 			for(var key in definition) {
/******/ 				if(__webpack_require__.o(definition, key) && !__webpack_require__.o(exports, key)) {
/******/ 					Object.defineProperty(exports, key, { enumerable: true, get: definition[key] });
/******/ 				}
/******/ 			}
/******/ 		};
/******/ 	})();
/******/ 	
/******/ 	
/******/ 	(() => {
/******/ 		__webpack_require__.o = (obj, prop) => (Object.prototype.hasOwnProperty.call(obj, prop))
/******/ 	})();
/******/ 	
/******/ 	
/******/ 	(() => {
/******/ 		// define __esModule on exports
/******/ 		__webpack_require__.r = (exports) => {
/******/ 			if(typeof Symbol !== 'undefined' && Symbol.toStringTag) {
/******/ 				Object.defineProperty(exports, Symbol.toStringTag, { value: 'Module' });
/******/ 			}
/******/ 			Object.defineProperty(exports, '__esModule', { value: true });
/******/ 		};
/******/ 	})();
/******/ 	
/************************************************************************/
var __webpack_exports__ = {};

(() => {
__webpack_require__.r(__webpack_exports__);
 var _tracker__WEBPACK_IMPORTED_MODULE_0__ = __webpack_require__("./src/tracker.ts");
 var _utils__WEBPACK_IMPORTED_MODULE_1__ = __webpack_require__("./src/utils.ts");
/**
 * SQMViews Page Views Tracker
 * Entry point for the tracking script
 */

const payload = (0,_utils__WEBPACK_IMPORTED_MODULE_1__.getPayload)();
if (payload) {
    new _tracker__WEBPACK_IMPORTED_MODULE_0__.SQMViewsTracker();
}

})();

/******/ })()
;