<?php
/*
Plugin Name: SparkAgent AI Chatbot
Plugin URI: https://www.sparkagentai.com/
Description: SparkAgent AI Chatbot adds a GPT-powered AI assistant to your WordPress site. Provide instant, 24/7 support and boost user engagement with live, conversational AI.
Version: 0.0.6
Author: sparkagentai
License: GPL2
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: sparkagent-ai-chatbot
*/

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Enqueue the SparkAgent AI chatbot script on the frontend.
 */
function sparkagentai_enqueue_chatbot_script() {
    $options     = get_option( 'sparkagentai_settings' );
    $chatbot_id  = isset( $options['chatbot_id'] ) ? $options['chatbot_id'] : '';
    $chatbot_url = 'https://chatbot.sparkagentai.com';

    if ( empty( $chatbot_id ) || empty( $chatbot_url ) ) {
        return;
    }

    // Register the external chatbot script.
    wp_register_script(
        'sparkagentai-chatbot',
        'https://www.sparkagentai.com/spark-agent-ai-chat-widget.js',
        [],
        '1.0.0',
        true
    );

    wp_enqueue_script( 'sparkagentai-chatbot' );
}
add_action( 'wp_enqueue_scripts', 'sparkagentai_enqueue_chatbot_script' );

/**
 * Add data attributes to the SparkAgent AI Chatbot script tag.
 *
 * This ensures that the custom data (chatbot ID and URL) is available to the third-party script
 * immediately when it is loaded.
 */
function sparkagentai_add_data_attributes( $tag, $handle, $src ) {
    if ( 'sparkagentai-chatbot' !== $handle ) {
        return $tag;
    }

    $options     = get_option( 'sparkagentai_settings' );
    $chatbot_id  = isset( $options['chatbot_id'] ) ? $options['chatbot_id'] : '';
    $chatbot_url = 'https://chatbot.sparkagentai.com';

    if ( empty( $chatbot_id ) || empty( $chatbot_url ) ) {
        return $tag;
    }

    // Modify the script tag to include data attributes.
    $tag = str_replace(
        '<script ',
        '<script data-chatbot-id="' . esc_attr( $chatbot_id ) . '" data-chatbot-url="' . esc_attr( $chatbot_url ) . '" ',
        $tag
    );

    return $tag;
}
add_filter( 'script_loader_tag', 'sparkagentai_add_data_attributes', 10, 3 );

/**
 * Add main menu item in the WordPress admin.
 */
function sparkagentai_add_admin_menu() {
    if ( function_exists( 'add_menu_page' ) ) {
        add_menu_page(
            'SparkAgent AI Settings',
            'SparkAgent AI Chatbot',
            'manage_options',
            'sparkagent-ai-chatbot_settings_page',
            'sparkagentai_settings_page',
            'dashicons-format-chat',
            6
        );
    }
}
add_action( 'admin_menu', 'sparkagentai_add_admin_menu' );

/**
 * Display the settings page content.
 */
function sparkagentai_settings_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }

    $options = get_option( 'sparkagentai_settings' );
    ?>
    <div class="wrap">
        <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>

        <form action="options.php" method="post">
            <?php settings_fields( 'sparkagentai_settings_group' ); ?>
            <table class="form-table">
                <tr valign="top">
                    <th scope="row">
                        <label for="sparkagentai_settings_chatbot_id"><?php esc_html_e( 'Chatbot ID', 'sparkagent-ai-chatbot' ); ?></label>
                    </th>
                    <td>
                        <input type="text" id="sparkagentai_settings_chatbot_id" name="sparkagentai_settings[chatbot_id]" value="<?php echo isset( $options['chatbot_id'] ) ? esc_attr( $options['chatbot_id'] ) : ''; ?>" class="regular-text" />
                        <p class="description"><?php esc_html_e( 'Enter your SparkAgent AI Chatbot ID.', 'sparkagent-ai-chatbot' ); ?></p>
                    </td>
                </tr>
            </table>
            <?php submit_button( esc_html__( 'Save Settings', 'sparkagent-ai-chatbot' ) ); ?>
        </form>
    </div>
    <?php
}

/**
 * Register plugin settings.
 */
function sparkagentai_register_settings() {
    if ( function_exists( 'register_setting' ) ) {
        register_setting(
            'sparkagentai_settings_group',
            'sparkagentai_settings',
            'sparkagentai_sanitize'
        );
    }
}
add_action( 'admin_init', 'sparkagentai_register_settings' );

/**
 * Sanitize plugin settings before saving.
 *
 * @param array $input Input data from settings form.
 * @return array Sanitized data.
 */
function sparkagentai_sanitize( $input ) {
    $sanitized = array();

    if ( isset( $input['chatbot_id'] ) ) {
        $sanitized['chatbot_id'] = sanitize_text_field( $input['chatbot_id'] );
    }

    if ( isset( $input['chatbot_url'] ) ) {
        $sanitized['chatbot_url'] = esc_url_raw( $input['chatbot_url'] );
    }

    return $sanitized;
}
