<?php
/*
Plugin Name: SpamScrubber
Plugin URI: https://spamscrubber.com
Description: A simple and robust anti-spam plugin that adds a submission delay, JavaScript token, and a honeypot field to your site's forms.
Version: 1.0.0
Author: Richard Phillips
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: spamscrubber
Domain Path: /languages
*/

if (!defined('ABSPATH')) exit;

/** Version & option keys */
define('SPAMSCRUBBER_VERSION', '1.0.0');

define('SPAMSCRUBBER_OPTION_DELAY',       'spamscrubber_delay_seconds');
define('SPAMSCRUBBER_OPT_DISABLED_CLICK', 'spamscrubber_disabled_click_count');
define('SPAMSCRUBBER_OPT_EARLY_SUBMIT',   'spamscrubber_early_submit_count');
define('SPAMSCRUBBER_OPT_NJS',            'spamscrubber_njs_spam_count');
define('SPAMSCRUBBER_OPT_ERROR_URL',      'spamscrubber_error_page_url');
define('SPAMSCRUBBER_OPT_HONEYPOT',       'spamscrubber_honeypot_count');
define('SPAMSCRUBBER_OPT_REMOTE',         'spamscrubber_remote_spam_count');

/** Helpers */
function spamscrubber_is_admin_or_login() {
    global $pagenow;
    return is_admin() || (isset($pagenow) && $pagenow === 'wp-login.php') || (defined('DOING_AJAX') && DOING_AJAX);
}

function spamscrubber_get_honeypot_options() {
    return [
        __("Middle Name", "spamscrubber"), __("Maiden Name", "spamscrubber"), __("Fax Number", "spamscrubber"), __("Pager Number", "spamscrubber"),
        __("ICQ Number", "spamscrubber"), __("AIM Handle", "spamscrubber"), __("Nickname", "spamscrubber"), __("Website", "spamscrubber"),
        __("Home Address Line 2", "spamscrubber"), __("Second Email Address", "spamscrubber"), __("Alternative Phone Number", "spamscrubber"),
        __("PO Box", "spamscrubber"), __("Skype ID", "spamscrubber"), __("Company", "spamscrubber"), __("Employer", "spamscrubber"),
        __("Mother’s Maiden Name", "spamscrubber"), __("Favorite Pet’s Name", "spamscrubber"), __("Social Security Number", "spamscrubber"),
        __("National ID Number", "spamscrubber"), __("Driver’s License Number", "spamscrubber"), __("Passport Number", "spamscrubber"),
        __("Tax ID", "spamscrubber"), __("Birthday (optional)", "spamscrubber"), __("Country of Birth", "spamscrubber"),
        __("Place of Issue", "spamscrubber"), __("Suffix", "spamscrubber"), __("Prefix", "spamscrubber"), __("Zip+4 Code", "spamscrubber"),
        __("Bank Account Number", "spamscrubber"), __("Routing Number", "spamscrubber"), __("Credit Card Number", "spamscrubber"),
        __("CVV", "spamscrubber"), __("Security Question", "spamscrubber"), __("Favorite Color", "spamscrubber"), __("Lucky Number", "spamscrubber"),
        __("Spouse Name", "spamscrubber"), __("Children’s Names", "spamscrubber"), __("Room Number", "spamscrubber"),
        __("Unit Number", "spamscrubber"), __("Floor", "spamscrubber"), __("Building Name", "spamscrubber"), __("Work Phone", "spamscrubber"),
        __("Fax Extension", "spamscrubber"), __("Pager Extension", "spamscrubber"), __("Website URL", "spamscrubber"), __("Personal Blog", "spamscrubber"),
        __("IRC Nick", "spamscrubber"), __("Google+ Profile", "spamscrubber"), __("MySpace ID", "spamscrubber"), __("Friendster Handle", "spamscrubber"),
        __("LiveJournal Name", "spamscrubber")
    ];
}

function spamscrubber_is_excluded_page() {
    if (is_user_logged_in()) return true;
    $exclude_ids = get_option('spamscrubber_exclude_ids', []);
    $exclude_ids = array_map('strval', (array)$exclude_ids);
    $current_id = strval(get_queried_object_id());
    if (is_singular() && in_array($current_id, $exclude_ids, true)) return true;
    return false;
}

/* === Styles (enqueued) === */
function spamscrubber_core_css_rules() {
    return '.spamscrubber-disabled,.spamscrubber-disabled:disabled{opacity:.6!important;pointer-events:auto!important;cursor:not-allowed!important}' .
           '.spamscrubber-extra-wrap{position:absolute!important;left:-9999px!important;width:1px!important;height:1px!important;overflow:hidden!important}';
}

add_action('wp_enqueue_scripts', 'spamscrubber_enqueue_frontend_styles');
function spamscrubber_enqueue_frontend_styles() {
    if (spamscrubber_is_excluded_page() || spamscrubber_is_admin_or_login()) return;
    // Register a "virtual" style handle and attach inline CSS to it.
    wp_register_style('spamscrubber', false, [], SPAMSCRUBBER_VERSION);
    wp_enqueue_style('spamscrubber');
    wp_add_inline_style('spamscrubber', spamscrubber_core_css_rules());
}

add_action('admin_enqueue_scripts', 'spamscrubber_enqueue_admin_styles');
function spamscrubber_enqueue_admin_styles($hook) {
    if ($hook !== 'settings_page_spamscrubber-settings') return;
    wp_register_style('spamscrubber-admin', false, [], SPAMSCRUBBER_VERSION);
    wp_enqueue_style('spamscrubber-admin');
    wp_add_inline_style('spamscrubber-admin', spamscrubber_core_css_rules());
}

/* === Admin menu & settings === */
add_action('admin_menu', 'spamscrubber_admin_menu');
function spamscrubber_admin_menu() {
    add_options_page(
        esc_html__('SpamScrubber Settings', 'spamscrubber'),
        esc_html__('SpamScrubber', 'spamscrubber'),
        'manage_options',
        'spamscrubber-settings',
        'spamscrubber_settings_page'
    );
}

add_action('admin_init', 'spamscrubber_admin_register_settings');
function spamscrubber_admin_register_settings() {
    register_setting('spamscrubber_settings_group', SPAMSCRUBBER_OPTION_DELAY, [
        'type' => 'integer',
        'default' => 4,
        'sanitize_callback' => 'absint'
    ]);
    register_setting('spamscrubber_settings_group', SPAMSCRUBBER_OPT_ERROR_URL, [
        'type' => 'string',
        'default' => '/submit-form',
        'sanitize_callback' => 'spamscrubber_sanitize_error_url'
    ]);
    register_setting('spamscrubber_settings_group', 'spamscrubber_exclude_ids', [
        'type' => 'array',
        'default' => [],
        'sanitize_callback' => 'spamscrubber_sanitize_exclude_ids'
    ]);
    register_setting('spamscrubber_settings_group', 'spamscrubber_honeypot_enabled', [
        'type' => 'boolean',
        'default' => true,
        'sanitize_callback' => 'spamscrubber_sanitize_boolean'
    ]);
    register_setting('spamscrubber_settings_group', 'spamscrubber_honeypot_field', [
        'type' => 'string',
        'default' => spamscrubber_get_honeypot_options()[0],
        'sanitize_callback' => 'spamscrubber_sanitize_honeypot_field'
    ]);
}
function spamscrubber_sanitize_error_url($v) {
    $v = trim((string)$v);
    if ($v === '') $v = '/submit-form';
    $path = wp_parse_url($v, PHP_URL_PATH);
    if (!$path) $path = '/submit-form';
    if (substr($path, 0, 1) !== '/') $path = '/' . $path;
    return untrailingslashit($path);
}
function spamscrubber_sanitize_exclude_ids($arr) {
    if (!is_array($arr)) $arr = [];
    return array_filter(array_map('absint', $arr));
}
function spamscrubber_sanitize_boolean($v) {
    return (bool)$v;
}
function spamscrubber_sanitize_honeypot_field($v) {
    $opts = spamscrubber_get_honeypot_options();
    return in_array($v, $opts, true) ? $v : $opts[0];
}

function spamscrubber_settings_page() {
    $honeypot_options = spamscrubber_get_honeypot_options();

    $disabled_click = (int) get_option(SPAMSCRUBBER_OPT_DISABLED_CLICK, 0);
    $early_submit   = (int) get_option(SPAMSCRUBBER_OPT_EARLY_SUBMIT, 0);
    $njs_spam       = (int) get_option(SPAMSCRUBBER_OPT_NJS, 0);
    $honeypot_count = (int) get_option(SPAMSCRUBBER_OPT_HONEYPOT, 0);
    $remote_spam    = (int) get_option(SPAMSCRUBBER_OPT_REMOTE, 0);
    $total          = $disabled_click + $early_submit + $njs_spam + $honeypot_count + $remote_spam;

    $error_url        = get_option(SPAMSCRUBBER_OPT_ERROR_URL, '/submit-form');
    $exclude_ids      = get_option('spamscrubber_exclude_ids', []);
    $honeypot_enabled = get_option('spamscrubber_honeypot_enabled', true);
    $honeypot_field   = get_option('spamscrubber_honeypot_field', $honeypot_options[0]);

    $pages = get_posts([
        'post_type'      => ['page', 'post'],
        'posts_per_page' => 200,
        'post_status'    => 'publish',
        'orderby'        => 'title',
        'order'          => 'ASC'
    ]);

    if (isset($_POST['spamscrubber_reset_stats']) && check_admin_referer('spamscrubber_reset_stats', 'spamscrubber_reset_stats_nonce')) {
        update_option(SPAMSCRUBBER_OPT_DISABLED_CLICK, 0);
        update_option(SPAMSCRUBBER_OPT_EARLY_SUBMIT, 0);
        update_option(SPAMSCRUBBER_OPT_NJS, 0);
        update_option(SPAMSCRUBBER_OPT_HONEYPOT, 0);
        update_option(SPAMSCRUBBER_OPT_REMOTE, 0);
        echo '<div class="updated notice is-dismissible"><p>' . esc_html__('SpamScrubber counters reset.', 'spamscrubber') . '</p></div>';
        $disabled_click = $early_submit = $njs_spam = $honeypot_count = $remote_spam = $total = 0;
    }
    ?>
    <div class="wrap">
        <h1><?php esc_html_e('SpamScrubber Settings', 'spamscrubber'); ?></h1>
        <div style="background: #fafbfc; border: 1px solid #ccc; padding: 16px; border-radius: 12px; margin-bottom: 24px; max-width: 500px;">
            <strong><?php esc_html_e('Spam Attempts Detected:', 'spamscrubber'); ?></strong>
            <ul style="margin-top:8px; margin-bottom:8px;">
                <li><b><?php esc_html_e('Disabled Submit Button Clicks:', 'spamscrubber'); ?></b> <span id="spamscrubber-disabled-click"><?php echo esc_html($disabled_click); ?></span></li>
                <li><b><?php esc_html_e('Early Form Submit Attempts:', 'spamscrubber'); ?></b> <span id="spamscrubber-early-submit"><?php echo esc_html($early_submit); ?></span></li>
                <li><b><?php esc_html_e('Non-Javascript Spam Attempts:', 'spamscrubber'); ?></b> <span id="spamscrubber-njs-count"><?php echo esc_html($njs_spam); ?></span></li>
                <li><b><?php esc_html_e('Honeypot Trap:', 'spamscrubber'); ?></b> <span id="spamscrubber-honeypot-count"><?php echo esc_html($honeypot_count); ?></span></li>
                <li><b><?php esc_html_e('Remote Spam Attempts:', 'spamscrubber'); ?></b> <span id="spamscrubber-remote-spam"><?php echo esc_html($remote_spam); ?></span></li>
                <li style="margin-top:10px"><b><?php esc_html_e('Total Attempts:', 'spamscrubber'); ?></b> <span id="spamscrubber-total"><?php echo esc_html($total); ?></span></li>
            </ul>
            <form method="post" style="display:inline">
                <?php wp_nonce_field('spamscrubber_reset_stats', 'spamscrubber_reset_stats_nonce'); ?>
                <input type="hidden" name="spamscrubber_reset_stats" value="1" />
                <button type="submit" class="button" onclick="return confirm('<?php echo esc_js(__('Reset all SpamScrubber stats?', 'spamscrubber')); ?>')"><?php esc_html_e('Reset Counters', 'spamscrubber'); ?></button>
            </form>
            <p><em><?php esc_html_e('SpamScrubber uses ARIA attributes and tabindex for better accessibility.', 'spamscrubber'); ?></em></p>
        </div>
        <form method="post" action="options.php">
            <?php settings_fields('spamscrubber_settings_group'); ?>
            <table class="form-table">
                <tr valign="top">
                    <th scope="row"><?php esc_html_e('Delay in seconds', 'spamscrubber'); ?></th>
                    <td>
                        <input type="number" min="1" name="<?php echo esc_attr(SPAMSCRUBBER_OPTION_DELAY); ?>" value="<?php echo esc_attr(get_option(SPAMSCRUBBER_OPTION_DELAY, 4)); ?>" />
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><?php esc_html_e('Error Page URL', 'spamscrubber'); ?></th>
                    <td>
                        <input type="text" name="<?php echo esc_attr(SPAMSCRUBBER_OPT_ERROR_URL); ?>"
                            value="<?php echo esc_attr($error_url); ?>" style="width:200px" />
                        <br>
                        <small><?php esc_html_e('Path for the spam error page (e.g. /submit-form). Must begin with a slash.', 'spamscrubber'); ?></small>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><?php esc_html_e('Exclude These Pages/Posts', 'spamscrubber'); ?></th>
                    <td>
                        <select name="spamscrubber_exclude_ids[]" multiple size="6" style="min-width:240px;">
                            <?php foreach ($pages as $p): ?>
                                <option value="<?php echo esc_attr($p->ID); ?>"<?php selected(in_array((string)$p->ID, (array)$exclude_ids, true)); ?>>
                                    <?php echo esc_html($p->post_title) . ' (' . esc_html($p->post_type) . ', ID:' . esc_html($p->ID) . ')'; ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <br>
                        <small><?php esc_html_e('Selected posts/pages will not have form obfuscation or delay. Hold Ctrl/Cmd to select multiple.', 'spamscrubber'); ?></small>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><?php esc_html_e('Enable Honeypot', 'spamscrubber'); ?></th>
                    <td>
                        <label>
                            <input type="checkbox" name="spamscrubber_honeypot_enabled" value="1" <?php checked($honeypot_enabled); ?> />
                            <?php esc_html_e('Enable honeypot anti-bot field', 'spamscrubber'); ?>
                        </label>
                    </td>
                </tr>
                <tr valign="top">
                    <th scope="row"><?php esc_html_e('Honeypot Field', 'spamscrubber'); ?></th>
                    <td>
                        <select name="spamscrubber_honeypot_field">
                            <?php foreach ($honeypot_options as $opt): ?>
                                <option value="<?php echo esc_attr($opt); ?>" <?php selected($honeypot_field, $opt); ?>>
                                    <?php echo esc_html($opt); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <br>
                        <small><?php esc_html_e("This invisible field is added to forms. If filled, it's spam!", 'spamscrubber'); ?></small>
                    </td>
                </tr>
            </table>
            <?php submit_button(); ?>
        </form>
    </div>
    <?php
}

/* === Front-end output transform === */
add_action('template_redirect', 'spamscrubber_template_redirect_buffer');
function spamscrubber_template_redirect_buffer() {
    if (spamscrubber_is_excluded_page() || spamscrubber_is_admin_or_login()) return;
    if (is_feed() || is_embed() || is_robots() || is_preview()) return;
    ob_start('spamscrubber_obfuscate_forms');
}
function spamscrubber_obfuscate_forms($content) {
    $error_url        = get_option(SPAMSCRUBBER_OPT_ERROR_URL, '/submit-form');
    $honeypot_enabled = get_option('spamscrubber_honeypot_enabled', true);
    $honeypot_field   = get_option('spamscrubber_honeypot_field', __('Middle Name', 'spamscrubber'));
    $honeypot_name    = strtolower(preg_replace('/[^a-z0-9]+/i', '_', $honeypot_field));
    $honeypot_label   = esc_html($honeypot_field);

    return preg_replace_callback('/<form\b([^>]*)>/i', function($matches) use ($error_url, $honeypot_enabled, $honeypot_name, $honeypot_label) {
        $attrs = $matches[1];

        // Skip GET forms
        if (preg_match('/\bmethod\s*=\s*([\'"])get\1/i', $attrs)) {
            return $matches[0];
        }

        // Replace action with error page, stash real action in data attribute
        if (preg_match('/action=[\'"]([^\'"]*)[\'"]/i', $attrs, $am)) {
            $original_action = $am[1];
            $new_attrs = preg_replace(
                '/action=[\'"][^\'"]*[\'"]/',
                'action="'.esc_attr($error_url).'" data-spamscrubber-action="'.esc_attr($original_action).'"',
                $attrs
            );
        } else {
            $original_action = '';
            $new_attrs = $attrs . ' action="'.esc_attr($error_url).'" data-spamscrubber-action=""';
        }

        $extra_html = '';
        if ($honeypot_enabled) {
            $extra_html = '<div class="spamscrubber-extra-wrap" aria-hidden="true">'.
                '<label for="spamscrubber_' . esc_attr($honeypot_name) . '">' . $honeypot_label . '</label>'.
                '<input type="text" name="' . esc_attr($honeypot_name) . '" id="spamscrubber_' . esc_attr($honeypot_name) . '" autocomplete="off" tabindex="-1" />'.
                '</div>';
        }
        return '<form' . $new_attrs . '>' . $extra_html;
    }, $content);
}

/* === Security checks (token + honeypot) === */
add_action('init', 'spamscrubber_check_js_and_honeypot');
function spamscrubber_check_js_and_honeypot() {
    if (spamscrubber_is_excluded_page()) return;
    if (spamscrubber_is_admin_or_login()) return;

    if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'POST') {
        $js_token = isset($_POST['spamscrubber_js_token'])
            ? sanitize_text_field( wp_unslash( $_POST['spamscrubber_js_token'] ) )
            : '';

        if (empty($js_token) || !wp_verify_nonce($js_token, 'spamscrubber_js_token')) {
            $remote = (int) get_option(SPAMSCRUBBER_OPT_REMOTE, 0);
            update_option(SPAMSCRUBBER_OPT_REMOTE, $remote + 1);
            $error_url = get_option(SPAMSCRUBBER_OPT_ERROR_URL, '/submit-form');
            wp_safe_redirect( home_url( $error_url ) );
            exit;
        }

        if (get_option('spamscrubber_honeypot_enabled', true)) {
            $field = get_option('spamscrubber_honeypot_field', __('Middle Name', 'spamscrubber'));
            $name = strtolower(preg_replace('/[^a-z0-9]+/i', '_', $field));
            if (!empty($_POST[$name])) {
                $hp = (int) get_option(SPAMSCRUBBER_OPT_HONEYPOT, 0);
                update_option(SPAMSCRUBBER_OPT_HONEYPOT, $hp + 1);
                $error_url = get_option(SPAMSCRUBBER_OPT_ERROR_URL, '/submit-form');
                wp_safe_redirect( home_url( $error_url ) );
                exit;
            }
        }
    }
}

/* === Front-end JS (using enqueue APIs & modern inline data) === */
add_action('wp_enqueue_scripts', 'spamscrubber_enqueue_frontend_script');
function spamscrubber_enqueue_frontend_script() {
    if (spamscrubber_is_excluded_page() || spamscrubber_is_admin_or_login()) return;

    $delay     = (int) get_option(SPAMSCRUBBER_OPTION_DELAY, 4);
    $token     = wp_create_nonce('spamscrubber_js_token');
    $ajaxNonce = wp_create_nonce('spamscrubber_spam_event');

    // WordPress 6.3+ supports 'strategy' in $args; earlier versions treat it like $in_footer.
    wp_register_script(
        'spamscrubber',
        plugin_dir_url(__FILE__) . 'spamscrubber.js',
        array('jquery'),
        SPAMSCRUBBER_VERSION,
        array('in_footer' => true, 'strategy' => 'defer')
    );

    // Add script loading strategy via metadata too, for graceful cross-version behavior.
    if (function_exists('wp_script_add_data')) {
        wp_script_add_data('spamscrubber', 'strategy', 'defer');
    }

    // Provide config BEFORE the script runs (preferred over wp_localize_script for non-i18n data).
    $cfg = array(
        'delay'    => $delay,
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce'    => $ajaxNonce,
        'js_token' => $token,
    );
    wp_add_inline_script('spamscrubber', 'window.SpamScrubberData = ' . wp_json_encode($cfg) . ';', 'before');

    wp_enqueue_script('spamscrubber');
}

function spamscrubber_verify_ajax_nonce() {
    if (!isset($_REQUEST['nonce'])) return false;
    $n = sanitize_text_field( wp_unslash( $_REQUEST['nonce'] ) );
    return (bool) wp_verify_nonce($n, 'spamscrubber_spam_event');
}

/* === AJAX handlers === */
add_action('wp_ajax_nopriv_spamscrubber_disabled_click', 'spamscrubber_handle_disabled_click');
add_action('wp_ajax_spamscrubber_disabled_click',        'spamscrubber_handle_disabled_click');
function spamscrubber_handle_disabled_click() {
    if (!spamscrubber_verify_ajax_nonce()) {
        wp_send_json_error(['message' => 'Bad nonce'], 403);
    }
    if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'POST') {
        $count = (int) get_option(SPAMSCRUBBER_OPT_DISABLED_CLICK, 0);
        update_option(SPAMSCRUBBER_OPT_DISABLED_CLICK, $count + 1);
    }
    wp_send_json_success(['count' => (int) get_option(SPAMSCRUBBER_OPT_DISABLED_CLICK, 0)]);
}

add_action('wp_ajax_nopriv_spamscrubber_early_submit', 'spamscrubber_handle_early_submit');
add_action('wp_ajax_spamscrubber_early_submit',        'spamscrubber_handle_early_submit');
function spamscrubber_handle_early_submit() {
    if (!spamscrubber_verify_ajax_nonce()) {
        wp_send_json_error(['message' => 'Bad nonce'], 403);
    }
    if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'POST') {
        $count = (int) get_option(SPAMSCRUBBER_OPT_EARLY_SUBMIT, 0);
        update_option(SPAMSCRUBBER_OPT_EARLY_SUBMIT, $count + 1);
    }
    wp_send_json_success(['count' => (int) get_option(SPAMSCRUBBER_OPT_EARLY_SUBMIT, 0)]);
}

/* === Error page rewrite & display === */
add_action('init', 'spamscrubber_add_error_page_rewrite');
function spamscrubber_add_error_page_rewrite() {
    $error_url = get_option(SPAMSCRUBBER_OPT_ERROR_URL, '/submit-form');
    $url_parts = wp_parse_url($error_url);
    $slug = isset($url_parts['path']) ? ltrim($url_parts['path'], '/') : '';
    if (!$slug) $slug = 'submit-form';
    add_rewrite_rule('^' . preg_quote($slug, '/') . '/?$', 'index.php?spamscrubber_blocked=1', 'top');
}
add_action('update_option_' . SPAMSCRUBBER_OPT_ERROR_URL, 'spamscrubber_flush_rewrite_on_url_change', 10, 2);
function spamscrubber_flush_rewrite_on_url_change($old, $new) {
    foreach ([$old, $new] as $url) {
        $url_parts = wp_parse_url($url);
        $slug = isset($url_parts['path']) ? ltrim($url_parts['path'], '/') : '';
        if (!$slug) $slug = 'submit-form';
        add_rewrite_rule('^' . preg_quote($slug, '/') . '/?$', 'index.php?spamscrubber_blocked=1', 'top');
    }
    flush_rewrite_rules();
}
add_filter('query_vars', 'spamscrubber_register_query_var');
function spamscrubber_register_query_var($vars) {
    $vars[] = 'spamscrubber_blocked';
    return $vars;
}

add_action('template_redirect', 'spamscrubber_display_blocked_page');
function spamscrubber_display_blocked_page() {
    if (get_query_var('spamscrubber_blocked')) {
        $njs = (int) get_option(SPAMSCRUBBER_OPT_NJS, 0);
        update_option(SPAMSCRUBBER_OPT_NJS, $njs + 1);

        status_header(403);
        nocache_headers();

        // Title as plain text, escaped at output time.
        $title = __('Submission Blocked', 'spamscrubber');

        // Body text is escaped per-string; then we allow only safe HTML tags at output.
        $msg   = '<p>' . esc_html__('Possible spam detected. Please make sure JavaScript is enabled and try submitting the form again.', 'spamscrubber') . '</p>'
               . '<p>' . esc_html__('If you continue to see this message, contact our support team.', 'spamscrubber') . '</p>';

        // ✅ Escape at output to satisfy PHPCS: message via wp_kses_post(), title via esc_html()
        wp_die(
            wp_kses_post($msg),
            esc_html($title),
            array('response' => 403, 'back_link' => true)
        );
    }
}

/* === Activation/deactivation === */
register_activation_hook(__FILE__, 'spamscrubber_plugin_activate');
function spamscrubber_plugin_activate() {
    $error_url = get_option(SPAMSCRUBBER_OPT_ERROR_URL, '/submit-form');
    $url_parts = wp_parse_url($error_url);
    $slug = isset($url_parts['path']) ? ltrim($url_parts['path'], '/') : '';
    if (!$slug) $slug = 'submit-form';
    add_rewrite_rule('^' . preg_quote($slug, '/') . '/?$', 'index.php?spamscrubber_blocked=1', 'top');
    flush_rewrite_rules();
}
register_deactivation_hook(__FILE__, 'spamscrubber_plugin_deactivate');
function spamscrubber_plugin_deactivate() {
    flush_rewrite_rules();
}
