<?php
/**
 * Plugin Name: Sort Options Label Editor for WooCommerce
 * Plugin URI:  https://wordpress.org/plugins/sort-options-label-editor-for-woocommerce
 * Description: Customize the display text and visibility of WooCommerce product sorting options with an easy-to-use admin interface.
 * Version:     1.0.4
 * Requires at least: 5.8
 * Tested up to: 6.9
 * Requires PHP: 7.4
 * Requires Plugins: woocommerce
 * Author:      Vinod
 * Author URI:  https://profiles.wordpress.org/vinodkudavoor/
 * License:     GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain: sort-options-label-editor-for-woocommerce
 * Domain Path: /languages
 */

if ( ! defined( 'WPINC' ) ) {
    die;
}

define( 'SOL_EDITOR_VERSION', '1.0.2' ); // Updated version here
define( 'SOL_EDITOR_PLUGIN_NAME', 'sort-options-label-editor-for-woocommerce' );

/**
 * The code that runs during plugin activation.
 * This action is documented in includes/class-sol-editor-activator.php
 */
function sol_editor_activate() {
    require_once plugin_dir_path( __FILE__ ) . 'includes/class-sol-editor-activator.php';
    SOL_Editor_Activator::activate();
}

/**
 * The code that runs during plugin deactivation.
 * This action is documented in includes/class-sol-editor-deactivator.php (though not created, just for documentation consistency)
 */
function sol_editor_deactivate() {
    // No specific deactivation tasks for this plugin, but you can add them here if needed.
}

register_activation_hook( __FILE__, 'sol_editor_activate' );
register_deactivation_hook( __FILE__, 'sol_editor_deactivate' );

/**
 * The core plugin class that is used to define internationalization,
 * admin-specific hooks, and public-facing site hooks.
 */
require plugin_dir_path( __FILE__ ) . 'admin/class-sol-editor-admin.php';

/**
 * Begins execution of the plugin.
 *
 * Since everything within the plugin is registered via hooks,
 * then there is no need to create an object and run an init method.
 */
function sol_editor_run_plugin() {
    $plugin_admin = new SOL_Editor_Admin();
    $plugin_admin->run();
}
sol_editor_run_plugin();

/**
 * Load the plugin text domain for internationalization.
 * This function is no longer needed when hosted on WordPress.org
 * because translations are loaded automatically based on the 'Text Domain' header.
 *
 * @since 1.0.0
 */
// REMOVED FOR WORDPRESS.ORG COMPLIANCE:
// function sol_editor_load_textdomain() {
//     load_plugin_textdomain(
//         'sort-options-label-editor-for-woocommerce',
//         false,
//         dirname( plugin_basename( __FILE__ ) ) . '/languages/'
//     );
// }
// add_action( 'plugins_loaded', 'sol_editor_load_textdomain' );


/**
 * Custom sorting filter.
 * This function retrieves the custom sorting texts and visibility settings from the plugin options
 * and applies them to the WooCommerce catalog orderby options.
 *
 * @param array $options An array of WooCommerce sorting options.
 * @return array The filtered array of WooCommerce sorting options.
 */
add_filter( 'woocommerce_catalog_orderby', 'sol_editor_options_filter' );

function sol_editor_options_filter( $options ) {
    $saved_options = get_option( 'sol_editor_options', array() );

    $custom_options = array();

    // Define all possible sorting keys and their default labels
    $all_sort_keys = array(
        'menu_order' => __( 'Default Sorting', 'sort-options-label-editor-for-woocommerce' ),
        'popularity' => __( 'Sort by popularity', 'sort-options-label-editor-for-woocommerce' ), // WooCommerce default
        'rating'     => __( 'Sort by average rating', 'sort-options-label-editor-for-woocommerce' ), // WooCommerce default
        'date'       => __( 'Sort by latest', 'sort-options-label-editor-for-woocommerce' ), // WooCommerce default
        'price'      => __( 'Sort by price: low to high', 'sort-options-label-editor-for-woocommerce' ), // WooCommerce default
        'price-desc' => __( 'Sort by price: high to low', 'sort-options-label-editor-for-woocommerce' ), // WooCommerce default
    );

    foreach ( $options as $key => $label ) {
        // Get custom label from options, fallback to plugin default, then WooCommerce default if not set.
        $custom_label = isset( $saved_options[ $key ] ) && ! empty( $saved_options[ $key ] )
                        ? sanitize_text_field( $saved_options[ $key ] )
                        : ( isset( $all_sort_keys[ $key ] ) ? $all_sort_keys[ $key ] : $label );

        // Check visibility. If 'hide' is set and true, skip this option.
        if ( isset( $saved_options[ $key . '_hide' ] ) && $saved_options[ $key . '_hide' ] === 'yes' ) {
            continue; // Skip this option if it's set to be hidden
        }

        $custom_options[ $key ] = $custom_label;
    }

    // This ensures that the original order from WooCommerce is preserved,
    // and only the labels and visibility are modified.
    return $custom_options;
}


/**
 * Add a direct link to the plugin settings page from the plugin listing page.
 *
 * @param array $links An array of plugin action links.
 * @return array The filtered array of plugin action links.
 */
function sol_editor_action_links( $links ) {
    $settings_link = '<a href="' . esc_url( admin_url( 'options-general.php?page=sol-editor-settings' ) ) . '">' . esc_html__( 'Settings', 'sort-options-label-editor-for-woocommerce' ) . '</a>';
    array_unshift( $links, $settings_link );
    return $links;
}
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'sol_editor_action_links' );

/**
 * Display an admin notice for donation.
 */
function sol_editor_admin_donate_notice() {
    $screen = get_current_screen();
    // Only display on the plugins page.
    if ( isset($screen->id) && 'plugins' === $screen->id ) {
        ?>
        <div class="notice notice-info is-dismissible">
            <p>
                <?php
                printf(
                    /* translators: %s: Link to Buy Me a Coffee page. */
                    esc_html__( 'If you find the Sort Options Label Editor for WooCommerce useful, please consider supporting its development via %s.', 'sort-options-label-editor-for-woocommerce' ),
                    '<a href="https://buymeacoffee.com/wemakeitlive" target="_blank">' . esc_html__( 'Buy Me a Coffee', 'sort-options-label-editor-for-woocommerce' ) . '</a>'
                );
                ?>
            </p>
        </div>
        <?php
    }
}
add_action( 'admin_notices', 'sol_editor_admin_donate_notice' );