<?php
/**
 * The admin-specific functionality of the plugin.

 * @package    Sort_Options_Label_Editor_For_WooCommerce
 * @subpackage Sort_Options_Label_Editor_For_WooCommerce/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and hooks to
 * register the admin-specific stylesheet and JavaScript.
 *
 * @package    Sort_Options_Label_Editor_For_WooCommerce
 * @subpackage Sort_Options_Label_Editor_For_WooCommerce/admin
 * @author     Vinod<https://profiles.wordpress.org/vinodkudavoor/>
 */
class SOL_Editor_Admin {

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     */
    public function __construct() {

    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function sol_editor_enqueue_styles() {
        // Enqueue styles for the Pro features section.
        wp_enqueue_style( 'sol-editor-admin', plugin_dir_url( __FILE__ ) . 'css/sol-editor-admin.css', array(), SOL_EDITOR_VERSION, 'all' );
    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function sol_editor_enqueue_scripts() {
        // You can add admin-specific JS here if needed.
        // wp_enqueue_script( 'sol-editor-admin', plugin_dir_url( __FILE__ ) . 'js/sol-editor-admin.js', array( 'jquery' ), SOL_EDITOR_VERSION, false );
    }

    /**
     * Add options page for the plugin.
     *
     * @since    1.0.0
     */
    public function sol_editor_add_plugin_admin_menu() {
        add_options_page(
            esc_html__( 'Sort Options Label Editor for WooCommerce', 'sort-options-label-editor-for-woocommerce' ),
            esc_html__( 'Product Sorting Labels', 'sort-options-label-editor-for-woocommerce' ),
            'manage_options',
            'sol-editor-settings', // Unique slug for the settings page
            array( $this, 'sol_editor_display_plugin_setup_page' )
        );
    }

    /**
     * Render the plugin's admin page.
     *
     * @since    1.0.0
     */
    public function sol_editor_display_plugin_setup_page() {
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Sort Options Label Editor for WooCommerce Settings', 'sort-options-label-editor-for-woocommerce' ); ?></h1>
            
            <form method="post" action="options.php">
                <?php
                settings_fields( 'sol_editor_options_group' ); // Option group
                do_settings_sections( 'sol-editor-settings' );     // Page slug
                submit_button();
                ?>
            </form>

            <div class="sol-pro-features">
                <h2><?php esc_html_e( 'Unlock More Features with Pro!', 'sort-options-label-editor-for-woocommerce' ); ?></h2>
                <p><?php esc_html_e( 'Upgrade to the Pro version for advanced control over your product sorting.', 'sort-options-label-editor-for-woocommerce' ); ?></p>
                
                <div class="pro-features-grid">
                    <div class="pro-feature-card">
                        <h3><span class="dashicons dashicons-lock"></span> <?php esc_html_e( 'Drag & Drop Reordering', 'sort-options-label-editor-for-woocommerce' ); ?></h3>
                        <p><?php esc_html_e( 'Easily reorder the sorting options by dragging and dropping them into your preferred sequence.', 'sort-options-label-editor-for-woocommerce' ); ?></p>
                        <ul class="sol-pro-sortable-list">
                            <li><?php esc_html_e( 'Default Sorting', 'sort-options-label-editor-for-woocommerce' ); ?></li>
                            <li><?php esc_html_e( 'Sort by popularity', 'sort-options-label-editor-for-woocommerce' ); ?></li>
                            <li><?php esc_html_e( 'Sort by latest', 'sort-options-label-editor-for-woocommerce' ); ?></li>
                        </ul>
                    </div>
                    
                    <div class="pro-feature-card">
                        <h3><span class="dashicons dashicons-lock"></span> <?php esc_html_e( 'Set a Custom Default', 'sort-options-label-editor-for-woocommerce' ); ?></h3>
                        <p><?php esc_html_e( 'Choose any sorting option as the default for your shop page when customers first visit.', 'sort-options-label-editor-for-woocommerce' ); ?></p>
                        <select disabled>
                            <option><?php esc_html_e( 'Sort by latest', 'sort-options-label-editor-for-woocommerce' ); ?></option>
                            <option><?php esc_html_e( 'Sort by price: high to low', 'sort-options-label-editor-for-woocommerce' ); ?></option>
                        </select>
                    </div>
                </div>

                <div class="sol-upgrade-cta">
                    <a href="https://kudavoor.gumroad.com/l/sort-options-label-editor-pro-for-woocommerce" target="_blank" class="button button-primary sol-upgrade-button">
                        <?php esc_html_e( 'Upgrade to Pro Now', 'sort-options-label-editor-for-woocommerce' ); ?>
                    </a>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Register and define the settings for the plugin.
     *
     * @since    1.0.0
     */
    public function sol_editor_register_settings() {
        register_setting(
            'sol_editor_options_group',              // Option group
            'sol_editor_options',                    // Option name
            array( $this, 'sol_editor_sanitize_options' ) // Sanitize callback
        );

        add_settings_section(
            'sol_editor_main_section',                   // ID
            esc_html__( 'Customize Sorting Display Text and Visibility', 'sort-options-label-editor-for-woocommerce' ), // Title
            array( $this, 'sol_editor_print_section_info' ), // Callback
            'sol-editor-settings'                        // Page
        );

        // Define all sorting options to manage dynamically
        $sort_options_to_manage = array(
            'menu_order' => esc_html__( 'Default Sorting', 'sort-options-label-editor-for-woocommerce' ),
            'popularity' => esc_html__( 'Sort by popularity', 'sort-options-label-editor-for-woocommerce' ),
            'rating'     => esc_html__( 'Sort by average rating', 'sort-options-label-editor-for-woocommerce' ),
            'date'       => esc_html__( 'Sort by latest', 'sort-options-label-editor-for-woocommerce' ),
            'price'      => esc_html__( 'Sort by price: low to high', 'sort-options-label-editor-for-woocommerce' ),
            'price-desc' => esc_html__( 'Sort by price: high to low', 'sort-options-label-editor-for-woocommerce' ),
        );

        foreach ( $sort_options_to_manage as $key => $label ) {
            add_settings_field(
                'sol_editor_' . $key,                        // ID (prefixed)
                $label,                                      // Title
                array( $this, 'sol_editor_text_and_checkbox_callback' ), // Callback to render input and checkbox
                'sol-editor-settings',                       // Page
                'sol_editor_main_section',                   // Section
                array( 'field' => $key, 'label' => $label )  // Args to pass to the callback
            );
        }
    }

    /**
     * Sanitize each setting field as needed.
     *
     * @param array $input Contains all settings fields as array keys.
     * @since    1.0.0
     * @return array The sanitized input.
     */
    public function sol_editor_sanitize_options( $input ) {
        $new_input = array();

        $sort_keys = array( 'menu_order', 'popularity', 'rating', 'date', 'price', 'price-desc' );

        foreach ( $sort_keys as $key ) {
            if ( isset( $input[ $key ] ) ) {
                $new_input[ $key ] = sanitize_text_field( $input[ $key ] );
            }
            // Sanitize checkbox for hiding
            $hide_key = $key . '_hide';
            if ( isset( $input[ $hide_key ] ) ) {
                $new_input[ $hide_key ] = 'yes'; // Only save 'yes' if checked
            } else {
                $new_input[ $hide_key ] = 'no'; // Explicitly save 'no' if unchecked
            }
        }
        return $new_input;
    }

    /**
     * Print the Section text.
     *
     * @since    1.0.0
     */
    public function sol_editor_print_section_info() {
        esc_html_e( 'Enter custom display text for each product sorting option and choose whether to hide it.', 'sort-options-label-editor-for-woocommerce' );
    }

    /**
     * Callback for text input and checkbox for each sorting option.
     *
     * @param array $args Arguments passed from add_settings_field (field key, label).
     * @since    1.0.0
     */
    public function sol_editor_text_and_checkbox_callback( $args ) {
        $field_key = $args['field'];
        $field_label = $args['label'];
        $options = get_option( 'sol_editor_options' );

        // Text input for custom label
        $value = isset( $options[ $field_key ] ) ? $options[ $field_key ] : '';

        printf(
            '<input type="text" id="%1$s" name="sol_editor_options[%1$s]" value="%2$s" class="regular-text" placeholder="%3$s" />',
            esc_attr( $field_key ),
            esc_attr( $value ), // FIX: Escaped the output value for security.
            esc_attr( $field_label )
        );

        // Checkbox for hiding the option
        $hide_key = $field_key . '_hide';
        $hidden = isset( $options[ $hide_key ] ) && $options[ $hide_key ] === 'yes' ? 'checked' : '';
        printf(
            '<label for="%1$s_hide" style="margin-left: 10px;"><input type="checkbox" id="%1$s_hide" name="sol_editor_options[%1$s_hide]" value="yes" %3$s /> %2$s</label>',
            esc_attr( $field_key ),
            esc_html__( 'Hide this option', 'sort-options-label-editor-for-woocommerce' ),
            esc_attr( $hidden )
        );
        echo '<p class="description">' . esc_html__( 'Leave blank to use the default WooCommerce label.', 'sort-options-label-editor-for-woocommerce' ) . '</p>';
    }

    /**
     * Run the admin functionality.
     *
     * @since    1.0.0
     */
    public function run() {
        add_action( 'admin_enqueue_scripts', array( $this, 'sol_editor_enqueue_styles' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'sol_editor_enqueue_scripts' ) );
        add_action( 'admin_menu', array( $this, 'sol_editor_add_plugin_admin_menu' ) );
        add_action( 'admin_init', array( $this, 'sol_editor_register_settings' ) );
    }
}