<?php
/**
 * Custom icons for this Social Snap.
 *
 * @package    Social Snap
 * @author     Social Snap
 * @since      1.2.0
 * @license    GPL-3.0+
 * @copyright  Copyright (c) 2020, Social Snap LLC
 */

/**
 * Do not allow direct script access.
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'SocialSnap_Icons' ) ) :
	/**
	 * Dynamically generate CSS code.
	 */
	class SocialSnap_Icons {

		/**
		 * Primary class constructor.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
		}

		/**
		 * GET SVG CODE
		 * Get the SVG code for the specified icon
		 *
		 * @param string $icon Icon name.
		 */
		public function get_svg( $icon, $fill = '' ) {

			$arr = self::get();

			if ( array_key_exists( $icon, $arr ) ) {
				$repl = '<svg class="ss-svg-icon" aria-hidden="true" role="img" focusable="false" ';
				$svg  = preg_replace( '/^<svg /', $repl, trim( $arr[ $icon ] ) ); // Add extra attributes to SVG code.
				$svg  = preg_replace( "/([\n\t]+)/", ' ', $svg ); // Remove newlines & tabs.
				$svg  = preg_replace( '/>\s*</', '><', $svg );    // Remove whitespace between SVG tags.

				// Make sure that only our allowed tags and attributes are included.
				$svg = wp_kses(
					$svg,
					array(
						'svg'     => array(
							'class'       => true,
							'xmlns'       => true,
							'width'       => true,
							'height'      => true,
							'viewbox'     => true,
							'aria-hidden' => true,
							'role'        => true,
							'focusable'   => true,
						),
						'path'    => array(
							'fill'      => true,
							'fill-rule' => true,
							'd'         => true,
							'transform' => true,
						),
						'polygon' => array(
							'fill'      => true,
							'fill-rule' => true,
							'points'    => true,
							'transform' => true,
							'focusable' => true,
						),
					)
				);

				if ( ! $svg ) {
					return null;
				}

				return $svg;
			}

			return null;
		}

		/**
		 * Get SVG icons array.
		 *
		 * @return array
		 */
		public function get() {
			return apply_filters( 'socialsnap_icons_svg', self::$icons );
		}

		/**
		 * Get all available SVG icons
		 *
		 * @return array
		 */
		public function get_all_svg_icons() {
			$arr = self::get();

			$return = array();

			foreach ( $arr as $icon => $svg ) {

				$svg = self::get_svg( $icon );

				if ( $svg ) {
					$return[ $icon ] = self::get_svg( $icon );
				}
			}

			return $return;
		}

		/**
		 * Icons.
		 * Store the code for all SVGs in an array.
		 *
		 * @var array
		 */
		public static $icons = array(
			'copy'              => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M25.313 28v-18.688h-14.625v18.688h14.625zM25.313 6.688c1.438 0 2.688 1.188 2.688 2.625v18.688c0 1.438-1.25 2.688-2.688 2.688h-14.625c-1.438 0-2.688-1.25-2.688-2.688v-18.688c0-1.438 1.25-2.625 2.688-2.625h14.625zM21.313 1.313v2.688h-16v18.688h-2.625v-18.688c0-1.438 1.188-2.688 2.625-2.688h16z"></path></svg>',
			'500px'             => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M7.904 21.024c0 0.032 0.16 0.48 0.256 0.704 0.512 1.248 1.28 2.368 2.24 3.328s2.080 1.728 3.328 2.24c1.28 0.544 2.656 0.832 4.064 0.832s2.784-0.288 4.064-0.832c1.248-0.512 2.368-1.28 3.328-2.24s1.728-2.080 2.24-3.328c0.544-1.28 0.832-2.656 0.832-4.064s-0.288-2.784-0.832-4.064c-0.512-1.248-1.28-2.368-2.24-3.328s-2.080-1.728-3.328-2.24c-1.28-0.544-2.656-0.832-4.064-0.832-1.44 0-2.848 0.288-4.128 0.832-1.024 0.416-2.752 1.536-3.744 2.592v0-8.672h14.496c0.512 0 0.512-0.736 0.512-0.992s0-0.992-0.544-0.992l-15.616 0.032c-0.416 0-0.704 0.352-0.704 0.672v12.128c0 0.384 0.48 0.672 0.928 0.768 0.896 0.192 1.088-0.096 1.312-0.384l0.032-0.032c0.32-0.48 1.376-1.536 1.376-1.536 1.6-1.6 3.776-2.496 6.080-2.496s4.448 0.896 6.048 2.496c1.632 1.632 2.496 3.776 2.496 6.048s-0.896 4.416-2.496 6.048c-1.6 1.6-3.808 2.496-6.080 2.496-1.536 0-3.040-0.416-4.32-1.184v-7.392c0-0.992 0.416-2.048 1.152-2.848 0.832-0.928 1.952-1.44 3.168-1.44 1.184 0 2.304 0.448 3.136 1.28 0.832 0.8 1.28 1.888 1.28 3.040 0 2.464-1.952 4.384-4.416 4.384-0.48 0-1.344-0.224-1.376-0.224-0.512-0.16-0.704 0.544-0.768 0.768-0.256 0.864 0.128 1.056 0.224 1.088 0.8 0.256 1.312 0.288 2.016 0.288 3.488 0 6.336-2.848 6.336-6.336 0-3.456-2.848-6.304-6.336-6.304-1.696 0-3.328 0.64-4.512 1.856-1.152 1.12-1.824 2.624-1.824 4.128v0.032c0 0.192 0 4.608 0 6.048v0c-0.672-0.736-1.312-1.856-1.728-2.976-0.16-0.448-0.544-0.352-1.056-0.192-0.288 0.096-0.992 0.288-0.832 0.8v0zM15.36 19.296c0 0.224 0.192 0.384 0.32 0.512v0.032c0.192 0.192 0.384 0.288 0.576 0.288 0.16 0 0.224-0.064 0.256-0.096 0.096-0.096 1.088-1.088 1.184-1.184l1.12 1.088c0.096 0.128 0.224 0.16 0.352 0.192 0.192 0 0.384-0.096 0.576-0.32 0.48-0.48 0.224-0.736 0.128-0.864l-1.12-1.12 1.152-1.184c0.256-0.288 0.032-0.576-0.192-0.8-0.32-0.32-0.64-0.416-0.832-0.224l-1.152 1.152-1.184-1.184c-0.064-0.064-0.128-0.096-0.224-0.096-0.16 0-0.352 0.096-0.544 0.32-0.352 0.352-0.448 0.608-0.256 0.8l1.184 1.184-1.152 1.184c-0.16 0.096-0.192 0.224-0.192 0.32zM17.92 3.424c-1.888 0-3.872 0.384-5.344 1.024-0.16 0.064-0.256 0.192-0.256 0.352-0.032 0.16 0.032 0.384 0.128 0.672 0.096 0.224 0.32 0.864 0.8 0.672 1.504-0.576 3.168-0.896 4.672-0.896 1.728 0 3.36 0.32 4.928 0.992 1.248 0.512 2.4 1.28 3.68 2.4 0.096 0.096 0.192 0.128 0.288 0.128 0.256 0 0.48-0.256 0.704-0.48 0.352-0.384 0.576-0.704 0.224-1.024-1.216-1.152-2.56-2.016-4.192-2.72-1.792-0.736-3.68-1.12-5.632-1.12zM28 26.592v0c-0.224-0.224-0.416-0.352-0.576-0.416s-0.32 0-0.448 0.096l-0.096 0.128c-1.152 1.152-2.528 2.080-4.032 2.72-1.568 0.672-3.232 0.992-4.928 0.992s-3.36-0.32-4.928-0.992c-1.504-0.64-2.848-1.536-4.032-2.72-1.216-1.216-2.112-2.56-2.72-4.032-0.576-1.408-0.768-2.496-0.832-2.848 0-0.032 0-0.064 0-0.064-0.064-0.416-0.448-0.448-0.992-0.352-0.224 0.032-0.928 0.128-0.864 0.64v0c0.192 1.152 0.512 2.272 0.96 3.36 0.736 1.728 1.792 3.296 3.104 4.608 1.344 1.344 2.88 2.4 4.608 3.104 1.792 0.768 3.712 1.152 5.664 1.152s3.872-0.384 5.664-1.152c1.728-0.736 3.296-1.792 4.608-3.104 0 0 0.064-0.064 0.128-0.128 0.192-0.128 0.32-0.416-0.288-0.992z"></path></svg>',
			'amazon'            => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M27.696 26.643c0.536-0.268 0.946 0.143 0.393 0.857s-5 4.786-12.464 4.786-13.179-5.107-14.929-7.214c-0.482-0.554 0.071-0.804 0.393-0.589 5.232 3.179 13.411 8.411 26.607 2.161zM31.393 24.589c0.268 0.357 0 1.929-0.464 3.071-0.464 1.125-1.143 1.911-1.518 2.214-0.393 0.321-0.679 0.196-0.464-0.268s1.375-3.321 0.911-3.929c-0.464-0.589-2.643-0.304-3.429-0.232-0.768 0.071-0.929 0.143-1-0.018-0.161-0.411 1.554-1.107 2.679-1.25 1.125-0.125 2.929-0.054 3.286 0.411zM24.357 16.679c0 1.964 2.304 3.768 2.304 3.768l-4.054 4c-1.589-1.5-2.786-2.75-2.786-2.75-0.179-0.179-0.321-0.393-0.446-0.589-3.232 5.054-13.107 4.732-13.107-3.089 0-7.286 8.625-8.268 12.107-8.393v-2.268c0-0.482 0.179-2.679-2.536-2.679 0 0-2.714 0-3.875 3.536l-5.25-0.482c0-3.518 3.339-7.446 9.625-7.446 6.268 0 8.018 4.071 8.018 5.875v10.518zM12.357 17.054c0 3.625 6.018 4.482 6.018-1.232v-2.893c-2.411 0.071-6.018 0.75-6.018 4.125z"></path></svg>',
			'analytics'         => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M29.333 15.333h-5.333c-0.575 0-1.083 0.367-1.267 0.908l-2.733 8.208-6.733-20.208c-0.183-0.542-0.692-0.908-1.267-0.908s-1.083 0.367-1.267 0.908l-3.692 11.092h-4.375c-0.733 0-1.333 0.6-1.333 1.333s0.6 1.333 1.333 1.333h5.333c0.575 0 1.083-0.367 1.267-0.908l2.733-8.208 6.733 20.208c0.183 0.542 0.692 0.908 1.267 0.908s1.083-0.367 1.267-0.908l3.692-11.092h4.375c0.733 0 1.333-0.6 1.333-1.333s-0.6-1.333-1.333-1.333z"></path></svg>',
			'api'               => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M30.667 16.667c0-8.008-6.458-14.542-14.442-14.667-0.142-0.025-0.283-0.025-0.425 0-7.992 0.108-14.467 6.642-14.467 14.667 0 8.008 6.458 14.542 14.442 14.667 0.067 0.008 0.142 0.017 0.208 0.017v0c0.067 0 0.142-0.008 0.208-0.017 8-0.108 14.475-6.642 14.475-14.667zM27.925 15.333h-5.325c-0.3-3.683-1.525-7.225-3.558-10.275 4.733 1.242 8.333 5.3 8.883 10.275zM15.992 27.942c-2.225-2.875-3.583-6.325-3.917-9.942h7.842c-0.333 3.633-1.7 7.083-3.925 9.942zM12.083 15.333c0.342-3.625 1.7-7.075 3.925-9.933 2.225 2.867 3.592 6.308 3.925 9.933h-7.85zM12.967 5.058c-2.033 3.050-3.258 6.592-3.567 10.275h-5.325c0.55-4.975 4.158-9.042 8.892-10.275zM4.075 18h5.325c0.3 3.675 1.525 7.208 3.55 10.275-4.725-1.242-8.325-5.308-8.875-10.275zM19.033 28.275c2.033-3.050 3.258-6.592 3.567-10.275h5.325c-0.55 4.975-4.158 9.042-8.892 10.275z"></path></svg>',
			'baidu'             => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M20.5 10.188c2.563 0.375 4.188-2.375 4.563-4.438 0.313-2.063-1.375-4.5-3.188-4.938-1.813-0.375-4.063 2.5-4.25 4.438-0.25 2.313 0.313 4.625 2.875 4.938zM5.75 16.875c3.5-0.75 3-4.938 2.938-5.813-0.188-1.438-1.875-3.875-4.063-3.688-2.813 0.25-3.25 4.313-3.25 4.313-0.375 1.875 0.938 5.938 4.375 5.188zM12.188 9.875c1.938 0 3.5-2.188 3.5-4.938s-1.563-4.938-3.5-4.938c-1.875 0-3.438 2.188-3.438 4.938s1.563 4.938 3.438 4.938zM9.438 24.125c-0.063 0.25-0.313 1-0.125 1.688 0.375 1.438 1.688 1.5 1.688 1.5h1.813v-4.438h-1.938c-0.875 0.25-1.313 0.938-1.438 1.25zM30.688 13.688c0-1-0.813-4-3.875-4-3.125 0-3.5 2.875-3.5 4.875 0 1.875 0.188 4.563 4 4.438 3.75-0.063 3.375-4.313 3.375-5.313zM26.813 22.438c0 0-4-3.063-6.313-6.375-3.125-4.938-7.625-2.938-9.125-0.438s-3.813 4.063-4.125 4.5c-0.313 0.375-4.813 2.813-3.813 7.188 1 4.438 4.5 4.313 4.5 4.313s2.563 0.25 5.5-0.375c3-0.688 5.563 0.125 5.563 0.125s7 2.375 8.875-2.125c1.875-4.5-1.063-6.813-1.063-6.813zM14.875 29.125h-4.5c-1.938-0.438-2.75-1.75-2.813-2-0.125-0.188-0.688-1.25-0.375-3.063 0.813-2.75 3.25-2.938 3.25-2.938h2.375v-3l2.063 0.063zM23.313 29.063h-5.188c-2.063-0.5-2.125-1.938-2.125-1.938v-5.75l2.125-0.063v5.188c0.125 0.563 0.813 0.688 0.813 0.688h2.125v-5.813h2.25z"></path></svg>',
			'behance'           => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M28.813 9.25h-8v-1.875h8zM15.563 16.813c0.625 0.875 0.938 1.875 0.938 3.125 0 1-0.188 1.875-0.563 2.563-0.438 0.75-0.938 1.313-1.563 1.813-0.688 0.438-1.438 0.75-2.25 1-0.813 0.188-1.688 0.313-2.563 0.313h-9.563v-19.875h9.25c0.938 0 1.813 0.188 2.625 0.375 0.75 0.188 1.375 0.5 1.938 0.875s1 0.938 1.313 1.563c0.313 0.563 0.438 1.375 0.438 2.313s-0.188 1.813-0.688 2.438c-0.438 0.688-1.125 1.25-2 1.625 1.25 0.375 2.125 1 2.688 1.875zM4.188 13.75h4.5c0.813 0 1.438-0.063 1.938-0.438s0.75-0.875 0.75-1.75c0-0.438-0.063-0.813-0.25-1.125-0.188-0.25-0.375-0.5-0.688-0.688-0.25-0.125-0.563-0.313-0.938-0.375s-0.75-0.125-1.125-0.125h-4.188zM12.188 19.688c0-1-0.313-1.688-0.875-2.125s-1.313-0.625-2.25-0.625h-4.875v5.438h4.75c0.438 0 0.875-0.063 1.25-0.125 0.375-0.125 0.688-0.25 1-0.438s0.563-0.438 0.688-0.813c0.188-0.313 0.313-0.75 0.313-1.313zM32 19.5h-10.313c0 1 0.375 2.188 1 2.75 0.563 0.563 1.438 0.875 2.5 0.875 0.813 0 1.5-0.25 2.063-0.688 0.563-0.375 0.875-0.688 1.063-1.313h3.438c-0.563 1.938-1.438 3-2.563 3.75s-2.5 1.188-4.125 1.188c-1.125 0-2.125-0.188-3.063-0.563-0.875-0.375-1.625-0.875-2.25-1.5-0.625-0.688-1.125-1.5-1.5-2.438-0.313-0.875-0.5-1.875-0.5-3 0-1.063 0.188-2.063 0.563-3 0.313-0.938 0.813-1.688 1.438-2.375 0.688-0.688 1.438-1.188 2.313-1.625 0.938-0.375 1.938-0.563 3-0.563 1.188 0 2.25 0.25 3.188 0.688 0.875 0.5 1.625 1.125 2.188 1.875 0.625 0.813 1 1.75 1.25 2.75s0.375 2.25 0.313 3.188zM28.063 16.938c-0.125-0.938-0.438-1.688-0.875-2.188-0.5-0.563-1.313-0.813-2.25-0.813-0.625 0-1.125 0.125-1.563 0.375-0.375 0.188-0.75 0.438-1 0.75s-0.375 0.625-0.5 1c-0.125 0.313-0.188 0.563-0.188 0.875z"></path></svg>',
			'blogger'           => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M29.75 12.188h-1.375c-1.375 0-2.125-0.938-2.125-1.938v-0.875c0-4.563-5.063-9.375-9.375-9.375h-6.75c-5.688 0-10.125 4.5-10.125 10.188v12.125c0 5.313 4.563 9.688 9.313 9.688h12.813c5.25 0 9.875-4.438 9.875-9.813v-7.688c0-1.438-0.5-2.313-2.25-2.313zM10.125 8.313h5.813c1 0 1.875 0.875 1.875 1.938 0 1-0.875 1.938-1.875 1.938h-5.813c-1 0-1.875-0.938-1.875-1.938s0.875-1.938 1.875-1.938zM21.688 23.688h-11.563c-1 0-1.875-0.75-1.875-1.75 0-1.063 0.875-1.75 1.875-1.75h11.563c1 0 1.875 0.688 1.875 1.75 0 1-0.875 1.75-1.875 1.75z"></path></svg>',
			'box'               => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M17.382 9.234c-2.816 0-5.392 1.034-7.382 2.736v-9.97c0-1.104-0.894-2-2-2s-2 0.896-2 2v18.618c0 6.276 5.106 11.382 11.382 11.382s11.382-5.106 11.382-11.382-5.104-11.384-11.382-11.384zM17.382 28c-4.070 0-7.382-3.312-7.382-7.382s3.312-7.382 7.382-7.382 7.382 3.312 7.382 7.382-3.31 7.382-7.382 7.382z"></path></svg>',
			'buffer'            => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M31.438 24.313c-0.063-0.063-0.188-0.125-0.25-0.188-0.813-0.375-1.688-0.75-2.5-1.125-0.688-0.375-1.438-0.375-2.188 0-3.125 1.438-6.25 2.938-9.375 4.375-0.75 0.313-1.5 0.375-2.188 0-3.125-1.438-6.313-2.938-9.438-4.375-0.75-0.375-1.5-0.375-2.25 0s-1.563 0.75-2.313 1.063c-0.25 0.125-0.625 0.25-0.625 0.625 0 0.313 0.375 0.375 0.625 0.5 4.625 2.188 9.25 4.375 13.938 6.5 0.75 0.375 1.5 0.375 2.188 0.063 4.688-2.188 9.375-4.375 14.063-6.563 0.125-0.063 0.25-0.125 0.313-0.188 0.313-0.188 0.313-0.5 0-0.688zM0.5 7.688c0.125 0.063 0.313 0.188 0.438 0.25 4.688 2.125 9.313 4.313 13.938 6.5 0.75 0.313 1.5 0.313 2.25 0 4.625-2.188 9.313-4.375 13.938-6.5 0.125-0.125 0.313-0.188 0.438-0.313 0.25-0.125 0.25-0.375 0-0.563-0.125-0.125-0.313-0.188-0.438-0.313-4.625-2.125-9.25-4.313-13.875-6.438-0.813-0.375-1.563-0.375-2.375 0-4.625 2.125-9.25 4.313-13.875 6.5-0.188 0.063-0.313 0.125-0.438 0.188-0.25 0.188-0.25 0.5 0 0.688zM31.438 15.688c-0.063-0.063-0.188-0.125-0.25-0.188-0.813-0.375-1.688-0.75-2.5-1.125-0.688-0.375-1.438-0.375-2.188 0-3.125 1.438-6.25 2.938-9.375 4.375-0.75 0.313-1.5 0.313-2.188 0-3.125-1.438-6.313-2.938-9.438-4.375-0.75-0.375-1.5-0.375-2.25 0s-1.563 0.75-2.313 1.063c-0.25 0.125-0.625 0.25-0.625 0.563 0 0.375 0.375 0.438 0.625 0.563 4.625 2.188 9.25 4.375 13.938 6.5 0.75 0.375 1.5 0.375 2.188 0.063 4.688-2.188 9.375-4.375 14.063-6.563 0.125-0.063 0.25-0.125 0.313-0.188 0.313-0.188 0.313-0.5 0-0.688z"></path></svg>',
			'close'             => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M17.883 16.667l7.058-7.058c0.517-0.517 0.517-1.367 0-1.883s-1.367-0.517-1.883 0l-7.058 7.058-7.058-7.058c-0.517-0.517-1.367-0.517-1.883 0s-0.517 1.367 0 1.883l7.058 7.058-7.058 7.058c-0.517 0.517-0.517 1.367 0 1.883 0.258 0.258 0.6 0.392 0.942 0.392s0.683-0.133 0.942-0.392l7.058-7.058 7.058 7.058c0.258 0.258 0.6 0.392 0.942 0.392s0.683-0.133 0.942-0.392c0.517-0.517 0.517-1.367 0-1.883l-7.058-7.058z"></path></svg>',
			'delicious'         => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M32 16h-16v16h-16v-16h16v-16h16z"></path></svg>',
			'deviantart'        => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M30.188 17.125c-1.5-1.813-3.375-2.938-5.5-3.813-2.75-1.125-5.625-1.563-8.563-1.625-0.188 0-0.438 0-0.75 0 0-1 0-1.938 0-3-1.313 0.125-2.563 0.313-4.188 0.438 0 0.188 0 0.25 0 0.375 0 0.75 0 1.563 0 2.313 0.063 0.25-0.063 0.375-0.313 0.438-2.125 0.438-4.125 1.125-6 2.25-1.438 0.875-2.688 1.938-3.625 3.375-1.125 1.688-1.563 3.563-1.063 5.5 5.125 0 10.063 0 15.188 0 0-3.188 0-6.125 0-9.188 0.625 0 1.313 0 1.938 0 1.625 0.125 3.25 0.375 4.75 1 2 0.938 3.375 2.375 3.938 4.5 0.063 0.125 0.063 0.5 0.125 0.5-1.813 0-3.375 0-5.313 0 0 0 0-0.313 0-0.438 0-1.188 0-2.438 0-3.688 0-0.25-0.063-0.375-0.313-0.375-0.813-0.188-1.5-0.313-2.313-0.438-0.5-0.063-1.25-0.063-1.563-0.125 0 2.813 0 5.688 0 8.25 5.125 0 10.063 0 15.188 0 0.063-0.625 0.188-0.938 0.188-1.375 0.063-1.875-0.625-3.438-1.813-4.875zM11.188 20.188c-1.938 0-3.5 0-5.25 0 0.125-2.25 2.688-5.063 5.25-5.375 0 1.75 0 3.438 0 5.375z"></path></svg>',
			'digg'              => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M9.625 10.875h3.188v10.875h-3.188zM5.125 6.375h3.188v15.375h-8.313v-10.875h5.125zM5.125 13.438h-1.938v5.75h1.938zM9.625 9.625h3.188v-3.25h-3.188zM14.063 10.875h8.313v14.75h-8.313v-2.563h5.125v-1.313h-5.125zM17.25 19.188h1.938v-5.75h-1.938zM32 10.875v14.75h-8.313v-2.563h5.125v-1.313h-5.125v-10.875zM28.813 13.438h-1.938v5.75h1.938z"></path></svg>',
			'discord'           => '<svg width="28" height="32" viewBox="0 0 28 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M18.576 15.2c0 0.976-0.72 1.776-1.632 1.776-0.896 0-1.632-0.8-1.632-1.776s0.72-1.776 1.632-1.776c0.912 0 1.632 0.8 1.632 1.776zM11.104 13.424c-0.912 0-1.632 0.8-1.632 1.776s0.736 1.776 1.632 1.776c0.912 0 1.632-0.8 1.632-1.776 0.016-0.976-0.72-1.776-1.632-1.776zM28 3.296v28.704c-4.031-3.562-2.742-2.383-7.424-6.736l0.848 2.96h-18.144c-1.808 0-3.28-1.472-3.28-3.296v-21.632c0-1.824 1.472-3.296 3.28-3.296h21.44c1.808 0 3.28 1.472 3.28 3.296zM23.44 18.464c0-5.152-2.304-9.328-2.304-9.328-2.304-1.728-4.496-1.68-4.496-1.68l-0.224 0.256c2.72 0.832 3.984 2.032 3.984 2.032-3.801-2.083-8.265-2.083-11.952-0.464-0.592 0.272-0.944 0.464-0.944 0.464s1.328-1.264 4.208-2.096l-0.16-0.192s-2.192-0.048-4.496 1.68c0 0-2.304 4.176-2.304 9.328 0 0 1.344 2.32 4.88 2.432 0 0 0.592-0.72 1.072-1.328-2.032-0.608-2.8-1.888-2.8-1.888 0.235 0.165 0.624 0.378 0.656 0.4 2.701 1.512 6.537 2.008 9.984 0.56 0.56-0.208 1.184-0.512 1.84-0.944 0 0-0.8 1.312-2.896 1.904 0.48 0.608 1.056 1.296 1.056 1.296 3.536-0.112 4.896-2.432 4.896-2.432z"></path></svg>',
			'dribbble'          => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M15.938 0.125c-8.813 0-16 7.188-16 16s7.188 16 16 16c8.875 0 16-7.188 16-16s-7.125-16-16-16zM26.563 7.5c1.875 2.313 3.063 5.25 3.063 8.5-0.438-0.125-4.938-1-9.5-0.438-0.125-0.25-0.188-0.5-0.313-0.688-0.25-0.688-0.563-1.375-0.875-2 5-2.063 7.313-5 7.625-5.375zM15.938 2.438c3.5 0 6.688 1.313 9.063 3.438-0.188 0.375-2.25 3.125-7.125 4.938-2.25-4.063-4.75-7.438-5.125-8 1.063-0.25 2.125-0.375 3.188-0.375zM10.188 3.75c0.313 0.5 2.75 3.875 5.063 7.938-6.438 1.688-12.063 1.625-12.688 1.625 0.938-4.25 3.75-7.75 7.625-9.563zM2.313 16.125c0-0.125 0-0.25 0-0.438 0.563 0.063 7.188 0.125 14.063-1.938 0.375 0.75 0.75 1.563 1.063 2.313-0.188 0.063-0.313 0.125-0.5 0.188-7.063 2.25-10.813 8.5-11.125 9-2.188-2.438-3.5-5.625-3.5-9.125zM15.938 29.813c-3.125 0-6.063-1.125-8.375-2.875 0.25-0.563 3-5.875 10.75-8.563 0 0 0.063 0 0.063-0.063 1.938 5 2.75 9.188 2.938 10.375-1.625 0.688-3.438 1.125-5.375 1.125zM23.625 27.438c-0.188-0.813-0.875-4.813-2.688-9.75 4.313-0.688 8.063 0.438 8.5 0.625-0.563 3.813-2.75 7.063-5.813 9.125z"></path></svg>',
			'edit'              => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M26.667 18.883c-0.733 0-1.333 0.6-1.333 1.333v7.117c0 0.733-0.6 1.333-1.333 1.333h-18.667c-0.733 0-1.333-0.6-1.333-1.333v-18.667c0-0.733 0.6-1.333 1.333-1.333h7.117c0.733 0 1.333-0.6 1.333-1.333s-0.6-1.333-1.333-1.333h-7.117c-2.208 0-4 1.792-4 4v18.667c0 2.208 1.792 4 4 4h18.667c2.208 0 4-1.792 4-4v-7.117c0-0.742-0.6-1.333-1.333-1.333zM30.275 7.725l-5.333-5.333c-0.517-0.517-1.367-0.517-1.883 0l-13.333 13.333c-0.25 0.25-0.392 0.592-0.392 0.942v5.333c0 0.733 0.6 1.333 1.333 1.333h5.333c0.35 0 0.692-0.142 0.942-0.392l13.333-13.333c0.525-0.517 0.525-1.367 0-1.883zM15.45 20.667h-3.45v-3.45l12-12 3.45 3.45-12 12z"></path></svg>',
			'envelope'          => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M32 12.964v14.179c0 1.571-1.286 2.857-2.857 2.857h-26.286c-1.571 0-2.857-1.286-2.857-2.857v-14.179c0.536 0.589 1.143 1.107 1.804 1.554 2.964 2.018 5.964 4.036 8.875 6.161 1.5 1.107 3.357 2.464 5.304 2.464h0.036c1.946 0 3.804-1.357 5.304-2.464 2.911-2.107 5.911-4.143 8.893-6.161 0.643-0.446 1.25-0.964 1.786-1.554zM32 7.714c0 2-1.482 3.804-3.054 4.893-2.786 1.929-5.589 3.857-8.357 5.804-1.161 0.804-3.125 2.446-4.571 2.446h-0.036c-1.446 0-3.411-1.643-4.571-2.446-2.768-1.946-5.571-3.875-8.339-5.804-1.268-0.857-3.071-2.875-3.071-4.5 0-1.75 0.946-3.25 2.857-3.25h26.286c1.554 0 2.857 1.286 2.857 2.857z"></path></svg>',
			'subscribers'       => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M32 12.964v14.179c0 1.571-1.286 2.857-2.857 2.857h-26.286c-1.571 0-2.857-1.286-2.857-2.857v-14.179c0.536 0.589 1.143 1.107 1.804 1.554 2.964 2.018 5.964 4.036 8.875 6.161 1.5 1.107 3.357 2.464 5.304 2.464h0.036c1.946 0 3.804-1.357 5.304-2.464 2.911-2.107 5.911-4.143 8.893-6.161 0.643-0.446 1.25-0.964 1.786-1.554zM32 7.714c0 2-1.482 3.804-3.054 4.893-2.786 1.929-5.589 3.857-8.357 5.804-1.161 0.804-3.125 2.446-4.571 2.446h-0.036c-1.446 0-3.411-1.643-4.571-2.446-2.768-1.946-5.571-3.875-8.339-5.804-1.268-0.857-3.071-2.875-3.071-4.5 0-1.75 0.946-3.25 2.857-3.25h26.286c1.554 0 2.857 1.286 2.857 2.857z"></path></svg>',
			'etsy'              => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M3.5 1.25c-0.063-0.438 0.125-0.75 0.75-0.75 0.563 0 6.625 0.188 10.688 0.188 8.375 0 10.438-0.688 11.875-0.688 0.75 0.063 0.938 0.625 0.938 1s-0.375 1.125-0.375 2.125c0 1 0.25 2 0.25 2.938 0 0.875-0.75 1.125-1.188 1.125-0.375 0-0.625-0.125-0.625-0.875-0.063-1.438 1-3.5-9.25-3.5-5.188 0-5.313 0.625-5.313 2.25 0 1.688 0 7.125 0 7.938 0 0.875 0.25 1.125 0.938 1.125s6 0 7.438 0c1.438 0 2.938-0.75 3.125-2.813 0.063-1.063 0.313-1.125 0.875-1.125s1 0.438 0.938 1.438c-0.125 1.063-0.313 2.125-0.313 3.563 0 1.5 0.313 2.938 0.313 4s-1.438 1.313-1.688 0.125c-0.25-1.063-0.688-3-3.938-3s-5.875 0-6.625 0c-0.75 0-1.063 0.438-1.063 1.625s0 7 0 8.438c0 1.5 0.438 3.063 3.375 3.063 3 0 8.375 0.313 10.875-1.625 1-0.813 1.375-2.063 1.688-4.063 0.188-0.875 2.188-0.875 2.063 1.25-0.063 1-1 3.063-1 4.438 0 1.063 0.25 2.625-0.75 2.625-0.875 0-0.5-0.5-4.375-0.5s-17.188 0.125-19.438 0.125c-0.813 0-1-0.25-1-0.75s0.313-0.875 1-0.875c0.688 0 3.688 0 3.688-3.75 0-3.813 0-17.625 0-19.688 0-2-0.438-4.063-2.938-4.313-1-0.125-0.938-0.5-0.938-1.063z"></path></svg>',
			'evernote'          => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M9.546 8.030c0.234 0 0.476-0.188 0.476-0.418 0-0.002 0-4.862 0-4.862 0-0.79 0.466-1.876 0.45-1.86l-7.94 7.774c0 0 0.928-0.634 2.468-0.634h4.546zM25 3h-5c-0.13-0.886-0.876-2.18-1.572-2.46-1.852-0.746-3.68-0.566-4.414-0.36-0.584 0.164-1.23 0.498-1.59 1.014-0.242 0.344-0.422 0.784-0.424 1.402 0 0 0 2.78 0 5.904 0 1.062-0.438 1.5-1.438 1.5h-6.562c-0.64 0-2 0.124-2 2.456 0 0.838 1.15 5.698 2.006 7.462 0 0 0.494 1.52 2.464 2.082 0 0 3.378 0.874 4.532 1 1.052 0.114 2.998 0 2.998-3v-2c0-0.552 0.448-1 1-1s1 0.448 1 1v1c0 3 3 2.968 3 2.968h2.124c1.876 0 2.76 1.032 2.76 3.032 0 3.432-1.108 3.928-2.93 4h-1.916c-0.538 0-1.038-0.436-1.038-0.976v-1.024c0-1.024 1-1 1-1s0.968 0 1 0c0.552 0 1-0.448 1-1s-0.448-1-1-1h-1.938c-1.14 0-2.062 0.922-2.062 2.060v1.94c0 3 2.334 4 4 4 0 0 1.572 0 2.458 0 3.5 0 7.542 0 7.542-12.722v-11.278c0-3-2.358-5-5-5zM26 16h-6v-1c0-1.656 1.344-3 3-3s3 1.344 3 3v1z"></path></svg>',
			'eye-closed'        => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M13.517 7.617c0.758-0.183 1.6-0.283 2.458-0.283h0.033c3.467 0 6.767 1.633 9.808 4.858 1.667 1.767 2.8 3.567 3.325 4.467-0.717 1.242-1.508 2.375-2.4 3.433-0.475 0.558-0.408 1.4 0.158 1.875 0.25 0.208 0.558 0.317 0.858 0.317 0.375 0 0.758-0.158 1.017-0.475 1.167-1.383 2.167-2.858 3.067-4.517 0.208-0.383 0.217-0.842 0.017-1.233-0.058-0.117-1.45-2.875-4.050-5.65-3.533-3.75-7.617-5.742-11.808-5.742h-0.033c-1.058 0-2.125 0.125-3.092 0.358-0.717 0.175-1.15 0.9-0.975 1.608 0.183 0.717 0.9 1.158 1.617 0.983zM31.608 30.392l-6.733-6.733c-0.008-0.008-0.017-0.017-0.033-0.033l-22.567-22.567c-0.517-0.517-1.367-0.517-1.883 0s-0.517 1.367 0 1.883l5.7 5.7c-2.392 2.058-4.383 4.542-5.933 7.383-0.208 0.383-0.217 0.842-0.025 1.233 0.058 0.117 1.45 2.875 4.058 5.65 3.533 3.767 7.617 5.758 11.817 5.758 0.008 0 0.017 0 0.025 0 2.792-0.050 5.442-0.85 7.733-2.342l5.95 5.95c0.258 0.258 0.6 0.392 0.942 0.392s0.683-0.133 0.942-0.392c0.533-0.517 0.533-1.367 0.008-1.883zM13.367 15.925l3.375 3.375c-0.267 0.092-0.55 0.133-0.842 0.133-1.467 0-2.667-1.2-2.667-2.667 0-0.292 0.050-0.575 0.133-0.842zM16 26c-3.467 0-6.767-1.642-9.817-4.858-1.667-1.767-2.8-3.558-3.325-4.467 1.367-2.358 3.092-4.417 5.125-6.142l3.4 3.4c-0.525 0.842-0.817 1.825-0.817 2.833 0 2.942 2.392 5.333 5.333 5.333 1.008 0 1.992-0.292 2.833-0.817l3.1 3.1c-1.758 1.025-3.75 1.583-5.833 1.617z"></path></svg>',
			'eye'               => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M31.858 16.067c-0.058-0.117-1.45-2.875-4.050-5.65-3.533-3.758-7.617-5.75-11.808-5.75s-8.275 1.992-11.808 5.758c-2.6 2.775-3.992 5.533-4.050 5.65-0.192 0.375-0.192 0.817 0 1.192 0.058 0.117 1.45 2.875 4.050 5.65 3.533 3.758 7.617 5.75 11.808 5.75s8.275-1.992 11.808-5.758c2.6-2.775 3.992-5.533 4.050-5.65 0.192-0.367 0.192-0.817 0-1.192zM25.817 21.142c-3.050 3.225-6.35 4.858-9.817 4.858s-6.767-1.633-9.817-4.858c-1.667-1.767-2.808-3.567-3.333-4.475 0.525-0.9 1.658-2.7 3.333-4.475 3.050-3.225 6.35-4.858 9.817-4.858s6.767 1.633 9.817 4.858c1.667 1.775 2.808 3.567 3.333 4.475-0.533 0.9-1.667 2.7-3.333 4.475zM16 11.333c-2.942 0-5.333 2.392-5.333 5.333s2.392 5.333 5.333 5.333c2.942 0 5.333-2.392 5.333-5.333s-2.392-5.333-5.333-5.333zM16 19.333c-1.467 0-2.667-1.2-2.667-2.667s1.2-2.667 2.667-2.667c1.467 0 2.667 1.2 2.667 2.667s-1.2 2.667-2.667 2.667z"></path></svg>',
			'facebook'          => '<svg width="32" height="32" viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg"><path d="M32 16.098C32 7.208 24.837 0 16 0S0 7.207 0 16.098C0 24.133 5.851 30.792 13.5 32V20.751H9.437v-4.653H13.5V12.55c0-4.034 2.389-6.263 6.043-6.263 1.751 0 3.582.315 3.582.315v3.961h-2.018c-1.987 0-2.607 1.241-2.607 2.514v3.02h4.438l-.71 4.653H18.5V32C26.149 30.792 32 24.133 32 16.098z"/></svg>',
			'flickr'            => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M7.25 23.25c-4 0-7.25-3.25-7.25-7.25s3.25-7.25 7.25-7.25c4 0 7.25 3.25 7.25 7.25s-3.25 7.25-7.25 7.25zM24.75 23.25c-4 0-7.25-3.25-7.25-7.25s3.25-7.25 7.25-7.25c4 0 7.25 3.25 7.25 7.25s-3.25 7.25-7.25 7.25z"></path></svg>',
			'flipboard'         => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M4.313 2.313v23.375h7.813v-7.813h7.75v-7.75h7.813v-7.813z"></path></svg>',
			'followers'         => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M22.483 21.458c-2.458-1.375-5.708-2.125-9.15-2.125s-6.692 0.75-9.15 2.125c-2.7 1.5-4.183 3.583-4.183 5.875v1.333c0 0.733 0.6 1.333 1.333 1.333s1.333-0.6 1.333-1.333v-1.333c0-1.258 1.025-2.55 2.817-3.55 2.067-1.15 4.85-1.783 7.85-1.783s5.783 0.633 7.85 1.783c1.792 1 2.817 2.292 2.817 3.55v1.333c0 0.733 0.6 1.333 1.333 1.333s1.333-0.6 1.333-1.333v-1.333c0-2.292-1.483-4.375-4.183-5.875zM13.392 16.608c3.675 0 6.667-2.992 6.667-6.667 0-0.333-0.025-0.667-0.067-0.967v-0.008c0-0.017 0-0.025-0.008-0.042-0.217-1.408-0.867-2.692-1.875-3.7s-2.292-1.658-3.7-1.875c-0.333-0.050-0.675-0.075-1.017-0.075-3.675 0-6.667 2.992-6.667 6.667s2.992 6.667 6.667 6.667zM13.392 5.942c0.208 0 0.408 0.017 0.608 0.050 1.7 0.258 3.075 1.625 3.342 3.325v0.008c0 0.017 0 0.025 0.008 0.042 0.025 0.175 0.042 0.375 0.042 0.575 0 2.208-1.792 4-4 4s-4-1.792-4-4 1.792-4 4-4zM30.667 14h-2.667v-2.667c0-0.733-0.6-1.333-1.333-1.333s-1.333 0.6-1.333 1.333v2.667h-2.667c-0.733 0-1.333 0.6-1.333 1.333s0.6 1.333 1.333 1.333h2.667v2.667c0 0.733 0.6 1.333 1.333 1.333s1.333-0.6 1.333-1.333v-2.667h2.667c0.733 0 1.333-0.6 1.333-1.333s-0.6-1.333-1.333-1.333z"></path></svg>',
			'foursquare'        => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M24.625 0c0 0-14.688 0-17.063 0-2.313 0-3 1.75-3 2.875s0 27.125 0 27.125c0 1.25 0.625 1.75 1 1.875 0.438 0.188 1.438 0.313 2.063-0.438 0 0 8.063-9.375 8.188-9.5 0.25-0.188 0.25-0.188 0.438-0.188 0.438 0 3.563 0 5.25 0 2.188 0 2.5-1.563 2.75-2.5 0.188-0.75 2.313-11.688 3.063-15.188 0.5-2.625-0.125-4.063-2.688-4.063zM24.25 19.25c0.188-0.75 2.313-11.688 3.063-15.188zM23.625 4.625l-0.688 3.75c-0.125 0.375-0.625 0.813-1.063 0.813-0.5 0-6.688 0-6.688 0-0.75 0-1.25 0.5-1.25 1.25v0.813c0 0.75 0.5 1.25 1.25 1.25 0 0 5.125 0 5.688 0 0.5 0 1 0.625 0.938 1.188-0.125 0.563-0.688 3.375-0.75 3.688-0.063 0.25-0.375 0.813-1.063 0.813-0.5 0-4.563 0-4.563 0-0.875 0-1.125 0.125-1.688 0.813s-5.563 6.75-5.563 6.75c-0.063 0.063-0.125 0.063-0.125 0v-21.188c0-0.5 0.438-1.063 1.063-1.063 0 0 13.125 0 13.688 0 0.5 0 0.938 0.5 0.813 1.125z"></path></svg>',
			'general'           => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M27.1 21.167c0.142-0.3 0.45-0.5 0.775-0.5h0.117c2.208 0 4-1.792 4-4s-1.792-4-4-4h-0.225c-0.292 0-0.55-0.142-0.708-0.375-0.017-0.092-0.050-0.175-0.083-0.258-0.050-0.108-0.075-0.233-0.075-0.358 0-0.225 0.083-0.442 0.242-0.6l0.075-0.075c0.758-0.758 1.175-1.758 1.175-2.833s-0.417-2.075-1.175-2.825c-0.758-0.758-1.758-1.175-2.833-1.175-1.067 0-2.075 0.417-2.833 1.175l-0.075 0.075c-0.158 0.158-0.375 0.242-0.6 0.242-0.117 0-0.242-0.025-0.342-0.067-0.017-0.008-0.033-0.017-0.050-0.017-0.308-0.142-0.5-0.45-0.5-0.775v-0.133c0-2.208-1.792-4-4-4s-4 1.792-4 4v0.225c0 0.292-0.142 0.55-0.375 0.708-0.092 0.017-0.175 0.050-0.258 0.083-0.108 0.050-0.233 0.075-0.358 0.075-0.225 0-0.442-0.083-0.6-0.242l-0.075-0.075c-0.758-0.758-1.758-1.175-2.833-1.175s-2.075 0.417-2.825 1.175c-0.742 0.758-1.158 1.758-1.158 2.833s0.417 2.075 1.175 2.833l0.075 0.075c0.158 0.158 0.242 0.375 0.242 0.6 0 0.117-0.025 0.242-0.067 0.342-0.017 0.033-0.033 0.075-0.042 0.117-0.125 0.317-0.425 0.533-0.775 0.542h-0.108c-2.208 0-4 1.792-4 4s1.792 4 4 4h0.225c0.35 0 0.65 0.208 0.8 0.533 0 0.008 0.008 0.017 0.008 0.025 0.042 0.1 0.067 0.217 0.067 0.325 0 0.225-0.083 0.442-0.242 0.6l-0.075 0.075c-0.758 0.758-1.175 1.758-1.175 2.833s0.417 2.075 1.175 2.825c0.758 0.758 1.758 1.175 2.833 1.175s2.075-0.417 2.825-1.175l0.075-0.075c0.158-0.158 0.375-0.242 0.6-0.242 0.117 0 0.242 0.025 0.342 0.067 0.033 0.017 0.075 0.033 0.117 0.042 0.317 0.125 0.533 0.425 0.542 0.775v0.1c0 2.208 1.792 4 4 4s4-1.792 4-4v-0.225c0-0.35 0.208-0.65 0.533-0.8 0.008 0 0.017-0.008 0.025-0.008 0.1-0.042 0.217-0.067 0.325-0.067 0.225 0 0.442 0.083 0.6 0.242l0.075 0.075c0.758 0.758 1.758 1.175 2.833 1.175s2.075-0.417 2.825-1.175c0.758-0.758 1.175-1.758 1.175-2.833 0-1.067-0.417-2.075-1.175-2.825l-0.075-0.075c-0.158-0.158-0.242-0.375-0.242-0.6 0-0.117 0.025-0.242 0.067-0.342 0.008-0.008 0.017-0.025 0.017-0.042zM24.65 20.117c-0.008 0.017-0.008 0.025-0.017 0.042-0.192 0.433-0.292 0.917-0.292 1.4 0 0.933 0.358 1.808 1.008 2.475 0 0 0.008 0.008 0.008 0.008l0.083 0.083c0.25 0.25 0.392 0.583 0.392 0.942s-0.142 0.692-0.392 0.942c-0.25 0.25-0.583 0.392-0.942 0.392s-0.692-0.142-0.942-0.392l-0.083-0.083c0 0-0.008-0.008-0.008-0.008-0.667-0.65-1.542-1.008-2.475-1.008-0.508 0-0.992 0.108-1.45 0.308-0.008 0.008-0.025 0.008-0.033 0.017-1.258 0.575-2.067 1.825-2.075 3.208v0.225c0 0.733-0.6 1.333-1.333 1.333s-1.333-0.6-1.333-1.333v-0.117c0-0.008 0-0.025 0-0.033-0.033-1.45-0.933-2.708-2.283-3.225-0.425-0.183-0.9-0.283-1.383-0.283-0.933 0-1.808 0.358-2.475 1.008 0 0-0.008 0.008-0.008 0.008l-0.083 0.083c-0.25 0.25-0.583 0.392-0.942 0.392s-0.692-0.142-0.942-0.392c-0.25-0.25-0.392-0.583-0.392-0.942s0.142-0.692 0.392-0.942l0.092-0.092c0.65-0.667 1.008-1.542 1.008-2.475 0-0.5-0.108-0.992-0.308-1.45-0.008-0.008-0.008-0.025-0.017-0.033-0.575-1.258-1.825-2.067-3.208-2.067h-0.217c-0.733 0-1.333-0.6-1.333-1.333s0.6-1.333 1.333-1.333h0.117c0.008 0 0.025 0 0.033 0 1.45-0.033 2.708-0.933 3.225-2.283 0.183-0.425 0.283-0.9 0.283-1.383 0-0.933-0.358-1.808-1.008-2.475 0 0-0.008-0.008-0.008-0.008l-0.083-0.083c-0.25-0.25-0.392-0.583-0.392-0.942s0.142-0.692 0.392-0.942c0.25-0.25 0.583-0.392 0.942-0.392s0.692 0.142 0.942 0.392l0.083 0.083c0 0 0.008 0.008 0.008 0.008 0.667 0.65 1.542 1.008 2.475 1.008 0.4 0 0.8-0.067 1.175-0.2 0.125-0.017 0.242-0.050 0.358-0.1 1.292-0.567 2.125-1.833 2.133-3.233v-0.225c0-0.733 0.6-1.333 1.333-1.333s1.333 0.6 1.333 1.333v0.125c0.008 1.4 0.833 2.667 2.117 3.225 0.008 0.008 0.025 0.008 0.033 0.017 0.433 0.192 0.917 0.292 1.4 0.292 0.933 0 1.808-0.358 2.475-1.008 0 0 0.008-0.008 0.008-0.008l0.083-0.083c0.25-0.25 0.583-0.392 0.942-0.392s0.692 0.142 0.942 0.392c0.25 0.25 0.392 0.583 0.392 0.942s-0.142 0.692-0.392 0.942l-0.083 0.083c0 0-0.008 0.008-0.008 0.008-0.65 0.667-1.008 1.542-1.008 2.475 0 0.4 0.067 0.8 0.2 1.175 0.017 0.125 0.050 0.242 0.1 0.358 0.567 1.292 1.833 2.125 3.233 2.133h0.225c0.733 0 1.333 0.6 1.333 1.333s-0.6 1.333-1.333 1.333h-0.125c-1.4 0-2.667 0.825-3.225 2.108zM16 11.333c-2.942 0-5.333 2.392-5.333 5.333s2.392 5.333 5.333 5.333c2.942 0 5.333-2.392 5.333-5.333s-2.392-5.333-5.333-5.333zM16 19.333c-1.467 0-2.667-1.2-2.667-2.667s1.2-2.667 2.667-2.667c1.467 0 2.667 1.2 2.667 2.667s-1.2 2.667-2.667 2.667z"></path></svg>',
			'github'            => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M30.25 11.125c0-2.25-0.75-4.125-2.125-5.563 0.25-0.5 0.938-2.625-0.188-5.5 0 0-1.75-0.563-5.688 2.125-1.688-0.438-3.438-0.688-5.25-0.688-1.75 0-3.5 0.25-5.188 0.688-4-2.688-5.688-2.125-5.688-2.125-1.188 2.875-0.438 5-0.25 5.5-1.313 1.438-2.125 3.313-2.125 5.563 0 8 4.875 9.75 9.5 10.313-0.625 0.5-1.188 1.438-1.375 2.75-1.188 0.563-4.188 1.438-6-1.75 0 0-1.125-1.938-3.188-2.125 0 0-2.063 0-0.188 1.25 0 0 1.375 0.688 2.313 3.063 0 0 1.25 3.688 7 2.438 0.063 1.75 0.063 4.438 0.063 4.938h10.313c0-0.688 0-4 0-6.75 0-1.938-0.625-3.188-1.375-3.875 4.625-0.5 9.438-2.25 9.438-10.25zM4.125 21.313c-0.063 0.063-0.25 0.125-0.375 0.063s-0.25-0.25-0.188-0.313c0.063-0.125 0.188-0.188 0.375-0.063 0.125 0.063 0.188 0.188 0.188 0.313zM4.938 22.25c-0.063 0.063-0.313 0-0.438-0.125s-0.125-0.313-0.063-0.438c0.125-0.063 0.313-0.063 0.438 0.125 0.125 0.125 0.188 0.313 0.063 0.438zM5.75 23.438c-0.125 0.063-0.313 0-0.438-0.188s-0.125-0.438 0-0.5c0.125-0.063 0.313 0 0.438 0.188s0.125 0.375 0 0.5zM6.875 24.563c-0.125 0.125-0.375 0.125-0.5-0.063-0.188-0.188-0.25-0.375-0.125-0.5 0.125-0.188 0.375-0.125 0.5 0.063 0.188 0.125 0.25 0.375 0.125 0.5zM8.438 25.25c-0.063 0.188-0.313 0.25-0.5 0.188-0.25-0.063-0.438-0.25-0.375-0.438s0.313-0.25 0.563-0.188c0.188 0.063 0.375 0.25 0.313 0.438zM9.688 25.688c-0.25 0-0.438-0.125-0.438-0.313s0.188-0.313 0.438-0.313c0.25 0 0.438 0.125 0.438 0.313s-0.188 0.313-0.438 0.313zM11.313 25.5c-0.25 0.063-0.438-0.063-0.5-0.25 0-0.125 0.188-0.313 0.438-0.375s0.438 0.063 0.5 0.25c0 0.125-0.188 0.313-0.438 0.375z"></path></svg>',
			'google'            => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M31.625 14.938c-0.063-0.625-0.125-1.313-0.25-1.938-5 0-10.063 0-15.063 0 0 2.063 0 4.188 0 6.25 0 0 0 0 0 0s0 0 0 0c2.938 0 5.813 0 8.75 0-0.375 2-1.563 3.813-3.25 4.938 0 0 0 0 0 0-1.063 0.688-2.25 1.188-3.5 1.375-1.25 0.25-2.563 0.25-3.813 0-1.313-0.25-2.5-0.813-3.563-1.563-1.563-1.063-2.813-2.688-3.5-4.438-0.063-0.125-0.125-0.313-0.188-0.438 0 0 0 0 0 0s0 0 0.063 0c-0.688-2-0.688-4.25-0.063-6.25 0.5-1.375 1.313-2.688 2.375-3.75 1.25-1.313 2.938-2.25 4.75-2.688 1.563-0.313 3.188-0.25 4.688 0.188 1.313 0.438 2.5 1.125 3.438 2.063 1-1 2-1.938 2.938-2.938 0.563-0.5 1.063-1.063 1.563-1.563 0 0 0 0 0 0s0 0 0 0c-1.5-1.438-3.25-2.5-5.125-3.188-3.438-1.313-7.313-1.313-10.75-0.125-3.938 1.313-7.25 4.25-9.125 7.938-0.625 1.25-1.063 2.625-1.375 4-0.688 3.5-0.188 7.188 1.375 10.375 1.063 2.063 2.563 3.875 4.375 5.313 1.688 1.375 3.688 2.375 5.813 2.938 2.688 0.75 5.5 0.75 8.188 0.125 2.438-0.563 4.688-1.75 6.563-3.438 1.938-1.75 3.313-4.125 4.063-6.625 0.563-2.125 0.813-4.313 0.625-6.563z"></path></svg>',
			'heart'             => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M16 30c-0.286 0-0.571-0.107-0.786-0.321l-11.143-10.75c-0.143-0.125-4.071-3.714-4.071-8 0-5.232 3.196-8.357 8.536-8.357 3.125 0 6.054 2.464 7.464 3.857 1.411-1.393 4.339-3.857 7.464-3.857 5.339 0 8.536 3.125 8.536 8.357 0 4.286-3.929 7.875-4.089 8.036l-11.125 10.714c-0.214 0.214-0.5 0.321-0.786 0.321z"></path></svg>',
			'houzz'             => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M16 31.938l8.938-5.438v-10.5l-8.938 5.375zM16 10.688v-10.75l-9.25 5.438v10.875l9.188-5.375 9 5.125v-10.625zM15.875 21.438l-9.125-5.188v10.563z"></path></svg>',
			'import'            => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M20.392 22.392l-3.058 3.058v-8.783c0-0.733-0.6-1.333-1.333-1.333s-1.333 0.6-1.333 1.333v8.783l-3.058-3.058c-0.517-0.517-1.367-0.517-1.883 0s-0.517 1.367 0 1.883l5.333 5.333c0.017 0.017 0.033 0.033 0.050 0.050 0 0 0 0 0 0s0.008 0.008 0.008 0.008c0 0 0 0 0.008 0.008 0 0 0 0 0.008 0.008 0 0 0.008 0.008 0.008 0.008s0 0 0 0 0.008 0.008 0.008 0.008c0 0 0 0 0 0 0.1 0.083 0.2 0.142 0.308 0.192 0 0 0 0 0.008 0 0 0 0 0 0.008 0 0.142 0.058 0.3 0.1 0.467 0.108 0 0 0 0 0 0 0.008 0 0.008 0 0.017 0 0 0 0 0 0 0s0.008 0 0.008 0c0 0 0.008 0 0.008 0s0.008 0 0.008 0 0.008 0 0.017 0 0.008 0 0.017 0c0 0 0.008 0 0.008 0s0.008 0 0.008 0c0 0 0.008 0 0.008 0s0 0 0 0c0.008 0 0.008 0 0.017 0 0 0 0 0 0 0 0.167-0.008 0.325-0.042 0.467-0.108 0 0 0 0 0 0s0.008 0 0.008 0c0 0 0.008 0 0.008 0s0 0 0 0c0.1-0.042 0.192-0.1 0.283-0.175 0 0 0 0 0 0s0.008-0.008 0.008-0.008c0 0 0 0 0 0s0.008-0.008 0.008-0.008c0 0 0 0 0.008-0.008 0 0 0 0 0.008-0.008 0 0 0.008-0.008 0.008-0.008s0 0 0 0c0.025-0.017 0.042-0.042 0.067-0.067l5.325-5.325c0.517-0.517 0.517-1.367 0-1.883-0.5-0.533-1.35-0.533-1.867-0.008zM24 11.333h-0.7c-1.7-4.767-6.217-8.017-11.317-8.017-6.617 0-12 5.383-12 12 0 2.942 1.075 5.767 3.025 7.967 0.492 0.55 1.333 0.6 1.883 0.108s0.6-1.333 0.108-1.883c-1.517-1.708-2.35-3.908-2.35-6.192 0-5.15 4.183-9.333 9.333-9.333 4.233 0 7.958 2.883 9.042 7.025 0.15 0.583 0.683 0.992 1.292 0.992h1.683c2.942 0 5.333 2.392 5.333 5.333 0 1.75-0.825 3.342-2.267 4.367-0.6 0.425-0.742 1.258-0.317 1.858 0.258 0.367 0.675 0.558 1.092 0.558 0.267 0 0.533-0.083 0.767-0.25 2.125-1.508 3.392-3.95 3.392-6.542 0.008-4.4-3.583-7.992-8-7.992z"></path></svg>',
			'instagram'         => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M16 2.875c4.313 0 4.75 0.063 6.438 0.125 1.625 0.063 2.438 0.313 3 0.5 0.75 0.313 1.313 0.688 1.813 1.188 0.563 0.563 0.938 1.125 1.188 1.813 0.25 0.625 0.5 1.438 0.563 3 0.063 1.75 0.125 2.188 0.125 6.5s-0.063 4.75-0.125 6.438c-0.063 1.625-0.313 2.438-0.5 3-0.313 0.75-0.688 1.313-1.188 1.813-0.563 0.563-1.125 0.938-1.813 1.188-0.625 0.25-1.438 0.5-3 0.563-1.75 0.063-2.188 0.125-6.5 0.125s-4.75-0.063-6.438-0.125c-1.625-0.063-2.438-0.313-3-0.5-0.75-0.313-1.313-0.688-1.813-1.188-0.563-0.563-0.938-1.125-1.188-1.813-0.25-0.625-0.5-1.438-0.563-3-0.063-1.75-0.125-2.188-0.125-6.5s0.063-4.75 0.125-6.438c0.063-1.625 0.313-2.438 0.5-3 0.313-0.75 0.688-1.313 1.188-1.813 0.563-0.563 1.125-0.938 1.813-1.188 0.625-0.25 1.438-0.5 3-0.563 1.75-0.063 2.188-0.125 6.5-0.125M16 0c-4.375 0-4.875 0.063-6.563 0.125-1.75 0.063-2.938 0.313-3.875 0.688-1.063 0.438-2 1-2.875 1.875s-1.438 1.813-1.875 2.875c-0.375 1-0.625 2.125-0.688 3.875-0.063 1.688-0.125 2.188-0.125 6.563s0.063 4.875 0.125 6.563c0.063 1.75 0.313 2.938 0.688 3.875 0.438 1.063 1 2 1.875 2.875s1.813 1.438 2.875 1.875c1 0.375 2.125 0.625 3.875 0.688 1.688 0.063 2.188 0.125 6.563 0.125s4.875-0.063 6.563-0.125c1.75-0.063 2.938-0.313 3.875-0.688 1.063-0.438 2-1 2.875-1.875s1.438-1.813 1.875-2.875c0.375-1 0.625-2.125 0.688-3.875 0.063-1.688 0.125-2.188 0.125-6.563s-0.063-4.875-0.125-6.563c-0.063-1.75-0.313-2.938-0.688-3.875-0.438-1.063-1-2-1.875-2.875s-1.813-1.438-2.875-1.875c-1-0.375-2.125-0.625-3.875-0.688-1.688-0.063-2.188-0.125-6.563-0.125zM16 7.75c-4.563 0-8.25 3.688-8.25 8.25s3.688 8.25 8.25 8.25c4.563 0 8.25-3.688 8.25-8.25s-3.688-8.25-8.25-8.25zM16 21.375c-2.938 0-5.375-2.438-5.375-5.375s2.438-5.375 5.375-5.375c2.938 0 5.375 2.438 5.375 5.375s-2.438 5.375-5.375 5.375zM26.438 7.438c0 1.063-0.813 1.938-1.875 1.938s-1.938-0.875-1.938-1.938c0-1.063 0.875-1.875 1.938-1.875s1.875 0.813 1.875 1.875z"></path></svg>',
			'layout'            => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M25.333 3.333h-18.667c-2.208 0-4 1.792-4 4v18.667c0 2.208 1.792 4 4 4h18.667c2.208 0 4-1.792 4-4v-18.667c0-2.208-1.792-4-4-4zM6.667 6h18.667c0.733 0 1.333 0.6 1.333 1.333v4h-21.333v-4c0-0.733 0.6-1.333 1.333-1.333zM5.333 26v-12h5.333v13.333h-4c-0.733 0-1.333-0.6-1.333-1.333zM25.333 27.333h-12v-13.333h13.333v12c0 0.733-0.6 1.333-1.333 1.333z"></path></svg>',
			'life-buoy'         => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M26.45 26.283c2.608-2.65 4.217-6.283 4.217-10.283s-1.608-7.633-4.217-10.283c-0.025-0.033-0.050-0.058-0.083-0.092-0.025-0.025-0.058-0.058-0.092-0.083-2.642-2.6-6.275-4.208-10.275-4.208s-7.633 1.608-10.283 4.217c-0.033 0.025-0.058 0.050-0.092 0.083-0.025 0.025-0.058 0.058-0.083 0.092-2.6 2.642-4.208 6.275-4.208 10.275s1.608 7.633 4.217 10.283c0.025 0.033 0.050 0.058 0.083 0.092 0.025 0.025 0.058 0.058 0.092 0.083 2.642 2.6 6.275 4.208 10.275 4.208s7.633-1.608 10.283-4.217c0.033-0.025 0.058-0.050 0.092-0.083 0.025-0.025 0.050-0.058 0.075-0.083zM25.375 23.483l-3.808-3.808c0.7-1.058 1.108-2.317 1.108-3.675s-0.408-2.617-1.108-3.675l3.808-3.808c1.642 2.050 2.625 4.658 2.625 7.483s-0.983 5.433-2.625 7.483zM12 16c0-2.208 1.792-4 4-4s4 1.792 4 4c0 2.208-1.792 4-4 4s-4-1.792-4-4zM23.483 6.625l-3.808 3.808c-1.058-0.7-2.317-1.108-3.675-1.108s-2.617 0.408-3.675 1.108l-3.808-3.808c2.050-1.642 4.658-2.625 7.483-2.625s5.433 0.983 7.483 2.625zM6.625 8.517l3.808 3.808c-0.7 1.058-1.108 2.317-1.108 3.675s0.408 2.617 1.108 3.675l-3.808 3.808c-1.642-2.050-2.625-4.658-2.625-7.483s0.983-5.433 2.625-7.483zM8.517 25.375l3.808-3.808c1.058 0.7 2.317 1.108 3.675 1.108s2.617-0.408 3.675-1.108l3.808 3.808c-2.050 1.642-4.658 2.625-7.483 2.625s-5.433-0.983-7.483-2.625z"></path></svg>',
			'linkedin'          => '<svg width="32" height="32" viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg"><path fill-rule="evenodd" clip-rule="evenodd" d="M29.632 0C30.939 0 32 1.032 32 2.307v27.384C32 30.966 30.939 32 29.632 32H2.36C1.058 32 0 30.966 0 29.69V2.308C0 1.032 1.058 0 2.36 0h27.272zM9.149 11.638h-4.77V27h4.77V11.638zm12.133-.382c-2.318 0-3.874 1.273-4.509 2.48h-.066v-2.098H12.14V27h4.76v-7.599c0-2.004.378-3.946 2.858-3.946 2.447 0 2.475 2.29 2.475 4.072V27H27v-8.423c0-4.138-.893-7.32-5.718-7.32v-.001zM6.764 4A2.766 2.766 0 004 6.769a2.765 2.765 0 105.528 0A2.767 2.767 0 006.765 4h-.001z"/></svg>',
			'livejournal'       => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M11 3.125c0 0 0 0 0 0s0 0 0 0 0 0 0 0zM16.938 1.875c-2.125 0-4.125 0.438-5.938 1.25 0.125 0.063 0.188 0.125 0.313 0.25 0.313 0.313 0.625 0.625 0.938 0.938 0.063 0.063 0.063 0.063 0.063 0.063l10.75 10.75c-3.313 1.938-5.875 4.563-7.75 8l-10.813-10.813c-0.063-0.063-0.063-0.063-0.125-0.125-0.313-0.375-0.688-0.688-1-1-0.063-0.063-0.125-0.125-0.188-0.25 1.625-3.563 4.188-6.188 7.875-7.75-0.063-0.063-0.063-0.063-0.063-0.063s0 0 0 0c-0.938-0.938-1.875-1.813-2.75-2.75-0.375-0.313-0.688-0.438-1.125-0.188-0.563 0.25-1.063 0.563-1.563 0.875-1.75 1.188-3.188 2.625-4.375 4.313-0.375 0.563-0.75 1.125-1.063 1.813-0.188 0.375-0.188 0.625 0.188 0.938 0.938 0.938 1.875 1.875 2.813 2.813-0.813 1.813-1.188 3.875-1.188 5.938 0 8.313 6.75 15.063 15 15.063 8.313 0 15.063-6.75 15.063-15.063 0-8.25-6.75-15-15.063-15zM18.5 24.313c1.25-2.875 3.25-4.813 6-6.125 0.5 2.563 1 5 1.5 7.688-2.625-0.563-5-1.063-7.5-1.563z"></path></svg>',
			'locker'            => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M25.333 14h-1.333v-4c0-4.408-3.592-8-8-8s-8 3.592-8 8v4h-1.333c-2.208 0-4 1.792-4 4v9.333c0 2.208 1.792 4 4 4h18.667c2.208 0 4-1.792 4-4v-9.333c0-2.208-1.792-4-4-4zM10.667 10c0-2.942 2.392-5.333 5.333-5.333s5.333 2.392 5.333 5.333v4h-10.667v-4zM26.667 27.333c0 0.733-0.6 1.333-1.333 1.333h-18.667c-0.733 0-1.333-0.6-1.333-1.333v-9.333c0-0.733 0.6-1.333 1.333-1.333h18.667c0.733 0 1.333 0.6 1.333 1.333v9.333z"></path></svg>',
			'login'             => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M25.15 21.458c-2.458-1.375-5.708-2.125-9.15-2.125s-6.692 0.75-9.15 2.125c-2.7 1.5-4.183 3.583-4.183 5.875v1.333c0 0.733 0.6 1.333 1.333 1.333s1.333-0.6 1.333-1.333v-1.333c0-1.258 1.025-2.55 2.817-3.55 2.067-1.15 4.85-1.783 7.85-1.783s5.783 0.633 7.85 1.783c1.792 1 2.817 2.292 2.817 3.55v1.333c0 0.733 0.6 1.333 1.333 1.333s1.333-0.6 1.333-1.333v-1.333c0-2.292-1.483-4.375-4.183-5.875zM16.058 16.608c3.675 0 6.667-2.992 6.667-6.667 0-0.333-0.025-0.667-0.067-0.967v-0.008c0-0.017 0-0.025-0.008-0.042-0.217-1.408-0.867-2.692-1.875-3.7s-2.292-1.658-3.7-1.875c-0.333-0.050-0.675-0.075-1.017-0.075-3.675 0-6.667 2.992-6.667 6.667s2.992 6.667 6.667 6.667zM16.058 5.942c0.208 0 0.408 0.017 0.608 0.050 1.7 0.258 3.075 1.625 3.342 3.325v0.008c0 0.017 0 0.025 0.008 0.042 0.025 0.175 0.042 0.375 0.042 0.575 0 2.208-1.792 4-4 4s-4-1.792-4-4 1.792-4 4-4z"></path></svg>
			',
			'medium'            => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M31.75 7.313l-10.125-5.063c-0.063 0-0.063 0-0.125 0-0.125 0-0.313 0.063-0.375 0.125l-6.188 10.188 6.938 11.313 10-16.188c0.063-0.125 0-0.313-0.125-0.375zM11.625 9v10.938l9.75 4.875zM22.188 25.188l8.938 4.5c0.438 0.25 0.875 0 0.875-0.563v-19.875zM10.5 7.375l-9.313-4.625-0.625-0.375c-0.063 0-0.125-0.063-0.188-0.063-0.125 0-0.188 0.063-0.25 0.125s-0.125 0.188-0.125 0.375v21.313c0 0.375 0.25 0.813 0.563 0.938l9.188 4.625c0.125 0.063 0.25 0.125 0.375 0.125 0.313 0 0.5-0.25 0.5-0.688v-21.563c0-0.063 0-0.125-0.125-0.188z"></path></svg>',
			'menu'              => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M28 15.333h-24c-0.733 0-1.333 0.6-1.333 1.333s0.6 1.333 1.333 1.333h24c0.733 0 1.333-0.6 1.333-1.333s-0.6-1.333-1.333-1.333zM4 10h24c0.733 0 1.333-0.6 1.333-1.333s-0.6-1.333-1.333-1.333h-24c-0.733 0-1.333 0.6-1.333 1.333s0.6 1.333 1.333 1.333zM28 23.333h-24c-0.733 0-1.333 0.6-1.333 1.333s0.6 1.333 1.333 1.333h24c0.733 0 1.333-0.6 1.333-1.333s-0.6-1.333-1.333-1.333z"></path></svg>',
			'messenger'         => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M16 0c-8.768 0-15.904 6.624-15.904 14.816 0 4.672 2.304 8.832 5.92 11.552v5.632l5.408-2.976c1.44 0.416 2.976 0.608 4.576 0.608 8.768 0 15.904-6.624 15.904-14.816s-7.136-14.816-15.904-14.816zM17.568 19.936l-4.032-4.32-7.904 4.32 8.704-9.28 4.16 4.352 7.808-4.352-8.736 9.28z"></path></svg>',
			'mewe'              => '<svg width="46" height="32" viewBox="0 0 46 32" xmlns="http://www.w3.org/2000/svg"><path d="M10.618 6.228a2.147 2.147 0 11-4.294 0 2.147 2.147 0 014.294 0zm10.455.007a2.148 2.148 0 11-4.292-.174 2.148 2.148 0 014.292.174zm12.356 0a2.15 2.15 0 11-4.301 0 2.15 2.15 0 014.301 0zm12.348 0a2.15 2.15 0 11-4.3 0 2.15 2.15 0 014.3 0zM0 12.117c0-.829.658-1.512 1.512-1.512h.317c.659 0 1.099.341 1.39.804l5.167 8.094 5.194-8.118c.317-.512.757-.78 1.363-.78h.316c.854 0 1.513.681 1.513 1.512v14.284a1.485 1.485 0 01-1.512 1.488 1.51 1.51 0 01-1.488-1.488V16.163L9.606 22.43c-.317.486-.708.78-1.268.78-.535 0-.926-.294-1.243-.78l-4.144-6.215v10.214c0 .829-.659 1.461-1.488 1.461a1.434 1.434 0 01-1.461-1.461v-14.31H0zm19 .498a1.585 1.585 0 01-.122-.586c0-.83.707-1.461 1.535-1.461.73 0 1.294.463 1.488 1.098l3.926 11.726 3.85-11.676c.245-.708.754-1.194 1.513-1.194h.218c.754 0 1.294.49 1.512 1.194l3.851 11.676 3.926-11.752c.222-.61.73-1.072 1.462-1.072.804 0 1.487.636 1.487 1.438-.01.21-.06.417-.146.61l-5.048 14.141c-.267.757-.829 1.243-1.561 1.243h-.29c-.731 0-1.27-.463-1.536-1.243L31.26 15.616l-3.803 11.141c-.267.78-.827 1.243-1.562 1.243h-.29c-.73 0-1.268-.463-1.562-1.243L19 12.615z"/></svg>',
			'mix'               => '<svg width="28" height="32" viewBox="0 0 28 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M0 4v21.806c0 3.512 5.5 3.631 5.5 0v-14.912c0.494-3.306 5.5-3.15 5.5 0.406v10.956c0 3.619 6 3.625 6 0v-7.256c0.331-3.419 5.5-3.281 5.5 0.269v1.487c0 3.744 5.5 3.538 5.5 0v-12.756h-28z"></path></svg>',
			'myspace'           => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M26.438 17.438c3.063 0 5.563 2.438 5.563 5.563 0 1 0 2.813 0 2.813h-11.188c0 0 0-1.813 0-2.813 0-3.125 2.5-5.563 5.625-5.563zM21.313 11.313c0-2.813 2.25-5.125 5.125-5.125 2.813 0 5.125 2.313 5.125 5.125 0 2.875-2.313 5.125-5.125 5.125-2.875 0-5.125-2.25-5.125-5.125zM14.875 18.313c2.75 0 5 2.25 5 5 0 0.875 0 2.5 0 2.5h-9.938c0 0 0-1.625 0-2.5 0-2.75 2.188-5 4.938-5zM10.313 12.75c0-2.563 2.063-4.625 4.563-4.625 2.563 0 4.625 2.063 4.625 4.625 0 2.5-2.063 4.563-4.625 4.563s-4.563-2.063-4.563-4.563zM4.438 19.063c2.5 0 4.5 2 4.5 4.5 0 0.813 0 2.25 0 2.25h-8.938c0 0 0-1.438 0-2.25 0-2.5 2-4.5 4.438-4.5zM0.313 14.063c0-2.313 1.875-4.125 4.125-4.125 2.313 0 4.125 1.813 4.125 4.125 0 2.25-1.813 4.125-4.125 4.125-2.25 0-4.125-1.875-4.125-4.125z"></path></svg>',
			'newsvine'          => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M17.125 27.375l2.75-2c0-0.125 0-0.188 0-0.25 0-1.25 1-2.313 2.25-2.313s2.313 1.063 2.313 2.313c0 1.25-1.063 2.25-2.313 2.25-0.438 0-0.875-0.125-1.188-0.375l-3.813 2.75v2.25h-2.25v-6.75l-3.875-2.813c-0.375 0.188-0.75 0.313-1.125 0.313-1.25 0-2.313-1.063-2.313-2.313s1.063-2.25 2.313-2.25c1.25 0 2.25 1 2.25 2.25 0 0.125 0 0.25-0.063 0.375l2.813 2.063v-6.75l-3.875-2.813c-0.375 0.188-0.75 0.313-1.125 0.313-1.25 0-2.313-1.063-2.313-2.313s1.063-2.25 2.313-2.25c1.25 0 2.25 1 2.25 2.25 0 0.125 0 0.313-0.063 0.438l2.813 2v-6.313l-3.938-2.875c-0.313 0.188-0.688 0.313-1.063 0.313-1.25 0-2.313-1-2.313-2.25s1.063-2.313 2.313-2.313c1.25 0 2.25 1.063 2.25 2.313 0 0.125 0 0.25-0.063 0.375l2.813 2.063v-5.063h2.25v9.563l2.75-2c0-0.125 0-0.25 0-0.313 0-1.25 1-2.313 2.25-2.313s2.313 1.063 2.313 2.313c0 1.25-1.063 2.25-2.313 2.25-0.438 0-0.813-0.125-1.188-0.313l-3.813 2.75v6.313l2.75-2c0-0.125 0-0.188 0-0.25 0-1.313 1-2.313 2.25-2.313s2.313 1 2.313 2.313c0 1.25-1.063 2.25-2.313 2.25-0.438 0-0.813-0.125-1.188-0.375l-3.813 2.813v6.688z"></path></svg>',
			'outlook'           => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M20 6.813v6.25l2.188 1.375c0.063 0 0.188 0 0.25 0l9.375-6.375c0-0.75-0.688-1.25-1.063-1.25zM20 15.375l2 1.375c0.25 0.188 0.625 0 0.625 0-0.375 0.188 9.188-6.125 9.188-6.125v11.5c0 1.25-0.813 1.75-1.688 1.75h-10.125v-8.5zM9.625 12.5c-0.688 0-1.188 0.313-1.625 0.938-0.375 0.625-0.563 1.5-0.563 2.563s0.188 1.875 0.563 2.5c0.438 0.625 0.938 0.938 1.625 0.938 0.625 0 1.188-0.313 1.563-0.875 0.375-0.625 0.563-1.438 0.563-2.563 0-1.063-0.188-1.938-0.563-2.563s-0.875-0.938-1.563-0.938zM0.188 3.625v24.5l18.625 3.875v-32zM12.625 20c-0.75 1-1.813 1.563-3.063 1.563s-2.25-0.5-3-1.5c-0.813-1-1.188-2.313-1.188-3.938 0-1.688 0.375-3.063 1.188-4.125s1.813-1.625 3.125-1.625c1.25 0 2.25 0.5 3 1.563 0.75 1 1.125 2.313 1.125 3.938 0 1.688-0.375 3.063-1.188 4.125z"></path></svg>',
			'parler'            => '<svg width="32" height="32" viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg"><path d="M18.669 26.66h-8v-2.718a2.603 2.603 0 012.605-2.604h5.398c4.416 0 8.006-3.581 8.006-7.981s-3.593-7.981-8.01-7.981h-.082l-1.696-.051H0A5.324 5.324 0 015.322 0h11.65l1.742.054C26.044.077 32 6.035 32 13.357c0 7.334-5.98 13.302-13.331 13.302z"/><path d="M5.322 32A5.321 5.321 0 010 26.678V15.837a5.171 5.171 0 015.171-5.171h13.501a2.662 2.662 0 010 5.324H7.952a2.627 2.627 0 00-2.627 2.628V32h-.003z"/></svg>',
			'patreon'           => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M32 12.175c0 6.331-5.15 11.488-11.488 11.488-6.356 0-11.525-5.15-11.525-11.488 0-6.35 5.169-11.519 11.525-11.519 6.338 0 11.488 5.169 11.488 11.519zM0 31.344h5.625v-30.688h-5.625v30.688z"></path></svg>',
			'pinterest'         => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M10.625 12.25c0-1.375 0.313-2.5 1.063-3.438 0.688-0.938 1.563-1.438 2.563-1.438 0.813 0 1.438 0.25 1.875 0.813s0.688 1.25 0.688 2.063c0 0.5-0.125 1.125-0.313 1.813-0.188 0.75-0.375 1.625-0.688 2.563-0.313 1-0.563 1.75-0.688 2.313-0.25 1-0.063 1.875 0.563 2.625 0.625 0.688 1.438 1.063 2.438 1.063 1.75 0 3.188-1 4.313-2.938 1.125-2 1.688-4.375 1.688-7.188 0-2.125-0.688-3.875-2.063-5.25-1.375-1.313-3.313-2-5.813-2-2.813 0-5.063 0.875-6.75 2.688-1.75 1.75-2.625 3.875-2.625 6.375 0 1.5 0.438 2.75 1.25 3.75 0.313 0.313 0.375 0.688 0.313 1.063-0.125 0.313-0.25 0.813-0.375 1.5-0.063 0.25-0.188 0.438-0.375 0.5s-0.375 0.063-0.563 0c-1.313-0.563-2.25-1.438-2.938-2.75s-1-2.813-1-4.5c0-1.125 0.188-2.188 0.563-3.313s0.875-2.188 1.625-3.188c0.75-1.063 1.688-1.938 2.688-2.75 1.063-0.813 2.313-1.438 3.875-1.938 1.5-0.438 3.125-0.688 4.813-0.688 1.813 0 3.438 0.313 4.938 0.938 1.5 0.563 2.813 1.375 3.813 2.375 1.063 1.063 1.813 2.188 2.438 3.5 0.563 1.313 0.875 2.688 0.875 4.063 0 3.75-0.938 6.875-2.875 9.313-1.938 2.5-4.375 3.688-7.375 3.688-1 0-1.938-0.188-2.813-0.688-0.875-0.438-1.5-1-1.875-1.688-0.688 2.938-1.125 4.688-1.313 5.25-0.375 1.438-1.25 3.188-2.688 5.25h-1.313c-0.25-2.563-0.188-4.688 0.188-6.375l2.438-10.313c-0.375-0.813-0.563-1.813-0.563-3.063z"></path></svg>',
			'plus'              => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M25.333 15.333h-8v-8c0-0.733-0.6-1.333-1.333-1.333s-1.333 0.6-1.333 1.333v8h-8c-0.733 0-1.333 0.6-1.333 1.333s0.6 1.333 1.333 1.333h8v8c0 0.733 0.6 1.333 1.333 1.333s1.333-0.6 1.333-1.333v-8h8c0.733 0 1.333-0.6 1.333-1.333s-0.6-1.333-1.333-1.333z"></path></svg>',
			'pocket'            => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M32 5.5c0-0.875 0.063-1.688-0.438-2.438-0.375-0.563-1-1.063-1.625-1.25-0.875-0.25-1.813-0.188-2.625-0.125-3.75 0-7.563-0.063-11.313-0.063-0.438 0-0.813 0-1.25 0-0.5 0-1 0-1.563 0-1.813 0-3.688 0-5.563 0-0.938 0-1.875 0-2.813 0-0.5 0-1 0-1.5 0-0.438 0-0.813 0-1.188 0.125-1.313 0.375-2.125 1.625-2.125 3 0 1.875 0 3.75 0 5.625 0 1.813-0.063 3.688 0.063 5.563 0.125 1.688 0.625 3.438 1.313 5 1.313 2.813 3.438 5.188 6.063 6.875 2.875 1.813 6.313 2.688 9.688 2.438 3.313-0.25 6.5-1.563 9.063-3.688 2.438-1.938 4.188-4.563 5.125-7.5 0.5-1.688 0.688-3.5 0.688-5.25 0-1.813 0-3.688 0-5.563 0-0.938 0-1.875 0-2.75zM25.625 12.75c-0.188 0.813-1 1.375-1.563 1.938-1.438 1.375-2.875 2.75-4.313 4.063-0.75 0.75-1.438 1.438-2.188 2.125-0.563 0.5-1.313 0.75-2.063 0.563-0.875-0.188-1.5-1-2.063-1.563-0.75-0.688-1.438-1.375-2.125-2-1.438-1.375-2.875-2.75-4.313-4.188-1.063-1.063-0.875-3 0.625-3.625 0.688-0.25 1.5-0.188 2.063 0.188 0.375 0.313 0.75 0.625 1.063 1 0.375 0.313 0.75 0.688 1.063 1 1.438 1.375 2.75 2.75 4.188 4 1.625-1.5 3.25-3.063 4.875-4.625 0.688-0.625 1.313-1.5 2.313-1.688 0.75-0.125 1.563 0.188 2.063 0.75 0.438 0.563 0.625 1.375 0.375 2.063z"></path></svg>',
			'print'             => '<svg width="30" height="32" viewBox="0 0 30 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M6.857 27.714h16v-4.571h-16v4.571zM6.857 16.286h16v-6.857h-2.857c-0.946 0-1.714-0.768-1.714-1.714v-2.857h-11.429v11.429zM27.429 17.429c0-0.625-0.518-1.143-1.143-1.143s-1.143 0.518-1.143 1.143 0.518 1.143 1.143 1.143 1.143-0.518 1.143-1.143zM29.714 17.429v7.429c0 0.304-0.268 0.571-0.571 0.571h-4v2.857c0 0.946-0.768 1.714-1.714 1.714h-17.143c-0.946 0-1.714-0.768-1.714-1.714v-2.857h-4c-0.304 0-0.571-0.268-0.571-0.571v-7.429c0-1.875 1.554-3.429 3.429-3.429h1.143v-9.714c0-0.946 0.768-1.714 1.714-1.714h12c0.946 0 2.25 0.536 2.929 1.214l2.714 2.714c0.679 0.679 1.214 1.982 1.214 2.929v4.571h1.143c1.875 0 3.429 1.554 3.429 3.429z"></path></svg>',
			'reddit'            => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M32 15.75c0-1.75-1.375-3.125-3.125-3.125-1 0-1.875 0.438-2.438 1.188-2.5-1.813-6-2.938-9.938-3l2.313-6.875 5.75 1.375c0.125 0 0.188 0 0.25 0 0.063 1.313 1.125 2.375 2.5 2.375s2.438-1.125 2.438-2.5c0-1.375-1.063-2.438-2.438-2.438-1.063 0-1.875 0.563-2.313 1.438 0 0-0.063-0.063-0.125-0.063l-6.313-1.5c-0.313-0.063-0.625 0.125-0.688 0.375l-2.688 7.813c-3.813 0.125-7.188 1.188-9.688 2.938-0.563-0.688-1.375-1.125-2.375-1.125-1.75 0-3.125 1.375-3.125 3.125 0 1.25 0.75 2.375 1.875 2.875-0.125 0.438-0.188 0.938-0.188 1.438 0 5.125 6.375 9.313 14.25 9.313s14.25-4.188 14.25-9.313c0-0.438-0.063-0.938-0.188-1.375 1.188-0.5 2-1.625 2-2.938zM8.688 18.25c0-1.25 1.063-2.313 2.313-2.313s2.313 1.063 2.313 2.313c0 1.25-1.063 2.313-2.313 2.313s-2.313-1.063-2.313-2.313zM21.188 24.625c-1.063 1.063-2.75 1.563-5.188 1.563s-4.125-0.5-5.188-1.563c-0.125-0.125-0.125-0.375 0-0.563 0.125-0.125 0.375-0.125 0.5 0 0.938 0.938 2.438 1.375 4.688 1.375 2.188 0 3.75-0.438 4.688-1.375 0.125-0.125 0.375-0.125 0.5 0 0.125 0.188 0.125 0.438 0 0.563zM21.063 20.563c-1.313 0-2.375-1-2.375-2.313 0-1.25 1.063-2.313 2.375-2.313 1.25 0 2.25 1.063 2.25 2.313 0 1.313-1 2.313-2.25 2.313z"></path></svg>',
			'rss'               => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M8.563 27.688c0 2.375-1.875 4.25-4.25 4.25s-4.25-1.875-4.25-4.25c0-2.313 1.875-4.25 4.25-4.25s4.25 1.938 4.25 4.25zM0.063 17c4 0 7.75 1.563 10.563 4.375s4.375 6.625 4.375 10.625h6.125c0-11.625-9.438-21.125-21.063-21.125zM0.063 6.125c14.188 0 25.75 11.625 25.75 25.875h6.125c0-17.625-14.313-32-31.875-32z"></path></svg>',
			'share'             => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M24 20.667c-1.475 0-2.817 0.6-3.783 1.575l-7.083-4.125c0.133-0.458 0.2-0.95 0.2-1.45s-0.067-0.992-0.2-1.45l7.083-4.133c0.967 0.975 2.308 1.583 3.783 1.583 2.942 0 5.333-2.392 5.333-5.333s-2.392-5.333-5.333-5.333-5.333 2.392-5.333 5.333c0 0.5 0.067 0.992 0.2 1.45l-7.083 4.133c-0.967-0.975-2.308-1.583-3.783-1.583-2.942 0-5.333 2.392-5.333 5.333s2.392 5.333 5.333 5.333c1.483 0 2.817-0.608 3.792-1.583l7.083 4.125c-0.133 0.467-0.2 0.95-0.2 1.458 0 2.942 2.392 5.333 5.333 5.333s5.333-2.392 5.333-5.333-2.4-5.333-5.342-5.333zM24 4.667c1.467 0 2.667 1.2 2.667 2.667s-1.2 2.667-2.667 2.667c-0.958 0-1.8-0.508-2.275-1.275-0.008-0.017-0.017-0.033-0.025-0.050s-0.025-0.033-0.033-0.050c-0.208-0.383-0.333-0.825-0.333-1.292 0-1.467 1.2-2.667 2.667-2.667zM8 19.333c-1.467 0-2.667-1.2-2.667-2.667s1.2-2.667 2.667-2.667c0.958 0 1.8 0.508 2.275 1.275 0.008 0.017 0.017 0.033 0.025 0.050s0.025 0.033 0.033 0.050c0.208 0.383 0.333 0.825 0.333 1.292s-0.125 0.908-0.333 1.292c-0.008 0.017-0.025 0.033-0.033 0.050s-0.017 0.033-0.025 0.050c-0.475 0.767-1.317 1.275-2.275 1.275zM24 28.667c-1.467 0-2.667-1.2-2.667-2.667 0-0.433 0.108-0.85 0.292-1.208 0.033-0.042 0.058-0.083 0.092-0.133 0.025-0.042 0.050-0.092 0.067-0.133 0.475-0.717 1.292-1.192 2.217-1.192 1.467 0 2.667 1.2 2.667 2.667s-1.2 2.667-2.667 2.667z"></path></svg>',
			'skype'             => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M30.875 19.375c0.25-1 0.313-2.125 0.313-3.188 0-8.313-6.688-15.063-15-15.063-0.875 0-1.75 0.063-2.563 0.188-1.375-0.813-2.938-1.313-4.688-1.313-4.813 0-8.688 3.938-8.688 8.75 0 1.563 0.438 3.125 1.188 4.375-0.188 1-0.313 2-0.313 3.063 0 8.313 6.75 15 15.063 15 0.938 0 1.813-0.063 2.75-0.25 1.188 0.688 2.625 1.063 4.125 1.063 4.813 0 8.688-3.938 8.688-8.75 0-1.375-0.313-2.688-0.875-3.875zM23.75 23.313c-0.688 1-1.688 1.75-3.063 2.313-1.313 0.563-2.875 0.813-4.688 0.813-2.125 0-3.938-0.375-5.375-1.125-1-0.5-1.875-1.25-2.5-2.188s-0.938-1.813-0.938-2.688c0-0.563 0.188-1.063 0.625-1.438s0.938-0.563 1.563-0.563c0.5 0 0.938 0.125 1.313 0.438s0.688 0.75 0.875 1.313c0.313 0.563 0.563 1.125 0.875 1.5s0.75 0.75 1.25 1c0.563 0.25 1.313 0.375 2.188 0.375 1.25 0 2.313-0.25 3.063-0.813 0.75-0.5 1.125-1.125 1.125-1.875 0-0.563-0.188-1.063-0.625-1.438-0.375-0.375-0.938-0.625-1.563-0.875-0.688-0.188-1.625-0.438-2.75-0.688-1.563-0.313-2.938-0.688-4-1.188-1.063-0.438-1.938-1.063-2.563-1.875s-1-1.813-1-3c0-1.125 0.375-2.188 1.063-3.063s1.625-1.563 2.938-2c1.25-0.5 2.688-0.688 4.375-0.688 1.375 0 2.563 0.125 3.5 0.438 1 0.313 1.875 0.75 2.5 1.25 0.688 0.5 1.188 1.063 1.5 1.625s0.438 1.188 0.438 1.75c0 0.5-0.188 1-0.625 1.438-0.375 0.438-0.938 0.625-1.563 0.625-0.5 0-0.938-0.125-1.25-0.375s-0.563-0.688-0.938-1.188c-0.375-0.75-0.813-1.313-1.313-1.688s-1.375-0.563-2.5-0.563c-1.125 0-2 0.188-2.625 0.625-0.625 0.375-0.938 0.875-0.938 1.438 0 0.313 0.063 0.625 0.313 0.875 0.188 0.25 0.5 0.5 0.875 0.688s0.813 0.375 1.188 0.438c0.438 0.125 1.125 0.313 2.125 0.563 1.188 0.25 2.313 0.5 3.313 0.813 1 0.375 1.875 0.75 2.563 1.188 0.75 0.5 1.313 1.063 1.688 1.813 0.438 0.688 0.625 1.625 0.625 2.625 0 1.25-0.375 2.375-1.063 3.375z"></path></svg>',
			'sms'               => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M24 10.688v-2.688h-16v2.688h16zM18.688 18.688v-2.688h-10.688v2.688h10.688zM8 12v2.688h16v-2.688h-16zM26.688 2.688c1.438 0 2.625 1.188 2.625 2.625v16c0 1.438-1.188 2.688-2.625 2.688h-18.688l-5.313 5.313v-24c0-1.438 1.188-2.625 2.625-2.625h21.375z"></path></svg>',
			'snapchat'          => '<svg width="30" height="32" viewBox="0 0 30 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M15.143 2.571c3.214-0.036 5.875 1.768 7.214 4.661 0.411 0.875 0.482 2.232 0.482 3.196 0 1.143-0.089 2.268-0.161 3.411 0.143 0.071 0.339 0.125 0.5 0.125 0.643 0 1.179-0.482 1.821-0.482 0.607 0 1.482 0.429 1.482 1.143 0 1.714-3.589 1.393-3.589 2.893 0 0.268 0.107 0.518 0.214 0.768 0.857 1.875 2.482 3.679 4.411 4.482 0.464 0.196 0.929 0.304 1.429 0.411 0.321 0.071 0.5 0.304 0.5 0.625 0 1.214-3.089 1.714-3.911 1.839-0.357 0.554-0.089 1.857-1.036 1.857-0.732 0-1.464-0.232-2.25-0.232-0.375 0-0.75 0.018-1.107 0.089-2.125 0.357-2.839 2.643-6.321 2.643-3.357 0-4.161-2.286-6.232-2.643-0.375-0.071-0.75-0.089-1.125-0.089-0.804 0-1.571 0.268-2.214 0.268-1 0-0.696-1.321-1.071-1.893-0.821-0.125-3.911-0.625-3.911-1.839 0-0.321 0.179-0.554 0.5-0.625 0.5-0.107 0.964-0.214 1.429-0.411 1.911-0.786 3.571-2.607 4.411-4.482 0.107-0.25 0.214-0.5 0.214-0.768 0-1.5-3.607-1.214-3.607-2.875 0-0.696 0.821-1.143 1.446-1.143 0.554 0 1.107 0.464 1.804 0.464 0.196 0 0.393-0.036 0.571-0.125-0.071-1.125-0.161-2.25-0.161-3.393 0-0.964 0.071-2.339 0.482-3.214 1.571-3.393 4.232-4.625 7.786-4.661z"></path></svg>',
			'social-id'         => '<svg width="37" height="32" viewBox="0 0 37 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M16 21.262c0 1.304-0.857 2.357-1.911 2.357h-7.607c-1.054 0-1.911-1.054-1.911-2.357 0-2.357 0.571-5.071 2.929-5.071 0.714 0.714 1.696 1.143 2.786 1.143s2.071-0.429 2.786-1.143c2.357 0 2.929 2.714 2.929 5.071zM13.714 13.333c0 1.893-1.536 3.429-3.429 3.429s-3.429-1.536-3.429-3.429 1.536-3.429 3.429-3.429 3.429 1.536 3.429 3.429zM32 20.762v1.143c0 0.321-0.25 0.571-0.571 0.571h-12.571c-0.321 0-0.571-0.25-0.571-0.571v-1.143c0-0.321 0.25-0.571 0.571-0.571h12.571c0.321 0 0.571 0.25 0.571 0.571zM25.143 16.191v1.143c0 0.321-0.25 0.571-0.571 0.571h-5.714c-0.321 0-0.571-0.25-0.571-0.571v-1.143c0-0.321 0.25-0.571 0.571-0.571h5.714c0.321 0 0.571 0.25 0.571 0.571zM32 16.191v1.143c0 0.321-0.25 0.571-0.571 0.571h-3.429c-0.321 0-0.571-0.25-0.571-0.571v-1.143c0-0.321 0.25-0.571 0.571-0.571h3.429c0.321 0 0.571 0.25 0.571 0.571zM32 11.619v1.143c0 0.321-0.25 0.571-0.571 0.571h-12.571c-0.321 0-0.571-0.25-0.571-0.571v-1.143c0-0.321 0.25-0.571 0.571-0.571h12.571c0.321 0 0.571 0.25 0.571 0.571zM34.286 26.476v-20h-32v20c0 0.304 0.268 0.571 0.571 0.571h30.857c0.304 0 0.571-0.268 0.571-0.571zM36.571 4.762v21.714c0 1.571-1.286 2.857-2.857 2.857h-30.857c-1.571 0-2.857-1.286-2.857-2.857v-21.714c0-1.571 1.286-2.857 2.857-2.857h30.857c1.571 0 2.857 1.286 2.857 2.857z"></path></svg>',
			'socialsnap-icon'   => '<svg width="30" height="32" viewBox="0 0 30 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M22.293 0.146l7.602 4.172c0.386 0.201 0.386 0.541 0 0.757l-16.688 9.147c-1.684 0.943-2.241 2.271-1.669 3.461 0 0.093 0 0.201-0.201 0.263-0.207 0.088-0.441 0.088-0.649 0l-10.399-5.702c-0.386-0.201-0.386-0.541 0-0.757l20.628-11.311c0.428-0.225 0.937-0.236 1.375-0.031zM7.892 31.854l-7.602-4.172c-0.386-0.201-0.386-0.541 0-0.757l16.688-9.147c1.684-0.943 2.241-2.271 1.669-3.461 0-0.093 0-0.201 0.201-0.263 0.207-0.088 0.442-0.088 0.649 0l10.399 5.702c0.386 0.201 0.386 0.541 0 0.757l-20.628 11.311c-0.428 0.225-0.937 0.237-1.375 0.031z"></path></svg>
			',
			'socialsnap-light'  => '<svg viewBox="0 0 189 32" xmlns="http://www.w3.org/2000/svg">
				<path d="m43.451 20.565c-0.001507-0.25756 0.14867-0.49178 0.38302-0.59733 0.12665-0.12589 0.30074-0.19182 0.47878-0.18133 0.25177 0.0031524 0.49083 0.11139 0.65965 0.29867 1.0727 1.4949 2.8132 2.3614 4.6495 2.3147 0.8736 0.019678 1.7343-0.21364 2.479-0.672 0.69172-0.39914 1.1025-1.1528 1.064-1.952 0.028561-0.75545-0.34381-1.4696-0.97884-1.8773-0.85138-0.49427-1.7737-0.85405-2.7344-1.0667-1.3749-0.30082-2.6837-0.85023-3.8622-1.6213-0.93502-0.67394-1.4609-1.7808-1.3938-2.9333-0.010852-0.78295 0.22769-1.549 0.68093-2.1867 0.47072-0.65934 1.1103-1.179 1.8513-1.504 0.83767-0.36869 1.7452-0.55066 2.6599-0.53333 0.88026 0.0013989 1.7528 0.16407 2.5748 0.48 0.8583 0.32619 1.6198 0.86575 2.213 1.568 0.17969 0.14436 0.27876 0.36684 0.26599 0.59733-0.0012558 0.2437-0.11039 0.47425-0.29791 0.62933-0.14953 0.12311-0.33859 0.18755-0.53198 0.18133-0.20171-0.0027652-0.39367-0.087444-0.53198-0.23467-0.46523-0.52802-1.0436-0.94373-1.6917-1.216-0.6733-0.28363-1.3976-0.42524-2.1279-0.416-0.8447-0.026215-1.6781 0.2003-2.3939 0.65067-0.65613 0.42979-1.0334 1.1781-0.98948 1.9627 0.012045 0.74992 0.41601 1.4384 1.064 1.8133 0.94531 0.51767 1.9591 0.89835 3.011 1.1307 0.87652 0.20634 1.7315 0.49563 2.5535 0.864 0.65424 0.30097 1.2216 0.76339 1.6491 1.344 0.45461 0.6751 0.67861 1.4798 0.63837 2.2933 0.035254 1.2222-0.54436 2.3804-1.5427 3.0827-1.1183 0.78927-2.4635 1.1901-3.8302 1.1413-2.2455 0.081726-4.3925-0.92701-5.7666-2.7093-0.12202-0.19587-0.19165-0.41997-0.20215-0.65067zm28.067-0.50133c-0.63377 1.1862-1.5842 2.1723-2.745 2.848-2.3707 1.3697-5.2897 1.3697-7.6605 0-1.1448-0.68229-2.0798-1.6677-2.7024-2.848-1.3046-2.5402-1.3046-5.5558 0-8.096 0.62593-1.1807 1.56-2.1688 2.7024-2.8587 2.3654-1.3897 5.2951-1.3897 7.6605 0 1.1424 0.68991 2.0765 1.678 2.7024 2.8587 1.3082 2.5393 1.3082 5.5567 0 8.096h0.042558zm-1.5427-7.264c-0.46968-0.92541-1.1836-1.7039-2.0641-2.2507-0.87809-0.5459-1.8927-0.83071-2.9259-0.82133-1.0266-0.010048-2.0345 0.27497-2.9046 0.82133-0.88059 0.54986-1.5972 1.3273-2.0747 2.2507-0.51142 0.9878-0.77094 2.0872-0.75541 3.2-0.018214 1.1131 0.24146 2.2131 0.75541 3.2 0.4741 0.92232 1.1871 1.6997 2.0641 2.2507 0.87005 0.54636 1.878 0.83138 2.9046 0.82133 2.1156 0.018815 4.0567-1.1737 5.0006-3.072 0.97991-2.0202 0.97991-4.3798 0-6.4zm15.332-4.4267c0.68974 0.16047 1.3431 0.44998 1.9258 0.85333 0.40514 0.252 0.66301 0.68553 0.69157 1.1627 1.611e-4 0.21967-0.079234 0.43193-0.22343 0.59733-0.13281 0.17057-0.33743 0.2692-0.55326 0.26667-0.17087 0.0075164-0.34042-0.033132-0.48942-0.11733-0.15438-0.10123-0.29711-0.21929-0.42558-0.352-0.2167-0.22694-0.46811-0.41778-0.74477-0.56533-0.37477-0.1612-0.76781-0.27583-1.1703-0.34133-0.51634-0.090499-1.0399-0.13334-1.564-0.128-1.055-0.011875-2.0917 0.27663-2.9897 0.832-0.89824 0.54945-1.6331 1.3304-2.1279 2.2613-0.52743 0.98259-0.79484 2.0843-0.77669 3.2-0.013619 1.1114 0.24195 2.2094 0.74477 3.2 0.96148 1.9174 2.9348 3.1119 5.0751 3.072 0.5729 0.014187 1.1452-0.046774 1.7023-0.18133 0.43847-0.12193 0.86286-0.28997 1.2661-0.50133 0.2503-0.16008 0.48855-0.33833 0.71285-0.53333 0.12924-0.11481 0.26807-0.2183 0.41494-0.30933 0.11589-0.069201 0.24814-0.10603 0.38302-0.10667 0.21115-0.0089267 0.41614 0.072504 0.5639 0.224 0.14518 0.15986 0.22164 0.37066 0.21279 0.58667-0.073561 0.44178-0.33329 0.83043-0.71285 1.0667-0.58829 0.44764-1.2511 0.78712-1.9577 1.0027-0.89386 0.28656-1.8279 0.42703-2.7663 0.416-1.3615 0.013107-2.6994-0.35641-3.8622-1.0667-1.1255-0.68292-2.0363-1.67-2.628-2.848-0.60502-1.2665-0.89696-2.6605-0.85116-4.064-0.017731-1.3844 0.30764-2.7515 0.94692-3.9787 1.278-2.47 3.8428-3.9996 6.6178-3.9467 0.8709-0.0098761 1.7396 0.090472 2.5854 0.29867zm6.022-2.496c-0.22344-0.21294-0.34377-0.51258-0.32983-0.82133v-0.29867c-0.020468-0.31202 0.10544-0.61576 0.34047-0.82133 0.27022-0.19921 0.60146-0.29732 0.93628-0.27733 0.79442 0 1.1916 0.35556 1.1916 1.0667v0.29867c0.019465 0.30548-0.092912 0.60462-0.30855 0.82133-0.2549 0.20605-0.57758 0.30881-0.90436 0.288-0.32932 0.030165-0.65833-0.060828-0.92564-0.256zm1.564 17.504c-0.25578 0.2618-0.64479 0.34105-0.98221 0.2001-0.33741-0.14095-0.5552-0.47368-0.54989-0.8401v-13.696c-0.03733-0.34148 0.12337-0.67429 0.41368-0.85676 0.29031-0.18247 0.65913-0.18247 0.94944 0 0.29031 0.18247 0.45101 0.51528 0.41368 0.85676v13.696c0.0071271 0.23961-0.08536 0.47142-0.25535 0.64h0.01064zm17.832-14.688c0.1701 0.17721 0.2621 0.41548 0.25535 0.66133v13.397c0.03733 0.34148-0.12337 0.67429-0.41368 0.85676-0.29031 0.18247-0.65913 0.18247-0.94944 0-0.29031-0.18247-0.45101-0.51528-0.41368-0.85676v-1.9947c-0.53392 0.93371-1.3 1.7128-2.2237 2.2613-0.9563 0.60284-2.0622 0.92438-3.1919 0.928-1.271 0.0091621-2.5157-0.36225-3.5749-1.0667-1.071-0.71134-1.931-1.6988-2.4897-2.8587-0.63045-1.2467-0.96198-2.6236-0.9682-4.0213-0.020983-1.4095 0.29649-2.8034 0.92564-4.064 0.56773-1.1489 1.4355-2.1221 2.5109-2.816 2.0636-1.296 4.6698-1.3533 6.788-0.14933 0.93493 0.51473 1.7058 1.2838 2.2237 2.2187v-1.8453c-0.006753-0.24585 0.085246-0.48412 0.25535-0.66133 0.1681-0.172 0.39818-0.26893 0.63837-0.26893 0.24019 0 0.47027 0.096928 0.63837 0.26893l-0.010639 0.010667zm-4.043 12.8c0.83487-0.56443 1.5006-1.3462 1.9258-2.2613 0.47198-0.99986 0.71205-2.0939 0.70221-3.2 0.017839-1.107-0.22263-2.2028-0.70221-3.2-0.42809-0.91036-1.0934-1.688-1.9258-2.2507-0.8224-0.54707-1.79-0.83326-2.7769-0.82133-0.98416-0.014353-1.9505 0.26404-2.7769 0.8-0.83481 0.55532-1.5011 1.3303-1.9258 2.24-0.47958 0.99724-0.72005 2.093-0.70221 3.2-0.011802 1.104 0.22076 2.1969 0.68093 3.2 0.41661 0.91739 1.0797 1.7004 1.9151 2.2613 1.7025 1.0923 3.8832 1.0923 5.5858 0v0.032zm10.671 1.888c-0.25578 0.2618-0.64479 0.34105-0.9822 0.2001-0.33741-0.14095-0.5552-0.47368-0.54989-0.8401v-20.267c0-0.49485 0.40013-0.896 0.89372-0.896s0.89372 0.40115 0.89372 0.896v20.267c0.007127 0.23961-0.08536 0.47142-0.25535 0.64zm3.5536-3.1147c0.059838-0.87041 0.7686-1.5533 1.6385-1.5787 0.45742 0.017924 0.88776 0.22209 1.1916 0.56533 0.52704 0.56656 1.1549 1.0296 1.8513 1.3653 0.73514 0.31682 1.5302 0.46969 2.3301 0.448 1.5463-0.056889 2.3194-0.64356 2.3194-1.76 0.015694-0.55397-0.2918-1.0664-0.78733-1.312-0.77774-0.39405-1.5991-0.69479-2.4471-0.896-0.95715-0.2506-1.8957-0.56782-2.8088-0.94933-0.73048-0.32226-1.3728-0.81619-1.8726-1.44-0.55761-0.74873-0.83616-1.6695-0.78733-2.6027-0.016872-0.89776 0.27211-1.7744 0.81924-2.4853 0.55881-0.71709 1.2913-1.2789 2.1279-1.632 0.88488-0.39087 1.8419-0.59075 2.8088-0.58667 1.0365 0.0036404 2.0638 0.19517 3.0323 0.56533 0.89451 0.32882 1.6845 0.89245 2.2875 1.632 0.24908 0.29919 0.38473 0.67697 0.38302 1.0667 0.004287 0.44586-0.20554 0.86659-0.5639 1.1307-0.26847 0.21585-0.60279 0.33259-0.94692 0.33067-0.39654 0.01344-0.78119-0.13696-1.064-0.416-0.37195-0.45864-0.84557-0.82388-1.3831-1.0667-0.60356-0.26407-1.2567-0.39502-1.9151-0.384-1.5037 0-2.2556 0.54756-2.2556 1.6427-0.018631 0.39843 0.13217 0.78608 0.41494 1.0667 0.30367 0.28633 0.66711 0.50131 1.064 0.62933 0.42558 0.14933 1.0249 0.33422 1.7981 0.55467 0.91113 0.24502 1.8036 0.55515 2.6705 0.928 0.71827 0.32299 1.3413 0.82633 1.8087 1.4613 0.52632 0.75814 0.78813 1.6693 0.74477 2.592 0.017787 0.92449-0.28703 1.8262-0.8618 2.5493-0.58219 0.73418-1.3448 1.3039-2.213 1.6533-0.87858 0.37285-1.8229 0.56509-2.7769 0.56533-1.2461 0.01148-2.4836-0.20915-3.6494-0.65067-1.1104-0.42456-2.0722-1.1663-2.7663-2.1333-0.15054-0.25766-0.21747-0.55586-0.19151-0.85333zm28.142-11.051c0.85183 1.5197 1.2541 3.2513 1.1597 4.992v7.616c0.016644 0.48388-0.1639 0.95371-0.50006 1.3013-0.71054 0.69371-1.843 0.69371-2.5535 0-0.33616-0.34762-0.5167-0.81745-0.50006-1.3013v-7.6267c0.052032-0.90774-0.18654-1.8084-0.68093-2.5707-0.58128-0.68268-1.4611-1.0339-2.3513-0.93867-0.99889-0.058263-1.9772 0.30086-2.7024 0.992-0.6556 0.66981-1.0093 1.5794-0.97884 2.5173v7.616c0.01789 0.48409-0.16283 0.9544-0.50006 1.3013-0.71054 0.69371-1.843 0.69371-2.5535 0-0.33723-0.34694-0.51795-0.81724-0.50006-1.3013v-12.288c-0.013736-0.48022 0.16659-0.94565 0.50006-1.2907 0.71053-0.69371 1.843-0.69371 2.5535 0 0.33347 0.34502 0.51379 0.81044 0.50006 1.2907v0.50133c0.58039-0.75034 1.3037-1.3776 2.1279-1.8453 0.8539-0.51576 1.8332-0.78521 2.8301-0.77867 2.0002 0.0071111 3.3834 0.61156 4.1494 1.8133zm19.162-1.3013c0.33996 0.34952 0.52091 0.82426 0.50006 1.312v12.587c0.016644 0.48388-0.1639 0.95371-0.50006 1.3013-0.33374 0.34586-0.79692 0.53548-1.2767 0.52267-0.47366 0.019341-0.93346-0.16272-1.2661-0.50133-0.33079-0.34404-0.51738-0.80216-0.52134-1.28-0.55945 0.62249-1.2402 1.1235-2.0002 1.472-0.82894 0.40292-1.7387 0.61087-2.6599 0.608-1.3146 0.012451-2.6046-0.35811-3.7132-1.0667-1.1223-0.72405-2.03-1.7372-2.628-2.9333-0.6621-1.3228-0.99433-2.7869-0.9682-4.2667-0.02446-1.4772 0.30002-2.9392 0.94692-4.2667 0.58233-1.1997 1.4799-2.2175 2.596-2.944 1.0837-0.70256 2.3482-1.0732 3.6387-1.0667 0.91597-0.0053612 1.8238 0.17232 2.6705 0.52267 0.78219 0.31036 1.5028 0.75828 2.1279 1.3227-0.015414-0.48368 0.16495-0.95304 0.50006-1.3013 0.33298-0.34698 0.79668-0.53681 1.2767-0.52267 0.4762-0.015818 0.93792 0.16548 1.2767 0.50133zm-4.1281 11.285c1.622-2.0968 1.622-5.0286 0-7.1253-0.78999-0.90688-1.9324-1.4272-3.1333-1.4272s-2.3433 0.52035-3.1333 1.4272c-1.6108 2.1006-1.6108 5.0247 0 7.1253 0.77234 0.92514 1.9252 1.4441 3.128 1.408 1.2052 0.033032 2.3601-0.48504 3.1387-1.408zm20.428-10.731c1.1223 0.72405 2.03 1.7372 2.628 2.9333 0.6617 1.323 0.9939 2.7869 0.9682 4.2667 0.02446 1.4772-0.30002 2.9392-0.94692 4.2667-0.58142 1.2003-1.4791 2.2183-2.596 2.944-1.0832 0.7037-2.348 1.0745-3.6387 1.0667-0.92916 0.0031738-1.8478-0.19704-2.6918-0.58667-0.78816-0.3369-1.5091-0.81391-2.1279-1.408v5.7813c0.016644 0.48388-0.1639 0.95371-0.50006 1.3013-0.33298 0.34698-0.79668 0.53681-1.2767 0.52267-0.4762 0.015818-0.93792-0.16548-1.2767-0.50133-0.33996-0.34952-0.52091-0.82426-0.50006-1.312v-18.219c-0.013736-0.48022 0.16659-0.94565 0.50006-1.2907 0.71053-0.69371 1.843-0.69371 2.5535 0 0.33347 0.34502 0.51379 0.81044 0.50006 1.2907v0.074667c0.54716-0.65575 1.2248-1.1898 1.9896-1.568 0.82143-0.44095 1.7387-0.67178 2.6705-0.672 1.3297-0.0053611 2.6318 0.38033 3.7451 1.1093zm-1.064 10.763c1.6108-2.1006 1.6108-5.0247 0-7.1253-0.80616-0.88259-1.945-1.3852-3.1387-1.3852s-2.3325 0.50259-3.1387 1.3852c-1.622 2.0968-1.622 5.0286 0 7.1253 0.78811 0.9101 1.9312 1.4328 3.1333 1.4328 1.2022 0 2.3452-0.5227 3.1333-1.4328h0.01064zm-161.52-18.229 7.171 3.9467c0.36174 0.192 0.36174 0.512 0 0.71467l-15.747 8.6613c-1.5959 0.896-2.1279 2.1333-1.5747 3.2747 0 0.096 0 0.192-0.18087 0.256-0.19713 0.084184-0.41996 0.084184-0.61709 0l-9.8097-5.44c-0.36174-0.192-0.36174-0.512 0-0.71467l19.46-10.667c0.40542-0.20756 0.88292-0.21933 1.298-0.032zm-13.587 29.995-7.171-3.9467c-0.36174-0.192-0.36174-0.512 0-0.71467l15.747-8.6613c1.5959-0.896 2.1279-2.1333 1.5747-3.2747 0-0.096 0-0.192 0.18087-0.256 0.19713-0.084184 0.41996-0.084184 0.61709 0l9.8097 5.3973c0.36174 0.192 0.36174 0.512 0 0.71467l-19.46 10.667c-0.39888 0.22133-0.87651 0.24881-1.298 0.074667z"/>
				</svg>',
			'socialsnap-dark'   => '<svg version="1.1" viewBox="0 0 189 32" xmlns="http://www.w3.org/2000/svg"><path id="Shape" d="m43.451 20.565c-0.001507-0.25756 0.14867-0.49178 0.38302-0.59733 0.12665-0.12589 0.30074-0.19182 0.47878-0.18133 0.25177 0.0031524 0.49083 0.11139 0.65965 0.29867 1.0727 1.4949 2.8132 2.3614 4.6495 2.3147 0.8736 0.019678 1.7343-0.21364 2.479-0.672 0.69172-0.39914 1.1025-1.1528 1.064-1.952 0.028561-0.75545-0.34381-1.4696-0.97884-1.8773-0.85138-0.49427-1.7737-0.85405-2.7344-1.0667-1.3749-0.30082-2.6837-0.85023-3.8622-1.6213-0.93502-0.67394-1.4609-1.7808-1.3938-2.9333-0.010852-0.78295 0.22769-1.549 0.68093-2.1867 0.47072-0.65934 1.1103-1.179 1.8513-1.504 0.83767-0.36869 1.7452-0.55066 2.6599-0.53333 0.88026 0.0013989 1.7528 0.16407 2.5748 0.48 0.8583 0.32619 1.6198 0.86575 2.213 1.568 0.17969 0.14436 0.27876 0.36684 0.26599 0.59733-0.0012558 0.2437-0.11039 0.47425-0.29791 0.62933-0.14953 0.12311-0.33859 0.18755-0.53198 0.18133-0.20171-0.0027652-0.39367-0.087444-0.53198-0.23467-0.46523-0.52802-1.0436-0.94373-1.6917-1.216-0.6733-0.28363-1.3976-0.42524-2.1279-0.416-0.8447-0.026215-1.6781 0.2003-2.3939 0.65067-0.65613 0.42979-1.0334 1.1781-0.98948 1.9627 0.012045 0.74992 0.41601 1.4384 1.064 1.8133 0.94531 0.51767 1.9591 0.89835 3.011 1.1307 0.87652 0.20634 1.7315 0.49563 2.5535 0.864 0.65424 0.30097 1.2216 0.76339 1.6491 1.344 0.45461 0.6751 0.67861 1.4798 0.63837 2.2933 0.035254 1.2222-0.54436 2.3804-1.5427 3.0827-1.1183 0.78927-2.4635 1.1901-3.8302 1.1413-2.2455 0.081726-4.3925-0.92701-5.7666-2.7093-0.12202-0.19587-0.19165-0.41997-0.20215-0.65067z" fill="#2A2E38"/>
				<path id="Shape" d="m71.518 20.064c-0.63377 1.1862-1.5842 2.1723-2.745 2.848-2.3707 1.3697-5.2897 1.3697-7.6605 0-1.1448-0.68229-2.0798-1.6677-2.7024-2.848-1.3046-2.5402-1.3046-5.5558 0-8.096 0.62593-1.1807 1.56-2.1688 2.7024-2.8587 2.3654-1.3897 5.2951-1.3897 7.6605 0 1.1424 0.68991 2.0765 1.678 2.7024 2.8587 1.3082 2.5393 1.3082 5.5567 0 8.096h0.042558zm-1.5427-7.264c-0.46968-0.92541-1.1836-1.7039-2.0641-2.2507-0.87809-0.5459-1.8927-0.83071-2.9259-0.82133-1.0266-0.010048-2.0345 0.27497-2.9046 0.82133-0.88059 0.54986-1.5972 1.3273-2.0747 2.2507-0.51142 0.9878-0.77094 2.0872-0.75541 3.2-0.018214 1.1131 0.24146 2.2131 0.75541 3.2 0.4741 0.92232 1.1871 1.6997 2.0641 2.2507 0.87005 0.54636 1.878 0.83138 2.9046 0.82133 2.1156 0.018815 4.0567-1.1737 5.0006-3.072 0.97991-2.0202 0.97991-4.3798 0-6.4z" fill="#2A2E38"/>
				<path id="Shape" d="m85.307 8.3733c0.68974 0.16047 1.3431 0.44998 1.9258 0.85333 0.40514 0.252 0.66301 0.68553 0.69157 1.1627 1.611e-4 0.21967-0.079234 0.43193-0.22343 0.59733-0.13281 0.17057-0.33743 0.2692-0.55326 0.26667-0.17087 0.0075164-0.34042-0.033132-0.48942-0.11733-0.15438-0.10123-0.29711-0.21929-0.42558-0.352-0.2167-0.22694-0.46811-0.41778-0.74477-0.56533-0.37477-0.1612-0.76781-0.27583-1.1703-0.34133-0.51634-0.090499-1.0399-0.13334-1.564-0.128-1.055-0.011875-2.0917 0.27663-2.9897 0.832-0.89824 0.54945-1.6331 1.3304-2.1279 2.2613-0.52743 0.98259-0.79484 2.0843-0.77669 3.2-0.013619 1.1114 0.24195 2.2094 0.74477 3.2 0.96148 1.9174 2.9348 3.1119 5.0751 3.072 0.5729 0.014187 1.1452-0.046774 1.7023-0.18133 0.43847-0.12193 0.86286-0.28997 1.2661-0.50133 0.2503-0.16008 0.48855-0.33833 0.71285-0.53333 0.12924-0.11481 0.26807-0.2183 0.41494-0.30933 0.11589-0.069201 0.24814-0.10603 0.38302-0.10667 0.21115-0.0089267 0.41614 0.072504 0.5639 0.224 0.14518 0.15986 0.22164 0.37066 0.21279 0.58667-0.073561 0.44178-0.33329 0.83043-0.71285 1.0667-0.58829 0.44764-1.2511 0.78712-1.9577 1.0027-0.89386 0.28656-1.8279 0.42703-2.7663 0.416-1.3615 0.013107-2.6994-0.35641-3.8622-1.0667-1.1255-0.68292-2.0363-1.67-2.628-2.848-0.60502-1.2665-0.89696-2.6605-0.85116-4.064-0.017731-1.3844 0.30764-2.7515 0.94692-3.9787 1.278-2.47 3.8428-3.9996 6.6178-3.9467 0.8709-0.0098761 1.7396 0.090472 2.5854 0.29867z" fill="#2A2E38"/>
				<path id="Shape" d="m91.329 5.8773c-0.22344-0.21294-0.34377-0.51258-0.32983-0.82133v-0.29867c-0.020468-0.31202 0.10544-0.61576 0.34047-0.82133 0.27022-0.19921 0.60146-0.29732 0.93628-0.27733 0.79442 0 1.1916 0.35556 1.1916 1.0667v0.29867c0.019465 0.30548-0.092912 0.60462-0.30855 0.82133-0.2549 0.20605-0.57758 0.30881-0.90436 0.288-0.32932 0.030165-0.65833-0.060828-0.92564-0.256zm1.564 17.504c-0.25578 0.2618-0.64479 0.34105-0.98221 0.2001-0.33741-0.14095-0.5552-0.47368-0.54989-0.8401v-13.696c-0.03733-0.34148 0.12337-0.67429 0.41368-0.85676 0.29031-0.18247 0.65913-0.18247 0.94944 0 0.29031 0.18247 0.45101 0.51528 0.41368 0.85676v13.696c0.0071271 0.23961-0.08536 0.47142-0.25535 0.64h0.01064z" fill="#2A2E38"/>
				<path id="Shape" d="m110.72 8.6933c0.1701 0.17721 0.2621 0.41548 0.25535 0.66133v13.397c0.03733 0.34148-0.12337 0.67429-0.41368 0.85676-0.29031 0.18247-0.65913 0.18247-0.94944 0-0.29031-0.18247-0.45101-0.51528-0.41368-0.85676v-1.9947c-0.53392 0.93371-1.3 1.7128-2.2237 2.2613-0.9563 0.60284-2.0622 0.92438-3.1919 0.928-1.271 0.0091621-2.5157-0.36225-3.5749-1.0667-1.071-0.71134-1.931-1.6988-2.4897-2.8587-0.63045-1.2467-0.96198-2.6236-0.9682-4.0213-0.020983-1.4095 0.29649-2.8034 0.92564-4.064 0.56773-1.1489 1.4355-2.1221 2.5109-2.816 2.0636-1.296 4.6698-1.3533 6.788-0.14933 0.93493 0.51473 1.7058 1.2838 2.2237 2.2187v-1.8453c-0.006753-0.24585 0.085246-0.48412 0.25535-0.66133 0.1681-0.172 0.39818-0.26893 0.63837-0.26893 0.24019 0 0.47027 0.096928 0.63837 0.26893l-0.010639 0.010667zm-4.043 12.8c0.83487-0.56443 1.5006-1.3462 1.9258-2.2613 0.47198-0.99986 0.71205-2.0939 0.70221-3.2 0.017839-1.107-0.22263-2.2028-0.70221-3.2-0.42809-0.91036-1.0934-1.688-1.9258-2.2507-0.8224-0.54707-1.79-0.83326-2.7769-0.82133-0.98416-0.014353-1.9505 0.26404-2.7769 0.8-0.83481 0.55532-1.5011 1.3303-1.9258 2.24-0.47958 0.99724-0.72005 2.093-0.70221 3.2-0.011802 1.104 0.22076 2.1969 0.68093 3.2 0.41661 0.91739 1.0797 1.7004 1.9151 2.2613 1.7025 1.0923 3.8832 1.0923 5.5858 0v0.032z" fill="#2A2E38"/>
				<path id="Shape" d="m117.35 23.381c-0.25578 0.2618-0.64479 0.34105-0.9822 0.2001-0.33741-0.14095-0.5552-0.47368-0.54989-0.8401v-20.267c0-0.49485 0.40013-0.896 0.89372-0.896s0.89372 0.40115 0.89372 0.896v20.267c0.007127 0.23961-0.08536 0.47142-0.25535 0.64z" fill="#2A2E38"/>
				<path id="Shape" d="m120.91 20.267c0.059838-0.87041 0.7686-1.5533 1.6385-1.5787 0.45742 0.017924 0.88776 0.22209 1.1916 0.56533 0.52704 0.56656 1.1549 1.0296 1.8513 1.3653 0.73514 0.31682 1.5302 0.46969 2.3301 0.448 1.5463-0.056889 2.3194-0.64356 2.3194-1.76 0.015694-0.55397-0.2918-1.0664-0.78733-1.312-0.77774-0.39405-1.5991-0.69479-2.4471-0.896-0.95715-0.2506-1.8957-0.56782-2.8088-0.94933-0.73048-0.32226-1.3728-0.81619-1.8726-1.44-0.55761-0.74873-0.83616-1.6695-0.78733-2.6027-0.016872-0.89776 0.27211-1.7744 0.81924-2.4853 0.55881-0.71709 1.2913-1.2789 2.1279-1.632 0.88488-0.39087 1.8419-0.59075 2.8088-0.58667 1.0365 0.0036404 2.0638 0.19517 3.0323 0.56533 0.89451 0.32882 1.6845 0.89245 2.2875 1.632 0.24908 0.29919 0.38473 0.67697 0.38302 1.0667 0.004287 0.44586-0.20554 0.86659-0.5639 1.1307-0.26847 0.21585-0.60279 0.33259-0.94692 0.33067-0.39654 0.01344-0.78119-0.13696-1.064-0.416-0.37195-0.45864-0.84557-0.82388-1.3831-1.0667-0.60356-0.26407-1.2567-0.39502-1.9151-0.384-1.5037 0-2.2556 0.54756-2.2556 1.6427-0.018631 0.39843 0.13217 0.78608 0.41494 1.0667 0.30367 0.28633 0.66711 0.50131 1.064 0.62933 0.42558 0.14933 1.0249 0.33422 1.7981 0.55467 0.91113 0.24502 1.8036 0.55515 2.6705 0.928 0.71827 0.32299 1.3413 0.82633 1.8087 1.4613 0.52632 0.75814 0.78813 1.6693 0.74477 2.592 0.017787 0.92449-0.28703 1.8262-0.8618 2.5493-0.58219 0.73418-1.3448 1.3039-2.213 1.6533-0.87858 0.37285-1.8229 0.56509-2.7769 0.56533-1.2461 0.01148-2.4836-0.20915-3.6494-0.65067-1.1104-0.42456-2.0722-1.1663-2.7663-2.1333-0.15054-0.25766-0.21747-0.55586-0.19151-0.85333z" fill="#2A2E38"/>
				<path id="Shape" d="m149.05 9.216c0.85183 1.5197 1.2541 3.2513 1.1597 4.992v7.616c0.016644 0.48388-0.1639 0.95371-0.50006 1.3013-0.71054 0.69371-1.843 0.69371-2.5535 0-0.33616-0.34762-0.5167-0.81745-0.50006-1.3013v-7.6267c0.052032-0.90774-0.18654-1.8084-0.68093-2.5707-0.58128-0.68268-1.4611-1.0339-2.3513-0.93867-0.99889-0.058263-1.9772 0.30086-2.7024 0.992-0.6556 0.66981-1.0093 1.5794-0.97884 2.5173v7.616c0.01789 0.48409-0.16283 0.9544-0.50006 1.3013-0.71054 0.69371-1.843 0.69371-2.5535 0-0.33723-0.34694-0.51795-0.81724-0.50006-1.3013v-12.288c-0.013736-0.48022 0.16659-0.94565 0.50006-1.2907 0.71053-0.69371 1.843-0.69371 2.5535 0 0.33347 0.34502 0.51379 0.81044 0.50006 1.2907v0.50133c0.58039-0.75034 1.3037-1.3776 2.1279-1.8453 0.8539-0.51576 1.8332-0.78521 2.8301-0.77867 2.0002 0.0071111 3.3834 0.61156 4.1494 1.8133z" fill="#2A2E38"/>
				<path id="Shape" d="m168.21 7.9147c0.33996 0.34952 0.52091 0.82426 0.50006 1.312v12.587c0.016644 0.48388-0.1639 0.95371-0.50006 1.3013-0.33374 0.34586-0.79692 0.53548-1.2767 0.52267-0.47366 0.019341-0.93346-0.16272-1.2661-0.50133-0.33079-0.34404-0.51738-0.80216-0.52134-1.28-0.55945 0.62249-1.2402 1.1235-2.0002 1.472-0.82894 0.40292-1.7387 0.61087-2.6599 0.608-1.3146 0.012451-2.6046-0.35811-3.7132-1.0667-1.1223-0.72405-2.03-1.7372-2.628-2.9333-0.6621-1.3228-0.99433-2.7869-0.9682-4.2667-0.02446-1.4772 0.30002-2.9392 0.94692-4.2667 0.58233-1.1997 1.4799-2.2175 2.596-2.944 1.0837-0.70256 2.3482-1.0732 3.6387-1.0667 0.91597-0.0053612 1.8238 0.17232 2.6705 0.52267 0.78219 0.31036 1.5028 0.75828 2.1279 1.3227-0.015414-0.48368 0.16495-0.95304 0.50006-1.3013 0.33298-0.34698 0.79668-0.53681 1.2767-0.52267 0.4762-0.015818 0.93792 0.16548 1.2767 0.50133zm-4.1281 11.285c1.622-2.0968 1.622-5.0286 0-7.1253-0.78999-0.90688-1.9324-1.4272-3.1333-1.4272s-2.3433 0.52035-3.1333 1.4272c-1.6108 2.1006-1.6108 5.0247 0 7.1253 0.77234 0.92514 1.9252 1.4441 3.128 1.408 1.2052 0.033032 2.3601-0.48504 3.1387-1.408z" fill="#2A2E38"/>
				<path id="Shape" d="m184.51 8.4693c1.1223 0.72405 2.03 1.7372 2.628 2.9333 0.6617 1.323 0.9939 2.7869 0.9682 4.2667 0.02446 1.4772-0.30002 2.9392-0.94692 4.2667-0.58142 1.2003-1.4791 2.2183-2.596 2.944-1.0832 0.7037-2.348 1.0745-3.6387 1.0667-0.92916 0.0031738-1.8478-0.19704-2.6918-0.58667-0.78816-0.3369-1.5091-0.81391-2.1279-1.408v5.7813c0.016644 0.48388-0.1639 0.95371-0.50006 1.3013-0.33298 0.34698-0.79668 0.53681-1.2767 0.52267-0.4762 0.015818-0.93792-0.16548-1.2767-0.50133-0.33996-0.34952-0.52091-0.82426-0.50006-1.312v-18.219c-0.013736-0.48022 0.16659-0.94565 0.50006-1.2907 0.71053-0.69371 1.843-0.69371 2.5535 0 0.33347 0.34502 0.51379 0.81044 0.50006 1.2907v0.074667c0.54716-0.65575 1.2248-1.1898 1.9896-1.568 0.82143-0.44095 1.7387-0.67178 2.6705-0.672 1.3297-0.0053611 2.6318 0.38033 3.7451 1.1093zm-1.064 10.763c1.6108-2.1006 1.6108-5.0247 0-7.1253-0.80616-0.88259-1.945-1.3852-3.1387-1.3852s-2.3325 0.50259-3.1387 1.3852c-1.622 2.0968-1.622 5.0286 0 7.1253 0.78811 0.9101 1.9312 1.4328 3.1333 1.4328 1.2022 0 2.3452-0.5227 3.1333-1.4328h0.01064z" fill="#2A2E38"/>
				<path id="a" d="m21.928 1.0027 7.171 3.9467c0.36174 0.192 0.36174 0.512 0 0.71467l-15.747 8.6613c-1.5959 0.896-2.1279 2.1333-1.5747 3.2747 0 0.096 0 0.192-0.18087 0.256-0.19713 0.084184-0.41996 0.084184-0.61709 0l-9.8097-5.44c-0.36174-0.192-0.36174-0.512 0-0.71467l19.46-10.667c0.40542-0.20756 0.88292-0.21933 1.298-0.032z" fill="#557CEB"/>
				<path d="m8.3414 30.997-7.171-3.9467c-0.36174-0.192-0.36174-0.512 0-0.71467l15.747-8.6613c1.5959-0.896 2.1279-2.1333 1.5747-3.2747 0-0.096 0-0.192 0.18087-0.256 0.19713-0.084184 0.41996-0.084184 0.61709 0l9.8097 5.3973c0.36174 0.192 0.36174 0.512 0 0.71467l-19.46 10.667c-0.39888 0.22133-0.87651 0.24881-1.298 0.074667z" fill="#557CEB"/></svg>
				',
			'socialsnap-circle' => '<svg width="80" height="80" viewBox="0 0 80 80" xmlns="http://www.w3.org/2000/svg"><path d="M40 80C17.91 80 0 62.09 0 40S17.91 0 40 0s40 17.91 40 40-17.91 40-40 40zm8.854-59.587c-.54-.254-1.164-.24-1.69.038L21.797 34.426c-.475.268-.475.688 0 .936l12.787 7.045c.255.11.543.11.798 0 .247-.076.247-.21.247-.324-.703-1.47-.02-3.11 2.052-4.275l20.52-11.302c.475-.268.475-.688 0-.936l-9.348-5.154zM31.146 59.587c.54.254 1.164.24 1.69-.038l25.366-13.975c.475-.268.475-.688 0-.936l-12.787-7.045c-.255-.11-.543-.11-.798 0-.247.076-.247.21-.247.324.703 1.47.02 3.11-2.052 4.275l-20.52 11.302c-.475.268-.475.688 0 .936l9.348 5.154z" fill="#3A424A"/></svg>',
			'soundcloud'        => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M0.625 21.125h-0.625v-3.875h0.625zM2.25 22.375h-0.625v-6.063h0.625zM3.5 22.75h-0.625v-7.063h0.625zM4.813 23.063h-0.625v-7.063h0.625zM6.375 23.063h-0.625v-9.313h0.625zM7.688 23.063h-0.625v-10.563h0.625zM9.25 23.063h-0.625v-11.25h0.625zM10.563 23.063h-0.625v-11.563h0.625zM12.188 23.063h-0.688v-11.25h0.688zM13.438 23.063h-0.625v-10.875h0.625zM14.75 23.063h-0.688v-12.188h0.688zM16.313 23.063h-0.625v-13.125h0.625zM28.063 22.938c-0.063 0-10.938 0-10.938 0-0.25 0-0.5-0.188-0.5-0.438v-12.5c0-0.25 0.125-0.375 0.438-0.5 0.75-0.313 1.625-0.438 2.5-0.438 3.625 0 6.625 2.75 6.938 6.313 0.5-0.188 1-0.313 1.563-0.313 2.188 0 3.938 1.75 3.938 3.938s-1.75 3.938-3.938 3.938z"></path></svg>',
			'spotify'           => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M25.813 25.938c-0.375 0.625-1.25 0.875-1.875 0.438-5.313-3.188-11.938-3.938-19.75-2.125-0.75 0.125-1.5-0.313-1.688-1.063s0.313-1.5 1.063-1.688c8.563-1.938 15.875-1.125 21.813 2.5 0.688 0.375 0.875 1.25 0.438 1.938zM28.563 19.813c-0.5 0.813-1.563 1.063-2.375 0.563-6.063-3.688-15.25-4.75-22.438-2.625-0.875 0.313-1.875-0.25-2.188-1.125-0.25-0.938 0.25-1.938 1.188-2.188 8.188-2.5 18.313-1.313 25.25 2.938 0.813 0.563 1.125 1.625 0.563 2.438zM31.688 12.688c-0.563 1-1.875 1.313-2.875 0.75-7.25-4.313-19.188-4.688-26.125-2.625-1.063 0.375-2.25-0.25-2.625-1.375-0.313-1.125 0.313-2.313 1.438-2.625 7.938-2.438 21.125-1.938 29.5 3 1 0.625 1.313 1.875 0.688 2.875z"></path></svg>
			',
			'telegram'          => '<svg width="28" height="32" viewBox="0 0 28 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M27.919 6.162l-4.225 19.925c-0.319 1.406-1.15 1.756-2.331 1.094l-6.438-4.744-3.106 2.988c-0.344 0.344-0.631 0.631-1.294 0.631l0.463-6.556 11.931-10.781c0.519-0.463-0.113-0.719-0.806-0.256l-14.75 9.287-6.35-1.988c-1.381-0.431-1.406-1.381 0.288-2.044l24.837-9.569c1.15-0.431 2.156 0.256 1.781 2.013z"></path></svg>',
			'tumblr'            => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M6.125 13.125v-4.563c1.25-0.375 2.375-1 3.25-1.75s1.625-1.688 2.125-2.75c0.563-1.063 0.938-2.438 1.125-4.063h4.563v8.063h7.625v5.063h-7.625v8.188c0 1.875 0.125 3.063 0.313 3.563 0.188 0.563 0.563 0.938 1.063 1.25 0.75 0.438 1.563 0.688 2.438 0.688 1.625 0 3.25-0.563 4.875-1.625v5.063c-1.375 0.625-2.625 1.125-3.75 1.375-1.063 0.25-2.313 0.375-3.563 0.375-1.5 0-2.75-0.188-3.875-0.563s-2.125-0.875-2.875-1.563c-0.813-0.688-1.375-1.438-1.688-2.188s-0.438-1.875-0.438-3.375v-11.188z"></path></svg>',
			'twitch'            => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M22.938 27.813h-6.25l-4.188 4.188h-4.125v-4.188h-7.688v-22.25l2.125-5.563h28.5v19.5zM28.5 18.063v-15.313h-22.938v20.188h6.25v4.188l4.188-4.188h7.625zM20.875 8.375v8.313h2.75v-8.313zM13.188 16.688h2.813v-8.313h-2.813z"></path></svg>',
			'twitter-outline'   => '<svg width="32" height="32" viewBox="0 0 24 24"><g><path d="M18.244 2.25h3.308l-7.227 8.26 8.502 11.24H16.17l-5.214-6.817L4.99 21.75H1.68l7.73-8.835L1.254 2.25H8.08l4.713 6.231zm-1.161 17.52h1.833L7.084 4.126H5.117z"></path></g></svg>',
			'twitter'           => '<svg width="32" height="32" viewBox="0 0 24 24"><g><path d="M18.244 2.25h3.308l-7.227 8.26 8.502 11.24H16.17l-5.214-6.817L4.99 21.75H1.68l7.73-8.835L1.254 2.25H8.08l4.713 6.231zm-1.161 17.52h1.833L7.084 4.126H5.117z"></path></g></svg>',
			'viber'             => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M18.438 8.063c-0.375-0.063-0.75 0.25-0.75 0.688-0.063 0.375 0.25 0.75 0.688 0.813 2.5 0.125 3.688 1.375 3.813 4 0.063 0.375 0.375 0.688 0.813 0.688 0 0 0 0 0 0 0.438 0 0.75-0.375 0.75-0.75-0.188-3.438-2-5.25-5.313-5.438zM26.875 15.563c0.375 0 0.75-0.313 0.75-0.688 0.063-3-0.813-5.5-2.688-7.5-1.813-2-4.375-3.125-7.625-3.438-0.438 0-0.75 0.313-0.813 0.75 0 0.375 0.313 0.75 0.688 0.813 2.938 0.188 5.063 1.125 6.625 2.875 1.625 1.688 2.375 3.813 2.313 6.438 0 0.375 0.313 0.75 0.75 0.75zM27.063 4.438c-2.938-2.875-6.688-4.375-11.125-4.438-0.438 0-0.75 0.375-0.75 0.75 0 0.438 0.313 0.813 0.75 0.813 4 0 7.438 1.375 10.125 4s4.063 6.188 4.063 10.875c0 0.375 0.375 0.75 0.813 0.75 0.375 0 0.75-0.375 0.75-0.75-0.125-5.063-1.625-9.125-4.625-12zM29.875 25.063c0 0-1.75-1.438-2.625-2.063-0.813-0.625-2.063-1.375-2.75-1.813-1.313-0.688-2.625-0.25-3.125 0.5l-1.125 1.375c-0.625 0.688-1.688 0.625-1.688 0.625-7.813-2-9.875-9.938-9.875-9.938s-0.125-1.063 0.563-1.625l1.438-1.125c0.688-0.563 1.125-1.875 0.438-3.125-0.375-0.688-1.188-2-1.75-2.813-0.688-0.813-2.125-2.563-2.125-2.563-0.688-0.813-1.688-1-2.813-0.438 0 0 0 0 0 0-1.063 0.563-2 1.313-2.875 2.313 0 0 0 0 0 0.063-0.688 0.75-1.063 1.563-1.188 2.375 0 0.063 0 0.188 0 0.313 0 0.375 0 0.688 0.125 1.063h0.063c0.313 1.188 1.188 3.188 3 6.563 1.188 2.125 2.375 3.938 3.625 5.438 0.625 0.813 1.313 1.625 2.188 2.438 0 0.063 0.063 0.125 0.063 0.125 0.063 0.063 0.063 0.063 0.125 0.125 0 0 0.063 0.063 0.063 0.063 0.063 0.063 0.063 0.063 0.125 0.125 0.813 0.813 1.625 1.5 2.438 2.125 1.5 1.25 3.313 2.438 5.5 3.625 3.313 1.875 5.313 2.688 6.5 3l0.063 0.063c0.313 0.063 0.625 0.188 1 0.188 0.125 0 0.25 0 0.375-0.063 0.75-0.125 1.563-0.5 2.313-1.125 0 0 0 0 0.063-0.063 1-0.875 1.75-1.875 2.313-2.875 0 0 0 0 0-0.063 0.563-1.063 0.375-2.063-0.438-2.813z"></path></svg>',
			'vimeo'             => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M20.688 9.125c-0.75 0-1.563 0.188-2.375 0.563 1.563-5.188 4.563-7.688 9-7.5 3.313 0.063 4.875 2.188 4.688 6.375-0.188 3.125-2.313 7.375-6.563 12.813-4.313 5.625-8 8.5-11 8.5-1.875 0-3.438-1.75-4.75-5.188-0.875-3.188-1.75-6.313-2.563-9.5-1-3.438-2-5.125-3.125-5.125-0.25 0-1.063 0.5-2.5 1.5l-1.5-1.938c1.563-1.438 3.125-2.813 4.688-4.188 2.063-1.813 3.688-2.75 4.75-2.875 2.438-0.25 4 1.438 4.563 5.125 0.625 3.938 1.063 6.375 1.313 7.313 0.688 3.25 1.5 4.875 2.375 4.875 0.688 0 1.688-1.063 3-3.125 1.313-2.125 2.063-3.75 2.188-4.875 0.188-1.813-0.563-2.75-2.188-2.75z"></path></svg>',
			'vkontakte'         => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M30.938 22.063c-0.938-1.25-1.75-2-2.938-3.125-0.5-0.5-1.188-1-1.25-1.688-0.063-0.938 0.813-1.813 1.25-2.438 0.063-0.063 0.125-0.125 0.188-0.188s0.063-0.063 0.125-0.125c0-0.063 0.063-0.063 0.063-0.125 0.063-0.063 0.063-0.125 0.125-0.125 0-0.063 0.063-0.063 0.063-0.063 0.063-0.063 0.063-0.125 0.125-0.188 0.813-1.063 1.625-2 2.438-3.25 0.25-0.438 1.063-2 0.813-2.625-0.375-0.75-2.313-0.5-3.375-0.5-1.313 0-2.75-0.25-3.688 0.125-0.563 0.188-0.938 1.438-1.25 2.125-0.5 1.063-0.938 1.938-1.438 2.75 0 0.063-0.063 0.125-0.125 0.188 0 0 0 0 0 0 0 0.063-0.063 0.125-0.125 0.188 0 0 0 0 0 0-0.063 0.125-0.125 0.25-0.188 0.313 0 0.063-0.063 0.063-0.063 0.063 0 0.063-0.063 0.063-0.063 0.125-0.063 0.063-0.063 0.063-0.063 0.125-0.063 0-0.063 0.063-0.125 0.125 0 0-0.063 0.063-0.063 0.125-0.063 0-0.063 0.063-0.063 0.063-0.063 0.063-0.125 0.125-0.188 0.25-0.438 0.625-0.875 1.375-1.688 1.5-0.063-0.063-0.063-0.063-0.125-0.063 0 0 0-0.063 0-0.063-0.063 0-0.063 0-0.063 0 0-0.063-0.063-0.063-0.063-0.063s0 0-0.063-0.063c0 0 0 0 0 0s0-0.063-0.063-0.063c0 0 0 0 0 0 0-0.063-0.063-0.063-0.063-0.063s0-0.063 0-0.063c0 0-0.063 0-0.063-0.063 0 0 0 0 0 0 0-0.063 0-0.063-0.063-0.063 0 0 0-0.063 0-0.063s0 0 0-0.063c0 0-0.063 0-0.063 0 0-0.063 0-0.063 0-0.125 0 0 0 0 0 0-0.063-0.063-0.063-0.063-0.063-0.063 0-0.063 0-0.063 0-0.063s0-0.063 0-0.063c0 0 0-0.063-0.063-0.063 0 0 0-0.063 0-0.063s0 0 0-0.063c0 0 0 0 0-0.063 0 0 0 0 0 0 0-0.063 0-0.063-0.063-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.125 0 0 0 0 0 0 0-0.063 0-0.063 0-0.125 0 0 0 0 0 0 0-0.063 0-0.063 0-0.125 0 0-0.063 0-0.063-0.063 0 0 0-0.063 0-0.063s0-0.063 0-0.063c0 0 0-0.063 0-0.063s0-0.063 0-0.063c0 0 0-0.063 0-0.063 0-0.063 0.063-0.063 0.063-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.063 0-0.125 0 0 0 0 0 0 0-0.063 0-0.063 0-0.125 0 0 0 0 0 0 0-0.188 0-0.313 0-0.5 0 0 0-0.063 0-0.063 0-0.125 0-0.313 0.063-0.438 0-1.375 0.438-3.188-0.188-4.25-0.563-0.938-2.688-0.813-4.188-0.813-1.813 0-3.25 0.125-3.813 1.188 0.625 0.438 1.313 0.438 1.625 1.063 0.375 0.688 0.438 2.125 0.375 3.438 0 0.063 0 0.125 0 0.188 0 1.125-0.063 2.938-0.813 3.063s-1.375-1-1.813-1.563c-0.063-0.125-0.125-0.188-0.188-0.313 0 0-0.063-0.063-0.063-0.063-0.063-0.063-0.063-0.125-0.125-0.25 0 0-0.063-0.063-0.063-0.063-0.063-0.063-0.125-0.125-0.125-0.188-0.063-0.063-0.063-0.063-0.063-0.125-0.063-0.063-0.125-0.188-0.188-0.25-0.563-0.938-1.125-1.875-1.625-3.063-0.375-0.688-0.813-1.875-1.188-2.063-0.75-0.375-2.25-0.188-3.25-0.188-1.125 0-2.313-0.188-3 0.375 0 1.25 0.625 2.313 1.125 3.313 0.688 1.375 1.375 2.688 2.125 3.938 0 0.063 0.063 0.125 0.063 0.188 0.125 0.125 0.188 0.313 0.313 0.438 0 0.063 0.063 0.125 0.063 0.188 0.125 0.125 0.188 0.313 0.313 0.438 0 0.063 0.063 0.125 0.125 0.188 0.125 0.188 0.25 0.375 0.375 0.563 0 0 0 0 0 0 0.125 0.188 0.25 0.438 0.375 0.625 0.063 0.063 0.063 0.125 0.125 0.125 0.125 0.188 0.188 0.313 0.313 0.438 0 0.063 0.063 0.125 0.125 0.188 0.063 0.125 0.188 0.313 0.313 0.438 0 0.063 0.063 0.125 0.125 0.188 0.125 0.188 0.25 0.375 0.438 0.563 1.938 2.438 4.125 4.75 8.438 5.25 1.375 0.125 3.375 0.25 3.813-0.563 0.625-1-0.063-3.375 1.313-3.563 0.938-0.188 1.75 1.25 2.313 1.813 0.938 1.063 2.063 2.188 3.688 2.5 0 0 0.063 0 0.125 0 1.438-0.625 5.75 0.688 6-1.125 0.063-0.625-0.688-1.625-1.063-2.063z"></path></svg>',
			'whatsapp'          => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M27.313 4.625c-3-3-7-4.625-11.25-4.625-8.75 0-15.875 7.125-15.875 15.875 0 2.75 0.75 5.5 2.125 7.938l-2.25 8.188 8.438-2.188c2.313 1.25 4.875 1.938 7.563 1.938v0c0 0 0 0 0 0 8.75 0 15.875-7.125 15.875-15.875 0-4.25-1.625-8.25-4.625-11.25zM16.063 29.063v0c-2.375 0-4.688-0.625-6.688-1.875l-0.5-0.25-5 1.313 1.313-4.875-0.313-0.5c-1.313-2.125-2-4.5-2-7 0-7.313 5.938-13.188 13.188-13.188 3.5 0 6.813 1.375 9.313 3.875s3.875 5.813 3.875 9.313c0 7.25-5.938 13.188-13.188 13.188zM23.313 19.188c-0.438-0.188-2.375-1.188-2.75-1.313-0.313-0.125-0.625-0.188-0.875 0.188-0.25 0.438-1 1.313-1.25 1.563s-0.438 0.313-0.875 0.125c-0.375-0.188-1.625-0.625-3.188-2-1.125-1-1.938-2.313-2.188-2.75-0.25-0.375 0-0.563 0.188-0.813 0.188-0.125 0.375-0.438 0.563-0.688 0.25-0.188 0.313-0.375 0.438-0.625s0.063-0.5-0.063-0.688c-0.063-0.188-0.875-2.188-1.188-2.938-0.375-0.813-0.688-0.688-0.938-0.688-0.188 0-0.5 0-0.75 0s-0.688 0.063-1.063 0.438c-0.375 0.438-1.375 1.375-1.375 3.313 0 2 1.438 3.875 1.625 4.125s2.813 4.25 6.75 6c0.938 0.375 1.688 0.625 2.25 0.813 1 0.313 1.813 0.25 2.5 0.188 0.813-0.125 2.375-1 2.688-1.938 0.375-0.875 0.375-1.688 0.25-1.875-0.125-0.125-0.375-0.25-0.75-0.438z"></path></svg>',
			'xing'              => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M7.688 22.438h-4.625c-0.25 0-0.438-0.125-0.563-0.313s-0.125-0.5 0-0.75l4.875-8.625c0 0 0 0 0-0.063l-3.125-5.375c-0.125-0.25-0.125-0.5 0-0.688s0.313-0.313 0.625-0.313h4.625c0.688 0 1.063 0.438 1.25 0.875 0 0 3.188 5.5 3.188 5.5-0.188 0.375-5 8.813-5 8.813-0.25 0.438-0.563 0.938-1.25 0.938zM29.5 1l-10.188 18.063c0 0 0 0.063 0 0.063l6.5 11.875c0.125 0.25 0.125 0.5 0 0.688s-0.313 0.313-0.625 0.313h-4.625c-0.688 0-1-0.5-1.25-0.875 0 0-6.563-12-6.563-12.063 0.313-0.563 10.25-18.188 10.25-18.188 0.25-0.438 0.563-0.875 1.25-0.875h4.688c0.25 0 0.5 0.125 0.563 0.313 0.125 0.188 0.125 0.438 0 0.688z"></path></svg>',
			'yahoo'             => '<svg width="27" height="32" viewBox="0 0 27 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M15.339 17.375l0.232 12.625c-0.607-0.107-1.232-0.196-1.875-0.196-0.625 0-1.25 0.089-1.875 0.196l0.232-12.625c-3.321-5.732-6.446-11.571-10.107-17.089 0.625 0.161 1.268 0.268 1.929 0.268s1.339-0.125 1.982-0.268c2.5 4.429 5.214 8.732 7.839 13.089 2.643-4.321 5.446-8.625 7.839-13.089 0.625 0.161 1.268 0.25 1.911 0.25 0.679 0 1.375-0.089 2.036-0.25v0 0c-1.429 1.964-2.607 4.107-3.839 6.196-2.125 3.625-4.214 7.25-6.304 10.893z"></path></svg>',
			'youtube'           => '<svg width="32" height="32" viewBox="0 0 32 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M31.688 9.625c0 0-0.313-2.25-1.25-3.188-1.25-1.313-2.625-1.313-3.25-1.375-4.438-0.313-11.188-0.313-11.188-0.313v0c0 0-6.75 0-11.188 0.313-0.625 0.063-2 0.063-3.25 1.375-0.938 0.938-1.25 3.188-1.25 3.188s-0.313 2.563-0.313 5.125v2.438c0 2.625 0.313 5.188 0.313 5.188s0.313 2.188 1.25 3.188c1.25 1.25 2.875 1.25 3.563 1.375 2.563 0.25 10.875 0.313 10.875 0.313s6.75 0 11.188-0.313c0.625-0.063 2-0.125 3.25-1.375 0.938-1 1.25-3.188 1.25-3.188s0.313-2.563 0.313-5.188v-2.438c0-2.563-0.313-5.125-0.313-5.125zM20.313 16.25v0l-7.563 3.938v-9l8.625 4.5z"></path></svg>',
			'yummly'            => '<svg width="49" height="32" viewBox="0 0 49 32" version="1.1" xmlns="http://www.w3.org/2000/svg"><path d="M4.576 0.128c-0.032 0.032-0.16 0.064-0.32 0.064-0.576 0.096-1.216 0.288-1.376 0.384-0.064 0.064-0.128 0.064-0.16 0.032s-0.032 0-0.032 0.064c0 0.064-0.032 0.064-0.064 0.064-0.096-0.096-1.504 0.672-1.888 1.024-0.16 0.16-0.32 0.288-0.32 0.256s-0.096 0.096-0.192 0.288c-0.16 0.256-0.192 0.416-0.192 0.672 0.032 0.224 0.096 0.448 0.16 0.512 0.032 0.064 0.064 0.096 0.032 0.096-0.096 0 0.064 0.288 0.384 0.768 0.16 0.224 0.32 0.416 0.384 0.416 0.032 0 0.096 0.032 0.064 0.064-0.032 0.128 0.384 0.416 0.576 0.416 0.128 0 0.32-0.128 0.544-0.32 0.16-0.192 0.352-0.32 0.352-0.32 0.032 0 0.192-0.096 0.384-0.224 0.704-0.48 1.6-0.608 1.856-0.288 0.064 0.064 0.16 0.128 0.192 0.096 0.064-0.032 0.064-0.032 0.032 0.032s-0.032 0.192 0 0.288c0.032 0.096 0.032 0.352 0 0.576-0.032 0.192-0.096 0.48-0.096 0.576-0.032 0.128-0.096 0.384-0.16 0.544-0.032 0.192-0.096 0.384-0.096 0.448-0.032 0.032-0.096 0.288-0.16 0.544-0.096 0.256-0.16 0.576-0.192 0.704s-0.064 0.256-0.064 0.288c-0.064 0.064-0.128 0.352-0.16 0.672-0.032 0.064-0.064 0.16-0.064 0.256-0.064 0.16-0.256 0.832-0.288 0.992 0 0.064-0.032 0.192-0.064 0.256-0.064 0.128-0.224 0.8-0.256 0.992-0.032 0.064-0.16 0.576-0.288 1.088-0.16 0.512-0.288 1.024-0.288 1.088-0.032 0.096-0.096 0.288-0.128 0.448-0.064 0.192-0.064 0.352-0.064 0.384 0.032 0.032 0.032 0.096-0.032 0.096-0.096 0.032-0.224 0.96-0.192 1.632 0 0.544 0.32 1.696 0.448 1.696 0.032 0 0.096 0.064 0.128 0.16 0.064 0.224 0.64 0.768 1.024 0.992 0.96 0.544 2.752 0.608 4.512 0.224 0.736-0.16 1.888-0.544 1.92-0.64 0-0.032 0.064-0.064 0.096-0.032 0.064 0.064 1.088-0.448 1.312-0.64 0.064-0.032 0.128-0.064 0.16-0.032 0.032 0-0.032 0.416-0.128 0.864-0.096 0.48-0.16 0.896-0.16 0.928 0 0.064-0.032 0.224-0.16 0.672 0 0.096-0.096 0.16-0.16 0.16-0.576 0-2.688 0.512-2.688 0.608 0 0.064-0.032 0.064-0.096 0.032s-1.184 0.384-1.28 0.512c-0.032 0.032-0.064 0.064-0.064 0s-0.672 0.288-0.768 0.384c0 0.032-0.064 0.096-0.096 0.096-0.16 0-0.704 0.352-0.672 0.416s0.032 0.064-0.032 0.032c-0.16-0.096-1.856 1.664-1.696 1.824 0.032 0.032 0 0.032-0.064 0.032-0.128 0-0.832 1.472-0.96 2.112-0.032 0.096-0.128 0.992-0.16 1.216-0.032 0.544 0.128 1.344 0.384 1.92 0.864 1.824 3.040 2.688 5.44 2.176 0.512-0.128 1.568-0.512 1.696-0.64 0.064-0.064 0.16-0.128 0.16-0.096 0.064 0.032 0.192 0 0.32-0.128 0.096-0.096 0.256-0.224 0.352-0.288 0.192-0.128 0.736-0.64 1.088-1.024 0.32-0.352 0.768-1.024 0.736-1.088-0.032-0.032-0.032-0.064 0.032-0.064s0.352-0.512 0.352-0.608c0-0.032 0.032-0.096 0.064-0.128 0.192-0.128 0.832-1.888 1.088-3.008 0.16-0.64 0.32-1.184 0.352-1.248 0.064-0.16 0.384-0.16 1.984 0.032 0.992 0.128 1.152 0.128 1.344 0.16 0.128 0.032 0.544 0.096 0.96 0.192 1.12 0.16 0.992 0.16 2.88 0.544 0.736 0.128 1.376 0.256 1.472 0.288 0.064 0 0.192 0.032 0.288 0.064 0.064 0.032 0.224 0.064 0.352 0.064 0.192 0.032 0.832 0.16 0.896 0.192 0.032 0 0.16 0.032 0.256 0.064 0.224 0.064 2.432 0.512 2.752 0.608 0.128 0 0.512 0.096 0.864 0.16 0.8 0.16 1.12 0.224 1.376 0.256 0.096 0 0.192 0.032 0.192 0.032 0 0.032 0.064 0.064 0.096 0.064 0.128 0 0.768 0.096 1.024 0.16 0.704 0.128 2.144 0.352 2.528 0.416 0.288 0.032 0.512 0.064 0.544 0.096 0.032 0 0.32 0.032 0.672 0.064 0.32 0.032 0.672 0.064 0.768 0.096 0.704 0.128 5.152 0.224 5.984 0.096 0.224-0.032 0.608-0.064 0.896-0.096s0.544-0.128 0.576-0.16c0.032-0.032 0.064-0.032 0.096-0.032 0.256 0.16 3.616-0.992 3.616-1.216 0-0.032 0.032-0.064 0.064-0.064 0.096 0.032 0.576-0.224 1.024-0.544 0.512-0.384 0.768-0.896 0.768-1.504 0-0.544-0.288-1.248-0.672-1.76-0.288-0.416-0.608-0.736-0.608-0.64 0 0.032-0.032 0-0.096-0.032-0.064-0.064-0.352 0.032-1.056 0.384-0.544 0.256-1.12 0.512-1.28 0.576s-0.416 0.16-0.544 0.224c-0.32 0.16-2.336 0.672-2.816 0.736-0.192 0.032-0.448 0.064-0.544 0.096-0.128 0-0.448 0.032-0.768 0.064-0.288 0.064-0.672 0.096-0.832 0.096-0.448 0.064-3.904 0.064-4.512 0-0.288 0-0.768-0.064-1.056-0.096-0.416-0.032-0.768-0.064-1.664-0.16-0.128 0-0.384-0.032-0.576-0.064-1.152-0.16-1.792-0.256-1.952-0.256-0.128-0.032-0.224-0.032-0.224-0.032 0-0.032-0.16-0.064-0.704-0.128-0.224-0.032-0.48-0.064-0.576-0.096-0.128-0.032-0.448-0.096-0.768-0.128-0.288-0.064-0.64-0.128-0.736-0.128-0.128-0.032-0.32-0.064-0.416-0.064-0.256-0.064-2.048-0.384-2.368-0.448-0.192-0.032-0.736-0.128-1.024-0.16-0.096 0-0.16-0.032-0.16-0.032 0-0.032-0.224-0.064-0.672-0.128-0.192 0-0.416-0.064-0.512-0.064-0.096-0.032-0.32-0.064-0.544-0.096-0.192-0.032-0.448-0.096-0.544-0.096-0.096-0.032-0.32-0.064-0.512-0.064-0.16-0.032-0.384-0.064-0.48-0.064-0.096-0.032-0.416-0.096-0.672-0.128-0.288-0.032-0.576-0.064-0.672-0.064-0.096-0.032-0.384-0.064-0.64-0.096-0.224-0.032-0.544-0.064-0.704-0.064-0.48-0.064-1.568-0.16-1.76-0.16-0.16 0.032-0.16 0-0.064-0.512 0.064-0.288 0.128-0.672 0.16-0.896 0.128-0.768 0.256-1.6 0.32-1.92 0.032-0.096 0.096-0.352 0.096-0.48 0.032-0.16 0.096-0.512 0.128-0.736 0.064-0.224 0.096-0.544 0.128-0.704 0-0.128 0.064-0.32 0.064-0.416 0.032-0.064 0.064-0.288 0.096-0.512 0.032-0.192 0.064-0.416 0.096-0.48 0-0.064 0.032-0.256 0.064-0.416s0.064-0.384 0.096-0.512c0.032-0.192 0.128-0.704 0.16-1.056 0.032-0.128 0.064-0.352 0.16-0.832 0.032-0.096 0.064-0.32 0.096-0.544 0.032-0.192 0.064-0.448 0.096-0.544 0.032-0.224 0.096-0.544 0.16-0.896 0.032-0.128 0.096-0.576 0.16-0.928 0.064-0.384 0.128-0.8 0.16-0.96s0.096-0.576 0.16-0.896c0.096-0.32 0.16-0.768 0.16-0.96 0.032-0.224 0.064-0.416 0.096-0.48 0.032-0.032 0.064-0.224 0.096-0.448 0-0.192 0.064-0.416 0.064-0.48 0.032-0.064 0.064-0.288 0.096-0.48 0-0.192 0.032-0.384 0.064-0.416 0.096-0.128 0.16-1.152 0.096-1.536-0.032-0.224-0.128-0.48-0.192-0.544-0.096-0.096-0.16-0.192-0.128-0.224 0.032 0-0.128-0.128-0.32-0.224-0.672-0.352-2.176-0.288-2.912 0.128-0.256 0.16-0.288 0.192-0.256 0.512 0.064 0.832 0.032 1.728-0.096 2.496-0.096 0.448-0.16 0.896-0.192 0.96 0 0.064-0.032 0.288-0.064 0.448-0.032 0.192-0.096 0.416-0.096 0.512-0.032 0.096-0.064 0.256-0.064 0.384-0.032 0.096-0.064 0.32-0.096 0.48s-0.128 0.8-0.256 1.376c-0.096 0.608-0.224 1.248-0.224 1.408-0.032 0.16-0.096 0.48-0.16 0.736-0.032 0.256-0.096 0.544-0.128 0.672 0 0.128-0.032 0.352-0.064 0.512s-0.16 0.768-0.256 1.376c-0.128 0.608-0.192 1.152-0.192 1.184 0 0.096-0.768 0.544-1.312 0.736-0.192 0.064-0.384 0.16-0.384 0.224s0 0.064-0.032 0.032-0.352 0.064-0.736 0.192c-1.632 0.544-3.008 0.608-3.488 0.128-0.192-0.192-0.192-0.256-0.192-0.928 0-0.736 0.032-0.896 0.992-4.448 0.064-0.256 0.096-0.512 0.064-0.512 0-0.032 0.032-0.096 0.064-0.096 0.032-0.032 0.096-0.16 0.128-0.288 0-0.128 0.032-0.256 0.032-0.256 0.032 0 0.064-0.128 0.128-0.48 0.096-0.48 0.096-0.512 0.128-0.544 0 0 0.032-0.032 0.032-0.064s0.064-0.224 0.128-0.448c0.064-0.192 0.128-0.448 0.128-0.544s0.032-0.224 0.064-0.288c0.48-1.344 0.48-3.2 0-4.256-0.352-0.8-1.216-1.408-2.208-1.6-0.416-0.064-1.696-0.096-1.76-0.032zM10.944 23.968c-0.032 0.192-0.128 0.416-0.16 0.544s-0.064 0.256-0.064 0.288c0 0.064 0 0.096-0.352 0.96-0.128 0.288-0.224 0.576-0.192 0.576 0.032 0.032 0 0.064-0.064 0.064s-0.224 0.32-0.192 0.448c0 0.032 0 0.032-0.032 0.032-0.064-0.032-0.16 0.064-0.224 0.224-0.192 0.288-0.608 0.704-0.832 0.768-0.064 0.032-0.128 0.064-0.128 0.096 0 0.096-0.064 0.096-0.704 0.128-0.544 0.032-0.736-0.064-0.96-0.512-0.16-0.224-0.128-0.992 0.032-1.408 0.064-0.16 0.128-0.352 0.128-0.416s0.032-0.096 0.064-0.064c0.064 0.032 0.096-0.032 0.128-0.128 0.064-0.192 0.832-0.928 0.992-0.928 0.032 0 0.032-0.032 0-0.064-0.032-0.064 0-0.096 0.032-0.096 0.096 0.064 0.704-0.256 0.8-0.384 0.032-0.032 0.064-0.032 0.064 0s0.064 0 0.16-0.032c0.16-0.096 1.28-0.416 1.472-0.416 0.064 0 0.064 0.096 0.032 0.32zM38.848 5.952c-0.16 0.032-0.384 0.064-0.512 0.096-0.288 0.064-1.44 0.48-1.504 0.576-0.032 0.064-0.064 0.064-0.064 0.032s-0.16 0.032-0.352 0.096c-0.32 0.192-0.576 0.224-0.544 0.128 0-0.128-0.128-0.352-0.256-0.352-0.064 0-0.096-0.032-0.096-0.064 0-0.16-0.608-0.352-1.312-0.384-0.48-0.064-2.048 0.384-1.92 0.512 0.032 0 0 0.096-0.032 0.128-0.064 0.096-0.064 0.224 0 0.448 0.096 0.416 0.096 1.248 0 1.632-0.064 0.16-0.096 0.416-0.128 0.64s-0.032 0.384-0.032 0.384c-0.032 0-0.064 0.16-0.128 0.576-0.032 0.16-0.064 0.384-0.096 0.48-0.032 0.128-0.064 0.32-0.064 0.48-0.032 0.16-0.064 0.384-0.096 0.512-0.064 0.224-0.448 2.496-0.512 2.816-0.032 0.256-0.096 0.576-0.192 1.184-0.064 0.256-0.128 0.608-0.128 0.768-0.032 0.128-0.064 0.384-0.096 0.48-0.288 1.6 0.032 2.368 1.056 2.592 0.48 0.128 1.728 0.032 1.888-0.096 0.032-0.032 0.128-0.064 0.224-0.064 0.064 0 0.128-0.032 0.128-0.096 0-0.032 0.032-0.064 0.064-0.064 0.352 0.064 0.448-0.224 0.32-0.8-0.128-0.544-0.096-0.896 0.096-1.856 0.064-0.416 0.16-0.864 0.16-0.992 0.032-0.128 0.064-0.352 0.096-0.448 0.096-0.544 0.128-0.64 0.16-0.896 0.032-0.128 0.064-0.352 0.064-0.48 0.032-0.16 0.064-0.384 0.096-0.512 0.224-1.152 0.288-1.568 0.256-1.632-0.032-0.032-0.032-0.096 0.032-0.128 0.032 0 0.064-0.128 0.064-0.256s0.064-0.48 0.128-0.736l0.096-0.512 0.576-0.256c0.64-0.32 0.992-0.416 1.44-0.32 0.544 0.096 0.704 0.704 0.512 1.824-0.032 0.192-0.096 0.416-0.096 0.512-0.032 0.096-0.064 0.256-0.064 0.384-0.032 0.096-0.064 0.32-0.096 0.48s-0.096 0.608-0.16 1.024c-0.064 0.384-0.16 0.832-0.16 1.024-0.032 0.16-0.064 0.32-0.096 0.384-0.032 0.032-0.064 0.224-0.096 0.48 0 0.224-0.064 0.48-0.096 0.608-0.224 1.152-0.288 2.112-0.16 2.56 0.128 0.32 0.608 0.8 0.736 0.768 0.064-0.032 0.192 0 0.32 0.064 0.256 0.16 1.376 0.096 1.984-0.096 0.576-0.16 0.8-0.384 0.736-0.672-0.032-0.128-0.096-0.32-0.096-0.448-0.032-0.128-0.064-0.256-0.064-0.288-0.032-0.064 0.032-0.512 0.128-1.024 0.064-0.512 0.16-1.056 0.192-1.216s0.096-0.448 0.128-0.704c0.096-0.448 0.16-0.896 0.192-1.152 0.032-0.096 0.096-0.544 0.16-0.96 0.096-0.448 0.16-0.896 0.192-0.992 0-0.128 0.064-0.352 0.064-0.512 0.032-0.16 0.128-0.544 0.16-0.864l0.096-0.576 0.544-0.288c0.768-0.384 1.344-0.48 1.664-0.288 0.16 0.096 0.32 0.288 0.352 0.416 0.096 0.256 0.032 1.088-0.128 1.984-0.064 0.288-0.096 0.448-0.288 1.664-0.224 1.312-0.256 1.472-0.32 1.792 0 0.16-0.032 0.352-0.064 0.416 0 0.064-0.064 0.256-0.064 0.384-0.032 0.16-0.096 0.416-0.096 0.544-0.224 1.216-0.224 1.76 0 2.24 0.128 0.224 0.288 0.352 0.576 0.512 0.384 0.192 0.512 0.192 1.216 0.192 0.608 0 0.896-0.032 1.28-0.192 0.608-0.224 0.672-0.352 0.544-0.928-0.096-0.512-0.032-1.408 0.192-2.464 0.032-0.224 0.096-0.576 0.128-0.736 0.032-0.192 0.096-0.608 0.16-0.928 0.064-0.288 0.096-0.576 0.096-0.608s0-0.064 0.032-0.096c0.032-0.096 0.064-0.288 0.096-0.672 0.032-0.224 0.096-0.448 0.096-0.512 0.032-0.064 0.064-0.256 0.096-0.416 0-0.16 0.064-0.384 0.064-0.448 0.032-0.128 0.064-0.32 0.096-0.48s0.064-0.352 0.064-0.448c0.224-1.088 0.32-2.272 0.192-2.848-0.096-0.512-0.128-0.576-0.256-0.832-0.416-0.8-1.12-1.184-2.432-1.248-0.608-0.032-2.464 0.448-2.464 0.64 0 0.064 0 0.064-0.032 0.032s-0.384 0.128-0.736 0.32l-0.704 0.352-0.16-0.224c-0.224-0.416-0.608-0.736-1.088-0.896-0.48-0.192-1.504-0.288-1.952-0.16zM21.536 6.048c0 0.032-0.224 0.064-0.48 0.064-0.544 0.064-1.184 0.224-1.184 0.352 0 0.032-0.032 0.032-0.064 0.032-0.032-0.032-0.128 0-0.224 0.096-0.128 0.16-0.128 0.192 0.032 0.832 0.064 0.224 0.032 0.864-0.064 1.376-0.064 0.288-0.128 0.608-0.128 0.768 0 0.128-0.064 0.416-0.128 0.64-0.032 0.224-0.096 0.544-0.128 0.704-0.064 0.544-0.096 0.704-0.16 0.864 0 0.064-0.064 0.32-0.064 0.544-0.032 0.224-0.064 0.416-0.096 0.448 0 0.032-0.064 0.224-0.096 0.48-0.032 0.224-0.064 0.48-0.064 0.544-0.032 0.064-0.064 0.256-0.096 0.416-0.096 0.544-0.128 0.8-0.16 1.024-0.16 0.896-0.096 2.208 0.096 2.624 0.064 0.096 0.128 0.256 0.192 0.384 0.16 0.352 0.704 0.896 1.056 1.024 0.928 0.384 1.664 0.416 2.784 0.128 0.224-0.064 0.416-0.128 0.448-0.128 0.064 0 0.16-0.064 0.256-0.096 0.704-0.288 1.472-0.544 1.504-0.512 0 0 0.064 0.128 0.128 0.256 0.288 0.608 0.896 0.864 1.888 0.864 0.64-0.032 1.536-0.256 1.632-0.416 0.032-0.032 0.096-0.064 0.128-0.096 0.032 0 0.064-0.16 0.064-0.384-0.096-1.248-0.096-1.472-0.032-1.76 0.032-0.16 0.096-0.416 0.128-0.576 0.064-0.576 0.128-0.96 0.16-1.12 0.032-0.096 0.064-0.288 0.096-0.448s0.064-0.384 0.096-0.48c0-0.096 0.032-0.288 0.064-0.448s0.096-0.576 0.16-0.896c0.064-0.288 0.128-0.576 0.096-0.608s0-0.096 0.032-0.16c0.032-0.096 0.096-0.32 0.128-0.512 0.032-0.224 0.064-0.48 0.096-0.608 0.032-0.192 0.32-1.792 0.416-2.272 0-0.16 0.064-0.352 0.064-0.48 0.128-0.448 0.064-1.312-0.096-1.632-0.128-0.288-0.736-0.768-0.832-0.672-0.032 0.032-0.096 0-0.16-0.032-0.096-0.096-1.088-0.128-1.44-0.032-0.448 0.096-1.056 0.32-1.184 0.416-0.096 0.096-0.096 0.192-0.032 0.544 0.096 0.448 0.096 1.312 0 1.824-0.032 0.16-0.128 0.576-0.192 0.928-0.064 0.32-0.096 0.64-0.096 0.64 0 0.032 0 0.096 0 0.128-0.032 0.064-0.096 0.352-0.128 0.576-0.032 0.192-0.128 0.704-0.192 1.024-0.032 0.096-0.064 0.32-0.064 0.48-0.032 0.16-0.064 0.32-0.064 0.384-0.032 0.032-0.064 0.256-0.096 0.48-0.064 0.224-0.096 0.448-0.096 0.48 0 0-0.064 0.32-0.128 0.704-0.096 0.576-0.128 0.64-0.352 0.736-0.128 0.064-0.224 0.128-0.192 0.128 0.032 0.032-0.032 0.064-0.096 0.064-0.096 0-0.32 0.064-0.512 0.16-0.864 0.32-1.856 0.064-1.92-0.512 0-0.096-0.032-0.224-0.032-0.288 0-0.16 0.16-1.472 0.192-1.472 0 0 0.032-0.064 0.032-0.16s0.032-0.288 0.064-0.416c0.032-0.128 0.096-0.384 0.096-0.512 0.128-0.704 0.192-1.152 0.256-1.44 0.032-0.128 0.064-0.384 0.096-0.544 0.032-0.192 0.128-0.8 0.256-1.344 0.288-1.632 0.16-2.432-0.416-2.784-0.16-0.064-0.288-0.16-0.352-0.16-0.512-0.096-0.864-0.16-0.896-0.16z"></path></svg>',
			'info'              => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" width="24" height="24" fill="#d8e1e6"><path d="M256 0C114.512 0 0 114.497 0 256c0 141.488 114.497 256 256 256 141.488 0 256-114.497 256-256C512 114.512 397.503 0 256 0zm0 477.867c-122.337 0-221.867-99.529-221.867-221.867S133.663 34.133 256 34.133 477.867 133.663 477.867 256 378.337 477.867 256 477.867z"/><path d="M249.507 311.95c-12.122 0-21.935 10.101-21.935 22.224 0 11.833 9.524 22.224 21.935 22.224s22.224-10.391 22.224-22.224c0-12.123-10.102-22.224-22.224-22.224z"/><path d="M253.259 143.679c-38.964 0-56.86 23.091-56.86 38.676 0 11.257 9.526 16.452 17.318 16.452 15.586 0 9.237-22.224 38.677-22.224 14.432 0 25.977 6.349 25.977 19.627 0 15.585-16.163 24.534-25.688 32.614-8.371 7.216-19.339 19.05-19.339 43.873 0 15.007 4.041 19.338 15.874 19.338 14.144 0 17.03-6.35 17.03-11.834 0-15.007.288-23.667 16.162-36.078 7.794-6.061 32.327-25.688 32.327-52.819s-24.533-47.625-61.478-47.625z"/></svg>',
			'info-dark'         => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" width="24" height="24"><path d="M256 0C114.512 0 0 114.497 0 256c0 141.488 114.497 256 256 256 141.488 0 256-114.497 256-256C512 114.512 397.503 0 256 0zm0 477.867c-122.337 0-221.867-99.529-221.867-221.867S133.663 34.133 256 34.133 477.867 133.663 477.867 256 378.337 477.867 256 477.867z"/><path d="M249.507 311.95c-12.122 0-21.935 10.101-21.935 22.224 0 11.833 9.524 22.224 21.935 22.224s22.224-10.391 22.224-22.224c0-12.123-10.102-22.224-22.224-22.224z"/><path d="M253.259 143.679c-38.964 0-56.86 23.091-56.86 38.676 0 11.257 9.526 16.452 17.318 16.452 15.586 0 9.237-22.224 38.677-22.224 14.432 0 25.977 6.349 25.977 19.627 0 15.585-16.163 24.534-25.688 32.614-8.371 7.216-19.339 19.05-19.339 43.873 0 15.007 4.041 19.338 15.874 19.338 14.144 0 17.03-6.35 17.03-11.834 0-15.007.288-23.667 16.162-36.078 7.794-6.061 32.327-25.688 32.327-52.819s-24.533-47.625-61.478-47.625z"/></svg>',
			'tiktok'            => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 32 32" width="32" height="32><path d="M16.707 0.027c1.747-0.027 3.48-0.013 5.213-0.027 0.107 2.040 0.84 4.12 2.333 5.56 1.493 1.48 3.6 2.16 5.653 2.387v5.373c-1.92-0.067-3.853-0.467-5.6-1.293-0.76-0.347-1.467-0.787-2.16-1.24-0.013 3.893 0.013 7.787-0.027 11.667-0.107 1.867-0.72 3.72-1.8 5.253-1.747 2.56-4.773 4.227-7.88 4.28-1.907 0.107-3.813-0.413-5.44-1.373-2.693-1.587-4.587-4.493-4.867-7.613-0.027-0.667-0.040-1.333-0.013-1.987 0.24-2.533 1.493-4.96 3.44-6.613 2.213-1.92 5.307-2.84 8.2-2.293 0.027 1.973-0.053 3.947-0.053 5.92-1.32-0.427-2.867-0.307-4.027 0.493-0.84 0.547-1.48 1.387-1.813 2.333-0.28 0.68-0.2 1.427-0.187 2.147 0.32 2.187 2.427 4.027 4.667 3.827 1.493-0.013 2.92-0.88 3.693-2.147 0.253-0.44 0.533-0.893 0.547-1.413 0.133-2.387 0.080-4.76 0.093-7.147 0.013-5.373-0.013-10.733 0.027-16.093z"></path></svg>',
			'line'              => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 32 32" width="32" height="32><path d="M25.82 13.151c0.465 0 0.84 0.38 0.84 0.841 0 0.46-0.375 0.84-0.84 0.84h-2.34v1.5h2.34c0.465 0 0.84 0.377 0.84 0.84 0 0.459-0.375 0.839-0.84 0.839h-3.181c-0.46 0-0.836-0.38-0.836-0.839v-6.361c0-0.46 0.376-0.84 0.84-0.84h3.181c0.461 0 0.836 0.38 0.836 0.84 0 0.465-0.375 0.84-0.84 0.84h-2.34v1.5zM20.68 17.172c0 0.36-0.232 0.68-0.576 0.795-0.085 0.028-0.177 0.041-0.265 0.041-0.281 0-0.521-0.12-0.68-0.333l-3.257-4.423v3.92c0 0.459-0.372 0.839-0.841 0.839-0.461 0-0.835-0.38-0.835-0.839v-6.361c0-0.36 0.231-0.68 0.573-0.793 0.080-0.031 0.181-0.044 0.259-0.044 0.26 0 0.5 0.139 0.66 0.339l3.283 4.44v-3.941c0-0.46 0.376-0.84 0.84-0.84 0.46 0 0.84 0.38 0.84 0.84zM13.025 17.172c0 0.459-0.376 0.839-0.841 0.839-0.46 0-0.836-0.38-0.836-0.839v-6.361c0-0.46 0.376-0.84 0.84-0.84 0.461 0 0.837 0.38 0.837 0.84zM9.737 18.011h-3.181c-0.46 0-0.84-0.38-0.84-0.839v-6.361c0-0.46 0.38-0.84 0.84-0.84 0.464 0 0.84 0.38 0.84 0.84v5.521h2.341c0.464 0 0.839 0.377 0.839 0.84 0 0.459-0.376 0.839-0.839 0.839zM32 13.752c0-7.161-7.18-12.989-16-12.989s-16 5.828-16 12.989c0 6.415 5.693 11.789 13.38 12.811 0.521 0.109 1.231 0.344 1.411 0.787 0.16 0.401 0.105 1.021 0.051 1.44l-0.219 1.36c-0.060 0.401-0.32 1.581 1.399 0.86 1.721-0.719 9.221-5.437 12.581-9.3 2.299-2.519 3.397-5.099 3.397-7.957z"></path></svg>',
			'bluesky'           => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path d="M111.8 62.2C170.2 105.9 233 194.7 256 242.4c23-47.6 85.8-136.4 144.2-180.2c42.1-31.6 110.3-56 110.3 21.8c0 15.5-8.9 130.5-14.1 149.2C478.2 298 412 314.6 353.1 304.5c102.9 17.5 129.1 75.5 72.5 133.5c-107.4 110.2-154.3-27.6-166.3-62.9l0 0c-1.7-4.9-2.6-7.8-3.3-7.8s-1.6 3-3.3 7.8l0 0c-12 35.3-59 173.1-166.3 62.9c-56.5-58-30.4-116 72.5-133.5C100 314.6 33.8 298 15.7 233.1C10.4 214.4 1.5 99.4 1.5 83.9c0-77.8 68.2-53.4 110.3-21.8z"/></svg>'
		);
	}
endif;
