<?php
/**
 * Plugin Name: Social Share Button for LINE
 * Plugin URI:  https://sakamoto.okinawa/social-share-button-for-line
 * Description: Adds a customizable "Share on LINE" button via [line_share_button] shortcode.
 * Version:     1.0.3
 * Author:      sakamoto
 * Author URI:  https://sakamoto.okinawa/
 * License:     GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: social-share-button-for-line
 * Domain Path: /languages
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}


/**
 * Sanitize settings callback.
 *
 * @param array $input Raw input.
 * @return array Sanitized values.
 */
function ssbfl_sanitize_settings( $input ) {
    return [
        'button_text' => sanitize_text_field( $input['button_text'] ?? '' ),
        'bg_color'    => sanitize_hex_color( $input['bg_color'] ?? '' ),
        'text_color'  => sanitize_hex_color( $input['text_color'] ?? '' ),
        'padding_v'   => sanitize_text_field( $input['padding_v'] ?? '' ),
        'padding_h'   => sanitize_text_field( $input['padding_h'] ?? '' ),
        'font_size'   => sanitize_text_field( $input['font_size'] ?? '' ),
        'font_family' => sanitize_text_field( $input['font_family'] ?? '' ),
    ];
}


class SSBFL_Plugin {
    private $option_key = 'social-share-button-for-line_settings';

    public function __construct() {
        add_action( 'init', [ $this, 'load_textdomain' ] );
        add_shortcode( 'line_share_button', [ $this, 'render_button' ] );
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_front_assets' ] );
        add_action( 'admin_menu', [ $this, 'add_settings_page' ] );
        add_action( 'admin_init', [ $this, 'register_settings' ] );
    }

    public function load_textdomain() {
        load_plugin_textdomain( 'social-share-button-for-line', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
    }

    public function enqueue_front_assets() {
        wp_enqueue_style(
            'ssbfl-style',
            plugins_url( 'assets/css/style.css', __FILE__ ),
            [],
            '1.0.3'
        );
    }

    public function render_button() {
        $opts = get_option( $this->option_key, [] );
        $text = $opts['button_text'] ?? __( 'このページをLINEで紹介する', 'social-share-button-for-line' );
        $share_url = 'https://social-plugins.line.me/lineit/share?url=' . rawurlencode( get_permalink() );

        $style = sprintf(
            'background:%1$s;color:%2$s;padding:%3$s %4$s;border-radius:8px;text-decoration:none;display:inline-block;font-size:%5$s;font-family:%6$s;',
            esc_attr( $opts['bg_color'] ?? '#00c300' ),
            esc_attr( $opts['text_color'] ?? '#ffffff' ),
            esc_attr( $opts['padding_v'] ?? '10px' ),
            esc_attr( $opts['padding_h'] ?? '20px' ),
            esc_attr( $opts['font_size'] ?? '16px' ),
            esc_attr( $opts['font_family'] ?? 'inherit' )
        );

        return sprintf(
            '<a href="%1$s" target="_blank" rel="noopener noreferrer" class="ssbfl-button" style="%2$s">%3$s</a>',
            esc_url( $share_url ),
            $style,
            esc_html( $text )
        );
    }

    public function add_settings_page() {
        add_options_page(
            'LINEシェアボタン設定',
            'LINEシェアボタン',
            'manage_options',
            'ssbfl-settings',
            [ $this, 'settings_page' ]
        );
    }

    public function register_settings() {
        register_setting(
            'ssbfl_settings_group',
            $this->option_key,
            'ssbfl_sanitize_settings'
        );
    }

    /* phpcs:disable Generic.PHP.DiscourageGoto.Found */
    public function settings_page() {
        $opts = get_option( $this->option_key, [] );
?>
<div class="wrap">
    <h1><?php esc_html_e( 'LINEシェアボタン設定', 'social-share-button-for-line' ); ?></h1>
    <form method="post" action="options.php">
        <?php settings_fields( 'ssbfl_settings_group' ); ?>
        <table class="form-table">
            <tr>
                <th><label for="ssbfl_button_text"><?php esc_html_e( 'ボタンの文言', 'social-share-button-for-line' ); ?></label></th>
                <td><input type="text" id="ssbfl_button_text" name="social-share-button-for-line_settings[button_text]" value="<?php echo esc_attr( $opts['button_text'] ?? '' ); ?>" class="regular-text"></td>
            </tr>
            <tr>
                <th><label for="ssbfl_bg_color">背景色</label></th>
                <td><input type="text" id="ssbfl_bg_color" name="social-share-button-for-line_settings[bg_color]" value="<?php echo esc_attr( $opts['bg_color'] ?? '#00c300' ); ?>" class="regular-text"></td>
            </tr>
            <tr>
                <th><label for="ssbfl_text_color">文字色</label></th>
                <td><input type="text" id="ssbfl_text_color" name="social-share-button-for-line_settings[text_color]" value="<?php echo esc_attr( $opts['text_color'] ?? '#ffffff' ); ?>" class="regular-text"></td>
            </tr>
            <tr>
                <th><label for="ssbfl_font_size">フォントサイズ</label></th>
                <td><input type="text" id="ssbfl_font_size" name="social-share-button-for-line_settings[font_size]" value="<?php echo esc_attr( $opts['font_size'] ?? '16px' ); ?>" class="regular-text"></td>
            </tr>
            <tr>
                <th><label for="ssbfl_padding_v">上下パディング</label></th>
                <td><input type="text" id="ssbfl_padding_v" name="social-share-button-for-line_settings[padding_v]" value="<?php echo esc_attr( $opts['padding_v'] ?? '10px' ); ?>" class="regular-text"></td>
            </tr>
            <tr>
                <th><label for="ssbfl_padding_h">左右パディング</label></th>
                <td><input type="text" id="ssbfl_padding_h" name="social-share-button-for-line_settings[padding_h]" value="<?php echo esc_attr( $opts['padding_h'] ?? '20px' ); ?>" class="regular-text"></td>
            </tr>
            <tr>
                <th><label for="ssbfl_font_family">フォントファミリ</label></th>
                <td><input type="text" id="ssbfl_font_family" name="social-share-button-for-line_settings[font_family]" value="<?php echo esc_attr( $opts['font_family'] ?? '' ); ?>" class="regular-text"></td>
            </tr>
        </table>
        <?php submit_button(); ?>
    </form>
</div>
<?php
    }
    /* phpcs:enable Generic.PHP.DiscourageGoto.Found */
}

new SSBFL_Plugin();
