<?php
/**
 * Plugin Name: SnapAura Extensions Manager
 * Description: Organize your plugins into custom categories with collapsible groups and filtering
 * Version: 1.0.0
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * Author: SnapAura
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: snapaura-extensions-manager
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Enqueue admin scripts and styles
 */
add_action('admin_enqueue_scripts', 'snapaura_exm_enqueue_admin_assets');
function snapaura_exm_enqueue_admin_assets($hook) {
    if ($hook !== 'plugins.php' || !current_user_can('activate_plugins')) {
        return;
    }

    wp_enqueue_script(
        'snapaura-extensions-manager',
        plugin_dir_url(__FILE__) . 'script.js',
        array(), // No jQuery - pure vanilla JavaScript
        time(), // Cache bust for testing
        true
    );

    wp_enqueue_style(
        'snapaura-extensions-manager',
        plugin_dir_url(__FILE__) . 'style.css',
        array(),
        time() // Cache bust for testing
    );

    // Get current user ID
    $user_id = get_current_user_id();

    // Get categories from user meta (per-user basis)
    $categories = get_user_meta($user_id, 'snapaura_exm_plugin_categories', true);
    $collapsed = get_user_meta($user_id, 'snapaura_exm_collapsed_categories', true);

    wp_localize_script('snapaura-extensions-manager', 'SnapAuraEXMData', array(
        'categories' => (object)(is_array($categories) && !empty($categories) ? $categories : array()),
        'collapsed' => is_array($collapsed) ? $collapsed : array(),
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('snapaura_exm_nonce'),
        'user_id' => $user_id
    ));
}

/**
 * Save plugin categories via AJAX
 */
add_action('wp_ajax_snapaura_exm_save_category', 'snapaura_exm_save_category');
function snapaura_exm_save_category() {
    // Check nonce
    if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'snapaura_exm_nonce')) {
        wp_send_json_error('Invalid nonce');
    }

    // Check permissions
    if (!current_user_can('activate_plugins')) {
        wp_send_json_error('No permission');
    }

    // Get user ID
    $user_id = get_current_user_id();

    // Get data
    $categories_json = isset($_POST['categories']) ? wp_unslash($_POST['categories']) : '{}';
    $categories = json_decode($categories_json, true);

    if (!is_array($categories)) {
        wp_send_json_error('Invalid data');
    }

    // Sanitize
    $sanitized = array();
    foreach ($categories as $slug => $category) {
        $clean_slug = sanitize_key($slug);
        $clean_category = sanitize_text_field($category);
        if ($clean_slug && $clean_category) {
            $sanitized[$clean_slug] = $clean_category;
        }
    }

    // Save to user meta
    $result = update_user_meta($user_id, 'snapaura_exm_plugin_categories', $sanitized);

    wp_send_json_success(array(
        'saved' => count($sanitized),
        'result' => $result,
        'data' => $sanitized
    ));

    // Sanitize
    $sanitized = array();
    foreach ($categories as $slug => $category) {
        $clean_slug = sanitize_key($slug);
        $clean_category = sanitize_text_field($category);
        if ($clean_slug && $clean_category) {
            $sanitized[$clean_slug] = $clean_category;
        }
    }

    // Save to user meta
    $result = update_user_meta($user_id, 'sapo_plugin_categories', $sanitized);

    wp_send_json_success(array(
        'saved' => count($sanitized),
        'result' => $result,
        'data' => $sanitized
    ));
}

/**
 * Save collapsed states via AJAX
 */
add_action('wp_ajax_snapaura_exm_save_collapsed', 'snapaura_exm_save_collapsed');
function snapaura_exm_save_collapsed() {
    if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'snapaura_exm_nonce')) {
        wp_send_json_error('Invalid nonce');
    }

    if (!current_user_can('activate_plugins')) {
        wp_send_json_error('No permission');
    }

    $user_id = get_current_user_id();
    
    $collapsed_json = isset($_POST['collapsed']) ? wp_unslash($_POST['collapsed']) : '[]';
    $collapsed = json_decode($collapsed_json, true);

    if (!is_array($collapsed)) {
        wp_send_json_error('Invalid data');
    }

    $sanitized = array_map('sanitize_text_field', $collapsed);
    
    update_user_meta($user_id, 'snapaura_exm_collapsed_categories', $sanitized);

    wp_send_json_success();
}
