(function() {
    'use strict';

    document.addEventListener('DOMContentLoaded', function() {
        const data = window.SnapAuraEXMData || {};
        
        // Force categoryMap to be an object
        let categoryMap = {};
        if (data.categories && typeof data.categories === 'object' && !Array.isArray(data.categories)) {
            categoryMap = data.categories;
        }
        
        let collapsedCategories = Array.isArray(data.collapsed) ? data.collapsed : [];

        const container = document.getElementById('the-list');
        if (!container) return;

        const pluginRows = Array.from(container.querySelectorAll('tr[data-slug]'));
        if (!pluginRows.length) return;

        // Add assign links FIRST
        addAssignLinks();

        // Add filter
        addFilter();

        // Organize plugins
        organizePlugins();

        // Setup events
        setupEvents();

        function addFilter() {
            if (document.getElementById('sapo-filter')) return;

            const applyButton = document.querySelector('.tablenav.top .apply');
            
            if (applyButton) {
                const select = document.createElement('select');
                select.id = 'sapo-filter';
                select.style.marginLeft = '10px';
                select.style.height = '32px';
                select.style.verticalAlign = 'top';
                select.style.minWidth = '150px';
                
                const option = document.createElement('option');
                option.value = 'All';
                option.textContent = 'All Categories';
                select.appendChild(option);
                
                applyButton.parentNode.insertBefore(select, applyButton.nextSibling);
            }
        }

        function organizePlugins() {
            // Group by category
            const grouped = {};
            
            pluginRows.forEach(row => {
                const slug = row.dataset.slug;
                const category = categoryMap[slug] || 'Uncategorized';
                
                row.dataset.category = category;
                
                if (!grouped[category]) {
                    grouped[category] = [];
                }
                grouped[category].push(row);
            });

            // Sort categories
            const sortedCats = Object.keys(grouped).sort((a, b) => {
                if (a === 'Uncategorized') return 1;
                if (b === 'Uncategorized') return -1;
                return a.localeCompare(b);
            });

            // Add to filter
            const filter = document.getElementById('sapo-filter');
            if (filter) {
                sortedCats.forEach(cat => {
                    const option = document.createElement('option');
                    option.value = cat;
                    option.textContent = cat;
                    filter.appendChild(option);
                });
            }

            // Build HTML string for performance
            const html = [];

            sortedCats.forEach(category => {
                const rows = grouped[category];
                const isCollapsed = collapsedCategories.includes(category);
                const arrow = isCollapsed ? '▶' : '▼';

                const actions = category === 'Uncategorized' ? '' : 
                    `<span class="sapo-actions-inline">
                        <a href="#" class="sapo-rename" data-cat="${esc(category)}" title="Rename category">✏️</a> 
                        <a href="#" class="sapo-delete" data-cat="${esc(category)}" title="Delete category">🗑️</a>
                    </span>`;

                html.push(`
                    <tr class="sapo-header" data-category="${esc(category)}">
                        <td colspan="3">
                            <span class="sapo-arrow" data-cat="${esc(category)}">${arrow}</span>
                            <strong class="sapo-cat-name">${esc(category)}</strong>
                            <span class="sapo-count">(${rows.length})</span>
                            ${actions}
                        </td>
                    </tr>
                `);

                rows.forEach(row => {
                    html.push(row.outerHTML);
                });
            });

            // Single DOM update
            container.innerHTML = html.join('');

            // Hide collapsed rows
            if (collapsedCategories.length > 0) {
                collapsedCategories.forEach(cat => {
                    const header = container.querySelector(`.sapo-header[data-category="${cat}"]`);
                    if (header) {
                        let nextRow = header.nextElementSibling;
                        while (nextRow && !nextRow.classList.contains('sapo-header')) {
                            nextRow.style.display = 'none';
                            nextRow = nextRow.nextElementSibling;
                        }
                    }
                });
            }
        }

        function addAssignLinks() {
            pluginRows.forEach(row => {
                const slug = row.dataset.slug;
                const actions = row.querySelector('.row-actions');
                
                if (actions && !actions.querySelector('.sapo-assign')) {
                    const span = document.createElement('span');
                    span.innerHTML = ' | <a href="#" class="sapo-assign" data-slug="' + slug + '">Assign Category</a>';
                    actions.appendChild(span);
                }
            });
        }

        function setupEvents() {
            // Event delegation for all clicks
            document.addEventListener('click', function(e) {
                // Assign
                if (e.target.classList.contains('sapo-assign')) {
                    e.preventDefault();
                    const slug = e.target.dataset.slug;
                    const current = categoryMap[slug] || '';
                    const newCat = prompt('Category name:', current);
                    
                    if (newCat && newCat.trim()) {
                        categoryMap[slug] = newCat.trim();
                        save();
                    }
                }

                // Rename
                if (e.target.classList.contains('sapo-rename')) {
                    e.preventDefault();
                    const oldCat = e.target.dataset.cat;
                    const newCat = prompt('Rename to:', oldCat);
                    
                    if (newCat && newCat !== oldCat) {
                        Object.keys(categoryMap).forEach(slug => {
                            if (categoryMap[slug] === oldCat) {
                                categoryMap[slug] = newCat;
                            }
                        });
                        save();
                    }
                }

                // Delete
                if (e.target.classList.contains('sapo-delete')) {
                    e.preventDefault();
                    const cat = e.target.dataset.cat;
                    
                    if (confirm('Delete "' + cat + '"? Plugins will move to Uncategorized.')) {
                        Object.keys(categoryMap).forEach(slug => {
                            if (categoryMap[slug] === cat) {
                                categoryMap[slug] = 'Uncategorized';
                            }
                        });
                        save();
                    }
                }

                // Collapse
                if (e.target.classList.contains('sapo-arrow')) {
                    e.preventDefault();
                    const cat = e.target.dataset.cat;
                    const header = e.target.closest('.sapo-header');
                    
                    // Toggle all rows until next header
                    let nextRow = header.nextElementSibling;
                    const rowsToToggle = [];
                    
                    while (nextRow && !nextRow.classList.contains('sapo-header')) {
                        rowsToToggle.push(nextRow);
                        nextRow = nextRow.nextElementSibling;
                    }

                    // Check if currently hidden
                    const isHidden = rowsToToggle[0] && rowsToToggle[0].style.display === 'none';
                    
                    // Toggle visibility
                    rowsToToggle.forEach(row => {
                        row.style.display = isHidden ? '' : 'none';
                    });

                    // Update arrow
                    e.target.textContent = isHidden ? '▼' : '▶';

                    // Update collapsed array
                    if (isHidden) {
                        collapsedCategories = collapsedCategories.filter(c => c !== cat);
                    } else {
                        if (!collapsedCategories.includes(cat)) {
                            collapsedCategories.push(cat);
                        }
                    }

                    console.log('Category:', cat, 'Collapsed:', !isHidden, 'All collapsed:', collapsedCategories);

                    // Save collapsed state
                    fetch(data.ajax_url, {
                        method: 'POST',
                        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                        body: new URLSearchParams({
                            action: 'snapaura_exm_save_collapsed',
                            collapsed: JSON.stringify(collapsedCategories),
                            nonce: data.nonce
                        })
                    }).then(response => response.json())
                      .then(result => {
                          console.log('Collapsed state saved successfully:', collapsedCategories);
                      })
                      .catch(error => {
                          console.error('Failed to save collapsed state:', error);
                      });
                }
            });

            // Filter
            const filter = document.getElementById('sapo-filter');
            if (filter) {
                filter.addEventListener('change', function() {
                    const val = this.value;
                    const allRows = container.querySelectorAll('tr');
                    
                    if (val === 'All') {
                        allRows.forEach(row => row.style.display = '');
                    } else {
                        allRows.forEach(row => {
                            if (row.dataset.category === val || row.classList.contains('sapo-header') && row.dataset.category === val) {
                                row.style.display = '';
                            } else {
                                row.style.display = 'none';
                            }
                        });
                    }
                });
            }
        }

        function save() {
            fetch(data.ajax_url, {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: new URLSearchParams({
                    action: 'snapaura_exm_save_category',
                    categories: JSON.stringify(categoryMap),
                    nonce: data.nonce
                })
            }).then(() => {
                location.reload();
            });
        }

        function esc(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
    });

})();
