<?php
/*
Plugin Name: smoove connector for Elementor forms
Description: Sends Elementor forms submitted contacts into your smoove account.
Version: 4.4.3
Author: smoove
Author URI: https://www.smoove.io
Stable tag: 4.4.3
Requires at least: 5.5.3
Requires PHP: 7.1
License: GPLv2 or later
Text Domain: smoove-elementor
*/

if( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

define( 'SMEF_DEV_MODE', false );
define( 'SMEF_PLUGIN_FILE', __FILE__ );
define( 'SMEF_PLUGIN_REL_PATH', dirname( plugin_basename( __FILE__ ) ) );
define( 'SMEF_PLUGIN_DIR', untrailingslashit( plugin_dir_url( __FILE__ ) ) );
define( 'SMEF_PLUGIN_PATH', untrailingslashit( plugin_dir_path( __FILE__ ) ) );
define( 'SMEF_FILE_VER', SMEF_DEV_MODE ? time() : '4.4.3' );

add_action( 'plugins_loaded', 'smef_plugin_init', 1 );

function smef_plugin_init() {
	load_plugin_textdomain( 'smoove-elementor', false, SMEF_PLUGIN_REL_PATH . '/languages' );

    if ( ! function_exists( 'is_plugin_active' ) ) {
        include_once ABSPATH . 'wp-admin/includes/plugin.php';
    }
	
	if ( !is_plugin_active( 'elementor/elementor.php' ) ) {
		add_action( 'admin_notices', 'smef_elementor_deactivated' );
		return;
	}

	smef_include_folder_files( SMEF_PLUGIN_PATH . '/inc/modules' );
	smef_include_folder_files( SMEF_PLUGIN_PATH . '/inc/helpers' );

	if( is_admin() ){
		smef_include_folder_files( SMEF_PLUGIN_PATH . '/inc/admin' );
		add_action( 'admin_notices', 'smef_show_validation_notices' );
	}
}

function smef_register_form_actions( $form_actions_registrar ) {
	smef_include_folder_files( SMEF_PLUGIN_PATH . '/inc/form-actions' );

	$form_actions_registrar->register( new \SMEF\Form_Actions\Smoove_Action_After_Submit() );
}

add_action( 'elementor_pro/forms/actions/register', 'smef_register_form_actions' );

// Hook into Elementor editor save to validate mailing field mapping
add_action( 'elementor/editor/after_save', 'smef_validate_form_mapping', 10, 2 );

function smef_elementor_deactivated() {
	echo '<div class="notice notice-warning">';
		/* translators: %1$s: current plugin name, required plugin name */
		echo '<p>' . sprintf( esc_html__( '%1$s requires %2$s to be installed and active.', 'smoove-elementor' ), '<strong>' . esc_html__('smoove connector for Elementor forms', 'smoove-elementor') . '</strong>', '<a href="https://elementor.com/" target="_blank">Elementor</a>' ) . '</p>';
	echo '</div>';
}

function smef_include_folder_files( $folder ){
	foreach( glob("{$folder}/*.php") as $filepath ){
		if( $filepath && is_readable( $filepath ) ){
			require_once $filepath;
		}
	}
}

function smef_plugin_activated(){
	smef_include_folder_files( SMEF_PLUGIN_PATH . '/inc/modules' );
	smef_include_folder_files( SMEF_PLUGIN_PATH . '/inc/helpers' );

	\SMEF\Helpers\DB::create_api_log_table();

	\SMEF\Helpers\Smoove::maybe_inform_plugin_usage('install');
}

register_activation_hook( SMEF_PLUGIN_FILE, 'smef_plugin_activated' );

function smef_plugin_deactivated(){
	smef_include_folder_files( SMEF_PLUGIN_PATH . '/inc/modules' );
	smef_include_folder_files( SMEF_PLUGIN_PATH . '/inc/helpers' );

	\SMEF\Helpers\Smoove::maybe_inform_plugin_usage('uninstall');
}

register_deactivation_hook( SMEF_PLUGIN_FILE, 'smef_plugin_deactivated' );

// Hook into the plugin update process
add_action('upgrader_process_complete', 'plugin_update_notice_after_update_multilang', 10, 2);

function plugin_update_notice_after_update_multilang($upgrader_object, $options) {
    // Check if the current process is for a plugin update
    if ($options['action'] == 'update' && $options['type'] == 'plugin') {
        // Get the plugin being updated
        $updated_plugins = $options['plugins'];
        
        // Check if the updated plugin is the one you want to show the message for
        if (in_array(plugin_basename(__FILE__), $updated_plugins)) {
            // Trigger a custom admin notice
            add_action('admin_notices', 'plugin_show_update_notice_multilang');
        }
    }
}

function plugin_show_update_notice_multilang() {
    // Get the site locale
    $locale = get_locale();
    
    // Define the notice message based on the locale
    if ($locale == 'he_IL') {
        // Hebrew message
        $message = 'עדכון חשוב: העדכון הזה דורש תשומת לב. אנא קרא בעיון את הערות העדכון.';
        $notice = '<strong>הודעת עדכון חשובה:</strong> שימו לב! לאחר עדכון הגרסה יש להוסיף לכל הטפסים שדה אישור דיוור וסמס (שדות מסוג acceptance או checkbox), אחרת אנשי הקשר יכנסו ללא הרשאות דיוור וסמס.';
    } else {
        // Default to English message
        $message = 'Important: This update requires some attention. Please read the update notes carefully.';
        $notice = "<strong>Important Update Notice:</strong> Pay attention! If you don't add email and SMS acceptance fields to all forms after the version update, all contacts will be transmitted without mailing permissions.";
    }
    
    ?>
    <div class="notice notice-success is-dismissible">
        <p><?php echo esc_html( $notice ); ?></p>
    </div>
    <script type="text/javascript">
        alert('<?php echo esc_html( $message ); ?>');
    </script>
    <?php
}

/**
 * Validate form mapping when Elementor data is saved.
 *
 * @since 4.2.0
 * @param int   $post_id     The ID of the post.
 * @param array $editor_data The editor data.
 */
function smef_validate_form_mapping( $post_id, $editor_data ) {
    if( !is_array( $editor_data ) || empty( $editor_data ) ) {
        return;
    }

    smef_include_folder_files( SMEF_PLUGIN_PATH . '/inc/helpers' );

    $validation_results = [];
    
    // Recursively check all elements for form widgets with smoove action
    $validation_results = smef_check_elements_for_form_validation( $editor_data );
    
    if( !empty( $validation_results ) ) {
        // Store validation results in post meta for admin display
        update_post_meta( $post_id, '_smef_form_validation_results', $validation_results );
        
        // Add admin notice for next page load
        $locale = get_locale();
        $notice_key = 'smef_validation_notice_' . $post_id;
        
        if ($locale == 'he_IL') {
            $notice_message = 'שימו לב! נמצאו טפסים עם פעולת smoove שלא ממפים שדות אישור דיוור. אנא ודאו שכל הטפסים כוללים מיפוי לשדות canReceiveEmails ו/או canReceiveSmsMessages.';
        } else {
            $notice_message = 'Attention! Forms with smoove action found that do not map mailing approval fields. Please ensure all forms include mapping for canReceiveEmails and/or canReceiveSmsMessages fields.';
        }
        
        set_transient( $notice_key, $notice_message, 300 ); // 5 minutes
    } else {
        // Clear validation results if all forms are valid
        delete_post_meta( $post_id, '_smef_form_validation_results' );
    }
}

/**
 * Recursively check elements for form validation.
 *
 * @since 4.2.0
 * @param array $elements Elementor elements data.
 * @return array Validation results.
 */
function smef_check_elements_for_form_validation( $elements ) {
    $validation_results = [];
    
    foreach( $elements as $element ) {
        if( isset( $element['elType'] ) && $element['elType'] === 'widget' && 
            isset( $element['widgetType'] ) && $element['widgetType'] === 'form' ) {
            
            $form_validation = smef_validate_single_form_element( $element );
            if( $form_validation ) {
                $validation_results[] = $form_validation;
            }
        }
        
        // Check nested elements (sections, columns, etc.)
        if( isset( $element['elements'] ) && is_array( $element['elements'] ) ) {
            $nested_results = smef_check_elements_for_form_validation( $element['elements'] );
            $validation_results = array_merge( $validation_results, $nested_results );
        }
    }
    
    return $validation_results;
}

/**
 * Validate a single form element for smoove mailing field mapping.
 *
 * @since 4.2.0
 * @param array $element Form element data.
 * @return array|false Validation result or false if no issues.
 */
function smef_validate_single_form_element( $element ) {
    if( !isset( $element['settings'] ) || !is_array( $element['settings'] ) ) {
        return false;
    }
    
    $settings = $element['settings'];
    
    // Check if smoove action is enabled
    if( !isset( $settings['submit_actions'] ) || 
        !is_array( $settings['submit_actions'] ) || 
        !in_array( 'smoove', $settings['submit_actions'] ) ) {
        return false;
    }
    
    // Check for mailing field mapping
    $has_email_mapping = false;
    $has_sms_mapping = false;
    $form_fields = isset( $settings['form_fields'] ) ? $settings['form_fields'] : [];
    
    foreach( $settings as $key => $value ) {
        if( strpos( $key, 'smoove_fields_map_input_' ) === 0 && !empty( $value ) ) {
            if( $value === 'canReceiveEmails' ) {
                $has_email_mapping = true;
            }
            if( $value === 'canReceiveSmsMessages' ) {
                $has_sms_mapping = true;
            }
        }
    }
    
    if( !$has_email_mapping && !$has_sms_mapping ) {
        return [
            'element_id' => isset( $element['id'] ) ? $element['id'] : 'unknown',
            'issue' => 'missing_mailing_fields',
            'form_fields_count' => is_array( $form_fields ) ? count( $form_fields ) : 0,
            'has_email_mapping' => false,
            'has_sms_mapping' => false
        ];
    }
    
    return false;
}

/**
 * Show validation notices in admin.
 *
 * @since 4.2.0
 */
function smef_show_validation_notices() {
    // Check for validation notices
    global $post;
    
    if( $post && isset( $post->ID ) ) {
        $notice_key = 'smef_validation_notice_' . $post->ID;
        $notice_message = get_transient( $notice_key );
        
        if( $notice_message ) {
            echo '<div class="notice notice-warning is-dismissible">';
            echo '<p><strong>' . esc_html__('Smoove Validation Alert', 'smoove-elementor') . ':</strong> ';
            echo esc_html( $notice_message ) . '</p>';
            echo '</div>';
            
            // Clear the transient after showing
            delete_transient( $notice_key );
        }
    }
}
