<?php
namespace SMEF\Helpers;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Validation helper class.
 *
 * Handles form validation for mailing field mapping.
 *
 * @since 4.2.0
 */
class Validation {
	
	/**
	 * Check if a form has proper mailing field mapping.
	 *
	 * @since 4.2.0
	 * @param array $form_settings Form settings from Elementor.
	 * @return array Validation result.
	 */
	public static function check_mailing_field_mapping( $form_settings ) {
		$result = [
			'is_valid' => true,
			'has_smoove_action' => false,
			'has_email_mapping' => false,
			'has_sms_mapping' => false,
			'missing_fields' => [],
			'message' => ''
		];
		
		// Check if smoove action is enabled
		if( !isset( $form_settings['submit_actions'] ) || 
			!is_array( $form_settings['submit_actions'] ) || 
			!in_array( 'smoove', $form_settings['submit_actions'] ) ) {
			return $result;
		}
		
		$result['has_smoove_action'] = true;
		
		// Check for mailing field mapping
		foreach( $form_settings as $key => $value ) {
			if( strpos( $key, 'smoove_fields_map_input_' ) === 0 && !empty( $value ) ) {
				if( $value === 'canReceiveEmails' ) {
					$result['has_email_mapping'] = true;
				}
				if( $value === 'canReceiveSmsMessages' ) {
					$result['has_sms_mapping'] = true;
				}
			}
		}
		
		// Validate mapping
		if( !$result['has_email_mapping'] && !$result['has_sms_mapping'] ) {
			$result['is_valid'] = false;
			$result['missing_fields'] = ['canReceiveEmails', 'canReceiveSmsMessages'];
			
			$locale = get_locale();
			if ($locale == 'he_IL') {
				$result['message'] = 'חסר מיפוי לשדות אישור דיוור (canReceiveEmails) ו/או הודעות SMS (canReceiveSmsMessages)';
			} else {
				$result['message'] = 'Missing mapping for mailing approval fields (canReceiveEmails and/or canReceiveSmsMessages)';
			}
			
			// Log validation error
			if (class_exists('\SMEF\Admin\Debug')) {
				\SMEF\Admin\Debug::log('validation_error', 'Form missing both required mailing fields', [
					'missing_fields' => $result['missing_fields']
				]);
			}
		} else if( !$result['has_email_mapping'] ) {
			$result['missing_fields'] = ['canReceiveEmails'];
			
			$locale = get_locale();
			if ($locale == 'he_IL') {
				$result['message'] = 'מומלץ להוסיף מיפוי לשדה אישור דיוור (canReceiveEmails)';
			} else {
				$result['message'] = 'Recommended to add mapping for email approval field (canReceiveEmails)';
			}
		} else if( !$result['has_sms_mapping'] ) {
			$result['missing_fields'] = ['canReceiveSmsMessages'];
			
			$locale = get_locale();
			if ($locale == 'he_IL') {
				$result['message'] = 'מומלץ להוסיף מיפוי לשדה אישור SMS (canReceiveSmsMessages)';
			} else {
				$result['message'] = 'Recommended to add mapping for SMS approval field (canReceiveSmsMessages)';
			}
		}
		
		return $result;
	}
	
	/**
	 * Get validation status for admin display.
	 *
	 * @since 4.2.0
	 * @param int $post_id Post ID.
	 * @return array|false Validation results or false if none.
	 */
	public static function get_post_validation_status( $post_id ) {
		return get_post_meta( $post_id, '_smef_form_validation_results', true );
	}
	
	/**
	 * Clear validation status for a post.
	 *
	 * @since 4.2.0
	 * @param int $post_id Post ID.
	 */
	public static function clear_post_validation_status( $post_id ) {
		delete_post_meta( $post_id, '_smef_form_validation_results' );
	}
	
	/**
	 * Generate admin notice for validation issues.
	 *
	 * @since 4.2.0
	 * @param array $validation_results Validation results.
	 * @return string HTML notice.
	 */
	public static function generate_admin_notice( $validation_results ) {
		if( empty( $validation_results ) ) {
			return '';
		}
		
		$locale = get_locale();
		$issue_count = count( $validation_results );
		
		if ($locale == 'he_IL') {
			$title = 'התראת וולידציה - Smoove';
			$message = sprintf( 
				'נמצאו %d טפסים עם פעולת smoove שחסרים מיפוי שדות אישור דיוור. ', 
				$issue_count 
			);
			$recommendation = 'מומלץ להוסיף מיפוי לשדות canReceiveEmails ו/או canReceiveSmsMessages בכל הטפסים.';
		} else {
			$title = 'Smoove Validation Alert';
			$message = sprintf( 
				'Found %d form(s) with smoove action missing mailing approval field mapping. ', 
				$issue_count 
			);
			$recommendation = 'Please add mapping for canReceiveEmails and/or canReceiveSmsMessages fields in all forms.';
		}
		
		$html = '<div class="notice notice-warning is-dismissible">';
		$html .= '<p><strong>' . esc_html( $title ) . ':</strong> ';
		$html .= esc_html( $message . $recommendation ) . '</p>';
		$html .= '</div>';
		
		return $html;
	}
}
