<?php
namespace SMEF\Admin;

if ( ! defined( 'ABSPATH' ) ) exit;

class Debug {
    
    public $debug_page_slug = 'smef-debug';
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_debug_menu'), 100);
        add_action('admin_enqueue_scripts', array($this, 'enqueue_debug_scripts'));
        add_action('wp_ajax_smef_get_debug_log', array($this, 'ajax_get_debug_log'));
        add_action('wp_ajax_smef_clear_debug_log', array($this, 'ajax_clear_debug_log'));
    }
    
    public function add_debug_menu() {
        add_submenu_page(
            'smef-connector-settings',
            __('Debug Monitor', 'smoove-elementor'),
            __('Debug Monitor', 'smoove-elementor'),
            'manage_options',
            $this->debug_page_slug,
            array($this, 'render_debug_page')
        );
    }
    
    public function enqueue_debug_scripts($hook) {
        if (strpos($hook, $this->debug_page_slug) === false) {
            return;
        }
        
        wp_enqueue_style('smef-debug-css', SMEF_PLUGIN_DIR . '/assets/css/debug.css', [], SMEF_FILE_VER);
        wp_enqueue_script('smef-debug-js', SMEF_PLUGIN_DIR . '/assets/js/debug.js', ['jquery'], SMEF_FILE_VER, true);
        
        wp_localize_script('smef-debug-js', 'smefDebug', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('smef_debug_nonce')
        ]);
    }
    
    public function render_debug_page() {
        ?>
        <div class="wrap smef-debug-page">
            <h1><?php _e('Smoove Debug Monitor', 'smoove-elementor'); ?></h1>
            
            <div class="smef-debug-controls">
                <button id="smef-refresh-log" class="button button-primary">
                    <span class="dashicons dashicons-update"></span>
                    <?php _e('Refresh Log', 'smoove-elementor'); ?>
                </button>
                <button id="smef-clear-log" class="button button-secondary">
                    <span class="dashicons dashicons-trash"></span>
                    <?php _e('Clear Log', 'smoove-elementor'); ?>
                </button>
                <label>
                    <input type="checkbox" id="smef-auto-refresh" checked>
                    <?php _e('Auto-refresh (every 3 seconds)', 'smoove-elementor'); ?>
                </label>
            </div>
            
            <div class="smef-debug-stats">
                <div class="smef-stat-box">
                    <h3><?php _e('API Requests', 'smoove-elementor'); ?></h3>
                    <div class="smef-stat-value" id="smef-api-count">0</div>
                    <div class="smef-stat-label"><?php _e('Last 24 hours', 'smoove-elementor'); ?></div>
                </div>
                <div class="smef-stat-box">
                    <h3><?php _e('Editor Loads', 'smoove-elementor'); ?></h3>
                    <div class="smef-stat-value" id="smef-editor-count">0</div>
                    <div class="smef-stat-label"><?php _e('Last 24 hours', 'smoove-elementor'); ?></div>
                </div>
                <div class="smef-stat-box">
                    <h3><?php _e('Validation Errors', 'smoove-elementor'); ?></h3>
                    <div class="smef-stat-value" id="smef-validation-count">0</div>
                    <div class="smef-stat-label"><?php _e('Last 24 hours', 'smoove-elementor'); ?></div>
                </div>
            </div>
            
            <div class="smef-debug-log">
                <h2><?php _e('Live Debug Log', 'smoove-elementor'); ?></h2>
                <div id="smef-log-container">
                    <p class="smef-loading"><?php _e('Loading...', 'smoove-elementor'); ?></p>
                </div>
            </div>
        </div>
        <?php
    }
    
    public function ajax_get_debug_log() {
        check_ajax_referer('smef_debug_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        $log_entries = get_option('smef_debug_log', []);
        $stats = $this->calculate_stats($log_entries);
        
        wp_send_json_success([
            'log' => array_reverse(array_slice($log_entries, -100)), // Last 100 entries
            'stats' => $stats
        ]);
    }
    
    public function ajax_clear_debug_log() {
        check_ajax_referer('smef_debug_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }
        
        delete_option('smef_debug_log');
        wp_send_json_success('Log cleared');
    }
    
    private function calculate_stats($log_entries) {
        $cutoff_time = time() - (24 * 60 * 60); // 24 hours ago
        
        $stats = [
            'api_requests' => 0,
            'editor_loads' => 0,
            'validation_errors' => 0
        ];
        
        foreach ($log_entries as $entry) {
            if ($entry['timestamp'] < $cutoff_time) {
                continue;
            }
            
            switch ($entry['type']) {
                case 'api_request':
                    $stats['api_requests']++;
                    break;
                case 'editor_load':
                    $stats['editor_loads']++;
                    break;
                case 'validation_error':
                    $stats['validation_errors']++;
                    break;
            }
        }
        
        return $stats;
    }
    
    public static function log($type, $message, $data = []) {
        $log_entries = get_option('smef_debug_log', []);
        
        // Keep only last 500 entries
        if (count($log_entries) > 500) {
            $log_entries = array_slice($log_entries, -500);
        }
        
        $log_entries[] = [
            'timestamp' => time(),
            'type' => $type,
            'message' => $message,
            'data' => $data,
            'user_id' => get_current_user_id(),
            'url' => isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '',
            'is_editor' => defined('ELEMENTOR_VERSION') && \Elementor\Plugin::$instance->editor->is_edit_mode()
        ];
        
        update_option('smef_debug_log', $log_entries, false);
    }
}

new Debug();
