<?php
/**
 * Plugin Name: Smarter Search for Products & Posts
 * Description: AI-powered smarter search for WordPress and WooCommerce. Converts user queries into structured JSON to improve product and content searches.
 * Version: 1.0.0
 * Author: Reinhard Stompe
 * Author URI: https://ai-assist.co.za
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: smarter-search-products-posts
 * Domain Path: /languages
 * Requires PHP: 7.4
 * Requires at least: 5.6
 * Tested up to: 6.9
 * Requires Plugins: woocommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('CRSH_VERSION', '1.0.0');
define('CRSH_URL', trailingslashit(plugin_dir_url(__FILE__)));

/**
 * Check WooCommerce dependency
 *
 * @return bool
 */
function crsh_check_woocommerce() {
    if (!class_exists('WooCommerce')) {
        add_action('admin_notices', function () {
            echo '<div class="error"><p>' . esc_html__('Smarter Search for WooCommerce and WordPress Posts requires WooCommerce to be installed and activated.', 'smarter-search-products-posts') . '</p></div>';
        });
        return false;
    }
    return true;
}

/**
 * Initialize plugin components
 */
function crsh_init() {
    // Always load admin
    require_once plugin_dir_path(__FILE__) . 'admin/class-smarter-search-products-posts-admin.php';
    new Crsh_Admin();

    // Load public only if WooCommerce is active
    if (crsh_check_woocommerce()) {
        require_once plugin_dir_path(__FILE__) . 'public/class-smarter-search-products-posts-public.php';
        new Crsh_Public();
    }
}
add_action('plugins_loaded', 'crsh_init');

// --- Activation / Deactivation --- //

register_activation_hook(__FILE__, function ($network_wide) {
    if ($network_wide && is_multisite()) {
        $sites = get_sites(['fields' => 'ids']);
        foreach ($sites as $site_id) {
            switch_to_blog($site_id);
            crsh_activate_site();
            restore_current_blog();
        }
    } else {
        crsh_activate_site();
    }
});

register_deactivation_hook(__FILE__, function ($network_wide) {
    if ($network_wide && is_multisite()) {
        $sites = get_sites(['fields' => 'ids']);
        foreach ($sites as $site_id) {
            switch_to_blog($site_id);
            crsh_deactivate_site();
            restore_current_blog();
        }
    } else {
        crsh_deactivate_site();
    }
});

add_action('wpmu_new_blog', function ($blog_id) {
    switch_to_blog($blog_id);
    crsh_activate_site();
    restore_current_blog();
});

function crsh_activate_site() {
    // Add default options with unique prefix
    add_option('crsh_api_key', '');
    add_option('crsh_llm_model', 'llama3-8b-8192');
    add_option('crsh_prompt', 'Extract search parameters from query. Return JSON with: name, category, min_price, max_price');
    add_option('crsh_temperature', 0.7);
    add_option('crsh_input_width', '300');
    add_option('crsh_input_height', '40');
    add_option('crsh_products_only', 'yes');

    crsh_add_rewrite_rules();
    flush_rewrite_rules();
}

function crsh_deactivate_site() {
    flush_rewrite_rules();
}

// --- Rewrite Rules --- //

function crsh_add_rewrite_rules() {
    add_rewrite_rule(
        '^search-results/?$',
        'index.php?crsh_search_results=1',
        'top'
    );
}
add_action('init', 'crsh_add_rewrite_rules');

function crsh_query_vars($vars) {
    $vars[] = 'crsh_search_results';
    return $vars;
}
add_filter('query_vars', 'crsh_query_vars');

function crsh_template_include($template) {
    if (get_query_var('crsh_search_results')) {
        $new_template = plugin_dir_path(__FILE__) . 'templates/search-results-template.php';
        if (file_exists($new_template)) {
            return $new_template;
        }
    }
    return $template;
}
add_filter('template_include', 'crsh_template_include');
